/************************************************************************************************************************

	PROGRAM:	HelpSystem

	FILENAME:	utils.c

	PURPOSE:	frequently used functions in all parts of the program

	FUNCTIONS:	in order of appearance

	get_color			- load the desired color
	load_colors			- load all colors used in the program
	create_information_box		- create a standard information box
	create_warning_box		- create a standard warning box
	create_error_box		- create a standard error box

************************************************************************************************************************/

#include "help.h"

/************************************************************************************************************************

 	FUNCTION	: get_color(colorName)
 
	PURPOSE		: load the desired color

	RETURNS		: the color as an unsigned long value (RGB-value) 

************************************************************************************************************************/

unsigned long get_color(colorName)
char *colorName;	/* color name */
{
	XColor	color;	/* color information */
	Status	status; /* status information */

	/* look if colorname is in color database */

	status = XParseColor(display, DefaultColormap(display, screenNr), colorName, &color);
	if (status == 0)
	{
		printf("Error: Colorname %s is not in database. Using color black.", colorName);
		return(BlackPixel(display, screenNr));
	}

	/* load color if color could be allocated */

	status = XAllocColor(display, DefaultColormap(display, screenNr), &color);
	if (status == 0)
	{
		printf("Error: Could not load color %s.", colorName);
		return(BlackPixel(display, screenNr));
	}
	return(color.pixel);
}

/************************************************************************************************************************

 	FUNCTION	: load_colors()
 
	PURPOSE		: load all colors needed in the help system

	RETURNS		: nothing

************************************************************************************************************************/

void load_colors()
{
	if (DefaultVisual(display,screenNr)->class == GrayScale || DefaultVisual(display,screenNr)->class == StaticGray)
	{	/* load colors for a monochrome display */

		textColor = (unsigned long) BlackPixel(display, screenNr);
		textBackgroundColor = (unsigned long) WhitePixel(display, screenNr);
		referenceColor = (unsigned long) BlackPixel(display, screenNr);
		glossaryColor = (unsigned long) BlackPixel(display, screenNr);
	}
	else	/* load colors for a color display */
	{
		textColor = get_color(textColour);
		textBackgroundColor = get_color(textBackgroundColour);
		referenceColor = get_color(referenceColour);
		glossaryColor = get_color(glossaryColour);
	}
} /* end of load_colors */

/************************************************************************************************************************

 	FUNCTION	: create_information_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard information box

	RETURNS		: widget id of information box 

************************************************************************************************************************/

Widget create_information_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	int		n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		informationDialog,
			informationWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	informationDialog = XmCreateDialogShell(parent, "informationDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_INFORMATION); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR("Information: ", defaultCharset);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR("Ok", defaultCharset);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	informationWidget = XmCreateMessageBox(informationDialog, "information", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(informationWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(informationWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(informationWidget, args, n);

	/* set symbol pixmap for an information dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(informationDialog), "default_xm_information", fg, bg)); n++;
	XtSetValues(informationWidget, args, n);

	/* return message box */

	return(informationWidget);

} /* end of create_information_box */

/************************************************************************************************************************

 	FUNCTION	: create_warning_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard warning box

	RETURNS		: widget id of warning box 

************************************************************************************************************************/

Widget create_warning_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	Cardinal	n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		warningDialog,
			warningWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	warningDialog = XmCreateDialogShell(parent, "warningDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_WARNING); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR("Warning: ", defaultCharset);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR("Ok", defaultCharset);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	warningWidget = XmCreateWarningDialog(warningDialog, "warning", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(warningWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(warningWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(warningWidget, args, n);

	/* set symbol pixmap for a warning dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(warningDialog), "default_xm_warning", fg, bg)); n++;
	XtSetValues(warningWidget, args, n);

	/* return message box */

	return(warningWidget);

} /* end of create_warning_box */

/************************************************************************************************************************

 	FUNCTION	: create_error_box(parent, dialogStyle, messageText)
 
	PURPOSE		: create a standard error box

	RETURNS		: widget id of error box 

************************************************************************************************************************/

Widget create_error_box(parent, dialogStyle, messageText)
Widget parent;			/* widget id */
unsigned char dialogStyle;	/* dialog style */
XmString messageText;		/* message text */
{
	Arg		args[10];		/* argument list for manipulating widgets */
	int		n;			/* used as argument counter for manipulating widgets */
	XmString 	xmstrings[10]; 		/* temporary storage for XmStrings */
	Widget		errorDialog,
			errorWidget,
			button;			/* intern widgets used to create the dialog box */
	int		fg, bg;			/* foreground and background color */

	/* create dialog shell */

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	errorDialog = XmCreateDialogShell(parent, "errorDialog", args, n);

	/* create message box */

	n = 0;
	XtSetArg(args[n], XmNdialogType, XmDIALOG_ERROR); n++;
	XtSetArg(args[n], XmNdialogStyle, dialogStyle); n++;
	xmstrings[0] = XmStringCreateLtoR("Error: ", defaultCharset);
	XtSetArg(args[n], XmNdialogTitle, xmstrings[0]); n++;
	XtSetArg(args[n], XmNmessageString, messageText); n++;
	xmstrings[1] = XmStringCreateLtoR("Ok", defaultCharset);
	XtSetArg(args[n], XmNokLabelString, xmstrings[1]); n++;
	errorWidget = XmCreateErrorDialog(errorDialog, "error", args, n);
	XmStringFree(xmstrings[0]);
	XmStringFree(xmstrings[1]);

	/* help button and cancel button are not needed, remove them */

	button = XmMessageBoxGetChild(errorWidget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(errorWidget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);

	/* get foreground and background colors */

	n = 0;
	XtSetArg(args[n], XmNforeground, &fg); n++;
	XtSetArg(args[n], XmNbackground, &bg); n++;
	XtGetValues(errorWidget, args, n);

	/* set symbol pixmap for an error dialog box */

	n = 0;
	XtSetArg(args[n], XmNsymbolPixmap, XmGetPixmap(XtScreen(errorDialog), "default_xm_error", fg, bg)); n++;
	XtSetValues(errorWidget, args, n);

	/* return message box */

	return(errorWidget);

} /* end of create_error_box */
