/************************************************************************************************************************

	PROGRAM:	HELPSERVER

	DATE:		01/10/1993

	LAST UPDATE:	19/11/1993

	VERSION:	1.0

	FILENAME:	help.c

	PURPOSE:	this file contains the main function of the program

	FUNCTIONS:	in order of appearance

	busy			- show a busy cursor if we are busy
	parse_line		- parse given line and return the status information of that line
	    			  delete format characters in given string
	read_helpFile		- read in a helpfile given by the filename
	quicksort_init		- initialization routine for quicksort
	quicksort		- sorts searchable themes
	reference_CB		- jumps to the reference selected by the button
	glossary_CB		- jumps to the glossary entry selected by the button
	find_themeNum_by_name	- finds the associated theme number
	get_nextSegment		- separates a logical text segment from current line
	show_current_theme	- shows the currently selected theme
	show_content	 	- shows the content page of the helpfile
	show_glossary		- shows the glossary section of the helpfile
	dialog_CB		- react to pressed buttons in dialog boxes
	show_theme_list_CB	- fill list of themes in search box
	show_theme_CB		- show the selected theme
	show_previous_theme_CB	- show the previously selected theme
	print_current_theme_CB	- print current theme to printer
	show_content_CB		- show the main index of the help file
	show_glossary_CB	- show the glossary section of the help file
	exit_helpServer		- terminates the help system
	exit_helpServer_CB	- terminates the help system
	askiccm			- parse a command received via X-Property (ICCM)
	main			- this is the main function of the help system

	COPYRIGHT:	(c) 1993 by Stefan Bergdoll, All rights reserved

************************************************************************************************************************/

#include "help.h"

void busy();
LineStatus parse_line();
Boolean read_helpFile();
void quicksort_init();
void quicksort();
void reference_CB();
void glossary_CB();
int find_themeNum_by_name();
LineStatus get_nextSegment();
void show_current_theme();
void show_content();
void show_glossary();
void dialog_CB();
void show_theme_list_CB();
void show_theme_CB();
void show_previous_theme_CB();
void print_current_theme_CB();
void show_content_CB();
void show_glossary_CB();
void exit_helpServer();
void exit_helpServer_CB();
void askiccm();
void main();

extern loadGIF();
extern load_colors();
extern Widget create_error_box();
extern Widget create_warning_box();
extern Widget create_information_box();

/* local variables */

static int	THEME_NUM;	/* theme number selected by previous theme list */

/************************************************************************************************************************

 	FUNCTION	: deIconifyWindow(display, w, screen)
 
	PURPOSE		: deiconify a window

	RETURNS		: nothing

************************************************************************************************************************/

Status deIconifyWindow (display, w, screen)
Display *display;
Window w;
int screen;
{
    XClientMessageEvent ev;
    Window root = RootWindow (display, screen);
    Atom prop;

    prop = XInternAtom (display, "WM_CHANGE_STATE", False);
    if (prop == None) return False;

    ev.type = ClientMessage;
    ev.window = w;
    ev.message_type = prop;
    ev.format = 32;
    ev.data.l[0] = NormalState;
    return (XSendEvent (display, root, False, SubstructureRedirectMask|SubstructureNotifyMask, (XEvent *)&ev));
}

/************************************************************************************************************************

 	FUNCTION	: busy(shell, flag)
 
	PURPOSE		: set busy cursor if we are busy

	RETURNS		: nothing

************************************************************************************************************************/

void busy(shell, flag)
Widget shell;	/* shell widget which calls busy */
Boolean flag;	/* True -> set busy cursor, else unset busy cursor */
{
	if (flag)
		XDefineCursor(display, XtWindow(mainWindow), busyCursor);
	else	XUndefineCursor(display, XtWindow(mainWindow));
	XFlush(display);

} /* end of busy */

/************************************************************************************************************************

 	FUNCTION	: parse_line(str)
 
	PURPOSE		: parse given line and return the status information of that line
			  delete format characters in given string

	RETURNS		: status information of given line

************************************************************************************************************************/

LineStatus parse_line(str)
char *str;
{
	char parseString[maxLineLength];	/* string without format characters */
	LineStatus state;			/* state of the line */
	register unsigned int i, j, k;		/* loop variables */
	char *ptr;				/* pointer to a string */
	char fontList[maxLineLength];		/* user font list */
	char fn[64];				/* current font name */

	/* initialize line status */

	state.index = state.theme = state.glossaryEntry = state.reference = state.newFont = state.newPage =
	state.boldFace = state.italic = state.fontLine = state.fileLine = state.filePath = FALSE;

	/* parse line and eliminate format characters */

	for (i = 0, j = 0; i < strlen(str); i++)
	{
		if (str[i] == '\\')	/* format character ? */
		{
			switch(str[++i])
			{
				/* check format characters */

				case 'G':	state.glossaryEntry = TRUE;
						break;

				case 'K':	state.italic = TRUE;
						break;

				case 'P':	state.newPage = TRUE;
						break;

				case 'Q':	state.reference = TRUE;
						break;

				case 'I':	state.index = TRUE;
						break;

				case 'T':	state.theme = TRUE;
						break;

				case 'Z':	state.newFont = TRUE;
						i++;	/* skip format character */
						while(isdigit((int) str[i]))	/* skip font number */
							i++;
						i--;
						break;

				case 'F':	/* check if bold face, graph file or fontlist */

						if ((str[i+1] == 'I') && (str[i+2] == 'L') && (str[i+3] == 'E') && (str[i+4] == ':'))
						{
							/* FILE: matched */

							state.fileLine = TRUE;
							i += 5;
							while(isspace((int) str[i]))	/* skip white space characters */
								i++;
							i--;
						}
						else if ((str[i+1] == 'i') && (str[i+2] == 'l') && (str[i+3] == 'e') &&
							(str[i+4] == 'p')  && (str[i+5] == 'a') && (str[i+6] == 't') &&
							(str[i+7] == 'h') && (str[i+8] == ':'))
						{
							/* Filepath: matched */

							state.filePath = TRUE;
							i += 9;
							while(isspace((int) str[i]))    /* skip white space characters */
								i++;

							/* set filePath */

							ptr = str;
							ptr += i;
							strcpy(filePath, ptr);
							filePath[strlen(ptr) - 1] = '\0'; /* substitude newline with end of string */
							i--;
						}
						else if ((str[i+1] == 'o') && (str[i+2] == 'n') && (str[i+3] == 't') &&
							(str[i+4] == 'l')  && (str[i+5] == 'i') && (str[i+6] == 's') &&
							(str[i+7] == 't') && (str[i+8] == ':'))
						{
							/* Fontlist: matched */

							state.fontLine = TRUE;
							i += 9;
							while(isspace((int) str[i]))    /* skip white space characters */
								i++;

							/* read in fontlist */

							ptr = str;
							ptr += i;
							strcpy(fontList, ptr);
							fontList[strlen(ptr) - 1] = '\0'; /* substitude newline with end of string */
							ptr = fontList;

							/* count user fonts */

							lastFontNr = 1;
							while (*ptr != '\0')
							{
								if (isspace((int) *ptr))
									lastFontNr++;
								ptr++;
							}

							/* read in user fonts */

							if (lastFontNr >= maxUserFonts)
								lastFontNr = maxUserFonts - 1;
							else	lastFontNr--;
							for (k = 1; k <= lastFontNr; k++)
							{
								sscanf(fontList, "%s", fn);
								userFont[k] = XLoadQueryFont(display, fn);
								ptr = fontList;
								while (*ptr != '\0' && ! isspace((int) *ptr)) ptr++;
								if (*ptr != '\0') ptr++;
								strcpy(fontList, ptr);
							}
							i--;
						}
						else /* bold face */
						{
							state.boldFace = TRUE;
						}
						break;

				default:	fprintf(stderr, "Error: Unknown format character %c in line %s\n", str[i], str);
						break;
			}
		}
		else	parseString[j++] = str[i];
	}
	if (parseString[j - 1] == '\n')
		parseString[j - 1] = '\0';	/* eliminate newline character and terminate parse string */
	else	parseString[j] = '\0';		/* terminate parse string */

	for (i = 0; i < strlen(parseString); i++)
		str[i] = parseString[i];	/* copy parse string to original string */
	str[i] = '\0';

	return(state);

} /* end of parse_line */

/************************************************************************************************************************

 	FUNCTION	: read_helpFile(filename)
 
	PURPOSE		: read in a helpfile given by the filename

	RETURNS		: TRUE, if help file could be read in successfully, FALSE else

************************************************************************************************************************/

Boolean read_helpFile(filename)
char *filename;
{
	FILE	*fp;			/* filepointer into helpfile */
	char	curLine[maxLineLength];	/* current line read in */
	Boolean	themeFound = FALSE;	/* used as flag in searching previously read in theme */
	Boolean	indexFound = FALSE;	/* TRUE, if we are in the main index section, else FALSE */
	Boolean	glossaryFound = FALSE;	/* TRUE, if we are in the glossary section, else FALSE */
	Boolean themeDescription = FALSE;/* TRUE, if we are in the description section of a theme */
	unsigned int curThemeNumber;	/* current theme number */
	register unsigned int ln;	/* index for linenumber, used for faster access */
	register unsigned int i;	/* use as loop index */

	/* open helpfile given by name */

	if (! (fp = fopen(filename, "r")))
	{
		fprintf(stderr, "Error: File %s not found. Exiting ...\n", filename);
		return(FALSE);
	}

	/* initialize help file structure */

	LineNumber = MaxLines = ThemeNumber = GlossaryNumber = GlossarBegin = GlossarEnd = 0;
	Line = (Zeile *) malloc((size_t) (sizeof(Zeile)) * lineBufCnt);
	Theme = (Entry *) malloc((size_t) (sizeof(Entry)) * themeBufCnt);
	Glossar = (Entry *) malloc((size_t) (sizeof(Entry)) * glossBufCnt);

	Theme[0].begin = 0;
	Theme[0].end = 0;
	strcpy(Theme[0].content, "");

	Glossar[0].begin = 0;
	Glossar[0].end = 0;
	strcpy(Glossar[0].content, "");

	/* read in helpfile */

	while (TRUE)
	{
		if (! fgets(curLine, maxLineLength, fp))
		{
			if (feof(fp))
				break;	/* leave reading while-loop since end of file was encountered */
			fprintf(stderr, "Error: While reading file %s an error has been encountered.\nLine %i not readable.\nLine will be skiped.\n", filename, ln);
		}
		else	/* line was read in successfully */
		{
			ln = ++LineNumber;

			/* realloc memory to have enough room to store new lines */

			if (ln % lineBufCnt == 0)
				Line = (Zeile *) realloc(Line, (sizeof(Zeile)) * (ln + lineBufCnt));
			strcpy(Line[ln].content, curLine);

			/* parse current line and eliminates the format characters */

			Line[ln].status = parse_line(curLine);

			/*** now curLine contains no format characters ***/
			/* check if line contains the main index */

			if (Line[ln].status.index)
			{
				/* the main index of the helpfile was found */

				if (indexFound)
				{
					/* the beginning of the main index was previously found, so this must be the end */

					indexFound = FALSE;
					IndexEnd = ln;
				}
				else
				{
					/* now we are entering the index section */

					indexFound = TRUE;
					IndexBegin = ln;
				}
			} /* end of main index section */
			
			/* check if line contains a theme or a glossary entry */

			if (Line[ln].status.theme)
			{
				/* check if we are in the glossary section */

				if (glossaryFound)
				{
					glossaryFound = FALSE; /* leave glossary section now */
					GlossarEnd = Glossar[GlossaryNumber].end = ln - 1;
				}
				else if (themeDescription)
					Theme[curThemeNumber].end = ln - 1; /* previous theme terminates one line before this */

				/* a theme defintion was read, so check if this theme was already be stored */

				if (! (i = find_themeNum_by_name(curLine)))
					themeFound = FALSE;
				else	themeFound = TRUE;

				/* check if theme is Glossar */

				if ((strcmp(curLine, "Glossar") == 0))
				{
					/* now the glossary section is reached */

					glossaryFound = TRUE;
					GlossarBegin = ln + 1;
				}
				else if (! themeFound)
				{
					/* theme isn't present */

					i = ++ThemeNumber;

					/* realloc memory to have enough room to store new themes */

					if (i % themeBufCnt == 0)
						Theme = (Entry *) realloc(Theme, (sizeof(Entry)) * (i + themeBufCnt));

					/* insert theme into theme table */

					strcpy(Theme[i].content, curLine);
					Theme[i].begin = ln;
					Theme[i].end = 0;
				}
				else
				{
					/* theme was earlier inserted, so this occurence contains the real stuff */

					Theme[i].begin = ln + 1;
					themeDescription = TRUE;
					curThemeNumber = i;
				}
			} /* end of theme section */
			else if (glossaryFound && Line[ln].status.glossaryEntry)
			{
				/* a glossary entry was read, so store it away */

				i = ++GlossaryNumber;

				/* realloc memory to have enough room to store new glossary entries */

				if (i % glossBufCnt == 0)
					Glossar = (Entry *) realloc(Glossar, (sizeof(Entry)) * (i + glossBufCnt));

				/* insert theme into theme table */

				strcpy(Glossar[i].content, curLine);
				Glossar[i].begin = ln;
				Glossar[i].end = 0;
				Glossar[i - 1].end = ln - 1;
			} /* end of glossary section */
		}
	} /* end of reading in helpfile */

	MaxLines = Theme[curThemeNumber].end = ln;

	/* set end of glossar section if not yet done */

	if (glossaryFound)
	{
		GlossarEnd = Glossar[GlossaryNumber].end = ln;
	}

	/* sort list of themes */

	ThemeList = (char **) malloc((size_t) (sizeof(char *)) * (ThemeNumber + 1));
	ThemeList[0] = NULL;
	for (i = 1; i <= ThemeNumber; i++)
		ThemeList[i] = Theme[i].content;

	quicksort_init(ThemeList, ThemeNumber);

	/* set search list status */

	newHelpSearchList = TRUE;
	currentTheme.number = 0;	/* no theme selected yet */
	THEME_NUM = 0;

	XtSetSensitive(contentButton, TRUE);
	XtSetSensitive(searchButton, TRUE);
	XtSetSensitive(backButton, TRUE);
	XtSetSensitive(previousButton, TRUE);
	XtSetSensitive(glossarButton, TRUE);
	XtSetSensitive(printButton, TRUE);
	XtSetSensitive(defineButton, TRUE);

	/* the helpfile was read in successfully, so return */

	return(TRUE);

} /* end of read_helpFile */

/************************************************************************************************************************

 	FUNCTION	: quicksort_init(item, count)
 
	PURPOSE		: initialization routine for quicksort

	RETURNS		: nothing

************************************************************************************************************************/

void quicksort_init(item, count)
char *item[];	/* array of char pointers to strings */
int count;	/* number of elements to be sorted */
{
	quicksort(item, 1, count);

} /* end of quicksort_init */

/************************************************************************************************************************

 	FUNCTION	: quicksort(item, left, right)
 
	PURPOSE		: sorts searchable themes

	RETURNS		: nothing

************************************************************************************************************************/

void quicksort(item, left, right)
char *item[];	/* array of char pointers to strings */
int left, right;/* left and right element to be sorted */
{
	register int i, j;	/* help variables for swapping */
	char *x, *y;		/* help strings */

	i = left;
	j = right;
	x = item[(left + right) / 2];

	do
	{
		while (strcmp(item[i], x) < 0 && i < right) i++;
		while (strcmp(item[j], x) > 0 && j > left)  j--;
		if (i <= j)	/* wrong order ? */
		{
			y = item[i];
			item[i] = item[j];
			item[j] = y;
			i++;
			j--;
		}
	} while (i <= j);

	if (left < j)
		quicksort(item, left, j);
	if (i < right)
		quicksort(item, i, right);

} /* end of quicksort */

/************************************************************************************************************************

 	FUNCTION	: reference_CB()
 
	PURPOSE		: jumps to the reference selected by the button

	RETURNS		: nothing

************************************************************************************************************************/

void reference_CB(w, destination, call_data)
Widget          w;		/*  widget id           */
XrmQuark(destination);		/*  hypertext link   */
caddr_t         call_data;	/*  data from widget class  */
{
	char	themeName[maxLineLength];	/* name of theme to jump to */
	Boolean	themeFound;			/* check if theme exists */
	int	i;				/* loop variable */

	busy(w, TRUE);	/* set busy cursor */
	sprintf(themeName, "%s", XrmQuarkToString(destination));

	if (! (i = find_themeNum_by_name(themeName)))
		themeFound = FALSE;
	else	themeFound = TRUE;

	if (! themeFound)
	{
		printf("Hypertext link: %s doesn't exist !\n", themeName);
	}
	else
	{
		currentTheme.number = i;
		show_current_theme();
	}
	busy(w, FALSE);	/* unset busy cursor */

} /* end of reference_CB */

/************************************************************************************************************************

 	FUNCTION	: glossary_CB()
 
	PURPOSE		: jumps to the glossary entry selected by the button

	RETURNS		: nothing

************************************************************************************************************************/

void glossary_CB(w, glossEntry, call_data)
Widget          w;		/*  widget id           */
XrmQuark(glossEntry);		/*  hypertext link   */
caddr_t         call_data;	/*  data from widget class  */
{
	char	glossName[maxLineLength];	/* name of glossary entry to show */
	char	curLine[maxLineLength];		/* current line */
	Boolean	glossFound;			/* check if glossary entry exists */
	int	i, j;				/* loop variables */
	XmString text;				/* compound string of info */

	busy(w, TRUE);	/* set busy cursor */
	sprintf(glossName, "%s", XrmQuarkToString(glossEntry));

	glossFound = FALSE;
	for (i = 1; i <= GlossaryNumber; i++)
	{
		if (strcmp(glossName, Glossar[i].content) == 0)
		{
			glossFound = TRUE;
			break;
		}
	}

	if (! glossFound)
	{
		printf("Glossary entry: %s doesn't exist !\n", glossName);
	}
	else
	{
		text = XmStringCreateLtoR("", defaultCharset);
		for (j = Glossar[i].begin; j <= Glossar[i].end; j++)
		{
			strcpy(curLine, Line[j].content);
			parse_line(curLine);
			strcat(curLine, "\n");
			text = XmStringConcat(text, XmStringCreateLtoR(curLine, defaultCharset));
		}
		XtManageChild(create_information_box(mainWindow, XmDIALOG_MODELESS, text));
		XmStringFree(text);
	}
	busy(w, FALSE);	/* unset busy cursor */

} /* end of glossary_CB */

/************************************************************************************************************************

 	FUNCTION	: find_themeNum_by_name(theme)
 
	PURPOSE		: finds the associated theme number

	RETURNS		: theme number

************************************************************************************************************************/

int find_themeNum_by_name(themeName)
char *themeName;	/* content of theme */
{
	register int	i;		/* loop variable */
	Boolean themeFound = FALSE;	/* status of search */

	for (i = 1; i <= ThemeNumber; i++)
	{
		if (strcmp(Theme[i].content, themeName) == 0)
		{
			themeFound = TRUE;
			break;
		}
	}

	if (themeFound)
		return(i);
	else	return(0);

} /* end of find_themeNum_by_name */

/************************************************************************************************************************

 	FUNCTION	: get_nextSegment(curLine, segment)
 
	PURPOSE		: separates a logical text segment from current line

	RETURNS		: status information of segment
			  current line contains now not the segment

************************************************************************************************************************/

LineStatus get_nextSegment(curLine, segment)
char *curLine;		/* current line */
char *segment;		/* current logical segment of line */
{
	int	i = 0, j = 0; 		/* loop variable */
	Boolean	segEnd = FALSE;		/* end of logical segment */
	LineStatus	state;		/* state of segment */
	char	*ptr;			/* pointer to a string */
	char	returnString[maxLineLength];	/* current line shortened by segment */

	state.index = state.theme = state.glossaryEntry = state.reference = state.newFont = state.newPage =
	state.boldFace = state.italic = state.fontLine = state.fileLine = state.filePath = FALSE;

	while((curLine[i] != '\0') && ! segEnd)
	{
		switch(curLine[i])
		{
			case '\n':	/* newline character */

					break;

			case '\\':	/* format character */

				switch(curLine[++i])
				{
					case 'F':	/* bold font */

						/* check if end of segment is reached */
						if (j > 0)
						{
							i -= 2;
							segEnd = TRUE;
							break;
						}
						else	/* no text before this format char */
						{
							if (strstr(charset, "bold"))
							{
								if (strstr(charset, "italic"))
									strcpy(charset, "italic");
								else	strcpy(charset, "normal");
								state.boldFace = FALSE;
							}
							else
							{
								if (strstr(charset, "italic"))
									strcpy(charset, "bolditalic");
								else	strcpy(charset, "bold");
								state.boldFace = TRUE;
							}
						}
						break;

					case 'K':	/* italic font */

						/* check if end of segment is reached */
						if (j > 0)
						{
							i -= 2;
							segEnd = TRUE;
							break;
						}
						else	/* no text before this format char */
						{
							if (strstr(charset, "italic"))
							{
								if (strstr(charset, "bold"))
									strcpy(charset, "bold");
								else	strcpy(charset, "normal");
								state.italic = FALSE;
							}
							else
							{
								if (strstr(charset, "bold"))
									strcpy(charset, "bolditalic");
								else	strcpy(charset, "italic");
								state.italic = TRUE;
							}
						}
						break;

					case 'Z':	/* user font */

						if (j > 0)
						{
							i -= 2;
							segEnd = TRUE;
							break;
						}

						state.newFont = TRUE;
						i++;    /* skip format character */
						userFontNr = 0;
						if (isdigit((int) curLine[i]))
							userFontNr = ((int) curLine[i] - (int) '0');
						else	state.newFont = FALSE;
						currentFont = userFont[userFontNr];

						/* the last font is the subscribe font, so insert a newline */

						if (userFontNr == lastFontNr)
							segment[j++] = '\n';

						break;

					case 'G':	/* glossary entry */

						if (state.glossaryEntry)
						{
							segEnd = TRUE;
						}
						else if (j == 0)        /* no text before this format char */
						{
							state.glossaryEntry = TRUE;
						}
						else
						{
							i -= 2;
							segEnd = TRUE;
						}
						break;

					case 'Q':	/* reference entry */

						if (state.reference)
						{
							segEnd = TRUE;
						}
						else if (j == 0)        /* no text before this format char */
						{
							state.reference = TRUE;
						}
						else
						{
							i -= 2;
							segEnd = TRUE;
						}
						break;

				}
				break;

			default:

				segment[j++] = curLine[i];
				break;
		}
		i++;
	}

	/* now a segment is extracted, so delete it from current line */

	segment[j] = '\0';	/* terminate segment */
	ptr = curLine;
	ptr += i;
	strcpy(returnString, ptr);
	strcpy(curLine, returnString);

	return(state);

} /* end of get_nextSegment */

/************************************************************************************************************************

 	FUNCTION	: show_current_theme()
 
	PURPOSE		: shows the currently selected theme

	RETURNS		: nothing

************************************************************************************************************************/

void show_current_theme()
{
	Arg	args[10];		/* arguments */
	int	ac;			/* argument counter */
	int	n;			/* current theme number */
	int	i;			/* loop variable */
	char	curLine[maxLineLength];	/* current line of theme */
	char	segment[maxLineLength];	/* segment of current line */
	char	fn[maxLineLength];	/* file name */
	LineStatus	status;		/* status of current segment */
	Widget	lineWindow;		/* window for a line */
	Widget	label, button;		/* hypertext links */
	XmString xs;			/* compound string */
	 
	/* destroy existing label widgets if present */

	if (helpRC != NULL && XtIsRealized(helpRC))
	{
		XtUnrealizeWidget(helpRC);
		XtDestroyWidget(helpRC);
	}

	/* now recreate container row column widget */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
	helpRC = XmCreateRowColumn(helpForm, "helpRC", args, ac);

	/* analyse line by line of theme and create label widgets
	   in the row column widget */
	
	n = currentTheme.number;

	/* show theme title */

	ac = 0;
	XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
	XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
	lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
	XtManageChild(lineWindow);

	xs = XmStringCreateLtoR(Theme[n].content, "theme");
	ac = 0;
	XtSetArg(args[ac], XmNlabelString, xs); ac++;
	XtSetArg(args[ac], XmNforeground, textColor); ac++;
	XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
	label = XmCreateLabelGadget(lineWindow, Theme[n].content, args, ac);
	XtManageChild(label);

	/* set normal font for start of page */

	strcpy(charset, "normal");
	
	for (i = Theme[n].begin; i <= Theme[n].end; i++)
	{
		ac = 0;
		XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
		XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
		lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
		XtManageChild(lineWindow);

		if (Line[i].status.fileLine)
		{
			/* read in graphic file */

			strcpy(curLine, Line[i].content);
			parse_line(curLine);	/* get filename */

			strcpy(fn, filePath);
			strcat(fn, "/");
			strcat(fn, curLine);

			loadGIF(fn); /* read in GIF-file into a XImage structure */

			/* create a bitmap to load in image */

			pixmap = XCreatePixmap(display, RootWindowOfScreen(screen), image->width, image->height, DefaultDepth(display, screenNr));

			/* convert image to a pixmap */

			XPutImage(display, pixmap, DefaultGC(display,screenNr), image, 0, 0, 0, 0, image->width, image->height);

			/* load bitmap in a label widget */

			ac = 0;
			XtSetArg(args[ac], XmNlabelType, XmPIXMAP); ac++;
			XtSetArg(args[ac], XmNlabelPixmap, pixmap); ac++;
			label = XmCreateLabelGadget(lineWindow, Line[i].content, args, ac);
			XtManageChild(label);

			/* free XImage */

			if (image)
			{
				free(image->data);
				image->data = NULL;
				XDestroyImage(image);
			}
		}
		else if (Line[i].status.theme)	/* check if current line is theme */
		{
			ac = 0;
			XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
			XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
			lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
			XtManageChild(lineWindow);

			strcpy(curLine, Line[i].content);
			parse_line(curLine);	/* get theme name */

			xs = XmStringCreateLtoR(curLine, charset);
			ac = 0;
			XtSetArg(args[ac], XmNlabelString, xs); ac++;
			XtSetArg(args[ac], XmNforeground, textColor); ac++;
			XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
			XtSetArg(args[ac], XmNmultiClick, XmMULTICLICK_DISCARD); ac++;
			button = XmCreatePushButton(lineWindow, curLine, args, ac);
			XtManageChild(button);
			XtAddCallback(button, XmNactivateCallback, reference_CB, XrmStringToQuark(curLine));

		}
		else	/* parse normal line */
		{
			strcpy(curLine, Line[i].content);

			strcpy(segment, "");
			do
			{
				status = get_nextSegment(curLine, segment);
				if (status.reference)
				{
					xs = XmStringCreateLtoR(segment, "reference");
					ac = 0;
					XtSetArg(args[ac], XmNlabelString, xs); ac++;
					XtSetArg(args[ac], XmNforeground, referenceColor); ac++;
					XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
					XtSetArg(args[ac], XmNmultiClick, XmMULTICLICK_DISCARD); ac++;
					button = XmCreatePushButton(lineWindow, segment, args, ac);
					XtManageChild(button);
					XtAddCallback(button, XmNactivateCallback, reference_CB, XrmStringToQuark(segment));
					continue;
				}

				if (status.glossaryEntry)
				{
					xs = XmStringCreateLtoR(segment, "glossary");
					ac = 0;
					XtSetArg(args[ac], XmNlabelString, xs); ac++;
					XtSetArg(args[ac], XmNforeground, glossaryColor); ac++;
					XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
					XtSetArg(args[ac], XmNmultiClick, XmMULTICLICK_DISCARD); ac++;
					button = XmCreatePushButton(lineWindow, segment, args, ac);
					XtManageChild(button);
					XtAddCallback(button, XmNactivateCallback, glossary_CB, XrmStringToQuark(segment));
					continue;
				}

				if (status.newFont)
				{
					xs = XmStringCreateLtoR(segment, defaultCharset);
					ac = 0;
					XtSetArg(args[ac], XmNlabelString, xs); ac++;
					XtSetArg(args[ac], XmNfontList, XmFontListCreate(currentFont, defaultCharset)); ac++;
					XtSetArg(args[ac], XmNforeground, textColor); ac++;
					XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
					label = XmCreateLabelGadget(lineWindow, segment, args, ac);
					XtManageChild(label);
					continue;
				}

				xs = XmStringCreateLtoR(segment, charset);
				ac = 0;
				XtSetArg(args[ac], XmNlabelString, xs); ac++;
				XtSetArg(args[ac], XmNforeground, textColor); ac++;
				XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
				label = XmCreateLabelGadget(lineWindow, segment, args, ac);
				XtManageChild(label);
			} while (strcmp(segment, "") != 0);
		}
	}
	XtManageChild(helpRC);

	/* add current theme to previous theme list */

	xs = XmStringCreateLtoR(Theme[n].content, defaultCharset);
	XmListAddItem(previousThemeList, xs, 0);

} /* end of show_current_theme */

/************************************************************************************************************************

 	FUNCTION	: show_content()
 
	PURPOSE		: shows the content page of the helpfile

	RETURNS		: nothing

************************************************************************************************************************/

void show_content()
{
	Arg	args[10];		/* arguments */
	int	ac;			/* argument counter */
	int	i;			/* loop variable */
	char	curLine[maxLineLength];	/* current line of theme */
	char	*contentLabel;		/* name of content button */
	Widget	lineWindow;		/* window for a line */
	Widget	label;			/* hypertext links */
	XmString xs;			/* compound string */
	 
	/* destroy existing label widgets if present */

	if (helpRC != NULL && XtIsRealized(helpRC))
	{
		XtUnrealizeWidget(helpRC);
		XtDestroyWidget(helpRC);
	}

	/* now recreate container row column widget */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
	helpRC = XmCreateRowColumn(helpForm, "helpRC", args, ac);

	/* show content title */

	ac = 0;
	XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
	XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
	lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
	XtManageChild(lineWindow);

	ac = 0;
	XtSetArg(args[ac], XmNlabelString, &xs); ac++;
	XtGetValues(contentButton, args, ac);

	XmStringGetLtoR(xs, defaultCharset, &contentLabel);
	xs = XmStringCreateLtoR(contentLabel, "theme");
	ac = 0;
	XtSetArg(args[ac], XmNlabelString, xs); ac++;
	XtSetArg(args[ac], XmNforeground, textColor); ac++;
	XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
	label = XmCreateLabelGadget(lineWindow, contentLabel, args, ac);
	XtManageChild(label);

	/* show all lines of content */
	
	for (i = IndexBegin; i <= IndexEnd; i++)
	{
		ac = 0;
		XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
		XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
		lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
		XtManageChild(lineWindow);

		strcpy(curLine, Line[i].content);
		parse_line(curLine);	/* eliminate format chars */

		xs = XmStringCreateLtoR(curLine, charset);
		ac = 0;
		XtSetArg(args[ac], XmNlabelString, xs); ac++;
		XtSetArg(args[ac], XmNforeground, textColor); ac++;
		XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
		XtSetArg(args[ac], XmNmultiClick, XmMULTICLICK_DISCARD); ac++;
		button = XmCreatePushButton(lineWindow, curLine, args, ac);
		XtManageChild(button);
		XtAddCallback(button, XmNactivateCallback, reference_CB, XrmStringToQuark(curLine));
	}
	XtManageChild(helpRC);

} /* end of show_content */

/************************************************************************************************************************

 	FUNCTION	: show_glossary()
 
	PURPOSE		: shows the glossary section of the helpfile

	RETURNS		: nothing

************************************************************************************************************************/

void show_glossary()
{
	Arg	args[10];		/* arguments */
	int	ac;			/* argument counter */
	int	i;			/* loop variable */
	char	curLine[maxLineLength];	/* current line of theme */
	char	*glossaryLabel;		/* name of glossary button */
	Widget	lineWindow;		/* window for a line */
	Widget	label;			/* label of title */
	Widget	button;			/* hypertext links */
	XmString xs;			/* compound string */
	 
	/* destroy existing label widgets if present */

	if (helpRC != NULL && XtIsRealized(helpRC))
	{
		XtUnrealizeWidget(helpRC);
		XtDestroyWidget(helpRC);
	}

	/* now recreate container row column widget */

	ac = 0;
	XtSetArg(args[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
	XtSetArg(args[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
	helpRC = XmCreateRowColumn(helpForm, "helpRC", args, ac);

	/* show glossary title */

	ac = 0;
	XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
	XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
	lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
	XtManageChild(lineWindow);

	ac = 0;
	XtSetArg(args[ac], XmNlabelString, &xs); ac++;
	XtGetValues(glossarButton, args, ac);

	XmStringGetLtoR(xs, defaultCharset, &glossaryLabel);
	xs = XmStringCreateLtoR(glossaryLabel, "theme");
	ac = 0;
	XtSetArg(args[ac], XmNlabelString, xs); ac++;
	XtSetArg(args[ac], XmNforeground, textColor); ac++;
	XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
	label = XmCreateLabelGadget(lineWindow, glossaryLabel, args, ac);
	XtManageChild(label);

	/* show all glossary entries */
	
	for (i = 1; i <= GlossaryNumber; i++)
	{
		ac = 0;
		XtSetArg(args[ac], XmNpacking, XmPACK_TIGHT); ac++;
		XtSetArg(args[ac], XmNorientation, XmHORIZONTAL); ac++;
		lineWindow = XmCreateRowColumn(helpRC, "lineWindow", args, ac);
		XtManageChild(lineWindow);

		xs = XmStringCreateLtoR(Glossar[i].content, charset);
		ac = 0;
		XtSetArg(args[ac], XmNlabelString, xs); ac++;
		XtSetArg(args[ac], XmNforeground, glossaryColor); ac++;
		XtSetArg(args[ac], XmNbackground, textBackgroundColor); ac++;
		XtSetArg(args[ac], XmNmultiClick, XmMULTICLICK_DISCARD); ac++;
		button = XmCreatePushButton(lineWindow, Glossar[i].content, args, ac);
		XtManageChild(button);
		XtAddCallback(button, XmNactivateCallback, glossary_CB, XrmStringToQuark(Glossar[i].content));
	}
	XtManageChild(helpRC);

} /* end of show_glossary */

/************************************************************************************************************************

 	FUNCTION	: dialog_CB(w, client_data, call_data)
 
	PURPOSE		: react to pressed buttons in dialog boxes

	RETURNS		: nothing

************************************************************************************************************************/

void dialog_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	Arg	args[20];				/* argument list for manipulating widgets */
	int	n;					/* used as argument counter for manipulating widgets */
	int	i;					/* loop variable */
	Boolean	ok;					/* status of file read operation */
	char	*filename = NULL;			/* help file name read in by file selection box */
	char	*themeName = NULL;			/* theme name to be searched */
	char	*bookMarkName = NULL;			/* name of book mark */
	XmString	text;				/* text for error messages */
	XmFileSelectionBoxCallbackStruct *selection;	/* call data for file selection box */
	XmListCallbackStruct *select;			/* call data for search list selection box */

	busy(w, TRUE);	/* set busy cursor */

	switch((int) client_data)
	{
	case OPEN_FILE:

		filename = NULL;
		selection = (XmFileSelectionBoxCallbackStruct *) call_data;
		XmStringGetLtoR(selection->value, defaultCharset, &filename);
		strcpy(fileName, filename);
		XtUnmanageChild(openShell);

		ok = read_helpFile(filename);
		if (! ok)
		{
			text = XmStringCreate(fileOpenErrorString, defaultCharset);
			XtManageChild(create_error_box(mainWindow, XmDIALOG_MODELESS, text));
		}
		
		show_content();

		break;

	case DEFINE_BOOKMARK:
	
		bookMarkName = XmTextGetString(defineBookMarkText);
		if (strcmp(bookMarkName, "") == 0)
			break;
		text = XmStringCreateLtoR(bookMarkName, defaultCharset);

		if (! XmListItemExists(defineBookMarkList, text))
			XmListAddItem(defineBookMarkList, text, 0);

		XtUnmanageChild(defineBB);
		n = 0;
		XtSetArg(args[n], XmNlabelString, text); n++;
		XtSetArg(args[n], XmNmultiClick, XmMULTICLICK_DISCARD); n++;
		button = XmCreatePushButton(bookMarkSubMenu, bookMarkName, args, n);
		XtAddCallback(button, XmNactivateCallback, show_theme_CB, currentTheme.number);
		XtManageChild(button);
		break;

	case SELECT_THEME:

		select = (XmListCallbackStruct *) call_data;
		THEME_NUM = select->item_position;
		XmTextSetString(searchText, ThemeList[THEME_NUM]);

		for (i = 1; i <= ThemeNumber; i++)
		{
			if (strcmp(Theme[i].content, ThemeList[THEME_NUM]) == 0)
				break;
		}
		THEME_NUM = i;

		break;

	case PREVIOUS_THEME:

		themeName = NULL;
		select = (XmListCallbackStruct *) call_data;
		XmStringGetLtoR(select->item, defaultCharset, &themeName);

		if (strcmp(themeName, "") == 0)
			break;

		XmListDeleteItem(previousThemeList, select->item);

		XtUnmanageChild(previousBB);

		currentTheme.number = find_themeNum_by_name(themeName);

		show_current_theme();

		break;

	case GOTO_THEME:

		if (THEME_NUM == 0)
			break;	/* nothing to do */

		/* add current theme to previous theme list */

		text = XmStringCreateLtoR(Theme[currentTheme.number].content, defaultCharset);
		XmListAddItem(previousThemeList, text, 0);

		XtUnmanageChild(searchBB);

		currentTheme.number = THEME_NUM;
		show_current_theme();

		break;
	}

	busy(w, FALSE);	/* unset busy cursor */

} /* end of dialog_CB */

/************************************************************************************************************************

 	FUNCTION	: show_theme_list_CB(w, client_data, call_data)
 
	PURPOSE		: fill list of themes in search box

	RETURNS		: nothing

************************************************************************************************************************/

void show_theme_list_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	Arg	args[20];	/* argument list for manipulating widgets */
	int	n;		/* used as argument counter for manipulating widgets */
	int	itemListCount;	/* number of items in the search list */
	int	i;		/* loop variable */

	/* check if search list have to be rebuild */

	if (! newHelpSearchList)
	{
		XtManageChild(searchBB);
		return;
	}
	else	newHelpSearchList = FALSE;

	busy(w, TRUE);	/* set busy cursor */

	n = 0;
	XtSetArg(args[n], XmNitemCount, &itemListCount); n++;
	XtGetValues(searchList, args, n);

	if (itemListCount > 0)	/* are there list elements ? */
	{
		/* delete search list to free all associated memory */

		for (i = 1; i <= itemListCount; i++)
		{
			XmListDeletePos(searchList, 0);	/* delete current theme */
		}
	}

	/* fill up search list with current theme list entries */

	for (i = 1; i <= ThemeNumber; i++)
	{
		XmListAddItem(searchList, XmStringCreateLtoR(ThemeList[i], defaultCharset), 0);
	}

	XtManageChild(searchBB);

	busy(w, FALSE);	/* unset busy cursor */

} /* end of show_theme_list_CB */

/************************************************************************************************************************

 	FUNCTION	: show_theme_CB(w, client_data, call_data)
 
	PURPOSE		: show the selected theme

	RETURNS		: nothing

************************************************************************************************************************/

void show_theme_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
int	        client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	busy(w, TRUE);	/* set busy cursor */

	currentTheme.number = client_data;
	show_current_theme();

	busy(w, FALSE);	/* unset busy cursor */

} /* end of show_theme_CB */

/************************************************************************************************************************

 	FUNCTION	: show_previous_theme_CB(w, client_data, call_data)
 
	PURPOSE		: show the previously selected theme

	RETURNS		: nothing

************************************************************************************************************************/

void show_previous_theme_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	busy(w, TRUE);	/* set busy cursor */

	XmListDeletePos(previousThemeList, 0);	/* delete current theme */
	XmListSelectPos(previousThemeList, 0, TRUE);	/* select last theme */

	busy(w, FALSE);	/* unset busy cursor */

} /* end of show_previous_theme_CB */

/************************************************************************************************************************

 	FUNCTION	: print_current_theme_CB(w, client_data, call_data)
 
	PURPOSE		: print current theme to printer

	RETURNS		: nothing

************************************************************************************************************************/

void print_current_theme_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	char *tempFileName;		/* unique temporary file name */
	char printString[128];		/* printer command */
	FILE *fp;			/* file pointer to temporary file */
	XmString xs;			/* compound string */
	char curLine[maxLineLength];	/* current line in theme for printing */
	int i, n;			/* help variables */

	busy(w, TRUE);	/* set busy cursor */

	tempFileName = mktemp("helpServer");
	if (! (fp = fopen(tempFileName, "w")))
	{
		xs = XmStringCreate(tempFileErrorString, defaultCharset);
		XtManageChild(create_error_box(mainWindow, XmDIALOG_MODELESS, xs));
	}

	n = currentTheme.number;
	fprintf(fp, "%s\n", Theme[n].content);

	for (i = Theme[n].begin; i <= Theme[n].end; i++)
	{
		strcpy(curLine, Line[i].content);
		parse_line(curLine);	/* eliminate format chars */
		fprintf(fp, "%s\n", curLine);
	}

	fclose(fp);

	strcpy(printString, printerCommand);
	strcat(printString, " ");
	strcat(printString, tempFileName);

	system(printString);	/* send temp file to printer */
	remove(tempFileName);	/* remove temp file */

	busy(w, FALSE);	/* unset busy cursor */
		
} /* end of print_current_theme_CB */

/************************************************************************************************************************

 	FUNCTION	: show_content_CB(w, client_data, call_data)
 
	PURPOSE		: show the main index of the help file

	RETURNS		: nothing

************************************************************************************************************************/

void show_content_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	busy(w, TRUE);	/* set busy cursor */

	show_content();

	busy(w, FALSE);	/* unset busy cursor */

} /* end of show_content_CB */

/************************************************************************************************************************

 	FUNCTION	: show_glossary_CB(w, client_data, call_data)
 
	PURPOSE		: show the glossary section of the help file

	RETURNS		: nothing

************************************************************************************************************************/

void show_glossary_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	busy(w, TRUE);	/* set busy cursor */

	show_glossary();

	busy(w, FALSE);	/* unset busy cursor */

} /* end of show_glossary_CB */

/************************************************************************************************************************

 	FUNCTION	: exit_helpServer()
 
	PURPOSE		: terminates the help system

	RETURNS		: nothing

************************************************************************************************************************/

void exit_helpServer()
{
	/* remove property from root window which indicates that the help system is runing */

	XDeleteProperty(display, rootWindow, ICCM_SERVERID);
	XDeleteProperty(display, rootWindow, ICCM_SERVICEID);
	XSync(display, FALSE);

	exit(0);

} /* end of exit_helpServer */

/************************************************************************************************************************

 	FUNCTION	: exit_helpServer_CB(w, client_data, call_data)
 
	PURPOSE		: terminates the help system

	RETURNS		: nothing

************************************************************************************************************************/

void exit_helpServer_CB(w, client_data, call_data)
Widget          w;		/*  widget id           */
caddr_t         client_data;	/*  data from application   */
caddr_t         call_data;	/*  data from widget class  */
{
	exit_helpServer();

} /* end of exit_helpServer_CB */

/************************************************************************************************************************

 	FUNCTION	: askiccm(display, window, topic)
 
	PURPOSE		: parse a command received via X-Property (ICCM)

	RETURNS		: nothing

************************************************************************************************************************/

void askiccm(display, window, topic)
Display *display;
Window  window;
Atom	topic;
{
	Boolean ok;
	char *data, *ptr, book[256], theme[256];
	char helpfile[64], title[512], text[512], message[1024];
	Atom type;
	int format;
	unsigned long count, rest;
	XmString xs;

	busy(mainWindow, TRUE);

	if (XGetWindowProperty(display, window, topic, 0L, 32L, TRUE,
		XA_STRING, &type, &format, &count, &rest,
		(unsigned char **) &data) == Success && type == XA_STRING)
	{
		deIconifyWindow(display, XtWindow(applShell), screenNr);
		XRaiseWindow(display, XtWindow(applShell));

		strcpy(book, "");
		strcpy(theme, "");
		ptr = strchr(data, ':');	/* find separator character */
		if (ptr)
		{
			ptr++;
			strcpy(theme, ptr);
			strcpy(book, data);
			book[strlen(data) - strlen(ptr) - 1] = '\0';
		}

		if (strlen(book) > 0 && strlen(theme) > 0)
		{
			sprintf(helpfile, "%s.hlp", book);
			ok = read_helpFile(helpfile);
			if (! ok)
			{
				xs = XmStringCreateLtoR(fileOpenErrorString, defaultCharset);
				XtManageChild(create_error_box(mainWindow, XmDIALOG_MODELESS, xs));
			}

			/* look for the special keywords Index and Glossar */

			if (strcmp(theme, "Index") == 0)
			{
				show_content();
			}
			else if (strcmp(theme, "Glossar") == 0)
			{
				show_glossary();
			}
			else if (! (currentTheme.number = find_themeNum_by_name(theme)))
			{
				sprintf(book, "Warning: Theme %s not found", theme);
				xs = XmStringCreateLtoR(book, defaultCharset);
				XtManageChild(create_warning_box(mainWindow, XmDIALOG_MODELESS, xs));
			}
			else	show_current_theme();
		}
		else	printf("Error in askiccm: %s\n", data);
	}

	busy(mainWindow, FALSE);

} /* end of askiccm */

/************************************************************************************************************************

 	FUNCTION	: main()
 
	PURPOSE		: this is the main function of the help system

	RETURNS		: nothing

************************************************************************************************************************/

void main(argc, argv)
unsigned int argc;	/* number of options in the command line */
char *argv[];		/* pointer to command line options */
{
	Arg args[10];				/* argument list for manipulating widgets */
	char geometry[20];			/* prefered geometry of help system */
	char *str_type[20];			/* used to read resource file */
	char *buffer;				/* buffer used to read in helptext file */
	char *iconname;				/* icon name of application */
	char hostname[256];			/* name of host */
	char topic[256];			/* used for inter client communication */
	char resourceFile[128];			/* name of resource file */
	XrmDatabase applicationDB;		/* application data base */
	XrmValue value;				/* value returned by resource manager */
	int n;					/* used as argument counter for manipulating widgets */
	int x, y;				/* x and y position of main window */
	int bytesToRead;			/* length of helptext file */
	unsigned int width, height;		/* width and height of main window */
	int SEND = 0;				/* flag if help server is present */
	int format;				/* format of property */
	int i = 0;				/* loop variable */
	int dummy;				/* variable not really used */
	unsigned long	num_gets,		/* returned from get window property */
			bytes_left,		/* returned from get window property */
			offset = 0L;		/* offset in 32-bit quantities for data retrieval */
	char dataForClient[512];		/* information send to client */
	char strbuffer[512];			/* temporary buffer */
	Atom type;				/* type of property */
	Window *server_data;			/* server data */
	Window applicationWindow;		/* window for inter client communication */
	XEvent event;				/* event in main application loop */
	XSizeHints sizeHints;			/* user prefered sizes of the top level window */
	XmString xs;				/* compound string */
	FILE *fp;				/* file pointer to helptext file */
	Atom xa_wm_delete_window;		/* atom for resource deleteResponse */

	display = XOpenDisplay(NULL);
	rootWindow = DefaultRootWindow(display);

	/* create Atom/Property to identify me as helpserver */

	gethostname(hostname, 256);
	strcpy(topic, "helpserver_slot");
	ICCM_SERVICEID = XmInternAtom(display, topic, FALSE);
	sprintf(topic, "%s_helpserver_slot", hostname);
	ICCM_SERVERID = XmInternAtom(display, topic, FALSE);

	/* look if help server is currently running, don't start it twice */

	do
	{
		if (XGetWindowProperty(display, rootWindow, ICCM_SERVERID, offset, 8L, FALSE,
			XA_WINDOW, &type, &format, &num_gets, &bytes_left,
			(unsigned char**) &server_data) == Success &&  type == XA_WINDOW)
		{
			for (i = 0; i < num_gets; i++)
				if (XGetGeometry(display, *(server_data + i), &rootWindow,
					&dummy, &dummy, &dummy, &dummy, &dummy, &dummy))
				{
					SEND = TRUE; /* communication to client is possible */
					break;
				}
			offset += 8;
		}
	} while (! SEND && bytes_left != 0);

	if (SEND)
	{
		printf("\nHelp Server is already running\n");
		exit(2);
	}

	/* create application shell */

	applShell = XtInitialize(argv[0], "HelpServer", (XrmOptionDescRec *) NULL, 0, &argc, argv);
	display = XtDisplay(applShell);
	rootWindow = DefaultRootWindow(display);
	screen = DefaultScreenOfDisplay(display);
	screenNr = DefaultScreen(display);
	xcmap = DefaultColormap(display, screenNr);

	/* check if there are not matched command line options */

	if (argc != 1)
	{
		fprintf(stderr, "HelpServer: unknown command line options, exiting ...");
		exit(1);	/* abort program with exit code = 1 */
	}

	/* convert registers for Motif Release 1.2 */

	XmRegisterConverters();

	/* String to unit type doesn't get added !! */

	XtAddConverter ( XmRString, XmRUnitType, XmCvtStringToUnitType, NULL, 0 );

	xa_wm_delete_window = XmInternAtom(display, "WM_DELETE_WINDOW", TRUE);

	/* use Editres-Protocol on X11R5 */

#ifdef X11R5
	XtAddEventHandler(applShell, (EventMask) 0, TRUE, _XEditResCheckMessages, NULL);
#endif

	/* add callback to close program correct in case of WM close */

	XmAddWMProtocolCallback(applShell, xa_wm_delete_window, exit_helpServer_CB, (caddr_t *) NULL);

	/* create busy cursor */

	busyCursor = XCreateFontCursor(display, XC_watch);

	/* initialize global variables */

	defaultCharset = (XmStringCharSet) XmSTRING_DEFAULT_CHARSET;
	newHelpSearchList = TRUE;
	currentTheme.number = 0;

	n = 0;
	XtSetArg(args[n], XmNallowShellResize, TRUE); n++;
	XtSetValues(applShell, args, n);

	/* get help system resources from resource file */

	buffer = getenv("HOME");
	strcpy(resourceFile, buffer);
	strcat(resourceFile, "/HelpServer");

	if (! (applicationDB = XrmGetFileDatabase(resourceFile)))
	{
		/* now try default application directory */

		strcpy(resourceFile, "/usr/lib/X11/app-defaults/HelpServer");
		if (! (applicationDB = XrmGetFileDatabase(resourceFile)))
		{
			fprintf(stderr, "Error: No resource file found ! Exit program ...\n");
			exit(3);
		}
	}

	if (XrmGetResource(applicationDB, "helpServer.printerCommand", "HelpServer.PrinterCommand", str_type, &value))
	{
		strncpy(printerCommand, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource printerCommand not found. Defaulting to lp\n");
		strcpy(printerCommand, "lp");
	}

	if (XrmGetResource(applicationDB, "helpServer.textColor", "HelpServer.TextColor", str_type, &value))
	{
		strncpy(textColour, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource textColor not found. Defaulting to black\n");
		strcpy(textColour, "black");
	}

	if (XrmGetResource(applicationDB, "helpServer.textBackgroundColor", "HelpServer.TextBackgroundColor", str_type, &value))
	{
		strncpy(textBackgroundColour, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource textBackgroundColour not found. Defaulting to white\n");
		strcpy(textBackgroundColour, "white");
	}

	if (XrmGetResource(applicationDB, "helpServer.referenceColor", "HelpServer.ReferenceColor", str_type, &value))
	{
		strncpy(referenceColour, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource referenceColor not found. Defaulting to blue\n");
		strcpy(referenceColour, "blue");
	}

	if (XrmGetResource(applicationDB, "helpServer.glossaryColor", "HelpServer.GlossaryColor", str_type, &value))
	{
		strncpy(glossaryColour, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource glossaryColor not found. Defaulting to red\n");
		strcpy(glossaryColour, "red");
	}

	/* load colors */

	load_colors();

	/* get default geometry of help system from resource file */

	if (XrmGetResource(applicationDB, "helpServer.geometry", "HelpServer.Geometry", str_type, &value))
	{
		strncpy(geometry, value.addr, (int) value.size);
	}
	else
	{
		fprintf(stderr, "Error: Resource geometry not found. Defaulting to =500x700+0+0\n");
		strcpy(geometry, "=500x700+0+0");
	}

	XParseGeometry(geometry, &x, &y, &width, &height);

	/* create other menus in main window */

	create_mainWindow ( applShell );
	create_explainHelpShell ( applShell );
	create_previousShell ( applShell );
	create_searchShell ( applShell );
	create_defineShell ( applShell );
	create_openShell ( applShell );

	/* information text over help system */

	xs = XmStringCreateLtoR("Help System Ver. 1.0\n\n(c) by Stefan Bergdoll, 1993, BASF-AG, All rights reserved\n", defaultCharset);

	/* create information message box */

	aboutHelpServer = create_information_box(mainWindow, XmDIALOG_MODELESS, xs);

	/* read in the help text for help over help */

	if (! (fp = fopen("./helptext", "r")))
	{
		xs = XmStringCreate(helpTextNotFoundString, defaultCharset);
		XtManageChild(create_error_box(mainWindow, XmDIALOG_MODELESS, xs));
	}
	else
	{
		fseek(fp, 0, 2);	/* seek end of file */
		bytesToRead = (int) ftell(fp);
		fseek(fp, 0, 0);	/* place file pointer to beginning of file */
		buffer = (char *) malloc((size_t) (sizeof(char)) * (bytesToRead + 1));
		fread(buffer, sizeof(char), bytesToRead, fp);
		fclose(fp);
		buffer[bytesToRead] = '\0';
		XmTextSetString(explainHelpWindow, buffer);
	}

	/* manage main window */

	XtManageChild(mainWindow);

	/* realize toplevel widgets */

	XtRealizeWidget(applShell);

	/* set size hints for the window manager */

	sizeHints.x = (int) x;
	sizeHints.y = (int) y;
	sizeHints.width = (int) width;
	sizeHints.height = (int) height;
	sizeHints.flags = USPosition | USSize;
	XSetNormalHints(display, XtWindow(applShell), &sizeHints);

	/* gray out unusable options */

	XtSetSensitive(contentButton, FALSE);
	XtSetSensitive(searchButton, FALSE);
	XtSetSensitive(backButton, FALSE);
	XtSetSensitive(previousButton, FALSE);
	XtSetSensitive(glossarButton, FALSE);
	XtSetSensitive(printButton, FALSE);
	XtSetSensitive(defineButton, FALSE);

	/* get window ids for the communication */

	applicationWindow = XtWindow(applShell);

	/* create the atoms for recieving commands */

	ICCM_COMMAND = XmInternAtom(display, "helpserver", FALSE);
	gethostname(hostname, 256);
	sprintf(topic, "%s_helpserver", hostname);
	ICCM_EXPLICIT = XmInternAtom(display, topic, FALSE);

	/* reset root window properties for help server */

	XDeleteProperty(display, rootWindow, ICCM_SERVERID);
	XDeleteProperty(display, rootWindow, ICCM_SERVICEID);

	XChangeProperty(display, rootWindow, ICCM_SERVERID, XA_WINDOW, 32, PropModeReplace, (unsigned char *) &applicationWindow, 1);
	XChangeProperty(display, rootWindow, ICCM_SERVICEID, XA_WINDOW, 32, PropModeReplace, (unsigned char *) &applicationWindow, 1);

	/* say X-Server that the HelpServer wants to be notified for property changes on the root window */

	XSelectInput(display, rootWindow, PropertyChangeMask);

	/* output all to display */

	XFlush(display);

	/* same as XtMainLoop plus EventHandler */

	askiccm(display, rootWindow, ICCM_EXPLICIT);
	askiccm(display, rootWindow, ICCM_COMMAND);

	while (TRUE)
	{
		XtNextEvent(&event);
		switch (event.type)
		{
			case PropertyNotify:
				if (event.xproperty.window == rootWindow)
				{
					if (event.xproperty.atom == ICCM_EXPLICIT)
						askiccm(display, rootWindow, ICCM_EXPLICIT);
					if (event.xproperty.atom == ICCM_COMMAND)
						askiccm(display, rootWindow, ICCM_COMMAND);
					else if (event.xproperty.atom == ICCM_SERVICEID &&
						 event.xproperty.state == PropertyDelete)
							XChangeProperty(display, rootWindow, ICCM_SERVICEID, XA_WINDOW, 32, PropModeAppend, (unsigned char *) &applicationWindow, 1);
					     else if (event.xproperty.atom == ICCM_SERVERID &&
						      event.xproperty.state == PropertyDelete)
							XChangeProperty(display, rootWindow, ICCM_SERVERID, XA_WINDOW, 32, PropModeAppend, (unsigned char *) &applicationWindow, 1);
				}
				else	XtDispatchEvent(&event);
				break;
			default:
				XtDispatchEvent(&event);
				break;
		}
	}

} /* end of main */
