% File: ddln.pl
% Created: 23-Feb-93
% Last Update: 
% Author: Afzal Ballim
% Purpose: dynamic double-linked nodes. This version uses the 
%	   "cell" package which uses asserta/1 and erase/1 to
%	   create and maintain dynamic "memory cells."
%
% Note: failure is used to indicate null pointers.
%
% Needs: cell

%============================================================
% Predicate: ddln_create/4
% Modes: ddln_create(+Contents,+Pred,+Succ,-NodeRef)
% 
% Purpose: NodeRef is the reference to a node whose 
%	   contents is Contents, and predecessor Pred,
%	   successor Succ. (Pred and Succ are references.)
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_create(Contents,Pred,Succ,Ref) :-
	cell_create([Pred,Contents,Succ],Ref).

%============================================================
% Predicate: ddln_destroy/1
% Modes: ddln_destroy(+Ref)
% 
% Purpose: destroy the node referred to by Ref.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_destroy(Ref) :-
	cell_destroy(Ref).

%============================================================
% Predicate: ddln_contents/2
% Modes: ddln_contents(+Ref,?Contents)
% 
% Purpose: The node referred to by Ref contains Contents.
%	   Note: cannot be used to modify the node.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_contents(Ref,Contents) :-
	cell_get(Ref,[_,Contents,_]).

%============================================================
% Predicate: ddln_update_contents/3
% Modes: ddln_update_contents(+Ref,?OldContents,+NewContents)
% 
% Purpose: change the contents of the node referred to by Ref
%	   from OldContents to NewContents.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_update_contents(Ref,OldContents,NewContents) :-
	cell_update(Ref,[P,OldContents,N],
			[P,NewContents,N]).

%============================================================
% Predicate: ddln_next/2
% Modes: ddln_next(+Ref,-NextRef)
% 
% Purpose: The node referred to by Ref has the successor
%	   NextRef.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_next(Ref,NextRef) :-
	cell_get(Ref,[_,_,NextRef]).

%============================================================
% Predicate: ddln_prev/2
% Modes: ddln_prev(+Ref,-PrevRef)
% 
% Purpose: The node referred to by Ref has the successor
%	   PrevRef.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_prev(Ref,PrevRef) :-
	cell_get(Ref,[PrevRef,_,_]).


%============================================================
% Predicate: ddln_update_next/3
% Modes: ddln_update_next(+Ref,?OldNext,+NewNext)
% 
% Purpose: change the successor reference of node referred to 
%	   by Ref from OldNext to NewNext
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_update_next(Ref,OldNext,NewNext) :-
	cell_update(Ref,[P,Contents,OldNext],
			[P,Contents,NewNext]).


%============================================================
% Predicate: ddln_update_prev/3
% Modes: ddln_update_prev(+Ref,?OldPrev,+NewPrev)
% 
% Purpose: change the predecessor reference of node referred to 
%	   by Ref from OldPrev to NewPrev
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_update_prev(Ref,OldPrev,NewPrev) :-
	cell_update(Ref,[OldPrev,Contents,P],
			[NewPrev,Contents,P]).


%============================================================
% Predicate: ddln_link/2
% Modes: ddln_link(+Ref1,+Ref2)
% 
% Purpose: link the nodes referred to by Ref1 and Ref2 such 
%	   that Ref1 is the predecessor of Ref2, and Ref2 is
%	   the successor of Ref1.
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_link(Ref1,Ref2) :-
	ddln_update_next(Ref1,_,Ref2),
	ddln_update_prev(Ref2,_,Ref1).

%============================================================
% Predicate: ddln_referenced/1
% Modes: ddln_referenced(+Ref)
% 
% Purpose: succeeds if Ref is referenced by a node (not itself?)
% 
% Licensed Use: 
% 
% Created: 23-Feb-93
% Last Update: 
%============================================================

ddln_referenced(Ref) :-
	ddln_prev(X,Ref);
	ddln_next(X,Ref).
