/* Test floating point operations */
/* This code assumues both unsigned & floats are 32 bits */
/* And that both unsigned long's and doubles are 64 bits */

/* Given a series of hex numbers on the command line,
   this program prints the floating point structure of each bit pattern
   Assumes that arguments are doubles, unless -f flag given
   Shows bit pattern of result of converting float->double or double->float,
   using both "official" conversion routines as well as your's.
*/

/* Interesting -f arguments include:
   00000000     Zero
   00000001     Smallest Positive Denormalized
   007FFFFF     Largest Denormalized
   00800000     Smallest Positive Normalized
   7F7FFFFF     Largest Normalized
   7F800000     Infinity
   7FFFFFFF     NaN
*/

#include <stdio.h>
#include <stdlib.h>
#include "fp.h"

/* The following union constructs makes it possible to extract & manipulate
   the bit representation of a float
 */

typedef union {
  float f;
  unsigned u;
} bit_float_t;

float bit2float(unsigned u) {
  bit_float_t arg;
  arg.u = u;
  return arg.f;
}

unsigned float2bit(float f) {
  bit_float_t arg;
  arg.f = f;
  return arg.u;
}

void float_parts(float x) {
  unsigned sign, exp;
  unsigned significand;
  int sig_bits = 23;
  int exp_bits = 8;
  int bias = IBITS(exp_bits-2);
  bit_float_t arg;
  arg.f = x;

  /* Get sign bit */
  sign = (arg.u >> (sig_bits + exp_bits)) & IBITS(0);
  /* Get exp bits */
  exp = (arg.u >> sig_bits) & IBITS(exp_bits-1);
  /* Get significand bits */
  significand = arg.u & IBITS(sig_bits-1);

  /* Split into cases */
  if (exp == 0) {
    /* Denormalized */
    printf(
     "Denormalized: s=%c [0x%x], E = %d [0x%x], m=%e [0x%x], value = %e\n",
     sign ? '-' : '+', sign,
     1 - bias, exp,
     (double) significand / (double) (1<<sig_bits), significand,
     x); 
  } else if (exp < IBITS(exp_bits-1)) {
    /* Normalized */
    printf(
     "Normalized: s=%c [0x%x], E = %d [0x%x], m=%e [0x%x], value = %e\n",
     sign ? '-' : '+', sign,
     exp - bias, exp,
     1.0 + (double) significand / (double) (1<<sig_bits), significand,
     x); 
  } else if (significand == 0) {
    /* Infinity */
    printf("Infinity: s=%c [0x%x], value = %e\n",
     sign ? '-' : '+', sign,	   
     x); 
  } else {
    /* NaN */
    printf("NaN: s=%d [0x%x], significand=%d [0x%x], value = %e\n",
     sign ? -1 : 1, sign,
     significand, significand,
     x); 
  }
}

int float_equal(float x1, float x2) {
  return float2bit(x1) == float2bit(x2);
}


/* The following union constructs makes it possible to extract & manipulate
   the bit representation of a double
 */

typedef union {
  double d;
  unsigned long u;
} bit_double_t;

double bit2double(unsigned long u) {
  bit_double_t arg;
  arg.u = u;
  return arg.d;
}

unsigned long double2bit(double d) {
  bit_double_t arg;
  arg.d = d;
  return arg.u;
}

void double_parts(double x) {
  unsigned sign, exp;
  unsigned long significand;
  int sig_bits = 52;
  int exp_bits = 11;
  int bias = IBITS(exp_bits-2);
  bit_double_t arg;
  arg.d = x;

  /* Get sign bit */
  sign = (arg.u >> (sig_bits + exp_bits)) & IBITS(0);
  /* Get exp bits */
  exp = (arg.u >> sig_bits) & IBITS(exp_bits-1);
  /* Get significand bits */
  significand = arg.u & LBITS(sig_bits-1);

  /* Split into cases */
  if (exp == 0) {
    /* Denormalized */
    printf(
     "Denormalized: s=%c [0x%x], E = %d [0x%x], m=%e [0x%lx], value = %e\n",
     sign ? '-' : '+', sign,
     1 - bias, exp,
     (double) significand / (double) (1L<<sig_bits), significand,
     x); 
  } else if (exp < IBITS(exp_bits-1)) {
    /* Normalized */
    printf(
     "Normalized: s=%c [0x%x], E = %d [0x%x], m=%e [0x%lx], value = %e\n",
     sign ? '-' : '+', sign,
     exp - bias, exp,
     1.0 + (double) significand / (double) (1L<<sig_bits), significand,
     x); 
  } else if (significand == 0) {
    /* Infinity */
    printf("Infinity: s=%c [0x%x], value = %e\n",
     sign ? '-' : '+', sign,	   
     x); 
  } else {
    /* NaN */
    printf("NaN: s=%d [0x%x], significand=%ld [0x%lx], value = %e\n",
     sign ? -1 : 1, sign,
     significand, significand,
     x); 
  }
}

int double_equal(double x1, double x2) {
  return double2bit(x1) == double2bit(x2);
}


main(int argc, char *argv[]) {
  int i;
  unsigned u;
  unsigned long ul;
  float f, fs;
  double d, ds;
  int double_mode = 1;
  if (argc < 2) {
    fprintf(stderr, "%s [-f|-d] x1 x2 ...", argv[0]);
    exit(1);
  }
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-' && argv[i][1] == 'f') {
      double_mode = 0;
    } else if (argv[i][0] == '-' && argv[i][1] == 'd') {
      double_mode = 1;
    } else if (double_mode) {
      sscanf(argv[i], "%lx", &ul);
      printf("Double bits: %lx\n", ul);
      d = bit2double(ul);
      double_parts(d);
      f = (float) d;
      u = float2bit(f);
      printf("-->Float bits (Alpha implementation): 0x%x\n", u);
      float_parts(f);
      fs = double_to_float(d);
      u = float2bit(fs);
      if (float_equal(fs,f)) {
	printf("-->Float bits (Your implementation): 0x%x matches!\n", u);
      } else {
	printf("-->Float bits (Your implementation): 0x%x Doesn't match\n", u);
	float_parts(fs);
      }
    } else {
      sscanf(argv[i], "%x", &u);
      printf("Float bits: %x\n", u);
      f =  bit2float(u);
      float_parts(f);
      d = (double) f;
      ul = double2bit(d);
      printf("-->Double bits: %lx\n", ul);
      double_parts(d);
      ds = float_to_double(f);
      ul = double2bit(ds);
      if (double_equal(ds, d)) {
	printf("-->Double bits (Your implementation): 0x%lx matches!\n", ul);
      } else {
	printf("-->Double bits (Your implementation): 0x%lx Doesn't match\n", ul);
	double_parts(ds);
      }
    }
  }
}
