/* Representation of Sparse matrices in compressed sparse row format */

#ifdef SNGL
typedef float ftype_t;
#else
#ifdef INT
typedef int ftype_t;
#else
typedef double ftype_t;
#endif
#endif
#define FSIZE sizeof(ftype_t)

#define FZERO ((ftype_t) 0)

/* Matrix types:
   UNSCALED: CSR, numeric column indices,
   SCALED:   CSR, byte offset column indices,
   DENSE:    Dense.  Can just use val array
   NEW_REP: New version created by you!
*/

typedef enum { UNSCALED, SCALED, DENSE, NEW_REP } matrix_t;
#define TCOUNT 4

#ifdef CSR_SOURCE
char *type_name[TCOUNT] = {"Unscaled", "Scaled  ", "Dense   ", "New_Rep "};
#else
extern char *type_name[TCOUNT];
#endif

/* Basic representation of matrix.
   In the scaled case, the entries in cindex are scaled by FSIZE
   to avoid having to scale during the array indexing */

typedef struct {
  matrix_t type;   /* Are the entries of cindex scaled by FSIZE? */
  int nrow;        /* Number of rows */
  int nentries;    /* Number of nonzero entries */ 
  ftype_t *val;    /* Set of non-zero values, in row-major order */
  int *cindex;     /* Column index for each element of val */
  int *rstart;    /* Index of starting element for row */
} csr_rec, *csr_ptr;

/* From csr.c */
void csr_mult(csr_ptr M, ftype_t *x, ftype_t *z);
void csr_dummy_mult(csr_ptr M, ftype_t *x, ftype_t *z);

/* From mdata.c */

csr_ptr gen_dense_matrix(int nrow, int nentries);
csr_ptr gen_matrix(int nrow, int nentries, matrix_t type);
csr_ptr retype_matrix(csr_ptr m, matrix_t new_type);
void free_matrix(csr_ptr m);
void mprint(csr_ptr m);
void vprint(ftype_t *v, int nrow);
void show_prod(csr_ptr m, ftype_t *v, ftype_t *result);
/* Get matrix element */
ftype_t get_ele(csr_ptr m, int r, int c); 
/* Generate random vector */
ftype_t *rvec(int nrow);

int test_mult(csr_ptr m, ftype_t *v);
