/* Code for Matrix * Vector computation */
/* Matrix is in Compressed Sparse row format */

#include <stdio.h>
#define CSR_SOURCE
#include "csr.h"

/* Compute z = M*x.  Most straightforward version.  Unscaled cindex */
static void csr_mult_smpl(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, ci;
  for (r = 0; r < M->nrow; r++) {
    z[r] = 0.0;
    for (ci = M->rstart[r]; ci < M->rstart[r+1]; ci++)
      z[r] += M->val[ci] * x[M->cindex[ci]];
  }
}

/* Compute z = M*x.  Dense matrix M */
static void dense_mult_smpl(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, c;
  int vi = 0;
  for (r = 0; r < M->nrow; r++) {
    z[r] = 0.0;
    for (c = 0; c < M->nrow; c++)
      z[r] += M->val[vi++] * x[c];
  }
}

/* Compute z = M*x.  Most straightforward version.  Scaled cindex */
static void scsr_mult_smpl(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, ci;
  for (r = 0; r < M->nrow; r++) {
    z[r] = 0.0;
    for (ci = M->rstart[r]; ci < M->rstart[r+1]; ci++)
      z[r] += M->val[ci] *
	*(ftype_t *) ((char *) x + M->cindex[ci]);
  }
}

/* Compute z = M*x.  Optimize with pointer code */
static void csr_mult_opt(csr_ptr M, ftype_t *x, ftype_t *z)
{
  ftype_t *val = M->val;
  int *cindex_start = M->cindex;
  int *cindex = M->cindex;
  int *rnstart = M->rstart+1;
  ftype_t *z_end = z+M->nrow;

  while (z < z_end) {
    ftype_t temp = 0.0;
    int *cindex_end = cindex_start + *(rnstart++);
    while (cindex < cindex_end)
      temp += *(val++) * x[*cindex++];
    *z++ = temp;
  }
}

/* Compute z = M*x.  Optimize with pointer code.
   Assume entries in cindex are scaled by FSIZE */
static void scsr_mult_opt(csr_ptr M, ftype_t *x, ftype_t *z)
{
  ftype_t *val = M->val;
  int *cindex_start = M->cindex;
  int *cindex = M->cindex;
  int *rnstart = M->rstart+1;
  ftype_t *z_end = z+M->nrow;

  while (z < z_end) {
    ftype_t temp = 0.0;
    int *cindex_end = cindex_start + *(rnstart++);
    while (cindex < cindex_end)
      temp += *(val++) *
	*(ftype_t *) ((char *) x + *cindex++);
    *z++ = temp;
  }
}

/* Compute z = M*x.  Dense matrix M.  Optimize with pointer code */
static void dense_mult_opt(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, c;
  ftype_t *v = M->val;
  ftype_t temp, *xnext;
  for (r = 0; r < M->nrow; r++) {
    temp = 0.0;
    xnext = x;
    for (c = 0; c < M->nrow; c++)
      temp += *v++ * *xnext++;
    *z++ = temp;
  }
}


/* Collect together optimized multiplies  as single procedure */
void csr_mult(csr_ptr M, ftype_t *x, ftype_t *z)
{
  switch (M->type) {
  case UNSCALED :
  case NEW_REP:
    csr_mult_opt(M, x, z);
    break;
  case SCALED:
    scsr_mult_opt(M, x, z);
    break;
  case DENSE:
    dense_mult_opt(M, x, z);
    break;
  default:
    fprintf(stderr, "Unknown matrix type %d\n", M->type);
  }
}

/* Dummy versions */

static void csr_mult_dummy(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, ci;
}

static void dense_mult_dummy(csr_ptr M, ftype_t *x, ftype_t *z)
{
  int r, c;
  int vi = 0;
}

void csr_dummy_mult(csr_ptr M, ftype_t *x, ftype_t *z)
{
  switch (M->type) {
  case UNSCALED:
  case SCALED:
  case NEW_REP:
    csr_mult_dummy(M, x, z);
    break;
  case DENSE:
    dense_mult_dummy(M, x, z);
    break;
  default:
    fprintf(stderr, "Unknown matrix type %d\n", M->type);
  }
}
