/******************************************************************************
 *	stages.h
 *
 *	defines the layout of the pipe-registers
 *	declares the functions that correspond to the pipeline stages
 ******************************************************************************/

#ifndef STAGES_H
#define STAGES_H

/******************************************************************************
 *	#includes
 ******************************************************************************/

#include "pipe.h"



/******************************************************************************
 *	typedefs
 ******************************************************************************/

/* Exception status possibilities */
typedef enum { X_AOK, X_BRK, X_ADR, X_ILI, X_CNT } x_status_t;

/* EX stage mux settings */
typedef enum { MUX_NONE, MUX_EX_A, MUX_EX_B, MUX_MEM_A, MUX_MEM_B, 
   MUX_WB_A, MUX_WB_R } mux_source_t;

/**** Define states held in pipe register stages ****/

/* Program counter */
typedef struct {
  unsigned pc;
} pc_ele, *pc_ptr;

/* IF/ID pipe register */
typedef struct {
  unsigned instr;
  unsigned incr_pc;
  x_status_t exception;  /* Exception status */
  /* The following are included only as debugging aids */
  unsigned char stage_pc_valid;
  unsigned stage_pc;
} if_id_ele, *if_id_ptr;

/* ID/EX Pipe Register */
typedef struct {
  unsigned incr_pc;
  unsigned instr;
  unsigned adata;
  unsigned bdata;
  x_status_t exception;  /* Exception status */
  /* The following are included only as debugging aids */
  unsigned char stage_pc_valid;
  unsigned stage_pc;
} id_ex_ele, *id_ex_ptr;

/* EX/MEM Pipe Register */
typedef struct {
  unsigned branch_target;
  unsigned char branch_flag;
  unsigned char mem_op; /* Memory operation */
  /* Need to keep track of source register for store  */
  unsigned char store_source_reg;
  unsigned alu_result;
  unsigned bdata;
  unsigned char write_back_dest;
  x_status_t exception;  /* Exception status */
  /* The following are included only as debugging aids */
  unsigned char stage_pc_valid;
  unsigned stage_pc;
} ex_mem_ele, *ex_mem_ptr;

/* MEM/WB Pipe Register */
typedef struct {
  unsigned char mem_op;
  unsigned char write_back_dest;
  unsigned alu_result;
  unsigned read_data;
  x_status_t exception;  /* Exception status */
  /* The following are included only as debugging aids */
  unsigned char stage_pc_valid;
  unsigned stage_pc;
} mem_wb_ele, *mem_wb_ptr;


/******************************************************************************
 *	global variable declarations
 ******************************************************************************/

/* mux settings are global to allow Tk to display source */
extern mux_source_t amux, bmux, smux;



/******************************************************************************
 *	function declarations
 ******************************************************************************/

/*
 * Routines that simulator uses to implement stage computations
 */

void do_if_stage(pc_ptr pc_next, if_id_ptr if_out);
void do_id_stage(id_ex_ptr id_out);
void do_ex_stage(ex_mem_ptr ex_out);
void do_mem_stage(mem_wb_ptr mem_out);

/*
 * This check is run after all the stages have been updated, so can use
 * both stage outputs and inputs.
 */
void do_stall_check(pc_ptr     pc_next,
                    if_id_ptr  if_out,
                    id_ex_ptr  id_out,
                    ex_mem_ptr ex_out,
                    mem_wb_ptr mem_out);

/******************************************************************************
 * Utility routine
 ******************************************************************************/

/* Return description for the different status possibilities */
char *x_name(x_status_t exception);

/******************************************************************************/

#endif /* STAGES_H */

