/******************************************************************************
 *	stages.c
 *
 *	Code for parts of the MIPS pipeline simulator that you can modify
 ******************************************************************************/

/******************************************************************************
 *	#includes
 ******************************************************************************/

/* sim.h includes both stages.h and mips.h */


#include "sim.h"



/******************************************************************************
 *	global variable definitions
 ******************************************************************************/

/* mux settings are global to allow Tk to display source */
mux_source_t amux, bmux, smux;



/******************************************************************************
 * Utility routine
 ******************************************************************************/

static char *exception_name[X_CNT] = {"AOK", "BRK", "ADR", "INS"};

/* Return description for the different status possibilities */
char *x_name(x_status_t exception)
{
  if (exception >= 0 && exception < X_CNT)
    return exception_name[exception];
  return "X??";
}

/******************************************************************************
 *	do_if_stage
 ******************************************************************************/

void do_if_stage(pc_ptr pc_next, if_id_ptr if_out)
{
  unsigned pc = pc_current->pc;
  unsigned pc4 = pc+4;
  
  /* IR <-- IMemory[PC] */
  if_out->exception = code_fetch_instruction(pc, &if_out->instr);
  if_out->incr_pc = pc4;

  if (mem_in->branch_flag)
  	/* PC <-- branch target */
  	pc_next->pc = mem_in->branch_target;
  else
    /* PC <-- PC + 4 */
    pc_next->pc  = pc4;
  
  /* debugging */
  if_out->stage_pc_valid = if_out->exception == X_AOK;
  if_out->stage_pc = pc;
}

/******************************************************************************
 *	do_id_stage
 ******************************************************************************/

void do_id_stage(id_ex_ptr id_out)
{
  /* Get parameters for write back */
  unsigned char write_reg = wb_in->write_back_dest;
  unsigned write_data = wb_in->mem_op == M_READ ?
    wb_in->read_data : wb_in->alu_result;

  /* Get parameters from instruction */ 
  unsigned instr = id_in->instr;
  unsigned char rs = GET_RS(instr);
  unsigned char rt = GET_RT(instr);

  /* Update stage outputs */
  id_out->incr_pc = id_in->incr_pc;
  id_out->instr = instr;
  
  /* Do register operations */
  reg_update(write_reg, write_data, rs, rt, &id_out->adata, &id_out->bdata);
  id_out->exception = id_in->exception;  /* Nothing new */

  /* debugging */
  id_out->stage_pc_valid = id_in->stage_pc_valid;
  id_out->stage_pc = id_in->stage_pc;
}

/******************************************************************************
 *	mux_out
 ******************************************************************************/

static unsigned mux_out(mux_source_t selector) {
  /* immediate is always computed from ex_in ! */
  unsigned instr = ex_in->instr;

  switch (selector)
  	{
  	case MUX_EX_A:  return ex_in->adata;
  	case MUX_EX_B:  return ex_in->bdata;
  	case MUX_WB_R:  return wb_in->read_data;
  	case MUX_WB_A:  return wb_in->alu_result;
  	case MUX_MEM_B: return mem_in->bdata;
  	case MUX_MEM_A: return mem_in->alu_result;
  	case MUX_NONE:  return 0;
  	}
  sim_log("ERROR: invalid MUX source in mux_out\n");
  return 0;
}

/******************************************************************************
 *	do_ex_stage
 ******************************************************************************/

void do_ex_stage(ex_mem_ptr ex_out)
{
  /* Examine instruction */
  unsigned instr = ex_in->instr;
  unsigned instr_op = GET_OPCODE(instr);
  unsigned instr_fun = GET_FUN(instr);
  unsigned char rt = GET_RT(instr);
  unsigned char rs = GET_RS(instr);
  unsigned char rd = GET_RD(instr);
  short i16 = GET_I16(instr);

  /* declare some local variables */
  unsigned char zero;         /* Is the ALU output == 0? */
  x_status_t alu_exception;   /* Status flag set by ALU  */
  unsigned adata, bdata;      /* A and B operand data    */
  x_status_t exception = ex_in->exception;

  /* Set up MUX's for extracting A and B operands */
  /* Default is to get from A and B fields of ex_in */ 
  amux = MUX_EX_A;
  bmux = MUX_EX_B;
  /*-------------------------- START FORWARDING ----------------------------*/
  /*
   * Handle RAW hazards by forwarding to EX, if necessary
   */
  if (sim_mode == S_FORWARD) {

    /*
     * Your (commented) solution goes here.
     * It's only effect should be to set the amux & bmux controls 
     */

  }
  /*------------------------------ END FORWARDING ----------------------------*/
  /* Get A and B operands from MUX's */
  adata = mux_out(amux);
  bdata = mux_out(bmux);
  /* Supply bdata to next stage */
  ex_out->bdata = bdata;

  {
    /*
     * ALU Operation.  Set ALU operands.  Compute ALU function
     * Set write_back_dest, alu output, and exception condition
     * You'll need to add code to this to implement the jumps.
     * Your code should just set the values of the following 4 variables:
     */
    /* Set default values */
    unsigned alu_fun = instr_fun;
    unsigned alu_adata = adata;
    unsigned alu_bdata = bdata;
    unsigned char write_back_dest = 0;

    switch (instr_op) {

    case I_OP:

      write_back_dest = rd;
      break;
    case I_ADDIU:
      alu_fun = O_ADDU;
      alu_bdata = (int) i16;
      write_back_dest = rt;
      break;
    case I_LW:
      alu_fun = O_ADDU;
      alu_bdata = (int) i16;
      write_back_dest = rt;
      break;
    case I_SW:
      alu_fun = O_ADDU;
      alu_bdata = (int) i16;      
      break;
    case I_BEQ:
      alu_fun = O_SUBU;
      break;
    case I_BNE:
      alu_fun = O_SUBU;
      break;
    case I_LUI:
      alu_fun = O_ADDU;
      alu_bdata = (int) i16 << 16;
      write_back_dest = rt;
      break;
    default:
      sim_log("ERROR: Illegal instruction encountered in EX %d\n", instr_op);
      exception = X_ILI;
      alu_fun = O_ADDU;
      break;
    }
    
    /* Do the ALU operation */
    alu_exception = alu_compute(alu_adata, alu_bdata, alu_fun,
				&ex_out->alu_result, &zero);
    if (alu_exception != X_AOK || exception != X_AOK)
      write_back_dest = 0; /* Turn into nop */
    ex_out->write_back_dest = write_back_dest;
  }
  
  {
    /*
     * Compute branch flag and branch target.
     * You'll need to extend this to implement jumps.  Easiest way
     * is to view as branch that is always taken.
     * Your code should just set the values of the following 3 variables:
     */
    int add_adata = 0;
    int add_bdata = 0;
    unsigned char branch_flag = 0;

    switch (instr_op) {

    case I_BEQ:
      branch_flag = zero;
      add_adata = ex_in->incr_pc;
      add_bdata = ((int) i16) << 2; /* sign extend */
      break;
    case I_BNE:
      branch_flag = !zero;
      add_adata = ex_in->incr_pc;
      add_bdata = ((int) i16) << 2; /* sign extend */
      break;
    default:
      break;
    }
    /* Compute branch target using second (add only) ALU */
    add_compute(add_adata, add_bdata, &ex_out->branch_target);
    ex_out->branch_flag = branch_flag;
  }
  /* Determine memory operation */
  if (instr_op==I_LW)
    ex_out->mem_op = M_READ;
  else if (instr_op==I_SW)
    ex_out->mem_op = M_WRITE;
  else
    ex_out->mem_op = M_NONE;

  /* fill pipe register */
  ex_out->exception = exception == X_AOK ? alu_exception : exception;
  ex_out->store_source_reg = rt;

  sim_log("EX Results: ALU:0x%x, BR:%u, MEM:%c, WB:0x%x, B:0x%x\n",
	  ex_out->alu_result, ex_out->branch_flag,
	  (ex_out->mem_op == M_WRITE) ? 'W' :
	  ((ex_out->mem_op == M_READ) ? 'R' : '-'),
	  ex_out->write_back_dest,
	  ex_out->bdata);

  /* debugging */
  ex_out->stage_pc_valid = ex_in->stage_pc_valid;
  ex_out->stage_pc = ex_in->stage_pc;
}

/******************************************************************************
 *	do_mem_stage
 ******************************************************************************/

void do_mem_stage(mem_wb_ptr mem_out)
{
  unsigned address = mem_in->alu_result;
  unsigned data = 0;
  unsigned sdata;
  x_status_t exception = mem_in->exception;
  smux = (mem_in->mem_op == M_WRITE) ? MUX_MEM_B : MUX_NONE; 

  /*------------------------- START FORWARDING -------------------------------*/
  /*
   * MEM_MEM Forwarding
   */
  if (sim_mode == S_FORWARD) {

    /*
     * your (commented) solution goes here,
     * It should only set the value of smux
     */

  }
  /*--------------------------- END FORWARDING -------------------------------*/

  sdata = mux_out(smux);

  if (mem_in->mem_op == M_READ)
    exception = data_read(address, &data);
  else if (mem_in->mem_op == M_WRITE)
    exception = data_write(address, sdata);
  mem_out->mem_op = mem_in->mem_op;
  mem_out->write_back_dest = mem_in->write_back_dest;
  mem_out->alu_result = mem_in->alu_result;
  mem_out->read_data = data;
  mem_out->exception = exception;

  /* debugging */
  mem_out->stage_pc_valid = mem_in->stage_pc_valid;
  mem_out->stage_pc = mem_in->stage_pc;
}

/******************************************************************************
 *	do_stall_check
 * 
 * Implementation of stalls
 *
 * This check is run after all the stages have been updated, so can use
 * both stage outputs and inputs. The outputs are passed as arguments, while
 * the inputs are already available as globals, as in the rest of the file.
 *
 * Stages can be canceled or stalled using sim_cancel_stage and sim_stall_stage
 * respectively.
 ******************************************************************************/

void do_stall_check(pc_ptr     pc_next,
                    if_id_ptr  if_out,
                    id_ex_ptr  id_out,
                    ex_mem_ptr ex_out,
                    mem_wb_ptr mem_out)
{
  /* Flags denoting which stage to initiate stall */
  unsigned char stall_if = 0;  /* Should I stall the instruction in IF? */
  unsigned char stall_id = 0;  /* Should I stall the instruction in ID? */
  unsigned char stall_ex = 0;  /* Should I stall the instruction in EX? */

  /*------------------------- START STALL MODE -------------------------------*/
  /*
   * Task 2: Handling hazards in stall mode 
   */
  if (sim_mode == S_STALL) {

    /*
     * your (commented) solution goes here
     * It should only alter the values of the stall flags
     */


  }
  /*-------------------------- END STALL MODE --------------------------------*/


  /*------------------------ START FORWARD MODE ------------------------------*/
  /*
   * Task 3: Handling hazards in forward mode
   */
  if (sim_mode == S_FORWARD) {

    /*
     * your (commented) solution goes here.
     * It can alter the stall flags or call sim_cancel_stage()
     * to cancel misfetched instructions
     */


  }
  /*--------------------------- END FORWARD MODE -----------------------------*/


  /***** End of stall detection logic *****/

  if (stall_ex) {
    /* Handle stall at EX */
    sim_stall_stage(IF_STAGE);
    sim_stall_stage(ID_STAGE);
    sim_stall_stage(EX_STAGE);
    /* Insert bubble into MEM */
    sim_cancel_stage(MEM_STAGE);
  }
  else if (stall_id) {
    /* Handle stall at ID */
    sim_stall_stage(IF_STAGE);
    sim_stall_stage(ID_STAGE);
    /* Insert bubble into EX */ 
    sim_cancel_stage(EX_STAGE);
  }
  else if (stall_if) {
    /* Handle stall at IF */
    sim_stall_stage(IF_STAGE);
    /* Insert bubble into ID */
    sim_cancel_stage(ID_STAGE);
  }
}

/******************************************************************************/

