/******************************************************************************
 * sim.h
 *
 * header file for sim.c
 * function declarations are given in order of their associated pipeline stage
 ******************************************************************************/

#ifndef SIM_H
#define SIM_H

/******************************************************************************
 *	#includes
 ******************************************************************************/

#include <stdio.h>
#include "mips.h"
#include "stages.h"



/******************************************************************************
 *	typedefs
 ******************************************************************************/

/**** Encodings with specific interpretations ****/

/* Memory operation possibilities */
typedef enum { M_NONE, M_READ, M_WRITE } m_status_t;

/* Simulator operating modes */
typedef enum { S_WEDGED, S_STALL, S_FORWARD } sim_mode_t;
 
/* Pipeline stage identifiers for stage operation control */
typedef enum { IF_STAGE, ID_STAGE, EX_STAGE, MEM_STAGE, WB_STAGE } stage_id_t;
 


/******************************************************************************
 *	global variable declarations
 ******************************************************************************/

/* stages.c access to the pipeline state */
extern pc_ptr pc_current;
extern if_id_ptr  id_in;
extern id_ex_ptr  ex_in;
extern ex_mem_ptr mem_in;
extern mem_wb_ptr wb_in;

/* Simulator operating mode */
extern sim_mode_t sim_mode;
extern FILE *dumpfile;



/******************************************************************************
 * pipeline control
 * These functions can be used to handle hazards
 ******************************************************************************/

/* cancel stage (has effect at next update) */
void sim_cancel_stage(stage_id_t stage);

/* stall stage (has effect at next update) */
void sim_stall_stage(stage_id_t stage);



/******************************************************************************
 * instruction memory operations
 ******************************************************************************/

/* Fetch instruction.  Signal exception. */
x_status_t code_fetch_instruction(unsigned pc, unsigned *resultp);

/* Load instructions from specified file.
   Return number of instructions read */
int code_load(FILE *code_file);

/* Dump code to specified file */
void code_dump(FILE *code_file);



/******************************************************************************
 * register-file operations
 ******************************************************************************/

/* Perform one write and two reads to register file */
void reg_update(unsigned char Waddr, unsigned Wdata,
		       unsigned char Aaddr, unsigned char Baddr,
		       unsigned *Adata, unsigned *Bdata);

/* Print register status */
void reg_display(FILE *outfile);



/******************************************************************************
 * ALU operations
 ******************************************************************************/

/* Perform ALU computation.  Signal Exception */
x_status_t alu_compute(unsigned dataA, unsigned dataB, unsigned char op,
		       unsigned *resultp, unsigned char *is_zero_ptr);

/* Perform address computation using dedicated adder */
void add_compute(unsigned dataA, unsigned dataB, unsigned *resultp);



/******************************************************************************
 * data memory operations
 ******************************************************************************/

/* Read from memory.  Signal exception */
x_status_t data_read(unsigned addr, unsigned *resultp);

/* Write to memory.  Signal exception */
x_status_t data_write(unsigned addr, unsigned data);

/* Load data from specified file.
   Return number of words read */
int data_load(FILE *data_file);

/* Dump data to specified file */
void data_dump(FILE *data_file);



/******************************************************************************
 * simulator
 ******************************************************************************/

/* Init simulator */
void sim_init();

/* Reset simulator state, including register, instruction, and data memories */
void sim_reset();

/* Run simulator until either hit exception, or reach maximum cycles */
x_status_t sim_run_pipe(int max_cycles);

/* If dumpfile set nonNULL, lots of status info printed out */
void sim_set_dumpfile(FILE *file);

/*
 * sim_log dumps a formatted string to the dumpfile, if it exists
 * accepts variable argument list
 */
void sim_log( const char *format, ... );



/******************************************************************************
 * User Interface functions (defined in tkSupport.c)
 ******************************************************************************/

void signal_register_update(unsigned char Waddr, unsigned Wdata);
void signal_sources();
void signal_register_clear();
void report_line(int addr, unsigned data, char *text);
void report_pc(unsigned fpc, unsigned char fpcv,
	       unsigned dpc, unsigned char dpcv,
	       unsigned epc, unsigned char epcv,
	       unsigned mpc, unsigned char mpcv,
	       unsigned wpc, unsigned char wpcv);
void report_state(char *id, int current, char *txt);



/******************************************************************************/

#endif /* SIM_H */
