/******************************************************************************
 *	sim.c
 *
 *	Simulator for pipelined implementation of MIPS
 ******************************************************************************/



/******************************************************************************
 *	#includes
 ******************************************************************************/

#include <stdarg.h> /* for sim_log */
#include "sim.h"
#include "pipe.h"

/******************************************************************************
 *	Workaround for some funkiness regarding linking X11 under Linux 
 ******************************************************************************/

#ifdef linux
int dlerror () {
  return 0;
}

int dlopen() {
  return 0;
}

int dlsym () {
  return 0;
}
#endif

/******************************************************************************
 *	global variables
 ******************************************************************************/

/* Pipe register inputs have limited access */
static pc_ptr pc_next;
static if_id_ptr if_out;
static id_ex_ptr id_out;
static ex_mem_ptr ex_out;
static mem_wb_ptr mem_out;

/* Pipe register outputs have global access (read only) */
pc_ptr pc_current;
if_id_ptr id_in;
id_ex_ptr ex_in;
ex_mem_ptr mem_in;
mem_wb_ptr wb_in;

/* The pipeline state */
static pipe_ptr pc_state, if_id_state, id_ex_state, ex_mem_state, mem_wb_state;

/* Simulator operating mode */
sim_mode_t sim_mode = S_WEDGED;

/* If dumpfile set nonNULL, lots of status info printed out */
FILE *dumpfile = NULL;

/* used for formatting instructions */
static char status_msg[128];



/******************************************************************************
 *	static variables
 ******************************************************************************/

/* Run time flags */
static int initialized = 0;

/* How many instruction words are valid? */
static int code_words = 0;

/* Register file contents */
static unsigned reg_state[32] = {0};

#define MEM_COUNT 128
static unsigned instr_mem[MEM_COUNT] = {0};
/* Set of strings describing each of the instructions */
static char *instr_code[MEM_COUNT] = {0};
static unsigned data_mem[MEM_COUNT] = {0}; 



/******************************************************************************
 * pipeline control
 * These functions can be used to handle hazards
 ******************************************************************************/

/* cancel stage (has effect at next update) */
void sim_cancel_stage(stage_id_t stage) {
  switch (stage)
    {
    case IF_STAGE : pc_state->op     = P_BUBBLE; break;
    case ID_STAGE : if_id_state->op  = P_BUBBLE; break;
    case EX_STAGE : id_ex_state->op  = P_BUBBLE; break;
    case MEM_STAGE: ex_mem_state->op = P_BUBBLE; break;
    case WB_STAGE : mem_wb_state->op = P_BUBBLE; break;
    }
}

/* stall stage (has effect at next update) */
void sim_stall_stage(stage_id_t stage) {
  switch (stage)
    {
    case IF_STAGE : pc_state->op     = P_STALL; break;
    case ID_STAGE : if_id_state->op  = P_STALL; break;
    case EX_STAGE : id_ex_state->op  = P_STALL; break;
    case MEM_STAGE: ex_mem_state->op = P_STALL; break;
    case WB_STAGE : mem_wb_state->op = P_STALL; break;
    }
}



/******************************************************************************
 * instruction memory operations
 ******************************************************************************/

/* Clear Instruction Memory */
static void code_clear()
{
  clear_state(instr_mem, instr_code, MEM_COUNT);
  code_words = 0;
}

/* Fetch instruction.  Signal exception. */
x_status_t code_fetch_instruction(unsigned pc, unsigned *resultp)
{
  char *code;
  unsigned wd = pc >> 2;
  if (pc & 0x3) {
    sim_log("ERROR: Unaligned instruction fetch 0x%x\n", pc);
    *resultp = 0;
    return X_ADR;
  }
  if (wd >= code_words) {
    sim_log("ERROR: Out of range instruction address 0x%x\n", pc);
    *resultp = 0;
    return X_ADR;
  }
  *resultp = instr_mem[wd];
  code = instr_code[wd];
  sim_log("Fetching instruction %x:0x%x %s\n",
	    pc, *resultp, code);
  return X_AOK;
}

/* Load instructions.  Return number of instructions read */
int code_load(FILE *code_file)
{
  if (!initialized) {
    initialized = 1;
    sim_init();
  }
  if (code_words)
    code_clear();
  code_words = load_state(instr_mem, instr_code, code_file, 1, 1);
  return code_words;
}

/* Dump code to specified file */
void code_dump(FILE *code_file) {
  dump_state(instr_mem, instr_code, MEM_COUNT, code_file, 1);
}



/******************************************************************************
 * register-file operations
 ******************************************************************************/

/* Perform one write and two reads to register file */
void reg_update(unsigned char Waddr, unsigned Wdata,
		       unsigned char Aaddr, unsigned char Baddr,
		       unsigned *Adata, unsigned *Bdata)
{
  /* First do the write */
  if (Waddr >= 32) {
    sim_log("WARNING: Attempt to write register %u\n", Waddr);
  }
  else
    if (Waddr > 0) {
      reg_state[Waddr] = Wdata;
      signal_register_update(Waddr, Wdata);
    }
  /* Now do the reads */
  if (Aaddr >= 32) {
    sim_log("WARNING: Attempt to read register %u\n", Aaddr);
  }
  else
    *Adata = reg_state[Aaddr];
  if (Baddr >= 32) {
    sim_log("WARNING: Attempt to read register %u\n", Baddr);
  }
  else
    *Bdata = reg_state[Baddr];
}

/* Print register status */
void reg_display(FILE *outfile)
{
  fprintf(outfile, "Nonzero register state\n");
  dump_state(reg_state, NULL, 32, outfile, 0);
}



/******************************************************************************
 * ALU operations
 ******************************************************************************/

x_status_t alu_compute(unsigned dataA, unsigned dataB, unsigned char op,
		       unsigned *resultp, unsigned char *is_zero_ptr)
{
  unsigned result = 0;
  x_status_t exception = X_AOK;
  switch (op) {
  case O_SLL:
    /* Don't implement actual shift */
    result = dataB;
    break;
  case O_ADDU:
    result = dataA + dataB;
    break;
  case O_NOR:
    result = ~(dataA | dataB);
    break;
  case O_OR:
    result = (dataA | dataB);
    break;
  case O_SLT:
    result = ((int) dataA < (int) dataB);
    break;
  case O_SLTU:
    result = (dataA < dataB);
    break;
  case O_SUBU:
    result = dataA - dataB;
    break;
  case O_XOR:
    result = dataA ^ dataB;
    break;
  case O_BREAK:
    result = 0;
    exception = X_BRK;
    break;
  default:
    sim_log("ERROR: Illegal ALU operation %u\n", op);
    exception = X_ILI;
    break;
  }
  if (is_zero_ptr)
    *is_zero_ptr = (result == 0);
  *resultp = result;
#if 0
  sim_log("  ALU: A=0x%x, B=0x%x, OP=%u, RES = 0x%x, STAT=%s\n",
	    dataA, dataB, op, result, exception_name[status]);
#endif
  return exception;
}

void add_compute(unsigned dataA, unsigned dataB, unsigned *resultp)
{
  *resultp = dataA + dataB;
}



/******************************************************************************
 * data memory operations
 ******************************************************************************/

x_status_t data_read(unsigned addr, unsigned *resultp)
{
  unsigned wd = addr >> 2;
  if (addr & 0x3) {
    sim_log("ERROR: Unaligned read 0x%x\n", addr);
    *resultp = 0;
    return X_ADR;
  }
  if (wd >= MEM_COUNT) {
    sim_log("ERROR: Out of range memory read address 0x%x\n", addr);
    *resultp = 0;
    return X_ADR;
  }
  *resultp = data_mem[wd];
  sim_log("Reading word 0x%x:0x%x\n", addr, *resultp);
  return X_AOK;
}

x_status_t data_write(unsigned addr, unsigned data)
{
  unsigned wd = addr >> 2;
  if (addr & 0x3) {
    sim_log("ERROR: Unaligned write 0x%x\n", addr);
    return X_ADR;
  }
  if (wd >= MEM_COUNT) {
    sim_log("ERROR: Out of range memory write address 0x%x\n", addr);
    return X_ADR;
  }
  data_mem[wd] = data;
  sim_log("Writing word 0x%x:0x%x\n", addr, data);
  return X_AOK;
}

/* Load data from specified file.
   Return number of words read */
int data_load(FILE *data_file)
{
  if (!initialized) {
    initialized = 1;
    sim_init();
  }
  return load_state(data_mem, NULL, data_file, 1, 0);
}

/* Dump data to specified file */
void data_dump(FILE *data_file) {
  dump_state(data_mem, NULL, MEM_COUNT, data_file, 1);
}



/******************************************************************************
 * reporting code
 ******************************************************************************/

static char *format_pc(pc_ptr state)
{
  char pstring[9];
  wstring(state->pc, 4, 32, pstring);
  sprintf(status_msg, "%s", pstring);
  return status_msg;
}

static char *format_if_id(if_id_ptr state)
{
  char istring[9];
  char pcstring[5];
  sprintf(pcstring, "0x%x", state->stage_pc);
  wstring(state->instr, 4, 32, istring);
  sprintf(status_msg, "%s %s %s", x_name(state->exception),
	  state->stage_pc_valid ? pcstring : "---",
	  istring);
  return status_msg;
}

static char *format_id_ex(id_ex_ptr state)
{
  char astring[9];
  char bstring[9];
  char pcstring[5];
  sprintf(pcstring, "0x%x", state->stage_pc);
  wstring(state->adata, 4, 32, astring);
  wstring(state->bdata, 4, 32, bstring);
  sprintf(status_msg, "%s %s %s %s", x_name(state->exception),
	  state->stage_pc_valid ? pcstring : "---",
	  astring, bstring);
  return status_msg;
}

static char *format_ex_mem(ex_mem_ptr state)
{
  char alustring[9];
  char bstring[9];
  char pcstring[5];
  sprintf(pcstring, "0x%x", state->stage_pc);
  wstring(state->alu_result, 4, 32, alustring);
  wstring(state->bdata, 4, 32, bstring);
  sprintf(status_msg, "%s %s %s %s %c %d %d %c",
	  x_name(state->exception),
	  state->stage_pc_valid ? pcstring : "---",
	  alustring, bstring,
	  state->mem_op == M_WRITE ? 'W' :
	    (state->mem_op == M_READ ? 'R' : '-'),
	  state->write_back_dest,
	  state->store_source_reg,
	  state->branch_flag ? 'Y' : 'N');
  return status_msg;
}

static char *format_mem_wb(mem_wb_ptr state)
{
  char alustring[9];
  char rstring[9];
  char pcstring[5];
  sprintf(pcstring, "0x%x", state->stage_pc);
  wstring(state->alu_result, 4, 32, alustring);
  wstring(state->read_data, 4, 32, rstring);
  sprintf(status_msg, "%s %s %s %s %c %d",
	  x_name(state->exception),
	  state->stage_pc_valid ? pcstring : "---",
	  alustring, rstring,
	  state->mem_op == M_READ ? 'R' : '-',
	  state->write_back_dest);
  return status_msg;
}

/* Report system state */
static void sim_report() {
  report_pc(pc_current->pc, 1,
	    id_in->stage_pc, id_in->stage_pc_valid,
	    ex_in->stage_pc, ex_in->stage_pc_valid,
	    mem_in->stage_pc, mem_in->stage_pc_valid,
	    wb_in->stage_pc, wb_in->stage_pc_valid);
  report_state("PC", 0, format_pc(pc_next));
  report_state("PC", 1, format_pc(pc_current));
  report_state("FD", 0, format_if_id(if_out));
  report_state("FD", 1, format_if_id(id_in));
  report_state("DE", 0, format_id_ex(id_out));
  report_state("DE", 1, format_id_ex(ex_in));
  report_state("EM", 0, format_ex_mem(ex_out));
  report_state("EM", 1, format_ex_mem(mem_in));
  report_state("MW", 0, format_mem_wb(mem_out));
  report_state("MW", 1, format_mem_wb(wb_in));
  signal_sources();
}



/******************************************************************************
 * simulator
 ******************************************************************************/

void sim_init()
{
  /* create 5 pipe registers */
  pc_state     = new_pipe(sizeof(pc_ele));
  if_id_state  = new_pipe(sizeof(if_id_ele));
  id_ex_state  = new_pipe(sizeof(id_ex_ele));
  ex_mem_state = new_pipe(sizeof(ex_mem_ele));
  mem_wb_state = new_pipe(sizeof(mem_wb_ele));
  
  /* connect them to the pipeline stages */
  pc_next      = pc_state->next;
  pc_current   = pc_state->current;
  
  if_out  = if_id_state->next;
  
  id_in   = if_id_state->current;
  id_out  = id_ex_state->next;
  
  ex_in   = id_ex_state->current;
  ex_out  = ex_mem_state->next;
  
  mem_in  = ex_mem_state->current;
  mem_out = mem_wb_state->next;
  
  wb_in   = mem_wb_state->current;

  sim_reset();
  code_clear();
}

void sim_reset()
{
  if (!initialized) {
    initialized = 1;
    sim_init();
  }
  clear_pipes();
  clear_state(reg_state, NULL, 32);
  signal_register_clear();
  clear_state(data_mem, NULL, MEM_COUNT);
  amux = bmux = smux = MUX_NONE;
  sim_report();
}

/* Run pipeline for one cycle */
/* Don't change this code */
static x_status_t sim_step_pipe()
{
  /* Update pipe registers */
  update_pipes();

  /* Behavior should be independent of ordering of following evaluations */
  do_if_stage (pc_next, if_out);
  do_id_stage (id_out);
  do_ex_stage (ex_out);
  do_mem_stage(mem_out);
  
  do_stall_check(pc_next, if_out, id_out, ex_out, mem_out);

  /* Report progress of instructions through pipe */
  sim_report();
  return wb_in->exception;
}

/* Run pipeline for count cycles or until exception signalled */
x_status_t sim_run_pipe(int max_cycles)
{
  unsigned cycles = 0;
  x_status_t exception = X_AOK;
  if (!initialized) {
    initialized = 1;
    sim_init();
  }
  if (!code_words)
    return X_ILI;
  while (max_cycles-- > 0 && exception == X_AOK) {
    exception = sim_step_pipe();
    cycles++;
    if (dumpfile) {
      sim_log("Nonzero register state after %d cycles\n", cycles);
      reg_display(dumpfile);
    }
  }
  return exception;
}

/* If dumpfile set nonNULL, lots of status info printed out */
void sim_set_dumpfile(FILE *df)
{
  dumpfile = df;
}

/*
 * sim_log dumps a formatted string to the dumpfile, if it exists
 * accepts variable argument list
 */
void sim_log( const char *format, ... ) {
  if (dumpfile) {
    va_list arg;
    va_start( arg, format );
    vfprintf( dumpfile, format, arg );
    va_end( arg );
    }
}

/******************************************************************************/

     

