/******************************************************************************
 *	tk_support.c
 *
 *	Simulation control for TK based simulator
 ******************************************************************************/


/******************************************************************************
 *	#includes
 ******************************************************************************/

#include <tk.h>
#include "pipe.h"
#include "sim.h"



/******************************************************************************
 *	static variables
 ******************************************************************************/

static char tcl_msg[256];

/* Keep track of the TCL Interpreter */
static Tcl_Interp *sim_interp = NULL;



/******************************************************************************
 *	function declarations
 ******************************************************************************/

int simResetCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[]);

int simLoadCodeCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[]);

int simLoadDataCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[]);

int simRunCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[]);

int simModeCmd(ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[]);

void addAppCommands(Tcl_Interp *interp);

/******************************************************************************
 *	tcl command definitions
 ******************************************************************************/

/* Implement command versions of the simulation functions */
int simResetCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  sim_interp = interp;
  if (argc != 1) {
    interp->result = "No arguments allowed";
    return TCL_ERROR;
  }
  sim_reset();
  interp->result = x_name(X_AOK);
  return TCL_OK;
}

int simLoadCodeCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  FILE *code_file;
  int code_count;
  sim_interp = interp;
  if (argc != 2) {
    interp->result = "One argument required";
    return TCL_ERROR;
  }
  code_file = fopen(argv[1], "r");
  if (!code_file) {
    sprintf(tcl_msg, "Couldn't open code file '%s'", argv[1]);
    interp->result = tcl_msg;
    return TCL_ERROR;
  }
  code_count = code_load(code_file);
  sprintf(tcl_msg, "%d", code_count);
  interp->result = tcl_msg;
  fclose(code_file);
  return TCL_OK;
}

int simLoadDataCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  FILE *data_file;
  int word_count;
  sim_interp = interp;
  if (argc != 2) {
    interp->result = "One argument required";
    return TCL_ERROR;
  }
  data_file = fopen(argv[1], "r");
  if (!data_file) {
    sprintf(tcl_msg, "Couldn't open data file '%s'", argv[1]);
    interp->result = tcl_msg;
    return TCL_ERROR;
  }
  word_count = data_load(data_file);
  sprintf(tcl_msg, "%d", word_count);
  interp->result = tcl_msg;
  fclose(data_file);
  return TCL_OK;
}


int simRunCmd(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
  int cycle_limit = 1;
  x_status_t status;
  sim_interp = interp;
  if (argc > 2) {
    interp->result = "At most one argument allowed";
    return TCL_ERROR;
  }
  if (argc >= 2 &&
      (sscanf(argv[1], "%d", &cycle_limit) != 1 ||
       cycle_limit < 0)) {
    sprintf(tcl_msg, "Cannot run for '%s' cycles!", argv[1]);
    interp->result = tcl_msg;
    return TCL_ERROR;
  }
  status = sim_run_pipe(cycle_limit);
  interp->result = x_name(status);
  return TCL_OK;
}

int simModeCmd(ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[])
{
  sim_interp = interp;
  if (argc != 2) {
    interp->result = "One argument required";
    return TCL_ERROR;
  }
  interp->result = argv[1];
  if (strcmp(argv[1], "wedged") == 0)
    sim_mode = S_WEDGED;
  else if (strcmp(argv[1], "stall") == 0)
    sim_mode = S_STALL;
  else if (strcmp(argv[1], "forward") == 0)
    sim_mode = S_FORWARD;
  else {
    sprintf(tcl_msg, "Unknown mode '%s'", argv[1]);
    interp->result = tcl_msg;
    return TCL_ERROR;
  }
  return TCL_OK;
}


/******************************************************************************
 *	registering the commands with tcl
 ******************************************************************************/

void addAppCommands(Tcl_Interp *interp)
{
  sim_interp = interp;
  Tcl_CreateCommand(interp, "simReset", simResetCmd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "simCode", simLoadCodeCmd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "simData", simLoadDataCmd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "simRun", simRunCmd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand(interp, "setSimMode", simModeCmd,
		    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
} 

/******************************************************************************
 *	tcl functionality called from within C
 ******************************************************************************/

/* Provide mechanism for simulator to update register display */
void signal_register_update(unsigned char Waddr, unsigned Wdata) {
  int code;
  sprintf(tcl_msg, "setReg %d %d 1", (int) Waddr, (int) Wdata);
  code = Tcl_Eval(sim_interp, tcl_msg);
  if (code != TCL_OK) {
    fprintf(stderr, "Failed to signal register set\n");
    fprintf(stderr, "Error Message was '%s'\n", sim_interp->result);
  }
}

char *rname[11] = {"none", "ea", "eb", "ma", "mb", "wa", "wr", "none", "none", "none", "none"};

/* provide mechanism for simulator to specify source registers */
void signal_sources() {
  int code;
  sprintf(tcl_msg, "showSources %s %s %s",
	  rname[amux], rname[bmux], rname[smux]);
  code = Tcl_Eval(sim_interp, tcl_msg);
  if (code != TCL_OK) {
    fprintf(stderr, "Failed to signal ALU/Mem sources\n");
    fprintf(stderr, "Error Message was '%s'\n", sim_interp->result);
  }
}

/* Provide mechanism for simulator to clear register display */
void signal_register_clear() {
  int code;
  code = Tcl_Eval(sim_interp, "clearReg");
  if (code != TCL_OK) {
    fprintf(stderr, "Failed to signal register clear\n");
    fprintf(stderr, "Error Message was '%s'\n", sim_interp->result);
  }
}

/* Provide mechanism for simulator to report instructions as they are 
   read in
*/

void report_line(int addr, unsigned data, char *text) {
  char sdata[9];
  void wstring(unsigned, int,int, char *);
  int code;
  /* Create hex representation of string */
  wstring(data, 4, 32, sdata);
  sprintf(tcl_msg, "addCodeLine %d {%s} {%s}", addr, sdata, text);
  code = Tcl_Eval(sim_interp, tcl_msg);
  if (code != TCL_OK) {
    fprintf(stderr, "Failed to report code line 0x%x\n", addr);
    fprintf(stderr, "Error Message was '%s'\n", sim_interp->result);
  }
}


/* Provide mechanism for simulator to report which instructions are in
   which stages */
void report_pc(unsigned fpc, unsigned char fpcv,
	       unsigned dpc, unsigned char dpcv,
	       unsigned epc, unsigned char epcv,
	       unsigned mpc, unsigned char mpcv,
	       unsigned wpc, unsigned char wpcv)
{
  int status;
  char addr[10];
  char code[12];
  Tcl_DString cmd;
  Tcl_DStringInit(&cmd);
  Tcl_DStringAppend(&cmd, "simLabel ", -1);
  Tcl_DStringStartSublist(&cmd);
  if (fpcv) {
    sprintf(addr, "%u", fpc);
    Tcl_DStringAppendElement(&cmd, addr);
  }
  if (dpcv) {
    sprintf(addr, "%u", dpc);
    Tcl_DStringAppendElement(&cmd, addr);
  }
  if (epcv) {
    sprintf(addr, "%u", epc);
    Tcl_DStringAppendElement(&cmd, addr);
  }
  if (mpcv) {
    sprintf(addr, "%u", mpc);
    Tcl_DStringAppendElement(&cmd, addr);
  }
  if (wpcv) {
    sprintf(addr, "%u", wpc);
    Tcl_DStringAppendElement(&cmd, addr);
  }
  Tcl_DStringEndSublist(&cmd);
  Tcl_DStringStartSublist(&cmd);
  sprintf(code, "%s %s %s %s %s",
	  fpcv ? "F" : "",
	  dpcv ? "D" : "",
	  epcv ? "E" : "",
	  mpcv ? "M" : "",
	  wpcv ? "W" : "");
  Tcl_DStringAppend(&cmd, code, -1);
  Tcl_DStringEndSublist(&cmd);
  /* Debug 
  fprintf(stderr, "Code '%s'\n", Tcl_DStringValue(&cmd));
  */
  status = Tcl_Eval(sim_interp, Tcl_DStringValue(&cmd));
  if (status != TCL_OK) {
    fprintf(stderr, "Failed to report pipe code '%s'\n", code);
    fprintf(stderr, "Error Message was '%s'\n", sim_interp->result);
  }
}

/* Report single line of pipeline state */
void report_state(char *id, int current, char *txt) {
  int status;
  sprintf(tcl_msg, "updateStage %s %d {%s}", id, current,txt);
  status = Tcl_Eval(sim_interp, tcl_msg);
  if (status != TCL_OK) {
    fprintf(stderr, "Failed to report pipe status\n");
    fprintf(stderr, "\tStage %s.%s, status '%s'\n",
	    id, current ? "current" : "next", txt);
    fprintf(stderr, "\tError Message was '%s'\n", sim_interp->result);
  }
}

/******************************************************************************/


