/******************************************************************************
 *	pipe.c
 *
 *	Code for implementing pipelined processor simulators
 ******************************************************************************/

/******************************************************************************
 *	#includes
 ******************************************************************************/

#include "pipe.h"
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

/******************************************************************************
 *	defines
 ******************************************************************************/

#define MAX_STAGE 10

/******************************************************************************
 *	static variables
 ******************************************************************************/

static pipe_ptr pipes[MAX_STAGE];
static int pipe_count = 0;

/******************************************************************************
 *	function definitions
 ******************************************************************************/

pipe_ptr new_pipe(int count)
{
  pipe_ptr result = (pipe_ptr) malloc(sizeof(pipe_ele));
  result->current = malloc(count);
  result->next = malloc(count);
  memset(result->current, '\0', count);
  memset(result->next, '\0', count);
  result->count = count;
  result->op = P_LOAD;
  pipes[pipe_count++] = result; 
  return result;
}

void update_pipes()
{
  int s;
  for (s = 0; s < pipe_count; s++) {
    pipe_ptr p = pipes[s];
    switch (p->op)
      {
      case P_BUBBLE:
      	/* insert a bubble into the next stage */
      	memset(p->current, '\0', p->count);
      	break;
      
      case P_LOAD:
      	/* copy calculated state from previous stage */
      	memcpy(p->current, p->next, p->count);
      	break;
      
      case P_STALL:
      default:
      	/* do nothing: next stage gets same instr again */
      	;
      }
    p->op = P_LOAD;
  }
}

void clear_pipes()
{
  int s;
  for (s = 0; s < pipe_count; s++) {
    pipe_ptr p = pipes[s];
    memset(p->current, '\0', p->count);
    memset(p->next, '\0', p->count);
    p->op = P_LOAD;
  }
}

/******************** Utility Code *************************/

/* Representations of digits */
static char digits[16] =
   {'0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

/* Print hex/oct/binary format with leading zeros */
/* bpd denotes bits per digit  Should be in range 1-4,
   pbw denotes bits per word.*/
void wprint(unsigned x, int bpd, int bpw, FILE *fp)
{
  int digit;
  unsigned mask = (1 << bpd) - 1;
  for (digit = (bpw-1)/bpd; digit >= 0; digit--) {
    unsigned val = (x >> (digit * bpd)) & mask;
    putc(digits[val], fp);
  }
}

/* Create string in hex/oct/binary format with leading zeros */
/* bpd denotes bits per digit  Should be in range 1-4,
   pbw denotes bits per word.*/
void wstring(unsigned x, int bpd, int bpw, char *str)
{
  int digit;
  unsigned mask = (1 << bpd) - 1;
  for (digit = (bpw-1)/bpd; digit >= 0; digit--) {
    unsigned val = (x >> (digit * bpd)) & mask;
    *str++ = digits[val];
  }
  *str = '\0';
}


/* Dump nonzero state of a memory, where
   state is an array of unsigned's denoting the memory state,
   count is the number of words.
   if text nonnull, prints associated text with each word
   Otherwise, puts in multicolumn format for legibility
   byte_address flag indicates that addresses should be printed X4 
*/
void dump_state(unsigned *state, char **text, int count, FILE *fp,
		int byte_address)
{
  int col = 0;
  int col_limit = text ? 1 : 4; 
  int bpa = 0;
  int i;
  /* figure out how many bits it takes to represent the address */ 
  while (1<<bpa < count)
    bpa++;
  for (i = 0; i < count; i++) {
    unsigned val = state[i];
    char *info = text ? text[i] : NULL;
    if (val != 0 || info != NULL) {
      fputs("  ", fp);
      wprint(byte_address ? i << 2 : i , 4, bpa, fp);
      fputs(":0x", fp);
      wprint(val, 4, 32, fp);
      if (info)
	fprintf(fp, "\t%s", info);
      if (++col >= col_limit) {
	col = 0;
	fputs("\n", fp);
      }
    }
  }
  if (col != 0)
    fputs("\n", fp);
}

/* Dump complete state of a memory, where
   state is an array of unsigned's denoting the memory state,
   count is the number of words.
   if text nonnull, prints associated text with each word
   Otherwise, puts in multicolumn format for legibility
   byte_address flag indicates that addresses should be printed X4 
*/
void dump_all_state(unsigned *state, char **text, int count, FILE *fp,
		int byte_address)
{
  int col = 0;
  int col_limit = text ? 1 : 4; 
  int bpa = 0;
  int i;
  /* figure out how many bits it takes to represent the address */ 
  while (1<<bpa < count)
    bpa++;
  for (i = 0; i < count; i++) {
    unsigned val = state[i];
    char *info = text ? text[i] : NULL;
    fputs("  ", fp);
    wprint(byte_address ? i << 2 : i , 4, bpa, fp);
    fputs(":0x", fp);
    wprint(val, 4, 32, fp);
    if (info)
      fprintf(fp, "\t%s", info);
    if (++col >= col_limit) {
      col = 0;
      fputs("\n", fp);
    }
  }
  if (col != 0)
    fputs("\n", fp);
}

/* Get string starting with first nonwhite character and ending with newline */
/* Convert tabs into spaces, assuming tab spacing of 8 */
static char *get_text(FILE *fp)
{
  char buffer[256];
  char *result;
  int i = 0;
  int  c = getc(fp);
  /* Skip over initial white space */
  while (c != EOF && c != '\n' && isspace(c))
    c = getc(fp);
  while (c != EOF && c != '\n') {
    if (c == '\t') {
      do
	buffer[i++] = ' ';
      while (i & 0x7);
    }
    else
      buffer[i++] = c;
    c = getc(fp);
  }
  buffer[i++] = '\0';
  result = malloc(i);
  strcpy(result, buffer);
  return result;
}

static void skip_junk(FILE *fp)
{
  /* Skip to first digit not enclosed in brackets */
  int c = getc(fp);
  while (!isdigit(c) && c != '[' && c != EOF)
    c = getc(fp);
  /* Look for bracket */
  if (c == '[') {
    /* Scan for matching close bracket */
    c = getc(fp);
    while (c != EOF && c != ']')
      c = getc(fp);
    while (!isdigit(c) && c != EOF)
      c = getc(fp); 
  }
  if (c != EOF)
    ungetc(c, fp);
}


/* Load state from memory */
/* Assumes stored as set of pairs AAA:0xDDDD, all in Hex format */
/* If text nonnull, parses quoted strings following numeric values
   as associated text information */
/* byte_address flag indicates word or byte addressing */
/* report flag indicates whether to report individual lines read */
/* Returns number of words read */

unsigned load_state(unsigned *state, char **text, FILE *fp,
		    int byte_address, int report)
{
  unsigned a, d;
  unsigned count = 0;
  void report_line(int a, unsigned d, char *t);
  int found;
  do {
    skip_junk(fp);
    found = fscanf(fp, " 0x%x: %x", &a, &d) == 2;
    if (found) {
      int wd = byte_address ? a >> 2 : a;
      state[wd] = d;
      if (text)
	text[wd] = get_text(fp);
      report_line(a, d, text ? text[wd] : NULL);
/* DEBUG 
      printf("Read word 0x%x: %x\t%s\n", wd, state[wd], text ? text[wd] : "");
*/
      count++;
    }
  } while (found);
  return count;
}

void clear_state(unsigned *state, char **text, int count)
{
  while (--count >= 0) {
    *state++ = 0;
    if (text && *text) {
      free(*text);
      *text++ = NULL;
    }
  }
}

/******************************************************************************/

