##########################################################################
# Parsing of command line flags                                          #
##########################################################################

proc flagVal {flag default} {
    global argv
    foreach t $argv {
	if {[string match "-$flag*" $t]} {return [string range $t 2 end]}
    }
    return $default
}

##########################################################################
# Register File Implementation.  Shown as array of 4 rows & 4 columns    #
##########################################################################


# Font used to display register contents
set fontSize [expr 10 * [flagVal "f" 12]]
set codeFontSize [expr 10 * [flagVal "c" 10]]
set dpyFont "*-courier-medium-r-normal--*-$fontSize-*-*-*-*-*-*"
set codeFont "*-courier-medium-r-normal--*-$codeFontSize-*-*-*-*-*-*"
# Background Color of normal register
set normalBg white
# Background Color of highlighted register
set specialBg LightSkyBlue

# Keep track of previous highlighted register
set lastId -1
proc setReg {id val highlight} {
    global lastId normalBg specialBg
    if {$lastId >= 0} {
	.r.reg$lastId config -bg $normalBg
	set lastId -1
    }
    if {$id < 0 || $id >= 32} {
	error "Invalid Register ($id)"
    }
    .r.reg$id config -text [format %8x $val]
    if {$highlight} {
	uplevel .r.reg$id config -bg $specialBg
	set lastId $id
    }
}

# Clear all registers
proc clearReg {} {
    global lastId normalBg
    if {$lastId >= 0} {
	.r.reg$lastId config -bg $normalBg
	set lastId -1
    } 
    for {set i 1} {$i < 32} {incr i 1} {
	.r.reg$i config -text ""
    }
}

# Create Window for registers
frame .r
pack .r -in . -side bottom
# Following give separate window for register file
# toplevel .r
# wm title .r "Register File"
label .r.lab -text "Register File"
pack .r.lab -in .r -side top
# Set up top row control panel (disabled)
# frame .r.cntl
# pack .r.cntl -fill x -in .r
# label .r.labreg -text "Register" -width 10
# entry .r.regid -width 3 -relief sunken -textvariable regId -font $dpyFont
# label .r.labval -text "Value" -width 10
# entry .r.regval -width 8 -relief sunken -textvariable regVal -font $dpyFont
# button .r.doset -text "Set" -command {setReg $regId $regVal 1} -width 6
# button .r.c -text "Clear" -command clearReg -width 6
# pack .r.labreg .r.regid .r.labval .r.regval .r.doset .r.c  -in .r.cntl -side left



# Create 4 rows of 8 register each
foreach i {0 1 2 3} {
    frame .r.row$i
    pack .r.row$i -side top -in .r
}

# Create 32 registers
for {set i 0} {$i < 32} {incr i 1} {
    label .r.reg$i -width 8 -font $dpyFont -relief ridge \
	    -bg $normalBg
}

# Pack them into 4 rows
for {set i 0} {$i < 4} {incr i 1} {
    for {set j [expr $i*8]} {$j < [expr ($i+1)*8] } {incr j 1} {
	pack .r.reg$j -in .r.row$i -side left
    }
}

# Initialize register 0
setReg 0 0 0


##############################################################################
#  Main Control Panel                                                        #
##############################################################################
wm title . "MIPS Simulator"

# Control Panel for simulator
set cntlBW 9
frame .cntl
pack .cntl
button .cntl.quit -width $cntlBW -text Quit -command exit
button .cntl.run -width $cntlBW -text Go -command simGo
button .cntl.stop -width $cntlBW -text Stop -command simStop
button .cntl.step -width $cntlBW -text Step -command simStep
button .cntl.reset -width $cntlBW -text Reset -command simResetAll
pack .cntl.quit .cntl.run .cntl.stop .cntl.step .cntl.reset -in .cntl -side left
# Simulation speed control
scale .spd -label {Simulator Speed (10*log Hz)} -from -10 -to 30 -length 10c \
  -orient horizontal -command setSpeed
pack .spd

# Simulation mode 
set simMode wedged

frame .md
pack .md
radiobutton .md.wedged -text Wedged -variable simMode \
	-value wedged -width 10 -command {setSimMode wedged}
radiobutton .md.stall -text Stall -variable simMode \
	-value stall -width 10 -command {setSimMode stall}
radiobutton .md.forward -text Forward -variable simMode \
	-value forward -width 10 -command {setSimMode forward}
pack .md.wedged .md.stall .md.forward -in .md -side left

set simStat "INI"
# Line to display simulation status (disabled)
#frame .stat
# pack .stat
#label .stat.statlab -width 10 -text "Status"
#label .stat.statdpy -width 3 -font $dpyFont -relief ridge -textvariable simStat
#pack .stat.statlab .stat.statdpy  -in .stat -side left


# simDelay defines #milliseconds for each cycle of simulator
# Initial value is 1000ms
set simDelay 1000
# Set delay based on rate expressed in log(Hz)
proc setSpeed {rate} {
  global simDelay
  set simDelay [expr round(1000 / pow(10,$rate/10.0))]
}

# Global variables controlling simulator execution
# Should simulator be running now?
set simGoOK 0

proc simStop  {} {
  global simGoOK
  set simGoOK 0
}

proc simStep {} {
    global simStat
    set simStat [simRun 1]
}

proc simGo {} {
    global simGoOK simDelay simStat
    set simGoOK 1
    # Disable the Go and Step buttons
    # Enable the Stop button
    while {$simGoOK} {
	# run the simulator 1 cycle
	after $simDelay
	set simStat [simRun 1]
	if {$simStat != "AOK"} {set simGoOK 0}
	update
    }
    # Disable the Stop button
    # Enable the Go and Step buttons
}

##############################################################################
#  Pipe Register Display                                                     #
##############################################################################

# Colors for Highlighting Data Sources
set adataBg LightPink
set bdataBg PaleGreen1
set sdataBg PaleTurquoise1

# Overall width of pipe register display
set pipeWidth 60
set labWidth 8

# Add labeled display to window 
proc addDisp {win width name} {
    global dpyFont
    set lname [string tolower $name]
    frame $win.$lname
    pack $win.$lname -in $win -side left
    label $win.$lname.t -text $name
    label $win.$lname.n -width $width -font $dpyFont -bg white -relief ridge
    label $win.$lname.c -width $width -font $dpyFont -bg white -relief ridge
    pack $win.$lname.c $win.$lname.t $win.$lname.n -in $win.$lname -side top
    return [list $win.$lname.n $win.$lname.c]
}

# Set text in display row
proc setDisp {wins txts} {
    for {set i 0} {$i < [llength $wins] && $i < [llength $txts]} {incr i} {
	set win [lindex $wins $i]
	set txt [lindex $txts $i]
	$win config -text $txt
    }
}

frame .p -width $pipeWidth 
pack .p -in . -side bottom
label .p.lab -text "Pipeline Registers"
pack .p.lab -in .p -side top
label .p.mem -text "MEM Stage" -width $pipeWidth -bg NavyBlue -fg White
label .p.ex -text "EX Stage" -width $pipeWidth -bg NavyBlue -fg White
label .p.id -text "ID Stage" -width $pipeWidth -bg NavyBlue -fg White
label .p.if -text "IF Stage" -width $pipeWidth -bg NavyBlue -fg White
frame .p.mw 
frame .p.em
frame .p.de
frame .p.fd
frame .p.pc
pack .p.mw .p.mem .p.em .p.ex .p.de .p.id .p.fd .p.if .p.pc -in .p -side top -anchor w -expand 1

proc addLabel { win nstage cstage } {
    global labWidth
    frame $win.lab
# Disabled
#    label $win.lab.t  -text "$nstage/$cstage"
    label $win.lab.t
    label $win.lab.n -width $labWidth -bg white -text "$nstage Out" -anchor w
    label $win.lab.c -width $labWidth -bg white -text "$cstage In" -anchor w
    pack $win.lab.c $win.lab.t $win.lab.n -in $win.lab -side top
    pack $win.lab -in $win -side left
}


addLabel .p.mw MEM WB
addLabel .p.em EX MEM
addLabel .p.de ID EX
addLabel .p.fd IF ID
addLabel .p.pc PC IF

proc addFill { win w } {
    frame $win.fill
    label $win.fill.t -text "" -width $w
    label $win.fill.n -bg white -text "" -width $w
    label $win.fill.c -bg white -text "" -width $w
    pack $win.fill.c $win.fill.t $win.fill.n -in $win.fill -side top -expand 1
    pack $win.fill -in $win -side right -expand 1
}

addFill .p.mw 11
addFill .p.de 23
addFill .p.fd 32
addFill .p.pc 42

# Take list of lists, and transpose nesting
# Assumes all lists are of same length
proc ltranspose {inlist} {
    set result {}
    for {set i 0} {$i < [llength [lindex $inlist 0]]} {incr i} {
	set nlist {}
	for {set j 0} {$j < [llength $inlist]} {incr j} {
	    set ele [lindex [lindex $inlist $j] $i]
	    set nlist [concat $nlist [list $ele]]
	}
	set result [concat $result [list $nlist]]
    }
    return $result
}

# Fields in PC displayed
set pwins(PC) [ltranspose [list [addDisp .p.pc 8 PC]]]

# Fields in IF/ID display
set pwins(FD) [ltranspose \
           [list [addDisp .p.fd 3 Exc]   \
	         [addDisp .p.fd 4 SPC] \
                 [addDisp .p.fd 8 Instr]]] 

# ID/EX Display
set pwins(DE) [ltranspose \
           [list [addDisp .p.de 3 Exc] \
	         [addDisp .p.de 4 SPC] \
                 [addDisp .p.de 8 Adata] \
                 [addDisp .p.de 8 Bdata]]]

# EX/MEM
set pwins(EM) [ltranspose \
           [list [addDisp .p.em 3 Exc] \
	         [addDisp .p.em 4 SPC] \
                 [addDisp .p.em 8 ALUdata] \
                 [addDisp .p.em 8 Bdata] \
                 [addDisp .p.em 5 Mop] \
                 [addDisp .p.em 5 Wdst] \
                 [addDisp .p.em 5 Ssrc] \
                 [addDisp .p.em 5 Bch]]] 


# MEM/WB
set pwins(MW) [ltranspose \
           [list [addDisp .p.mw 3 Exc] \
	         [addDisp .p.mw 4 SPC] \
                 [addDisp .p.mw 8 ALUdata] \
                 [addDisp .p.mw 8 Rdata] \
                 [addDisp .p.mw 5 Mop] \
                 [addDisp .p.mw 5 Wdst]]]


# update status line for specified pipe register
proc updateStage {name current txts} {
    set Name [string toupper $name]
    global pwins
    set wins [lindex $pwins($Name) $current]
    setDisp $wins $txts
}   

# Create Array of windows corresponding to data sources
set rwins(wa) .p.mw.aludata.c
set rwins(wr) .p.mw.rdata.c
set rwins(ma) .p.em.aludata.c
set rwins(mb) .p.em.bdata.c
set rwins(ea) .p.de.adata.c
set rwins(eb) .p.de.bdata.c

# Highlight Data Source Registers for ALU A, ALU B, and Store
proc showSources { a b s } {
    global rwins adataBg bdataBg sdataBg
    # Set them all to white
    foreach w [array names rwins] {
	$rwins($w) config -bg White
    }
    if {$a != "none"} { $rwins($a) config -bg $adataBg }
    if {$b != "none"} { $rwins($b) config -bg $bdataBg }
    if {$s != "none"} { $rwins($s) config -bg $sdataBg }

    # Indicate forwarding destinations by their color
    .p.de.adata.t config -bg $adataBg
    .p.de.bdata.t config -bg $bdataBg
    .p.em.bdata.t config -bg $sdataBg
}

##########################################################################
#                    Instruction Display                                 #
##########################################################################

toplevel .c
wm title .c "Program Code"
frame .c.cntl 
pack .c.cntl -in .c -side top -anchor w
label .c.filelab -width 10 -text "File"
entry .c.filename -width 20 -relief sunken -textvariable codeFile \
	-font $dpyFont -bg white
button .c.loadbutton -width $cntlBW -command {loadCode $codeFile} -text Load
pack .c.filelab .c.filename .c.loadbutton -in .c.cntl -side left

proc clearCode {} {
    simLabel {} {}
    destroy .c.t
    destroy .c.t0
    destroy .c.t1
    destroy .c.t2
    destroy .c.t3
}

proc createCode {} {
    # Create Code Structure
    frame .c.t
    pack .c.t -in .c -side top -anchor w
    # Support up to 4 columns of code, each 32 lines long
    frame .c.t0
    frame .c.t1
    frame .c.t2
    frame .c.t3
    pack .c.t0 .c.t1 .c.t2 .c.t3 -in .c.t -side left -anchor nw
}

proc loadCode {file} {
    # Kill old code window
    clearCode
    # Create new one
    createCode
    simCode $file
    simResetAll
}

# Start with initial code window, even though it will be destroyed.
createCode

# Add a line of code to the display
proc addCodeLine {addr op text} {
    # Create new line in display
    set col [expr $addr / 128]
    global codeFont
    frame .c.t$col.$addr
    pack .c.t$col.$addr -in .c.t$col -side top -anchor w
    label .c.t$col.$addr.a -width 5 -text [format "0x%x" $addr] -font $codeFont
    label .c.t$col.$addr.i -width 8 -text $op -font $codeFont
    label .c.t$col.$addr.s -width 2 -text "" -font $codeFont -bg white
    label .c.t$col.$addr.t -text $text -font $codeFont
    pack .c.t$col.$addr.a .c.t$col.$addr.i .c.t$col.$addr.s \
	    .c.t$col.$addr.t -in .c.t$col.$addr -side left
}

# Keep track of which instructions have stage labels

set oldAddr {}

proc simLabel {addrs labs} {
    global oldAddr
    set newAddr {}
    # Clear away any old labels
    foreach a $oldAddr {
	set col [expr $a / 128]
	.c.t$col.$a.s config -text ""
    }
    for {set i 0} {$i < [llength $addrs]} {incr i} {
	set a [lindex $addrs $i]
	set t [lindex $labs $i]
	set col [expr $a / 128]
	if {[winfo exists .c.t$col.$a]} {
	    .c.t$col.$a.s config -text $t
	    set newAddr [concat $newAddr $a]
	}
    }
    set oldAddr $newAddr
}

proc simResetAll {} {
    global simStat
    set simStat "INI"
    simReset
    simLabel {} {}
}

###############################################################################
#    Command Line Initialization                                              #
###############################################################################

# Get code file name from input

# Find file with specified extension
proc findFile {tlist ext} {
    foreach t $tlist {
	if {[string match "*.$ext" $t]} {return $t}
    }
    return ""
}


set codeFile [findFile $argv O]
if {$codeFile != ""} { loadCode $codeFile}
