// Automated testing of LSP support code.
// These test provide some corner conditions to check
// properly handing of disconnecting components & early buffering

package lsp

import (
	"fmt"
	"json"
	"rand"
	"testing"
	"time"
)

// Convert integer to byte array
func auxi2b(i int) []byte {
	b, _ := json.Marshal(i)
	return b
}

// Convert byte array back to integer
func auxb2i(b []byte) int {
	var i int
	json.Unmarshal(b, &i)
	return i
}

// Different test options
const (
	doslowstart = iota
	doclientstop
	doserverstop
	)

var ModeName = map[int] string { doslowstart: "Slow Start", doclientstop: "Client Stop", doserverstop : "Server Stop" }

// Test components
type AuxTestSystem struct {
	Server *LspServer
	Clients []*LspClient
	Mode int // One of the modes enumerated above
	Port int
	// Use to synchronize
	ClientChan chan bool // Clients
	ServerChan chan bool // Server
	TimeChan chan bool   // Overall time limit
	RunFlag bool
	Nclients int
	Nmessages int
	Tester *testing.T
}

// Delay for fixed amount of time
func auxdelay(sleeps float32) {
	if sleeps > 0.0 {
		t := int64(sleeps * 1e9)
		time.Sleep(t)
	}
}

// Generate random port number
func randport() int {
	rand.Seed(time.Nanoseconds())
	return 3000 + rand.Intn(1000)
}

// Try to create server on designated port.  Return false if fail
func (ts *AuxTestSystem) GenerateServer() bool {
	ts.Server = NewLspServer(ts.Port)
	return ts.Server != nil 
}

// Try to create next client.  Return false if fail
func (ts *AuxTestSystem) GenerateClient(index int) bool {
	ts.Clients[index] = NewLspClient("localhost", ts.Port)
	return ts.Clients[index] != nil
}

func NewAuxTestSystem(t *testing.T, nclients int, mode int, maxtime float32) {
	fmt.Printf("Testing: Mode %s, %d clients\n", ModeName[mode], nclients)
	ts := new(AuxTestSystem)
	ts.Mode = mode
	ts.ClientChan = make(chan bool)
	ts.ServerChan = make(chan bool)
	ts.TimeChan = make(chan bool)
	ts.Clients = make([]*LspClient, nclients)
	ts.Nclients = nclients
	ts.Nmessages = 10
	ts.Port = randport()
	ts.RunFlag = true
	ts.Tester = t
	go ts.BuildServer()
	for i := 0; i < nclients; i++ {
		go ts.BuildClient(i)
	}
	go ts.runtimeout(maxtime)

	switch ts.Mode {
	case doclientstop:
		// Wait for server or timer to complete
		select {
			case sok := <- ts.ServerChan:
			if !sok {
				ts.Tester.Logf("Server error\n")
				ts.Tester.FailNow()
			}
			case <- ts.TimeChan:
			ts.Tester.Logf("Test timed out waiting for server\n")
			ts.Tester.FailNow()
		}
		fmt.Printf("Server completed\n")
		// Wait for the clients
		for i := 0; i < nclients; i++ {
			select {
				case cok := <- ts.ClientChan:
				if !cok {
					ts.Tester.Logf("Client error\n")
					ts.Tester.FailNow()
				}
				case <- ts.TimeChan:
				ts.Tester.Logf("Test timed out waiting for client\n")
				ts.Tester.FailNow()
			}
		}
		fmt.Printf("Clients completed\n")
	default:  // Includes stopserver
		// Wait for the clients
		for i := 0; i < nclients; i++ {
			select {
				case cok := <- ts.ClientChan:
				if !cok {
					ts.Tester.Logf("Client error\n")
					ts.Tester.FailNow()
				}
				case <- ts.TimeChan:
				ts.Tester.Logf("Test timed out waiting for client\n")
				ts.Tester.FailNow()
			}
		}
		// Wait for server or timer to complete
		select {
			case sok := <- ts.ServerChan:
			if !sok {
				ts.Tester.Logf("Server error\n")
				ts.Tester.FailNow()
			}
			case <- ts.TimeChan:
			ts.Tester.Logf("Test timed out waiting for server\n")
			ts.Tester.FailNow()
		}
	}
}

// Create server and and run test

func (ts *AuxTestSystem) BuildServer() {
	if ts.Mode == doslowstart {
		// Delay start so that clients can start first
		auxdelay(3 * epochlength)
	}
	if !ts.GenerateServer() {
		fmt.Printf("Couldn't create server on port %d\n", ts.Port)
		ts.ServerChan <- false
		return
	}
	ndead := 0;
	necho := 0;
	for ts.RunFlag {
		id, data := ts.Server.Read()
		if data == nil {
			fmt.Printf("Detected dead connection %d\n", id)
			ndead ++
			if ts.Mode == doclientstop && ndead == ts.Nclients {
				fmt.Printf("Detected termination by all %d clients\n", ndead)
				ts.ServerChan <- true
				// Shut down server after completion
				ts.Server.Close(id)
				return
			}
		} else {
			ts.Server.Write(id, data)
			necho ++
			if necho == ts.Nclients * ts.Nmessages {
				fmt.Printf("Echoed all %d X %d messsages\n",
					ts.Nclients, ts.Nmessages)
				if ts.Mode == doserverstop {
					// Tell server to stop
					fmt.Printf("Stopping server\n")
					ts.Server.Close(id)
					ts.ServerChan <- true
					return
				}
				if ts.Mode != doclientstop {
					ts.ServerChan <- true
					// Shut down server after completion
					ts.Server.Close(id)
					return
				}
			}
		}
	}
	// Reached time limit
	if ts.Mode == doclientstop {
		fmt.Printf("Server only detected termination of %d clients\n", ndead)
		ts.ServerChan <- false
	} else {
		fmt.Printf("Server only received %d messages.  Expected %d X %d\n",
			necho, ts.Nclients, ts.Nmessages)
		ts.ServerChan <- false
	}
	return
}

// Create client.  Have it generate messages and check that they are echoed.
func (ts *AuxTestSystem) BuildClient(clienti int) {
	if !ts.GenerateClient(clienti) {
		fmt.Printf("Couldn't create client %d on port %d\n", clienti, ts.Port)
		ts.ClientChan <- false
		return
	}
	cli := ts.Clients[clienti]
	var mcount int
	for mcount = 0; ts.RunFlag && mcount < ts.Nmessages; mcount++ {
		tv := mcount * 100 + rand.Intn(100)
		cli.Write(auxi2b(tv))
		b := cli.Read()
		if b == nil {
			fmt.Printf("Client %d detected server termination after only %d messages\n",
				clienti, mcount)
			ts.ClientChan <- false
			// Shut down client after completion
			cli.Close()
			return
		}
		v := auxb2i(b)
		if v != tv {
			fmt.Printf("Client %d got %d.  Expected %d\n",
				clienti, v, tv)
			ts.ClientChan <- false
			// Shut down client after completion
			cli.Close()
		}
	}
	fmt.Printf("Client #%d completed %d messages\n", clienti, ts.Nmessages)
	if ts.Mode == doclientstop {
		// Shut down client
		fmt.Printf("Stopping client\n")
		cli.Close()
		ts.ClientChan <- true
		return
	}
	if ts.Mode == doserverstop {
		b := cli.Read()
		if b == nil {
			fmt.Printf("Client %d detected server termination after %d messages\n",
				clienti, mcount)
			ts.ClientChan <- true
			// Shut down client after completion
			cli.Close()
			return
		} else {
			fmt.Printf("Client %d received unexpected data from serve\n",
				clienti)
			ts.ClientChan <- false
			// Shut down client after completion
			cli.Close()
			return
		}
	}
	ts.ClientChan <- true
	cli.Close()
}

// Time Out test
func (ts *AuxTestSystem) runtimeout(timeouts float32) {
	// Delay for 1 second
	auxdelay(timeouts)
	ts.RunFlag = false
	ts.TimeChan <- true
}

func TestSlowStart1(t *testing.T) {
	SetVerbose(6)
	SetEpochLength(0.5)
	NewAuxTestSystem(t, 1, doslowstart, 5 * epochlength)
	SetEpochLength(2.0)
	SetVerbose(0)

}

func TestSlowStart2(t *testing.T) {
	SetVerbose(1)
	SetEpochLength(0.5)
	NewAuxTestSystem(t, 3, doslowstart, 5 * epochlength)
	SetEpochLength(2.0)
	SetVerbose(0)

}

func TestStopServer1(t *testing.T) {
	SetVerbose(1)
	SetEpochLength(0.5)
	// Cannot do stop server for more than 1 client
	NewAuxTestSystem(t, 1, doserverstop, 10 * epochlength)
	SetEpochLength(2.0)
	SetVerbose(0)
}

func TestStopClient1(t *testing.T) {
	SetVerbose(2)
	SetEpochLength(0.5)
	NewAuxTestSystem(t, 1, doclientstop, 10 * epochlength)
	SetEpochLength(2.0)
	SetVerbose(0)

}

func TestStopClient2(t *testing.T) {
	SetVerbose(2)
	SetEpochLength(0.5)
	NewAuxTestSystem(t, 3, doclientstop, 15 * epochlength)
	SetEpochLength(2.0)
	SetVerbose(0)
}
