// Automated testing of LSP support code

package lsp

import (
	"fmt"
	"json"
	"rand"
	"testing"
	"time"
)

// General parameters
var DefaultVerbosity = 0


// Convert integer to byte array
func i2b(i int) []byte {
	b, _ := json.Marshal(i)
	return b
}

// Convert byte array back to integer
func b2i(b []byte) int {
	var i int
	json.Unmarshal(b, &i)
	return i
}

// Use channel for synchronization
type CommChan chan int

// Test components
type TestSystem struct {
	Server *LspServer
	Clients []*LspClient
	RunFlag bool  // Set to false to get clients and server to stop
	Cchan CommChan   // Use to synchronize
	Nclients int
	Nmessages int
	Maxsleeps float32
	Tester *testing.T
	Description string
	DropRate float32
}

func (ts *TestSystem) SetMaxsleeps(v float32) {
	ts.Maxsleeps = v
}

func (ts *TestSystem) SetNmessages(n int) {
	ts.Nmessages = n
}

func (ts *TestSystem) SetDropRate(r float32) {
	ts.DropRate = r
	SetDropRate(r)
}

func (ts *TestSystem) SetDescription(t string) {
	ts.Description = t
}

// Delay for fixed amount of time
func delay(sleeps float32) {
	if sleeps > 0.0 {
		t := int64(sleeps * 1e9)
		time.Sleep(t)
	}
}

// Delay for a random amount of time up to limit
func rdelay(maxsleeps float32) {
	delay(rand.Float32() * maxsleeps)
}

func NewTestSystem(t *testing.T, nclients int) *TestSystem {
	t.Logf("Testing: %d clients", nclients)
	ts := new(TestSystem)
	// Try to create server for different port numbers
	ntries := 5
	port := 0
	var errmsg string = "Failed to create server on ports"
	var i int
	for i = 0; i < ntries && ts.Server == nil; i++ {
		rand.Seed(time.Nanoseconds())
		port = 2000 + rand.Intn(1000)
		ts.Server = NewLspServer(port)
		errmsg = fmt.Sprintf("%s %d", errmsg, port)
	}
	if ts.Server == nil {
		t.Logf(errmsg)
		return nil
	}
	// Create clients
	ts.Clients = make([]*LspClient, nclients)
	for i = 0; i < nclients; i++ {
		ts.Clients[i] = NewLspClient("localhost", port)
		if ts.Clients[i] == nil {
			t.Logf("Couldn't open client on port %d", port)
			return nil
		}
	}
	ts.RunFlag = true
	ts.Cchan = make(CommChan, nclients + 1)
	ts.Nclients = nclients
	ts.Nmessages = 50
	ts.Maxsleeps = 0.0
	ts.Tester = t
	ts.Description = ""
	return ts
}


// Set up server as an echo.

func (ts *TestSystem) runserver() {
	ndead := 0;
	for ts.RunFlag {
		id, data := ts.Server.Read()
		if data == nil {
			ndead ++
		} else {
			rdelay(ts.Maxsleeps)
			ts.Server.Write(id, data)
		}
	}
}

// Have client generate n messages and check that they are echoed.
func (ts *TestSystem) runclient(clienti int) {
	if clienti > ts.Nclients {
		ts.Tester.Logf("Invalid client number %d\n", clienti)
		ts.Tester.FailNow()
	}
	cli := ts.Clients[clienti]
	for i := 0; i < ts.Nmessages && ts.RunFlag; i++ {
		wt := rand.Intn(100)
		cli.Write(i2b(i+wt))
		v := b2i(cli.Read())
		if v != wt + i {
			ts.Tester.Logf("Client got %d.  Expected %d\n",
				v, i+wt)
			ts.RunFlag = false
			ts.Tester.FailNow()
		}
	}
	fmt.Printf("Client #%d completed %d messages\n", clienti, ts.Nmessages)
	ts.Cchan <- 0
}

// Time Out test by sending -1 to Cchan
func (ts *TestSystem) runtimeout(timeouts float32) {
	// Delay for 1 second
	delay(timeouts)
	ts.Cchan <- -1
}

func (ts *TestSystem) runtest(timeouts float32) {
	if ts.Description != "" {
		fmt.Printf("Testing: %s\n", ts.Description)
	}
	go ts.runserver()
	for i := 0; i < ts.Nclients; i++ {
		go ts.runclient(i)
	}
	go ts.runtimeout(timeouts)
	for i := 0; i < ts.Nclients; i++ {
		v := <- ts.Cchan
		if v < 0 {
			ts.RunFlag = false
			ts.Tester.Logf("Test timed out after %f secs\n",
				timeouts)
			ts.Tester.FailNow()
		}
	}
	ts.RunFlag = false
	fmt.Printf("Passed: %d clients, %d messages/client, %.2f maxsleep, %.2f drop rate\n",
		ts.Nclients, ts.Nmessages, ts.Maxsleeps, ts.DropRate)
}


func TestBasic1(t *testing.T) {
	SetVerbose(2)
	ts := NewTestSystem(t, 1)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Short client/server interaction")
	ts.SetNmessages(3)
	ts.runtest(1.0)
}

func TestBasic2(t *testing.T) {
	SetVerbose(2)
	ts := NewTestSystem(t, 1)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Long client/server interaction")
	ts.SetNmessages(50)
	ts.runtest(1.0)
}

func TestBasic3(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	ts := NewTestSystem(t, 2)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Two client/one server interaction")
	ts.SetNmessages(50)
	ts.runtest(1.0)
}

func TestBasic4(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	ts := NewTestSystem(t, 10)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("10 clients, long interaction")
	ts.SetNmessages(50)
	ts.runtest(2.0)
}

func TestBasic5(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	ts := NewTestSystem(t, 4)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Random delays by clients & server")
	ts.SetNmessages(10)
	ts.SetMaxsleeps(0.1)
	ts.runtest(15.0)
}

func TestSendReceive1(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(5.0)
	ts := NewTestSystem(t, 1)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("No epochs.  Single client")
	ts.SetNmessages(6)
	ts.runtest(5.0)
	SetEpochLength(2.0)
}

func TestSendReceive2(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(5.0)
	ts := NewTestSystem(t, 4)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("No epochs.  Multiple clients")
	ts.SetNmessages(6)
	ts.runtest(5.0)
	SetEpochLength(2.0)
}

func TestSendReceive3(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(10.0)
	ts := NewTestSystem(t, 4)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("No epochs.  Random delays inserted")
	ts.SetNmessages(6)
	ts.SetMaxsleeps(0.1)
	ts.runtest(10.0)
	SetEpochLength(2.0)
}


func TestRobust1(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(0.05)
	SetEpochCount(20)
	ts := NewTestSystem(t, 1)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Single client.  Some packet dropping")
	ts.SetDropRate(0.1)
	ts.SetNmessages(10)
	ts.runtest(10.0)
	ts.SetDropRate(0.0)
	SetEpochLength(2.0)
	SetEpochCount(5)
}

func TestRobust2(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(0.05)
	SetEpochCount(20)
	ts := NewTestSystem(t, 3)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Three clients.  Some packet dropping")
	ts.SetDropRate(0.1)
	ts.SetNmessages(10)
	ts.runtest(15.0)
	ts.SetDropRate(0.0)
	SetEpochLength(2.0)
	SetEpochCount(5)
}

func TestRobust3(t *testing.T) {
	SetVerbose(DefaultVerbosity)
	SetEpochLength(0.05)
	SetEpochCount(20)
	ts := NewTestSystem(t, 4)
	if ts == nil {
		t.FailNow()
	}
	ts.SetDescription("Four clients.  Some packet dropping")
	ts.SetDropRate(0.1)
	ts.SetNmessages(8)
	ts.runtest(15.0)
	ts.SetDropRate(0.0)
	SetEpochLength(2.0)
	SetEpochCount(5)
}
