/*

  driver.c

  Driver program for matrix multiply functions.

  DO NOT MODIFY ANYTHING ELSE IN THIS FILE

*/

#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "defs.h"
#include "mxm.h"

#define MICROSECONDS_PER_SECOND 1000000
#define BYTES_PER_MEGABYTE 1048576

typedef struct {
  int rows;
  int cols;
} dim_t;

static dim_t test_dim_A[] = {
  {7, 7},
  {32, 32},
  {64, 64},
  {103, 103},
  {128, 128},
  {256, 256},
  {61, 253},
  {137, 57}};

static dim_t test_dim_B[] = {
  {7, 7},
  {32, 32},
  {64, 64},
  {103, 103},
  {128, 128},
  {256, 256},
  {253, 253},
  {57, 192}};

/* Pre-declaration of print() */

void print(array_t A);

void
create(array_t *a_p,
       int a_rows,
       int a_cols,
       array_t *b_p,
       int b_rows,
       int b_cols,
       array_t *c_p)
{
  int i, j;

  ARRAY_CREATE(*a_p, a_rows, a_cols);
  for (i = 0; i < a_rows; i++){
    for (j = 0; j < a_cols; j++){
      ARRAY_ELEM(*a_p, i, j) = i + j;
    }
  }

  ARRAY_CREATE(*b_p, b_rows, b_cols);
  for (i = 0; i < b_rows; i++){
    for (j = 0; j < b_cols; j++){
      ARRAY_ELEM(*b_p, i, j) = i - j;
    }
  }

  /* We can deduce the size of the c array. */

  ARRAY_CREATE(*c_p, a_rows, b_cols);
}

void
destroy(array_t *a_p,
	array_t *b_p,
	array_t *c_p)
{
  ARRAY_DESTROY(*a_p);
  ARRAY_DESTROY(*b_p);
  ARRAY_DESTROY(*c_p);
}

void
verify(array_t A,
       array_t B,
       array_t C)
{
  int i, j, k, sum;

  if (ARRAY_ROWS(C) != ARRAY_ROWS(A)) {
    printf("dimension mismatch: C has %d rows, A has %d rows\n",
	   ARRAY_ROWS(C),
	   ARRAY_ROWS(A));
    exit(EXIT_FAILURE);
  }

  if (ARRAY_COLS(C) != ARRAY_COLS(B)) {
    printf("dimension mismatch: C has %d cols, B has %d cols\n",
	   ARRAY_COLS(C),
	   ARRAY_COLS(B));
    exit(EXIT_FAILURE);
  }

  for (i = 0; i < ARRAY_ROWS(C); i++){
    for (j = 0; j < ARRAY_COLS(C); j++){
      sum = 0;
      for (k = 0; k < ARRAY_COLS(A); k++){
	sum += ARRAY_ELEM(A, i, k) * ARRAY_ELEM(B, k, j);
      }
      if (ARRAY_ELEM(C, i, j) != sum){
	printf("error in multiplication: C[%d][%d] = %d, "
	       "correct value is %d\n",
	       i, j, ARRAY_ELEM(C, i, j),
	       sum);
	print(A);
	printf("\n");
	print(B);
	printf("\n");
	print(C);
	printf("\n");
	exit(EXIT_FAILURE);
      }
    }
  }
}

void
print(array_t A)
{
  int i, j;

  for (i = 0; i < ARRAY_ROWS(A); i++){
    for (j = 0; j < ARRAY_COLS(A); j++){
      printf("%d ",ARRAY_ELEM(A, i, j));
    }
    printf("\n");
  }  
}

int
main(int argc, char *argv[])
{
  struct timeval t_start, t_end;
  array_t A, B, C;
  double time, throughput;
  int test_num;

  if (argc < 2) {
    printf("insufficient arguments to %s: need to supply a test number\n",
	   basename(argv[0]));
    exit(EXIT_FAILURE);
  }

  test_num = atoi(argv[1]);
  if (test_num < 1 || test_num > sizeof(test_dim_A) / sizeof(dim_t)) {
    printf("invalid argument to %s: need to supply a test number between "
	   "1 and %d\n",
	   basename(argv[0]),
	   sizeof(test_dim_A) / sizeof(dim_t));
    exit(EXIT_FAILURE);
  }
  test_num--;

  printf("Team %s test %d\n", team.team, (test_num + 1));
  printf("C-rows %d C-cols %d\n",
	 test_dim_A[test_num].rows,
	 test_dim_B[test_num].cols);

  /* Cache test. */

  create(&A,
	 test_dim_A[test_num].rows,
	 test_dim_A[test_num].cols,
	 &B,
	 test_dim_B[test_num].rows,
	 test_dim_B[test_num].cols,
	 &C);
  mxm(A,B,C);
  verify(A, B, C);
  destroy(&A, &B, &C);

  /* Throughput test. */

  create(&A,
	 test_dim_A[test_num].rows,
	 test_dim_A[test_num].cols,
	 &B,
	 test_dim_B[test_num].rows,
	 test_dim_B[test_num].cols,
	 &C);

  /* Run it once to warm it up. */

  mxm_through(A,B,C);

  /* ... and again to time it. */

  gettimeofday(&t_start, NULL);
  mxm_through(A,B,C);
  gettimeofday(&t_end, NULL);

  verify(A, B, C);
  destroy(&A, &B, &C);

  time = (double)(((t_end.tv_sec - t_start.tv_sec)*MICROSECONDS_PER_SECOND) +
		  (t_end.tv_usec - t_start.tv_usec))/MICROSECONDS_PER_SECOND;
  throughput = ((ARRAY_ROWS(C) * ARRAY_COLS(C) * sizeof(int)) / time )
    / BYTES_PER_MEGABYTE;
  printf("Throughput %f\n", throughput);

  exit(EXIT_SUCCESS);
}
