;;  CogTool Copyright Notice and Distribution Terms
;;  CogTool 1.0, Copyright (c) 2005-2006 Carnegie Mellon University
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).
;; 
;;  CogTool is free software; you can redistribute it and/or modify
;;  it under the terms of the GNU Lesser General Public License as published by
;;  the Free Software Foundation; either version 2.1 of the License, or
;;  (at your option) any later version.
;; 
;;  CogTool is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU Lesser General Public License for more details.
;; 
;;  You should have received a copy of the GNU Lesser General Public License
;;  along with CogTool; if not, write to the Free Software
;;  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
;;
;;  
;;  CogTool makes use of several third-party components, with the 
;;  following notices:
;;
;;  Eclipse SWT 3.1
;;  Eclipse GEF Draw2D 3.1
;;
;;  Unless otherwise indicated, all Content made available by the Eclipse 
;;  Foundation is provided to you under the terms and conditions of the Eclipse 
;;  Public License Version 1.0 ("EPL"). A copy of the EPL is provided with this 
;;  Content and is also available at http://www.eclipse.org/legal/epl-v10.html.
;;
;;  clisp 2.35
;;
;;  Copyright (c) 1992-2004 Bruno Haible and Michael Stoll
;;  This software is distributed under the terms of the FSF Gnu Public License.
;;  See COPYRIGHT file in clisp installation folder for more information.
;;  
;;  ACT-R 5.1
;;
;;  Copyright (c) 1998-2005 Dan Bothell, Mike Byrne, Christian Lebiere & 
;;                          John R Anderson. 
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).
;;
;;  Apache Batik 1.6
;;  Apache Jakarta Commons-Collections 3.1
;;                        -Concurrent 1.3.2
;;                        -Lang 2.1
;;                        -Logging 1.6.2
;;
;;  This product contains software developed by the Apache Software Foundation
;;  (http://www.apache.org/)
;;
;;  Hibernate 3.0.5
;;
;;  This software is distributed under the terms of the FSF Lesser
;;  Gnu Public License (see LGPL.txt).

(defvar *klm-time* 0.0)

;; Debug flag -- set to 1 for basic tracing, 2 for verbose, and 0 for none
(defvar *cogtool-debug* 0)

(defmacro cogtool-debug (level message &rest args)
  ;; Note that this is a macro, so neither message nor args are eval'd
  ;; unless we are at or above the given debug level.
  `(when (>= *cogtool-debug* ,level)
     (format t "~&~5TCogTool: ~@?~%" ,message ,@args)))

;; Set the infer transition's flag to try and infer which transition to follow
;; this is broken.
(defvar *infer-transitions* nil )

;; Declare a global variable, in which we will store the current design
(defvar *cogtool-design* nil)

;; Set up a disembodied plist to pass side-band information from CogTool.
(defvar *cogtool-plist* nil)

(defmacro get-cogtool-plist (key)
  `(getf *cogtool-plist* ,key))

;; When updating the plist we defer changes to :destination-frame, as it needs to be kept around in
;; some circumstances.
(defun set-cogtool-plist (&rest keys 
			  &key (destination-frame (getf *cogtool-plist* :destination-frame) df-supplied)
			  &allow-other-keys)
  (setf *klm-time* (pm-time))
  (unless df-supplied
    (setq keys (list* :destination-frame destination-frame keys)))
  (setq *cogtool-plist* keys))

(defun clear-cogtool-plist-element (key)
  (when (get-cogtool-plist key)
    (let ((result (copy-seq *cogtool-plist*)))
      (setf (getf result key) nil)
      (setq *cogtool-plist* result))))

(defun set-cogtool-delay (seconds)
  (setf (delay-until *cogtool-design*) (+ *klm-time* (max 0 seconds))))


;;;;;;;;;; Define cogtool model classes

(defclass cogtool-device (device-interface) 
  ;; init position really happens in act-simple.lisp "(run-model)"
  ((frames :accessor frames :initarg :frames :initform 
           (make-hash-table :test #'equal)
           :documentation "hashtable of (string => frame)")
   (cursor-x :accessor cursor-x :initarg :cursor-x :initform 0)
   (cursor-y :accessor cursor-y :initarg :cursor-y :initform 0)
   (curframe :accessor curframe :initarg :curframe :initform nil
             :documentation "the currently-active frame")
   (curwidget :accessor curwidget :initarg :curwidget :initform nil
              :documentation "the widget currently under the cursor")
   (curkeys :accessor curkeys :initarg :curkeys :initform nil
            :documentation "the list of keys pressed thus far")
   (curgraffiti :accessor curgraffiti :initarg :curgraffiti :initform nil
            :documentation "the list of graffiti strokes entered thus far")
   (curclicks :accessor curclicks :initarg :curclicks :initform nil
            :documentation "the list of clicks thus far")
   (delayedframe :accessor delayedframe :initarg :delayedframe :initform nil
                 :documentation
                 "the frame that should be activated at delay-until")
   (delay-until :accessor delay-until :initarg :delay-until :initform 0
                :documentation 
                "the earliest time that the next visual icon update can occur")
   ;; the following slot is ignored, and just included here to ease
   ;; compatibility with the ACT-R 6 version
   (initial-frame-name :initarg :initial-frame-name)
  )
)

(defclass cogtool-frame () 
  ((name :accessor name :initarg :name :initform ""
         :documentation "name of the frame")
   (widgets :accessor widgets :initarg :widgets :initform nil
            :documentation "list of widgets")
   (voicecoms :accessor voicecoms :initarg :voicecoms
              :initform (make-hash-table :test #'equal))
   (keycoms :accessor keycoms :initarg :keycoms
            :initform (make-hash-table :test #'equal))
  )
)

;;;         TODO: all except label are just rects for now
;;;         TODO: explictly declare/set widget being acted upon
(defclass cogtool-widget () 
  ((name :accessor name :initarg :name :initform ""
         :documentation "name of the widget")
   (title :accessor title :initarg :title :initform ""
         :documentation "title of the widget")
   (x :accessor x :initarg :x :initform 0) ; x,y are the upper left corner, not center!
   (y :accessor y :initarg :y :initform 0)
   (width :accessor width :initarg :width :initform 0)
   (height :accessor height :initarg :height :initform 0)
   (level :accessor level :initarg :level :initform 0)
   (wtype :accessor wtype :initarg :wtype :initform "cogtool-button")
   (transitions :accessor transitions :initarg :transitions 
                :initform (make-hash-table :test #'equal)
                :documentation "hashtable of (action => transition)")
  )
)

(defclass cogtool-transition () 
  ((target :accessor target :initarg :target :initform ""
           :documentation "name of the target frame")
  )
)

;; TODO: change ACT-R handling to respect buttons and up/down actions
;; TODO: respect modifier state
;; TODO: change key handling to recognize key modifiers and press type

;;;;;;;;;; Actual ACT-R device-interface methods

(defmethod device-update ((design cogtool-device) time)
  ;; TODO: handle various types of timed triggers
  (cogtool-debug 1 "Device update at time: ~D" time)
  (if (< time (delay-until design))
      ;; a delay is set, and we need to wait
      (progn
        ;; Clear the visual icon, if it hasn't been already
        (if (curframe design)
          (progn
	    (cogtool-debug 1 "Delaying frame: ~A at time: ~D to time: ~D" 
                        (curframe design) time (delay-until design))
                        
            (setf (delayedframe design) (curframe design))
            (setf (curframe design) nil)
            (pm-proc-display)
          )
          (progn
	    (cogtool-debug 1 "Still waiting for delay to expire")
            (pm-proc-display)
            )
        )
      )
      ;; the delay has expired
      (progn
	(cogtool-debug 2 "time's up for frame: ~A" (delayedframe design))
        ;; if a frame has been delayed, set it now
        (if (delayedframe design)
          (progn
	    (cogtool-debug 1 "Restoring delayed frame at time: ~D" 
                      time)
            (setf (curframe design) (delayedframe design))
            (setf (delayedframe design) nil)
            (pm-proc-display)
          )
          ;; On the start up case where delayedFrame abd curFrame are null, get the
          ;; cur frame from the plist
          
            (if (curframe design) 
              () ;; Do nothing
              (if *cogtool-plist*
              (progn
		(cogtool-debug 2 "Transitioning to ~A" (get-cogtool-plist :current-frame))
                (do-transition (get-cogtool-plist :current-frame) design)
              ))
            )
          
        )
      )
  )
)

(defmethod build-features-for ((design cogtool-device) (vis-mod vision-module))
  (cogtool-debug 2 "Calling build-features-for ~A" (curframe design))
  (if (not (curframe design))
      ;; if there is no curframe, clear the visual module
      (progn
	(cogtool-debug 1 "Clearing visual icon at time: ~D" (pm-time))
        (clear vis-mod)
        nil
      )
      
      ;; there is a current frame, so tell the vis-mod about all widgets
      (let ((features nil))
	(cogtool-debug 1 "Building visual icon at time: ~D" 
                  (pm-time))
        (dolist (widget (widgets (curframe design)) features)
          ;; NOTE: this reverses the widget-list
	  (cogtool-debug 2 "Making feature for widget: ~A" (name widget))
          (setf features (cons (make-feature widget design vis-mod) features))
        )
      )
  )
)

(loop for wtype in '(cogtool-menu cogtool-submenu cogtool-menu-item cogtool-text
		     cogtool-text-box cogtool-pull-down-list cogtool-pull-down-item
		     cogtool-list-box-item cogtool-radio-button cogtool-checkbox
		     cogtool-non-interactive cogtool-graffiti)
    do (setf (get wtype 'surrogate-widget-type) 'cogtool-button))

;; Helper method for build-features-for -- turns a widget into an icon-feature
;; cogtool-widget -> icon-feature
(defun make-feature (widget design vis-mod)
  ;; use the vision-module.lisp classes and functions, including:
  ;;     (build-string-feats)
  ;;     (new-other-sound)
  ;;     (make-instance 'icon-feature :kind 'visual-object)
  ;;
  ;; TODO: support oval-shaped and other widgets
  ;;
  ;; Note that the x,y position for an icon-feature needs to be its
  ;; center rather than its upper left corner.
  ;; XXX: do we care about round here?
  (make-instance 'icon-feature
      :value (name widget)
      :dmo-id (gentemp (name widget))
      :x (+ (x widget) (floor (width widget) 2))
      :y (+ (y widget) (floor (height widget) 2))
      :kind (let ((typ (wtype widget))) (or (get typ 'surrogate-widget-type) typ))
      :screen-obj design
      :height (height widget) 
      :width (width widget)))

;; Sets the cursor to a new location.
;; cogtool-device (vector x y) => nil
(defmethod device-move-cursor-to ((design cogtool-device) (xyloc vector))
  (cogtool-debug 1 "Move cursor to x: ~D y: ~D" 
            (px xyloc) (py xyloc))
  
  ;; save position info
  (setf (cursor-x design) (px xyloc))
  (setf (cursor-y design) (py xyloc))

  ;; Follow a transition if needed.
  (if (not *infer-transitions*) 
          (do-transition (get-cogtool-plist :destination-frame) design  )
  )
   
  (update-cursor design nil)
)

;; Gets the current location of the cursor.
;; cogtool-device => (vector x y)
(defmethod get-mouse-coordinates ((design cogtool-device))
  (cogtool-debug 2 "Getting mouse coords x: ~D y: ~D" 
            (cursor-x design) (cursor-y design))
  (vector (cursor-x design) (cursor-y design))
)

(defmethod device-speak-string ((design cogtool-device) str)
  (cogtool-debug 1 "Speaking string: ~A" str)

  (if (not *infer-transitions*) 
    ( do-transition (get-cogtool-plist :destination-frame) design  )
    
    ;; Do inference
    (progn 
      ;; lookup voice transition by action (voice string)
      (let ((trans (gethash (make-instance 'cogtool-voice-action :command str)
                            (voicecoms (curframe design)))))
        (if trans
          ;; if exists, follow transition
          (follow-trans trans design)
          ;; XXX: no such voice command -- do nothing? report error?
        )
      )
    )
  ) ;; end if
)

(defmethod device-handle-graffiti ((design cogtool-device) key)
  (cogtool-debug 1 "Graffiti: ~A" key)
    
  (if (not *infer-transitions*) 
    ( do-transition (get-cogtool-plist :destination-frame) design  )
    
    ;; do inference
    (progn
      (setf (curgraffiti design) (append (curgraffiti design) (list key)))
      ;; lookup key transition by action (key character list)
      ;; TODO: use modifiers and key press type here
      (let ((trans (gethash (list 'cogtool-graffiti-action 
                              (curgraffiti design))
                            (transitions (curwidget design)))))
        (if trans
          ;; if exists, follow transition
          (progn
	    (cogtool-debug 1 "Recognized Graffiti: ~A" (curgraffiti design))
            (setf (curgraffiti design) nil)
              (follow-trans trans design)
              )
          ;; no such key command -- cache current sequence for future lookup
          (progn
	    (cogtool-debug 1 "Unrecognized Graffiti: ~A" (curgraffiti design))
            )
        )
      )
      
      ;; TODO: lookup keyboard transition by key-press type
      ;;       if exists, follow transition
      ;;       else       check for text-input widget
      ;;                  if exists, put text into widget (maybe a nop?)
    )
  )
)

(defmethod device-handle-keypress ((design cogtool-device) key)
  (cogtool-debug 1 "Keypress: ~A" key)
    
  (if (not *infer-transitions*) 
    (do-transition (get-cogtool-plist :destination-frame) design  )
    
    (progn
    
      (setf (curkeys design) (append (curkeys design) (list key)))
      ;; lookup key transition by action (key character list)
      ;; TODO: use modifiers and key press type here
      (let ((trans (gethash (list 'cogtool-keyboard-action 
                              (curkeys design))
                            (keycoms (curframe design)))))
        (if trans
          ;; if exists, follow transition
          (progn
	    (cogtool-debug 2 "Recognized keys: ~A" (curkeys design))
            (setf (curkeys design) nil)
              (follow-trans trans design)
              )
          ;; no such key command -- cache current sequence for future lookup
          (progn
	    (cogtool-debug 2 "Unrecognized keys: ~A" (curkeys design))
            )
        )
      )
      ;; TODO: check modifier keys state
      ;;       lookup keyboard transition by (key press type)
      ;;       if exists, follow transition
      ;;       else       check for text-input widget
      ;;                  if exists, put text into widget (maybe a nop?)
    )
  )  
)

(defmethod device-handle-click ((design cogtool-device))
  (cogtool-debug 0 "Clicked")
  (if (not *infer-transitions*) 
    ( do-transition (get-cogtool-plist :destination-frame) design  )
    
    (progn

          ;; if mouse is over a widget...
          (let ((curwidget (curwidget design)))
            (if curwidget
              (progn 
		(cogtool-debug 1 "Clicked on widget: ~A" (name curwidget))
                
                ;; lookup transition by action (click)
                ;; TODO: figure out how to differentiate buttons
                (setf (curclicks design) (append (curclicks design) 
                                                 (list (list 'click 'left))))
                (let ((trans (gethash (curclicks design) (transitions curwidget))))
                  (if trans
                    ;; if exists, follow transition
                    (progn
		      (cogtool-debug 2 "Recognized clicks: ~A" (curclicks design))
                      (setf (curclicks design) nil)
                        (follow-trans trans design)
                        )
                    ;; no such click transition
                    ;; -- cache current sequence for future lookup
                    (progn
		      (cogtool-debug 2 "Unrecognized clicks: ~A" (curclicks design))
                      )
                  )
                )
              )
              ;; XXX: click on empty space -- do nothing? report error?
              ;;      this might be caused by a system delay but no (look-at)
	      (cogtool-debug 1 "Clicked on nothing at x: ~D y: ~D"
                  (cursor-x design) (cursor-y design)
              )
            )
        )
    )
  )
)


;;;;;;;;;; Helper methods

;; find a widget by location within a frame
;; number number cogtool-frame => cogtool-widget
(defun lookup-widget (x y frame)
  (cogtool-debug 2 "Looking for a widget at x: ~D y: ~D in frame: ~A"
      x y (name frame)
  )
  ;; stupid theta(n) linear search
  (let ((hits nil))
    (dolist (widget (widgets frame))
      ;; first check x coord
      (if (and (>= x (x widget))
               (< x (+ (width widget) (x widget))))
          ;; x is in the widget box -- check y coord
          (if (and (>= y (y widget))
                   (< y (+ (height widget) (y widget))))
            ;; y is in the widget box -- save to hits list
            (setf hits (cons widget hits))
            ;; y is not in the widget box -- ignore it
          )
          ;; x is not in the widget box -- ignore it
      )
    )
    
    (cogtool-debug 2 "Found hits: ~S" (mapcar #'name hits))
    
    ;; now hits contains all widgets that include the point -- pick one
    ;; XXX: for now, this uses layering only and ignores widget shape
    ;; FIXME: this breaks if there are more than 10,000 widgets on a frame
    (let ((topmost nil)
          (toplevel 10000))
      (dolist (widget hits)
        ;; check if the widget is more frontward than the current top level
        (if (< (level widget) toplevel)
          ;; this widget wins! store it as topmost
          (progn
            (setf toplevel (level widget))
            (setf topmost widget)
          )
        )
      )
      (cogtool-debug 2 "Topmost at level: ~D is widget: ~S" 
          toplevel (if topmost (name topmost))
      )
      topmost
    )
  )
)
  
;; handle a change in cursor, whether because of mouse move or frame change
;; cogtool-device => nil
(defun update-cursor (design suppress-tap) 
  ;; if new location is over a widget...
  (let ((widget (lookup-widget (cursor-x design)
                               (cursor-y design)
                               (curframe design))))
    (if widget
      (progn
	(cogtool-debug 1 "Rolled over widget: ~A" (name widget))
      
        ;; save this widget for later use
        (setf (curwidget design) widget)
        
        (if (not *infer-transitions*) 
          nil ;; do nothing if not inferring transition.
          
          (progn
        
            ;; lookup transition by action (rollover)
            ;; XXX: should rollovers be different for different key modifiers?
            (let ((trans (gethash 'rollover (transitions widget))))
              (if trans
                ;; if exists, follow transition
                (follow-trans trans design)
                ;; no rollover transition -- this must be a 'tap
                ;;                           (but not if it's a Graffiti area)
                (unless (or (eq (wtype widget) 'cogtool-graffiti)
                            suppress-tap)
                  (progn
		    (cogtool-debug 1 "Implicit tap on widget: ~A" (name widget))
                    (device-handle-click design)
                  )
                )
              )
            )
          )
        )
      )
      ;; no widget at mouse location -- clear curwidget
      (progn
        (setf (curwidget design) nil)
	(cogtool-debug 1 "Moved to empty space (normal after a transition)")
        ;; XXX: move-mouse to empty space -- report error?
        ;;      this might happen when system delay has cleared the screen
      )
    )
  )
)

;; follow transition
;; cogtool-transition cogtool-device => nil
(defun follow-trans (trans design)
  (cogtool-debug 2 "follow-trans called on ~S" trans)
    
  ;; immediately perform the transition
  (do-transition (target trans) design)

)

;; Actually perform the transition to the specified frame
;; cogtool-frame cogtool-design => nil
(defun do-transition (frame-name design)
  ;; do the transition
  
  (format t "~&ATTEMPT TO TRANSITION ~A:" frame-name)
  
  ;; only do the tarnsition if not nil.
  (if frame-name      
    ;; get the frame from the name
    (let ((frame (gethash frame-name (frames design))))
    
      (cogtool-debug 1 "Transition to frame: ~A at time: ~D" 
                  (name frame) (pm-time))
        (setf (curframe design) frame)
        
        ;; tell Act-R to update visual Icon
        (pm-proc-display)
        
        ;; (delayed) transitions can change the curwidget!
        (update-cursor design t)
        
        ;; clear the dest frame
        (clear-cogtool-plist-element :destination-frame)
    )
  )
)

;; Actually perform the transition to the specified frame
;; cogtool-frame cogtool-design => nil
;; Simplification of the 2 param and do-transition which does not specify
;; the design. get it from the device-interface
(defun do-transition-default-frame (frame-name)
  ;; do the transition
  (do-transition frame-name (device (device-interface *mp*)) ) 
)

;;;
;;;  Run Code
;;;

;; This is the default set of values passed to sgp below; by setting 
;; *overridden-global-parameters* to a list of alternating keywords and values
;; before klm-p is called, one or more of those default global parameter values
;; can be overridden.
(defparameter *default-global-parameters* '(:v t :ct t :pt nil :ot t :esc t :er t))
(defvar *overridden-global-parameters* '())

(defun cogtool-run-model (start-with-mouse timeout debug &rest load-files)
  (let ((*cogtool-debug* (max *cogtool-debug* debug)))
    (clear-all)
    (pm-reset)
    (chunk-type (cogtool-Text (:include visual-object)) ())
    (chunk-type (cogtool-Button (:include visual-object)) ())
    (chunk-type (drag (:include motor-command)) hand)
    (chunk-type (graffiti-gesture (:include motor-command)) key)
    (sgp-fct (loop :with result := (copy-seq *default-global-parameters*)
		:for (key val) :on *overridden-global-parameters* :by #'cddr
		:do (setf (getf result key) val)
		:finally (return result)))
    (when start-with-mouse 
      (pm-set-hand-location right 28 2)) ;;TODO: why not call (pm-start-hand-at-mouse)?
    (chunk-type klm state)
    (mapcar #'load load-files)
    (install-device *cogtool-design*)
    (add-dm (goal isa klm state 1))
    (goal-focus goal)
    (setf *actr-enabled-p* t)
    (pm-set-params :real-time nil :randomize-time nil :trace-modules t)
    (pm-set-cursor-position 0 0)
    (pm-proc-display)
    (let ((*cogtool-plist*))
      (pm-run timeout))))

;; XXX: Why is this overriding default ACT-R behavior?
(defmethod update-attended-loc ((vis-mod vision-module)) nil)

(defun install-device (&rest args) (apply #'pm-install-device args))
