;; -*- LISP -*-

(in-package :huh)

;;.@chapter Top Level Functions

;;.HUH's top level functions provide a variety of ways of customizing
;;.and modifying parsers and understanders.  The functions described
;;.here handle reading text with and without script-based
;;.expectations.  The functions for reading with out expectations will
;;.be described first; everything said about them also applies to the
;;.understanding with expectations, which is described afterwards.

;;.Three variables are particular useful for modifying the parser:
;;.@table @code
;;.@item *sentence-parser*
;;.This is the default parser (the second argument initially passed to
;;.@code{parse}) used on input sentences.  The default is
;;.@code{HUH::SENTENCE}.
;;.@item *sense-selector*
;;.This is the default function for choosing between multiple parses;
;;.the default is @code{default-sense-selector} which picks the parse
;;.that uses most of the sentence yet involves the smallest
;;.expression.
;;.@item *sentence-evaluator*
;;.This is the function used to evaluate the forms returned by the
;;.parser.  By default it is @code{sentence-eval} which calls
;;.@code{EVAL} and issues a trace statement to
;;.@code{*top-level-parse-trace-stream*}.
;;.@end table

(defparameter *sentence-parser* 'sentence
  "This is the default parser to use on sentences.")
(defparameter *sense-selector* 'default-sense-selector
  "This is the default function for choosing between multiple parses.")
(defparameter *sentence-evaluator* 'sentence-eval
  "This is the default function for evaluating sentence parse results.")

(defvar *top-level-parse-trace-stream* *trace-output*
  "Output stream for top level parse tracing.")

;;.The top level function for reading text is @code{read-text}; it
;;.normally establishes a new discourse-context, but if provided an
;;.optional second argument, uses that as a discourse context to
;;.continue with.  It returns whatever discourse context (new or
;;.continuing) it understood its argument in.
(defun read-text (text &optional context)
  (if context
      (continuing-with context (reading-text text))
    (with-continuity (reading-text text))))

;;.For instance, (note that the pretty printing is implementation dependent)
;;.@example
;;.> (read-text "John is a man.")
;;.; Successfully parsed: JOHN IS A MAN . 
;;.; Translation to LISP:
;;.(HUH::COPY-OBJECT (HUH::CREATE-NAMED-OBJECT (GENSYMBOL (QUOTE WORDS::JOHN))
;;.		      (MEMBER-OF 'HUMANS) (MEMBER-OF 'MEN)
;;.                   (FIRST-NAME 'WORDS::JOHN))
;;.		  (CREATE-OBJECT (MEMBER-OF 'MEN)))
;;.(HUH::STORY MEN.1 JOHN.1)
;;.@end example
;;.which creates two descriptions (a description of JOHN) and a
;;.description of some anonymous man whose properties are copied onto
;;.JOHN.  It also evaluates these descriptions and returns a discourse
;;.context (the list starting with @code{HUH::STORY}) containing both
;;.the prototype and @code{JOHN.1} himself.

;;.@code{Read-text} calls @code{reading-text} which iteratively parses
;;.the input text, one sentence at a time, calling @code{parse} with
;;.@code{*sentence-parser*}, choosing between the returned parses with
;;.@code{*sense-selector*} and evaluating the final result by
;;.@code{*sentence-evaluator*}.

(defun reading-text (text)
  "Iteratively parses sentences from text and selects and evals one of
the parses."
  (let ((input (read-words-from-string text)))
    (loop (if (null input) (return (values)))
      (let ((parse
	     (funcall *sense-selector* (parse input *sentence-parser*))))
	(when parse
	  (when (or *top-level-parse-trace-stream* *parser-trace-stream*)
	    (format (or *top-level-parse-trace-stream* *parser-trace-stream*)
		    "~&\; Successfully parsed: ")
	    (do ((words input (cdr words)))
		((eq words (pr-yet-to-parse parse)))
	      (format T "~A " (car words))))
	  (setq input (pr-yet-to-parse parse))
	  (funcall *sentence-evaluator* (pr-object parse)))
	(unless parse
	  (format T "~&\; Couldn't parse ~S" input)
	  (return input))))))


;;;;.Default methods

;;.The default sense selector, aptly named
;;.@code{default-sense-selector} sorts the senses by the length of
;;.unparsed text (number of words) and the total size (in number of
;;.atoms) of the lisp expression it yielded.  This first criterion
;;.prefers a parse which absorbs all of `John went to the store.'
;;.rather than just an initial segment (like `John went') which is
;;.nonetheless a valid sentence.  The second criterion is an @emph{ad
;;.hoc} characterization of characterizing the `simplest parse' which
;;.particular prefers references to existing objects rather than
;;.constructions of new objects.

(defun default-sense-selector (parses)
  "Picks the simplest parse that uses the most words."
  (and parses (first (sort parses 'prefer-interpretation-p))))

(defun prefer-interpretation-p (parse-1 parse-2)
  "Returns true if PARSE-1 is judged to be a better interpretation than PARSE-2.
Longer parses are preferred over shorter ones and shorter descriptions
are preferred over longer ones.  This is --- of course --- a stopgap measure."
  (or (< (length (pr-yet-to-parse parse-1)) (length (pr-yet-to-parse parse-2)))
      (not (>= (length (pr-yet-to-parse parse-1)) (length (pr-yet-to-parse parse-2))))
      (< (count-atoms (pr-object parse-1)) (count-atoms (pr-object parse-2)))))

(defun count-atoms (list)
  "Counts the atoms in a tree."
  (if (consp list)
      (+ (count-atoms (car list)) (count-atoms (cdr list)))
    1))

;;.The default sentence evaluator, called @code{sentence-eval} prints
;;.the form being evaluated and then evaluates it.

(defun sentence-eval (expression)
  "Evaluates a sentence with a print function."
  (when (or *top-level-parse-trace-stream* *parser-trace-stream*)
    (format (or *top-level-parse-trace-stream* *parser-trace-stream*)
	    "~&\; Translation to LISP: ~S" expression))
  (eval expression))


;;;;.Reading stories.

;;.Reading stories is more complicated; a story has a discourse
;;.context, an expectation context and must connect with the
;;.expectation mechanisms of @pxref{Expecting Stories}.

;;.The top level function for reading stories is @code{read-story}
;;.which takes a text (a string) and (optionally) a script to apply;
;;.if no script is supplied, @code{guess-script} selects one based on
;;.hints determined from words occuring in the story.
;;.@code{Read-story} dynamically binds @code{*expectations*},
;;.@code{*discourse-context*}, and defines @code{*object-introducer*}
;;.and @code{*sentence-evaller*} to interpret with reference to an
;;.existing story.

(defun read-story (text &optional (script (guess-script text)))
  (if (null script) (read-text text)
      (let* ((*expectations* '())
	     (*last-event* nil)
	     (*object-introducer* 'introduce-expected-object)
	     (*sentence-evaluator* 'story-eval)
	     (story (make-unit (gensymbol script)
		      (member-of script)
		      (story-text text)))
	     (*discourse-context* (list story)))
	(declare (special *last-event*))
	(satisfy-expectation `(((member-of ,script))) story)
	(read-text text)
	(assertion story 'discourse-context *discourse-context*)
	(assertion story 'expectation-context *expectations*)
	(assertion story 'last-event *last-event*)
	story)))
;;.@code{Read-story} returns the story description constructed from
;;.the text.  This can then be used to continue the story with more
;;.events or questions via @code{continue-story}.

;;.The function @code{continue-story} takes a story and a text and
;;.understands a text in the expectation and discourse contexts
;;.already set up for the story.

(defun continue-story (story text &optional special-parser)
  (let ((*expectations* (get-value story 'expectation-context))
	(*discourse-context* (get-value story 'discourse-context))
	(*last-event* (get-value story 'last-event))
	(*object-introducer* 'introduce-expected-object)
	(*sentence-evaluator* 'story-eval))
    (declare (special *last-event*))
    (unless special-parser
      (read-text text)
      (assertion story 'discourse-context *discourse-context*)
      (assertion story 'expectation-context *expectations*)
      (assertion story 'last-event *last-event*))
    (if special-parser (parse text special-parser) story)))

;; For debugging.
(defmacro inside-story (story &body body)
  `(let ((*expectations* (get-value ,story 'expectation-context))
	 (*discourse-context* (get-value ,story 'discourse-context)))
    ,@body))

;;.In understanding a story, it is assumed that events mentioned after
;;.each other occur after each other; the @code{story-eval} function
;;.(bound to @code{*sentence-evaluator*} while a story is being read)
;;.asserts @code{starts-after} relations between top level events.
;;.This helps the matching process since part of the knowledge that
;;.allows the matching of events with a script is knowledge about
;;.temporal relations between events.

(defvar *last-event* nil
  "Support for the sequencing of events in stories.")
(define-unit last-event
  (member-of 'single-valued-slots)
  (makes-sense-for 'stories)
  (must-be 'processes))

(defun story-eval (expression)
  (when (or *top-level-parse-trace-stream* *parser-trace-stream*)
    (format (or *top-level-parse-trace-stream* *parser-trace-stream*)
	    "~&\; Translation to LISP: ~S" expression))
  (let ((expression (if (and *last-event* (eq (car expression) 'create-object))
			;; This is an embarassing kludge neecessitated
			;; by the lack of any really good time representation.
			`(create-object ,@(cdr expression) (starts-after ',*last-event*))
			expression)))
    (let ((new-unit (eval expression)))
      (when (and (unitp new-unit)
		 (satisfies? new-unit 'processes) (not (satisfies? new-unit 'things))
		 (not (get-value new-unit 'putative)))
	(setq *last-event* new-unit)))))

