;;; -*- LISP -*-

;;.@chapter Reference

;;.In HUH, `reference' means the manner in which english descriptions
;;.--- once parsed --- are translated into ARLOtje structures.  More
;;.sophisticated story understanding systems gain their power by being
;;.better and better at reference; the underlying parsing mechanisms
;;.are not likely to change.

;;.In the simplest analysis, there are two sorts of reference:
;;.reference to objects which have been mentioned before and
;;.references to objects newly mentioned.  HUH's expectation mechanism
;;.expands this with references to objects which are `expected'; but
;;.this will be discussed when the expectation mechanism is introduced.

;;.The reference mechanism is built around three functions used by
;;.code produced from English by HUH: @code{create-object},
;;.@code{create-named-object}, and @code{extend-object}; and one
;;.parser method @code{discourse-reference} which returns a set of
;;.candidates from existing objects which match a particular
;;.specification.

;;.The function @code{create-object} is a macro which takes any number
;;.of slot/value pairs and creates a corresponding unit; it attempts
;;.to guess a useful name for the unit by looking for the collections
;;.it will be a member of and using the collection name as the root
;;.for generating a symbol.  The function @code{create-named-object}
;;.is like @code{create-object} but its first argument is a symbol
;;.which is used as the name of the created unit.

;;.The macro @code{extend-object} is like @code{create-object} but
;;.with a first argument which is a full-fledged unit upon which the
;;.corresponding slots and values are to be stored.

;;.The parser macro @code{discourse-reference} takes a list of
;;.properties (a single argument) and specifies interpretations
;;.corresponding to each object created by the above functions.  The
;;.user level function @code{find-discourse-references} returns the
;;.list of references.

;;.The scope of references is organized and delimited by `discourse
;;.contexts'; the macro function @code{with-continuity} creates a discourse
;;.context and establishes newly created objects in this context and
;;.searches for unknown references in it.  This macro function returns
;;.a list describing the context and the macro function
;;.@code{continuing-with} takes this structure as a first argument and
;;.evaluates its remaining forms in the same context reestablished.

(in-package :huh)


;;;;.Tracing reference

;;.HUH's reference mechanism can be traced by calling the procedure
;;.@code{trace-reference}; with no arguments, it toggles tracing on
;;.and off, defaulting to the standard output.  With a single
;;.argument, it directs trace output to a particular stream.

(defvar *REFERENCE-TRACE-STREAM* nil
  "The stream to which object reference information goes.")
(defmacro reference-trace (format-string &rest format-args)
  "Prints a trace from the HUH reference mechanism if tracing is activated."
  `(when *REFERENCE-TRACE-STREAM*
    (format *REFERENCE-TRACE-STREAM* "~&\;\;\; ")
    (format *REFERENCE-TRACE-STREAM* ,format-string ,@format-args)
    (force-output  *REFERENCE-TRACE-STREAM*)))

(defun trace-reference (&optional (stream (and (not *REFERENCE-TRACE-STREAM*) *standard-output*)))
  "Turns on reference tracing."
  (if *reference-trace-stream*
      (cond ((eq stream *reference-trace-stream*))
	    (stream (format *reference-trace-stream*
			    "~&\; Switching reference trace from ~S to ~S~&"
			    stream *reference-trace-stream*)
		    (format stream
			    "~&\; Switching reference trace from ~S to ~S~&"
			    stream *reference-trace-stream*))
	    (T (format *reference-trace-stream*
		       "~&\; Switching off reference trace (from ~S)~&"
		       *reference-trace-stream*)))
    (format stream "~&\; Tracing reference on stream ~S~&" stream))
  (setq *REFERENCE-TRACE-STREAM* stream))


;;;;.Discourse contexts

;;.Discourse contexts are simply lists of objects which have been
;;.constructed during their duration; searches for previously referred
;;.to objects look at these objects to see whether they might be the
;;.object of attention.  The LISP variable @code{*discourse-context*}
;;.contains a list whose @code{car} is the symbol @code{STORY} and
;;.whose @code{cdr} is a list of objects mentioned during the contexts
;;.tenure.

(defvar *discourse-context* NIL
  "A structure of objects constructed or referenced during the current discourse.")

;;.The function @code{discourse-reference!} adds an object to the
;;.current discourse context.
(defun discourse-reference! (thing)
  "Adds thing to the discourse context (if there is one)."
  (if *discourse-context*
      (pushnew thing (cdr *discourse-context*)))
  thing)

;;.The macro function @code{with-continuity} creates a new discourse
;;.context and evaluates a number of forms in it.  It finally returns
;;.the assembled context.
(defmacro with-continuity (&body body)
  "Establishes a discourse context for the execution of BODY, returning the final context."
  `(cond
    ((null *discourse-context*)
     (let ((*discourse-context* (list 'STORY)))
       ,@body
       *discourse-context*))
    (T ,@body *discourse-context*)))

;;.The macro function @code{continuing-with} takes a first argument
;;.which is a discourse context and dynamically binds
;;.@code{*discourse-context*} to it.
(defmacro continuing-with (context &body body)
  "Continues the discourse context CONTEXT for the execution of BODY."
  `(let ((*discourse-context* ,context))
     ,@body
     *discourse-context*))

;;.HUH provides the function @code{INTRODUCE!} to interface to various
;;.expectation and understanding systems.  It calls the function
;;.stored in @code{*object-introducer*} to integrate a new object into
;;.some current story; the default value of @code{*object-introducer*}
;;.is @code{DISCOURSE-REFERENCE!} which simply adds the object to the
;;.current discourse context.  

(defvar *object-introducer* 'discourse-reference!
  "This is the function called when new objects are introduced.")

(defun introduce! (object)
  (funcall *object-introducer* object))


;;;;.Creating objects

;;.The macro functions @code{create-object} and
;;.@code{create-named-object} both create objects with particular
;;.properties.  The specification of properties for these objects
;;.allow the assertion of a single computed value on multiple slots by
;;.specifying a list of slot names in the right hand side of the
;;.slot/value pair.  For instance,
;;.@example
;;.(create-object (member-of 'being-moved)
;;.               ((initiator thing-moved)
;;.                (create-object (member-of 'men) (first-name 'ken))))
;;.@end example
;;.creates a member of @code{being-moved} whose @code{initiator} and
;;.@code{thing-moved} slots both contain the single unit created by
;;.the @code{create-object} form.  Note that the obvious translation
;;.@example
;;.(create-object (member-of 'being-moved)
;;.               (initiator
;;.                (create-object (member-of 'men) (first-name 'ken)))
;;.               (thing-moved
;;.                (create-object (member-of 'men) (first-name 'ken)))))
;;.@end example
;;.would make distinct individuals for the two slots.  This interpretation
;;.slot/value pairs is handled by the function @code{sentence-assert}
;;.which is just like @code{assert-value} but that the @var{slot}
;;.argument may actually be a list of slots.

(defmacro create-object (&body properties)
  "Creates a new object with particular properties and adds it to the discourse context."
  `(let ((unit (declare-unit (gensymbol ',(or (guess-name-from-properties properties) 'foobar)))))
    ,@(mapcar #'(lambda (x) `(sentence-assert unit ',(car x) ,(cadr x)))
       properties)
    (multiple-value-bind (unit expected?)  (introduce! unit)
      (if expected? (assert-value unit 'putative NIL)))
    (retraction unit 'putative T)
    unit))

;;.@code{Create-named-object} is just like @code{create-object} except
;;.that it provides for a first argument being a name of the description
;;.being created.
(defmacro create-named-object (name &body properties)
  "Creates a new object with particular properties and adds it to the discourse context."
  `(let ((unit (declare-unit ,name)))
    ,@(mapcar #'(lambda (x) `(sentence-assert unit ',(car x) ,(cadr x)))
       properties)
    (multiple-value-bind (unit expected?)  (introduce! unit)
      (if expected? (assert-value unit 'putative NIL)))
    unit))

;;.Sometimes objects created by parsing will be @emph{putative}
;;.meaning that their existence is assumed rather than explicit; for
;;.instance, various expectation mechanisms will often return objects
;;.annotated in this way.  But objects created by @code{create-object}
;;.and @code{create-named-object} always have this property retracted
;;.(or, more correctly, set to NIL).

;;.@code{Extend-object} takes an object and a list of properties and
;;.assigns those properties to the object.  These properties are just
;;.like the properties passed to @code{create-object} in that their
;;.right hand side may be a list of slot names rather than a single
;;.slot name.
(defmacro extend-object (object &body properties)
  `(let ((obj ,object))
    (reference-trace "Adding properties to ~S: ~S" obj ',properties)
    ,@(mapcar #'(lambda (prop) `(sentence-assert obj ',(car prop) ,(cadr prop)))
       properties)
    obj))

(defmacro copy-object (o1 o2)
  `(funcall *object-copier* ,o1 ,o2))

(defvar *object-copier* 'default-copy-object
  "This is the default function for equating objects (i.e. with IS).")

(defun sentence-assert (unit slot value &rest properties)
  "Asserts a slot value dependent on the current text context."
  (reference-trace "  asserting ~S ~S ~{\; ~S~}" slot value properties)
  (dolist (slot (if (listp slot) slot (list slot)))
    (unless (satisfies? unit (get-value slot 'makes-sense-for))
      (funcall (get-value (get-value slot 'makes-sense-for) 'to-enforce-test)
	       unit (get-value slot 'makes-sense-for)))
    (unless (satisfies? value (get-value slot 'must-be))
      (funcall (get-value (get-value slot 'must-be) 'to-enforce-test)
	       value (get-value slot 'must-be)))
    (apply #'assertion unit slot value properties)))

;;.The function @code{create-new-unit} takes a name and creates the
;;.corresponding unit, recording it in the current discourse context.
(defun create-new-unit (name)
  "Makes a new unit and adds it to the discourse context recording its text context."
  (let ((unit (declare-unit name)))
    (reference-trace "Creating ~S" unit)
    unit))

(defun guess-name-from-properties (properties)
  "Tries to figure out what collections an object created from PROPERTIES would be in."
  (let ((pair (assoc 'member-of properties)))
    (and pair (listp (cadr pair)) (eq (car (cadr pair)) 'quote)
	 (cadr (cadr pair)))))

(defvar *dumb-properties-to-copy* '(creation-id type-code))

(defun default-copy-object (object-1 object-2)
  (reference-trace "Copying properties from ~S to ~S" object-2 object-1)
  (dovalues object-2 (slot value)
    (unless (member slot *dumb-properties-to-copy*)
      (sentence-assert object-1 slot value)))
  object-1)


;;;;.Finding references.

;;.Discourse references are resolved by returning all the entries in
;;.the current discourse context which pass a particular slot
;;.definition.  The properties passed to @code{discourse-reference}
;;.have right hand sides which may be evaluated; the procedure
;;.@code{MATCH?} looks at those properties whose values are definitely
;;.known (i.e. are quoted symbols) and sees if the corresponding
;;.properties hold for objects in the discourse context.

;;.The function @code{find-discourse-reference} returns all such
;;.objects.
(defun find-discourse-reference (properties)
  "Tries to find an object in the current discourse context which has PROPERTIES."
  (let ((refs (remove-if-not #'(lambda (x) (match? x properties))
			     (cdr *discourse-context*))))
    (if refs (reference-trace "Found references ~S for ~S" refs properties)
      (reference-trace "Failed to find references for ~S" properties))
    refs))

;;.The parser macro @code{discourse-reference} returns a set of
;;.interpretations corresponding to each match for use by the parser.
(define-parser-macro discourse-reference (properties)
  `(mapcar #'(lambda (ref) (parse-result ref sentence)) (find-discourse-reference ,properties)))

;;.The function @code{match} takes an object and a list of properties
;;.(as passed to @code{discourse-reference}) and returns those objects
;;.which satisfy all of them that are evaluable.
(defun match? (object properties)
  (every #'(lambda (prop) 
	     (if (and (listp (cadr prop)) (eq (car (cadr prop)) 'quote))
		 (if (listp (car prop))
		     (every #'(lambda (p)
				(and (satisfies? object (get-value p 'makes-sense-for))
				     (query object p (cadr (cadr prop)))))
			    (car prop))
		     (and (satisfies? object (get-value (car prop) 'makes-sense-for))
			  (query object (car prop) (cadr (cadr prop)))))
	       T))
	 properties))


