;;; -*- LISP -*-

;;.@chapter Expecting Stories

;;.HUH's script-based story understanding is a simplified version of
;;.the script understanding used in systems like Schank's SAM.  It is
;;.organized around having a set of expectations each of which
;;.corresponds to one slot of a `story' structure.

;;.The basic element in story understanding is a @dfn{story} which is
;;.an ARLO unit in a collection identified as a @dfn{script}.  Each
;;.@code{script} has a variety of slots which make sense for it; as a
;;.story is understood, the expectation system tries to place events,
;;.settings, and characters into these slots.  The systems constructs
;;.an expectation context of the slots expecting to be filled, what
;;.sorts of things should fill them and (in particular) the relations
;;.which their fillers should bear to other elements of the story.

;;.For instance, in understanding a story about going to restaurants,
;;.the expectation system might expect two players: a patron and a
;;.waitperson.  It also expects there to be some food and money and
;;.that these are likely to change hands, with the waiter giving the
;;.food to the patron and the patron giving the money to the
;;.waitperson.  This relational knowledge is crucial in figuring out
;;.who is who in filling in a script.

(in-package :huh)

;;.While a story is being understood, you can trace the activities of
;;.the expectation mechanism by the function @code{trace-expectation};
;;.it announces the internal activities of the expectation mechanism.
;;.If you give the function a stream as an argument, expectation
;;.information is output to that stream.

(defvar *EXPECTATION-TRACE-STREAM* nil
  "The stream to which object expectation information goes.")
(defmacro expectation-trace (format-string &rest format-args)
  "Prints a trace from the HUH expectation mechanism if tracing is activated."
  `(when *EXPECTATION-TRACE-STREAM*
    (format *EXPECTATION-TRACE-STREAM* "~&\;\;\; ")
    (format *EXPECTATION-TRACE-STREAM* ,format-string ,@format-args)
    (force-output  *EXPECTATION-TRACE-STREAM*)))

;(setq  huh::*trace-expectation-match* T)
(defvar *trace-expectation-match* NIL
  "Whether individual match information is reported.")

(defun trace-expectation (&optional (stream (and (not *EXPECTATION-TRACE-STREAM*) *standard-output*)))
  "Turns on expectation tracing."
  (if *expectation-trace-stream*
      (cond ((eq stream *expectation-trace-stream*))
            (stream (format *expectation-trace-stream*
                            "~&\; Switching expectation trace from ~S to ~S~&"
                            stream *expectation-trace-stream*)
                    (format stream
                            "~&\; Switching expectation trace from ~S to ~S~&"
                            stream *expectation-trace-stream*))
            (T (format *expectation-trace-stream*
                       "~&\; Switching off expectation trace (from ~S)~&"
                       *expectation-trace-stream*)))
    (format stream "~&\; Tracing expectation on stream ~S~&" stream))
  (setq *EXPECTATION-TRACE-STREAM* stream))


;;;;.Representing stories

;;.In order to support story understanding, a collection called
;;.@code{stories} is introduced; its subsets will be various special
;;.stories like @code{restaurant-script}, etc.  All stories have an
;;.@code{expectation-context} (maintained and used by the mechanisms
;;.implemented in this file), a @code{discourse-context} (maintained
;;.and used by the reference mechanisms), and a story text containing
;;.the texts parsed `into' the story.

;;.The two context slots are both single valued slots which take lists
;;.as arguments; while a story is being read, these slot's value are
;;.accessed as the variables @code{*discourse-context*}
;;.@pxref{Discourse contexts} and @code{*expectations*}
;;.@pxref{Fulfilling Expectations}.  The functions for creating and
;;.manipulating stories were describe before @pxref{Top Level
;;.Functions}.

(define-unit stories
  (member-of 'collections)
  (supersets 'processes))

(define-unit expectation-context
  (works-like 'prototypical-slot)
  (makes-sense-for 'stories)
  (must-be 'listp))

(define-unit discourse-context
  (works-like 'prototypical-slot)
  (makes-sense-for 'stories)
  (must-be 'listp))

(define-unit story-text
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'stories)
  (must-be 'stringp))


;;;; Representing Scripts

;;.Scripts are a type of collections; members of scripts always have
;;.the @code{supersets} slot of @code{stories}, meaning that they are
;;.types of stories.  The slots of a script are either
;;.@code{script-roles}, @code{script-settings}, or
;;.@code{script-scenes}.  @code{Script-roles} are filled by
;;.@code{things} (including people, food, etc); @code{script-settings}
;;.are filled by @code{places}; @code{script-scenes} are filled by
;;.@code{processes} including actions.  These all are subsets of the
;;.collection @code{script-slots}; another subset of
;;.@code{script-slots} is the collection @code{script-options}.  The
;;.difference between these is that @code{script-options} are slots
;;.which `fail' unless they are explicitly asserted; other slots will
;;.`automatically fill' if a value is requested from them.

(define-unit scripts
  (member-of 'collections)
  (supersets 'collections)
  (members-have '(supersets stories)))

(define-unit prototypical-script-slot
  (works-like 'prototypical-slot)
  (makes-sense-for 'scripts)
  (to-compute-value 'fill-in-script-slot))
(define-unit script-slots
  (member-of 'collections)
  (generalizations 'slotp)
  (members-have '(works-like prototypical-script-slot)))
(define-unit script-options
  (member-of 'collections)
  (supersets 'script-slots)
  (members-have '(to-compute-value fail)))

(defun fill-in-script-slot (unit slot)
  "Fills in a slot of a script and adds it to the discourse context."
  (declare (ignore unit))
  (break)
  (let ((filler (make-unit (gensymbol slot)
                  (member-of (get-value slot 'must-be)))))
    (unless (failurep (get-value slot 'inverse-slot))
      (assert-value filler (get-value slot 'inverse-slot) unit))
    (introduce! filler)
    filler))

(define-unit script-roles
  (member-of 'collections)
  (supersets 'script-slots)
  (members-have '(must-be things)))

(define-unit script-scenes
  (member-of 'collections)
  (supersets 'script-slots)
  (members-have '(must-be processes)))

(define-unit script-settings
  (member-of 'collections)
  (supersets 'script-slots)
  (members-have '(must-be spatial-extents)))

(define-unit excluded-slots
  (member-of 'many-valued-slots)
  (makes-sense-for 'stories)
  (must-be 'script-slots))

;;.Scripts are associated with stories by keywords.  The slot
;;.@code{script-hints} contains strings whose occurence in a story
;;.suggests a particular script.  These are inversely stored on a list
;;.stored in @code{*script-hints*} which is used by the procedure
;;.@code{guess-script} that takes a text and looks for significant
;;.keywords.  Admittedly, this is a pretty poor method....

(defvar *script-hints* '()
  "Tricks for guessing what script to apply to a story.")

(defun guess-script (text)
  (dolist (hint *script-hints*)
    (when (search (car hint) text)
      (return (cadr hint)))))

(define-unit script-hints
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'scripts)
  (must-be 'stringp)
  (put-demons '(record-script-hint %unit% %value%)))

(defun record-script-hint (script value)
  "Records a hint on *SCRIPT-HINTS*."
  (pushnew (list value script) *script-hints* :test 'equal))


;;;;.Defining new scripts

;;.New scripts can be either defined piecemeal with @code{define-unit}
;;.or @code{define-collection} and @code{define-slot} or by the macro
;;.function @code{define-script}; @pxref{Defining the Restaurant
;;.Script} describes one use of @code{define-script} in detail.

;;.The first argument to @code{define-script} is the name of the
;;.script being defined; the second is a list of `keyword' strings
;;.whose presence in a story indicates that story may be of this type.
;;.Following this are a list of `slot specifications'.  Each slot
;;.specification begins with either a slot name to be defined or one
;;.of a special set of script slot types:
;;.@table @code
;;.@item role
;;.The slot contains a `thing' which is part of the script.
;;.@item setting
;;.The slot contains a `place' which is part of the script.
;;.@item scene
;;.The slot contains an `action' or `process' which is part of the script.
;;.@item optional-role
;;.The slot contains a `thing' which may be part of the script.
;;.@item optional-setting
;;.The slot contains a `place' which may be part of the script.
;;.@item optional-scene
;;.The slot contains an `action' or `process' which may be part of the script.
;;.@end table
;;.If the slot specification begins with one of these types, the
;;.second element of the specification is the name of the
;;.corresponding slot and the remaining elements are slot-value
;;.specifications (just as for @code{define-unit}) specifying
;;.additional properties of the slot.  In particular, this is where
;;.@code{structure} relations are asserted between slots.  Because
;;.these definitions are @code{bundled} together, earlier definitions
;;.can refer to later ones.

(defmacro define-script (name hints &body parts)
  `(aj::bundling
    (define-unit ,name (member-of 'scripts)
                 ,@(mapcar #'(lambda (x) `(script-hints ,x)) hints))
    ,@(mapcar #'(lambda (part) (part->definition part name)) parts)))
(export '(define-script))

(defparameter *slot-types->slot-collections*
  '((role script-roles)
    (scene script-scenes)
    (setting script-settings)
    (optional-role script-roles script-options)
    (optional-scene script-scenes script-options)
    (optional-setting script-settings script-options))
  "This is an a-list mapping script slot types into slot collections.")

(defun part->definition (part name)
  (flet ((same-name-p (x y) (string-equal (symbol-name x) (symbol-name y))))
    (multiple-value-bind (slot-name slot-props)
        (if (assoc (car part) *slot-types->slot-collections* :test #'same-name-p)
            (values (cadr part)
                    (cons `(member-of ',(cadr (assoc (car part) *slot-types->slot-collections*
                                               :test #'same-name-p)))
                          (cddr part)))
            (values (car part) (cdr part)))
      `(define-unit ,slot-name
        (makes-sense-for ',name)
        ,@slot-props))))

;;.For instance, the following might be a simple script for boarding a
;;.subway train:
;;.@example
;;. (define-script boarding-the-subway
;;.     ("subway" " T " "MBTA" "red line" "green line" "orange line")
;;.   (role boarder (member-of 'humans))
;;.   (role train (member-of 'artifacts))
;;.   (role turnstile (member-of 'artifacts))
;;.   (role token (member-of 'artifacts))
;;.   (setting t-station (member-of 'places))
;;.   (scene arriving (member-of 'being-moved)
;;.      (structure '(initiator boarder))
;;.      (structure '(destination t-station))
;;.      (structure '(thing-moved boarder)))
;;.   (scene pay-token (member-of 'being-moved)
;;.      (structure '(initiator boarder))
;;.      (structure '(destination turnstile))
;;.      (structure '(thing-moved token)))
;;.   (scene going-to-train (member-of 'being-moved)
;;.      (structure '(initiator boarder))
;;. 	 (structure '(destination train))
;;. 	 (structure '(thing-moved boarder)))
;;.   (scene train-leaving (member-of 'being-moved)
;;. 	 (structure '(initiator train))
;;. 	 (structure '(origin t-station))
;;. 	 (structure '(thing-moved train))))
;;.@end example

;;;;.Fulfilling Expectations

;;.The variable @code{*expectations*} stores a list of all the
;;.currently expectations held by the system for the current story;
;;.each is a list of the form @code{(props slot unit)} which `means'
;;.that the system is expecting an object having @code{props} which
;;.will fill @code{slot} of @code{unit}.  When a new object is
;;.introduced (i.e. when @code{introduce!} is called on a unit), this
;;.list of expectations is searched and the expectation which has no
;;.@code{props} conflicting with the object and a maximal number of
;;.@code{props} agreeing with the object is selected.

(defvar *expectations* '()
  "This is a list of expectations in the current `story'.")

;;.The function @code{find-expectation} takes an object and returns
;;.the matching expectation; it filters the expectations first to
;;.remove expectations which are no longer valid by checking if the
;;.slot of the expectation still makes sense for the object for which
;;.it is expected.  This is important because the progress of the
;;.story may invalidate some of the expectations which the system
;;.originally had.  Once the valid expectations have been identified,
;;.the function @code{compatability} is called to match the object
;;.with the expectation and the expectation with the maximal
;;.compatability is returned.  If no expectation can be found,
;;.@code{find-expectation} returns NIL.

(defun find-expectation (object)
  "Finds the best expectation matching OBJECT."
  (labels ((invalid-expectation? (expectation)
	     (member (cadr expectation) (get-value (caddr expectation) 'excluded-slots)))
	   (score-with (expectation)
	     (let ((score (if (not (invalid-expectation? expectation))
			      (compatability object (car expectation))
			      0)))
	       (when *trace-expectation-match*
		 (expectation-trace "Match of ~S to ~S of ~S: ~D"
				    object (cadr expectation) (caddr expectation)
				    score))
	       score)))
    (when *expectations*
      (multiple-value-bind (match score) (maximize #'score-with *expectations*)
	(when (> score 0)
	  (expectation-trace "Found expectation (~D) for ~S: ~S of ~S"
			     score object (cadr match) (caddr match))
	  (values match score))))))

;;.The function @code{compatability} takes an object and the list of
;;.properties it expects.  Each property is either of the form
;;.@code{(slot value)} or @code{(slot other-slot object)}.  The first
;;.is matched if the object has a @code{slot} of @code{value}; the
;;.second is matched if the object has a @code{slot} which is the
;;.@code{other-slot} of @code{value}, if such a slot-value has been
;;.explicitly created already.  These properties are computed based on
;;.the @code{structure} slot of script roles, scenes, and settings.
;;.The compatability score is the number of properties which match the
;;.object in question.
(defun compatability (object properties)
  "Computes the compatability of an object with the properties from an expectation."
  (flet ((property-match? (prop)
	   (and (get object (car prop))
		(if (null (cddr prop))
		    (and (satisfies? object (get-value (car prop) 'makes-sense-for))
			 (query object (car prop) (cadr prop)))
		  (and (not (aj::%hasnt (caddr prop) (cadr prop)))
		       (let ((resolves-to (get-value (caddr prop) (cadr prop))))
			 (unless (failurep resolves-to)
			   (setf (cdr prop) (list resolves-to))
			   (and (satisfies? object (get-value (car prop) 'makes-sense-for))
				(query object (car prop) resolves-to)))))))))
    (count-if #'property-match? properties)))

;;.Once an expectation has been found, @code{satisfy-expectation}
;;.places the object in the appropriate slot and also asserts all of
;;.the properties in the expectations @code{props}, though structual
;;.inferences should have asserted most of them as a result of the
;;.original assertion.  It then removes the expectation from
;;.@code{*expectations*} and calls @code{generate-expectations*}.

(defun satisfy-expectation (expectation object)
  "Does the work involved in assuming that OBJECT satisfies EXPECTATION."
  (unless (null (cdr expectation))
    (sentence-assert (caddr expectation) (cadr expectation) object))
  (dolist (prop (car expectation))
    (if (null (cddr prop))
	(sentence-assert object (car prop) (cadr prop))))
  (setq *expectations* (remove expectation *expectations*))
  (generate-new-expectations object)
  object)

;;.@code{Generate-new-expectations} take an object and generates
;;.expectations from it; this is done by going over the objects
;;.@code{possible-slots} and creating an expectation for each one
;;.which satisfies @code{script-slots}.  This expectation has a set of
;;.properties which beings with the @code{must-be} of the slot and
;;.adds each of the structural implications of the slot; its relation
;;.to other slots of the object.  This expectation is then pushed onto
;;.@code{*expectations*}.
(defun generate-new-expectations (realization)
  "Generates new expectations from REALIZATION."
  (let ((new-expectations '()))
    (dolist (prop (get-possible-script-slots realization))
      (push `(((member-of ,(get-value prop 'must-be))
	       ,@(mapcar #'(lambda (rel) `(,(car rel) ,(cadr rel) ,realization))
		  (get-value prop 'structure)))
	      ,prop ,realization)
	    new-expectations))
    (if (null new-expectations)
	(expectation-trace "New expectations: none")
      (let ((*print-pretty* t))
	(expectation-trace "New expectations: ~{~&\;\;\;~{~*~S of ~S~}~}" new-expectations)))
   (setq *expectations* (append new-expectations *expectations*))
   realization))

(defun get-possible-script-slots (object)
  "Returns the script slots which might make sense for this object."
  (let ((possibles '()) (types-considered '()))
    (dolist (collection (get-value object 'minimal-member-of))
      (unless (member collection types-considered)
	(push collection types-considered)
	(dolist (slot (get-value collection 'sensible-slots))
	  (if (satisfies? slot 'script-slots) (pushnew slot possibles)))
	(dolist (speczn (get-value collection 'specializations))
	  (unless (member speczn types-considered)
	    (push speczn types-considered)
	    (dolist (slot (get-value speczn 'sensible-slots))
	      (if (satisfies? slot 'script-slots) (pushnew slot possibles)))))))
      possibles))

(defun maximize (fcn over)
  (let* ((best (car over)) (max (funcall fcn best)))
    (dolist (elt (cdr over))
      (let ((score (funcall fcn elt)))
	(if (>= score max) (setq best elt max score))))
    (values best max)))


;;;;.Connecting to HUH

;;.The functionality above is introduced by defining a function
;;.@code{introduce-expected-object} which will be dynamically bound to
;;.@code{*object-introducer*} when a story is being understood.  
(defun introduce-expected-object (new-unit)
  "Creates a new object with particular properties and adds it to the discourse context."
  (discourse-reference! new-unit)
  (let ((expectation (find-expectation new-unit)))
    (when expectation
      (satisfy-expectation expectation new-unit))
    (values new-unit expectation)))

(define-unit putative
  (english-description
   "Whether this came from the text or from some expectation mechanism.")
  (works-like 'prototypical-primitive-slot)
  (to-get-value 'get)
  (to-put-value 'arlotje::%put)
  (makes-sense-for 'processes)
  (must-be 'booleanp))


;;(defun flesh-out-story (story)
;;  (dolist (slot (get-value story 'possible-slots))
;;    (if (member-of slot 'script-slots)
;;	(get-value story slot)))
;;  story)

