;;; -*- Mode:  LISP; Package: ARLOTJE -*-

;;.@section Speeding up collections
(in-package :arlotje)

;;.The implementation of collections is relatively slow since it has
;;.to look at each element of a unit's @code{member-of} slot and check
;;.it for lattice subsumption.  ARLOtje speeds up this implementation
;;.by using bit vectors (actually, large integers) to represent
;;.membership information.  This implementation operates through a set
;;.of @code{type-mask} and @code{type-id} slots on individual
;;.collections and a @code{type-code} slot on each @code{member-of}
;;.value description.

;;.ARLOtje maintains (in the variable @code{*type-count*}) a counter
;;.of declared collections.  This is used to compute the collection's
;;.@code{type-mask} slot.  The type mask of a collection is the
;;.`inclusive OR' of its generalizations' type masks along with the
;;.bit corresponding to its @code{type-code} set.  The
;;.@code{type-code} of a value description is the `inclusive OR' of
;;.all the type masks of the collections stored as its recorded
;;.elements.

;;.Every collection has a unique identifying integer stored as its
;;.@code{type-id} slot.  This is generated by the defaulting procedure
;;.@code{generate-type-count} based on a counter stored in the
;;.variable @code{*type-count*}.
(define-unit type-id
  (works-like 'prototypical-slot)
  (makes-sense-for 'collectionp)
  (must-be 'integerp)
  (to-compute-value 'generate-type-count))
;;.@vindex{type-id (slot)}

(defvar *type-count* 0)
;;.@findex{*type-count* (variable)}
(defun generate-type-count (unit slot)
  "Generates a TYPE-ID for a collection."
  (declare (ignore unit slot))
  (incf *type-count*))
;;.@findex{generate-type-count (default method)}

;;.The type ids are used to compute the @dfn{type mask} for a
;;.collection.  This type mask is stored in a collection's
;;.@code{type-mask} slot; it is computed by @code{compute-type-mask}
;;.which logically inclusive ORs together the type masks of the
;;.collection's generalizations and finally sets the @code{type-id}th
;;.bit of this product.
(define-unit type-mask
  (works-like 'prototypical-slot)
  (makes-sense-for 'testp)
  (must-be 'integerp)
  (to-compute-value 'compute-type-mask)
  (put-demons '(%put (%get %unit% type-mask) constant-value T)))
;;.@vindex{type-mask (slot)}
;;.The computation of the type mask goes through ARLOtje's standard
;;.slot computation mechanism, so invalidation and updating works.  In
;;.general, a type mask will depend on the @code{type-id} of a
;;.collection and the type masks of each of the collection's
;;.generalizations.

(defun compute-type-mask (unit slot)
  "Computes the type mask for a slot."
  (declare (ignore slot))
  (apply #'logior
	 (if (satisfies? unit 'collections) (ash 1 (get-value unit 'type-id)) 0)
	 (mapcar #'(lambda (x) (get-value x 'type-mask))
		 (get-value unit 'supersets))))
;;.@findex{compute-type-mask (default method)}

;;.Member-of value descriptions have computed @code{type-code} slots
;;.based on the collections of which they are members.  This is simply
;;.the logical inclusive OR of each of the unit's @code{member-of}
;;.values.
(define-unit type-code
  (works-like 'prototypical-slot)
  (makes-sense-for 'unitp)
  (must-be 'integerp)
  (to-compute-value 'compute-type-code))
;;.@vindex{type-code (slot)}

(defun compute-type-code (unit slot)
  "Computes the type code for a unit from its MEMBER-OF slot."
  (declare (ignore slot))
  (dependency! (annotated-value unit 'member-of))
  (apply #'logior 0
	 (let ((masks '()))
	   (do-members (type unit 'member-of)
	     (push (get-value type 'type-mask) masks))
	   masks)))
;;.@findex{compute-type-code (default method)}

;;..It is neccessary to ensure that all of ARLOtje's collections have
;;..type masks before installing the fast implementation of
;;..collections.  Otherwise, attempts to type check the implementation
;;..will yield recursive errors.
(dolist (collection (get-value 'collections 'members))
  (recklessly (get-value collection 'type-mask)))

(defun fast-collection-check-value (unit slot value &rest annotations)
  (declare (ignore slot annotations))
  (and (logbitp (get-value value 'type-id) (get-value unit 'type-code))
       (find-recorded-element unit slot value)))
(put-value 'member-of 'to-find-value 'fast-collection-check-value)

(defun satisfies? (x test)
  "Returns true if X satisfies the test or is in the collection TEST."
  (cond (*reckless?* t)
	((collectionp test)
	 (and (symbolp x) (not (%hasnt x 'creation-id))
	      (logbitp (get-value test 'type-id) (get-value x 'type-code))))
	((testp test) (funcall test x))
	(T (error "~S isn't a valid test" test))))