;;; -*- LISP -*-

;;.@chapter Describing LISP functions
(in-package :arlotje)

;;.ARLOtje is implemented in LISP and parts of its semantics are
;;.defined by lisp procedures attached to one place or another among
;;.its representations of its representations.  Sometimes these LISP
;;.functions are hand coded bits of definition, but often they are
;;.standard formulaic combinations of existing definitions.  ARLOtje
;;.provides mechanisms for describing both these sorts of definitions.

;;.A lisp function named @var{foo} is represented by a unit named
;;.@var{foo}; (actually, since units are currently symbols, the
;;.function name and the unit are identical) with a
;;.@code{LAMBDA-DEFINIITON} slot.  This is a list of the form
;;.@code{(LAMBDA @var{formals} . @var{body})}.  The
;;.@code{COMPILED-DEFINITION} slot of the function contains an
;;.uninterned symbol whose function cell contains a compiled version
;;.of the @code{LAMBDA-DEFINITION}.  Dependency tracking is done on
;;.this value, so if the lambda definition changes (or anything upon
;;.which the lambda definition depended), the compiled definition is
;;.invalidated.

;;.When a unit is given a @code{LAMBDA-DEFINITION} property when the
;;.corresponding function is undefined, it defines a version of the
;;.function which gets and calls the @code{COMPILED-DEFINITION} slot
;;.of the function description; thus, calling a described function
;;.forces an update of its @code{COMPILED-DEFINITION} slot.

;;.On the other side, of these transactions, the
;;.@code{LAMBDA-DEFINITION} slot is often computed by default; this
;;.computation refers to the definition's @code{LISP-CODER} slot.
;;.This slot contains a function which generates a
;;.@code{LAMBDA-DEFINITION} from the unit, accessing the unit's
;;.individual slots.  For instance, the lisp coder
;;.@code{TEST-INTERSECTION} gets the value of a definition's
;;.@code{intersection-of} slot and uses this to construct its
;;.@code{lambda-definition}.  For instance, using @code{define-unit},
;;.we could define a function @code{negative-integerp} thus:
;;.@example
;;.(define-unit negative-integerp
;;.  (lisp-coder 'test-intersection)
;;.  (intersection-of 'minusp)
;;.  (intersection-of 'integerp))
;;.@end example
;;.calling the generated function by a form like
;;.@example
;;.(funcall (get-value 'negative-integerp 'compiled-definition) -3)
;;.@end example
;;.After we have gotten the compiled definition (and thus also the
;;.lambda definition), ARLOtje will automatically define the
;;.corresponding function @code{negative-integerp}, allowing us to say simply
;;.@example
;;.(negative-integerp -3)
;;.@end example

;;.While these functions can be defined with @code{define-unit}, the
;;.preferrable way to define them is with the macro @code{deffcn}
;;.which forces the lambda and compiled definitions to be computed and
;;.also supports the specification of hand coded functions.
;;.@code{Deffcn} has two forms: the `coder' form and the
;;.`hand-crafted' form.  The coder form is just like
;;.@code{define-unit} but with a third argument (before the slot
;;.specifications) specifying the coder to be used:
;;.@example
;;.(deffcn negative-integerp test-intersection
;;.  (intersection-of 'minusp)
;;.  (intersection-of 'integerp))
;;.@end example
;;.the remaining arguments are `define-unit' style slot specifications
;;.(including the `@code{(@var{unit} @var{slot})}' syntax for slot names).

;;.The hand crafted format of @code{deffcn} is just like @code{defun}
;;.syntactically except that the body of the definition may include an
;;.@code{ANNOTATIONS} expression which is a list of slot/value pairs
;;.applying to the definition.  
;;.@example
;;.(deffcn unit-package-name (unit)
;;.  (annotations (domain unitp) (range stringp))
;;.  (package-name (symbol-package unit)))
;;.@end example
;;.which gives the corresponding definition explicit domain and range
;;.information.  Such information may also appear in the lambda
;;.definitions generated by coders.


;;;; Defining functions.

;;.The slot @code{Lisp-Coder} contains the function used to generate
;;.the lambda definition for a function description.  New types of
;;.definitions are constructed by defining new lisp coders.
;;.@vindex{Lisp-Coder (slot)}
(define-unit Lisp-Coder
  (english-description "This is the automatic definition used to compile this.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'function-namep)
  (must-be 'function-namep))

;;.The inverse of the @code{Lisp-coder} slot is the
;;.@code{Coded-definitions} slot which contains all the definitions
;;.generated by a particular coder.  This slot is often used with
;;.ARLOtje's generic inheritance mechanism to give properties to the
;;.definitions yielded by a particular coder.
;;.@vindex{Coded-Definitions (slot)}
(define-unit Coded-Definitions
  (english-description "This is the automatic definition used to compile this.")
  (works-like 'prototypical-set-slot)
  (makes-sense-for 'function-namep)
  (must-be 'function-namep)
  (inverse-slot 'lisp-coder))

;;.For instance, the following coder specifies a predicate over a
;;.range of numbers:
;;.@example
;;.(deffcn range-test (rt)
;;.  (annotations ((coded-definitions associated-inferences)
;;.                   (domain numberp))
;;.                ((coded-definitions associated-inferences)
;;.                   (range booleanp)))
;;.  ;; Here is the actual definition....
;;.  `(lambda (x) (< (get-value rt 'range-bottom) x (get-value rt 'range-top))))
;;.@end example
;;.where the @code{associated-inferences} for a unit define its domain
;;.and range.

;;.Specifying Lambda definitions

;;.The @code{lambda-definition} of a described function contains a
;;.list of the form @code{(LAMBDA @var{formals} . @var{body})}.  This
;;.is given as an argument to the compiler when it defines the
;;.compiled definition.  Storing a lambda definition invokes two put
;;.demons: @code{define-lisp-caller} and @code{record-lambda-annotations}.
;;.@vindex{lambda-definition (slot)}
(define-unit lambda-definition
  (english-description "A described function's lambda definition.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'function-namep)
  (must-be 'lambda-definitionp)
  (to-compute-value 'apply-lisp-coder)
  (put-demons '(define-lisp-caller %unit% %value%))
  (put-demons '(record-lambda-annotations %unit% %value%)))

;;.The @code{lambda-definition} slot only accepts values satisfying
;;.@code{lambda-definitionp}, a predicate which returns true for lists
;;.starting with @code{lambda}.
(defun lambda-definitionp (x)
  "Returns true for valid lambda definitions."
  (and (consp x) (eq (car x) 'lambda)))

;;.When a function's @code{lambda-definition} slot is not specified, it
;;.is defaulted by the procedure @code{apply-lisp-coder} which calls
;;.the @code{lisp-coder} slot of the description on the description to
;;.yield a lambda definition.
(defun apply-lisp-coder (fcn slot)
  "Applies a LISP coder to yield a lambda definition."
  (declare (ignore slot))
  (funcall (get-value fcn 'lisp-coder) fcn))

;;.The procedure @code{define-lisp-caller} is a put demon for
;;.@code{lambda-definition} which sets up a `caller function' to
;;.invoke a described function.  Suppose that a unit @code{FOO} is
;;.given a @code{lambda-definition} slot; providing that @code{FOO}
;;.does not already have a function binding (a special case),
;;.@code{define-lisp-caller} defines a function like the following:@refill
;;.@example
;;.(defun foo (&rest args)
;;.  (apply (get-value 'foo 'compiled-definition) args))
;;.@end example
;;.which gets the compiled definition from a description and then
;;.calls @emph{that} to yield a value.
(defun define-lisp-caller (fcn lambda-def)
  "Defines a lisp function which calls the function defined for FCN."
  (unless (fboundp fcn)
    (let ((lambda-args (cadr lambda-def)))
      (multiple-value-bind (just-args lexpr?) (arglist->args lambda-args)
	(compile fcn `(lambda ,lambda-args
		       (,(if lexpr? 'apply 'funcall)
			(get-value ',fcn 'compiled-definition)
			,@just-args)))))))

(defun arglist->args (arglist)
  (flet ((&keywordp (x) (and (symbolp x) (char= #\& (char (symbol-name x) 0)))))
    (values (mapcar #'(lambda (x) (if (listp x) (car x) x))
		    (remove-if #'&keywordp arglist))
	    (member '&rest arglist))))

;;.The advantage of this --- instead of defining @code{foo} directly
;;.using the given @code{lambda-definition} --- is that defaults
;;.computed using the function will depend on the function's
;;.implementation.  Thus, suppose that computing a default for a slot
;;.@code{bar} ended up calling @code{foo} and at some later point
;;.@code{foo} was transformed to compute something different; if the
;;.procedure @code{foo} actually fetched and executed @code{foo}'s
;;.@code{compiled-definition} slot, then a dependency would be
;;.recorded for the value computed for the @code{bar} slot.

;;.The procedure @code{record-lambda-annotations} allows the user to
;;.specify the properties of a function within the function's lambda
;;.definition.@refill
;;.@example
;;.(deffcn unit-package-name (unit)
;;.  (annotations (domain unitp) (range stringp))
;;.  (package-name (symbol-package unit)))
;;.@end example
;;.which defines a function @code{unit-package-name} and assigns to
;;.its function description (the same description with the
;;.@code{lambda-definition} and @code{compiled-definition}
;;.properties), the properties of @code{domain} and @code{range}
;;.specified in the @code{annotations} form within the body.
(defun record-lambda-annotations (unit lambda-defn)
  "Records the annotations of a lambda definition stored on a unit."
  (dolist (annotations (remove-if-not #'(lambda (x) (and (listp x) (eq (car x) 'annotations)))
				      (cddr lambda-defn)))
    (dolist (annotation (cdr annotations))
      (if (listp (first annotation))
	  (apply #'assert-value unit (apply #'get-value (first annotation))
		 (second annotation) 'depends-on (support-set)
		 (rest (rest annotation)))
	(apply #'assert-value unit (first annotation) (second annotation)
	       'depends-on (support-set) (rest (rest annotation)))))))



;;;;.Compiling Function Descriptions

;;.Definitions are compiled by the Common LISP compile function
;;.applied to their lambda definitions.  The
;;.@code{compiled-definition} slot of a function description contains
;;.an uninterened symbol whose function cell contains a compiled
;;.version of the description's lambda definition.  The name of this
;;.unintered symbol is the same as of the function definition.
(define-unit compiled-definition
  (english-description "A described function's lambda definition.")
  (works-like 'prototypical-slot)
  (makes-sense-for 'function-namep)
  (must-be 'symbolp)
  (to-compute-value 'compile-lambda-definition))

;;.If the variable @code{*compile-deffcn*} is set to @code{nil} (the
;;.default is @code{t}), compiled definitions are not really compiled
;;.but only defined.  This may occasionally be useful for debugging.
(defvar *compile-deffcn* t
  "Whether DEFFCN'd functions are really compiled.")

;;.Compiled definitions are produced by the defaulting function
;;.@code{compile-lambda-definition} which gets a unit's lambda
;;.definition and calls the LISP compiler on it and an uninterned
;;.copy of the unit's name.
(defun compile-lambda-definition (fcn slot)
  "Compiles a functions lambda definition onto a gensym'd symbol."
  (declare (ignore slot))
  (if *compile-deffcn*
      (compile (make-symbol (symbol-name fcn))
	       `(lambda ,@(cdr (get-value fcn 'lambda-definition))))
      (let ((temp-name (make-symbol (symbol-name fcn))))
	(setf (symbol-function temp-name) (get-value fcn 'lambda-definition))
	temp-name)))


;;;;.Major macros

;;.The @code{deffcn} macro is used for specifying both hand-crafted
;;.and automatically coded functions.  For hand-crafted functions, it
;;.looks and acts just like @code{defun} except that it constructs an
;;.`indirect' calling function and sets up a unit description with an
;;.appropriate @code{lambda-definition} slot.
(defmacro deffcn (name args-or-coder &body body)
  "Defines a function description (and function) for ARLOTJE."
  (if (listp args-or-coder)
      (let ((temp-name (make-symbol (symbol-name name))))
	(multiple-value-bind (args lexpr?) (arglist->args args-or-coder)
	  `(progn
	    (defun ,temp-name ,args-or-coder ,@body)
	    (defun ,name ,args-or-coder
	      (,(if lexpr? 'apply 'funcall)
		(get-value ',name 'compiled-definition) ,@args))
	    (define-internal-unit ,name
	      (lambda-definition '(lambda ,args-or-coder ,@body)))
	    (assert-value ',name 'compiled-definition
	     ',temp-name
	     'depends-on (list (get ',name 'lambda-definition)
			  (get 'compiled-definition 'to-compute-value )))
	    ',name)))
    `(progn (define-internal-unit ,name
	      (lisp-coder ',args-or-coder)
	      ,@body)
      (get-value ',name 'lambda-definition))))
;;..In fact, @code{deffcn} does a little bit more in its
;;..macroexpansion; in particular, it defines a direct compiled version of
;;..the function and stores this on the definition immediately (rather
;;..than waiting for it to be defaulted) with dependencies on the
;;..installed lambda definition.  This is so that when a @code{deffcn}
;;..is compiled by the `file compiler,' the work of compilation can be
;;..done immediately and not postponed until load or call time.

;;.The body of a @code{deffcn} looks just like a regular @code{defun}
;;.except that the @code{annotations} occuring in it have actual
;;.effect on the constructed unit description.

;;.The @code{annotations} macro does nothing (it is identical to a
;;.many-argument `quote') from the point of view of LISP proper, but
;;.the @code{lambda-definition} slot extracts the @code{annotations}
;;.expressions in a lambda body and asserts them as properties on the
;;.description of the function.
(defmacro annotations (&body annotations)
  `',annotations)

;;.The @code{fcn} macro is an inline generator for conded definitions.
;;.Its first argument is a lisp coder and its body is a list of slots
;;.and values to be assigned to the corresponding unit.  It expands
;;.into an expression which constructs a gensymed description name and
;;.assigns that name the expected @code{lisp-coder} slot and the other
;;.slots neccessary for generating a complete lambda definition; it
;;.then extracts this lambda definition.
(defmacro fcn (coder &body body)
  "Defines a function description (and function) for ARLOTJE."
  `(let ((unit (make-internal-unit (gensymbol ',coder)
		 (lisp-coder ',coder)
		 ,@body)))
    (get-value unit 'lambda-definition)
    unit))
;;.An example use of @code{fcn} might be:
;;.@example
;;.(assertion 'bachelor-eligibility 'makes-sense-for
;;.           (fcn test-intersection
;;.		(intersection-of 'unmarried)
;;.		(intersection-of 'men)))
;;.@end example
