;;; -*- LISP -*-

;;.@chapter Caching slot values

(in-package :arlotje)

;;.Annotated values are used by ARLOtje to describe slots which
;;.dynamically compute their values on demand and store the results
;;.for quick access on subsequent requests.  The problem with any such
;;.scheme is how to detect when a stored value needs to be recomputed;
;;.ARLOtje handles this problem by annotating the descriptions of
;;.values with dependency information indicating the sources of
;;.information from which a value was computed.

;;.In some representation languages, these dependencies are set up by
;;.having a fixed set of `inference methods' which compute the slot's
;;.value based on other slots.  ARLOtje instead allows the use of any
;;.LISP function to compute the default but dynamically keeps track of
;;.the slot's accessed during the computation.  This is done by using
;;.annotated values to describe referenced values; whenever an
;;.annotated value is `reduced' to its actual value, the value is
;;.added to the list @code{*current-computation*}; when a default
;;.value is being computed


;;;;.Default value descriptions

;;.A default value description is an annotated value which is accessed
;;.by its @code{default-value} slot.  The @code{default-value} gets
;;.its value by referring to the corresponding slot's
;;.@code{to-compute-value} slot; this is a function which it calls to
;;.compute an actual value.  During this computation, slot accesses
;;.are tracked and the corresponding dependencies set up when the
;;.returned value is actually stored as the @code{current-value} slot
;;.of the annotated description.  Even though cached slots access
;;.their values through the @code{default-value} slot, no unit
;;.(symbol) actually has a @code{default-value} property; the
;;.@code{default-value} slot stores its value in the
;;.@code{current-value} property.

(define-unit default-value
  (works-like 'prototypical-primitive-slot)
  (to-get-value 'compute-default-value)
  (to-put-value 'invalidate-and-store))

;;.Defaults are computed by a `default method' stored either on the
;;.annotated description itself (as a @code{default-method} slot) or
;;.on the @code{to-compute-value} slot of the slot-value which the
;;.annotated description describes.  This allows the specification of
;;.distinct computation methods for particular slots of particular
;;.units.  The default method is a function of two arguments, a unit
;;.and a slot, and returns a value to be cached on the corresponding
;;.description.  Slots accessed during this computation are recorded
;;.and the value description annotated with them as dependencies.

;;.This functionality is all implemented within the procedure
;;.@code{compute-default-value} which first tries to return any
;;.currently valid value (with @code{get-if-valid}) and otherwise gets
;;.the appropriate default method and calls it to yield a value which
;;.is locally stored along with the appropriate dependencies.
(defun compute-default-value (av slot)
  "Computes a default value for a slot from its annotated description."
  (declare (ignore slot))
  (dependency! av)
  (let ((attempt (get-if-valid av 'default-value)))
    (if (failurep attempt)
	(let ((unit (annotated-value-unit av)) (slot (annotated-value-slot av)))
	  (multiple-value-bind (result dependencies)
	      (return-dependencies
	       (let ((method (get-value slot 'to-compute-value)))
		 (if (failurep method) method
		   (funcall method unit slot))))
	    (unless (failurep result) 
	      (put-value unit slot result)
	      ;; Equivalent to (put-value av 'depends-on dependencies)
	      (store-dependencies av 'depends-on dependencies))
	    result))
      attempt)))

;;..This is the slot method.
(define-unit to-compute-value
  (works-like 'prototypical-slot)
  (must-be 'function-namep)
  (makes-sense-for 'slotp)
  (value-in-slot 'default-value)
  (accumulates-in-slot 'default-value))


;;;;.Caching Prototype

;;.The unit @code{prototypical-cached-slot} is a prototype for cached
;;.slots which defines a new default @code{value-in-slot} slot of
;;.@code{default-value}.  This is inherited by any slot which
;;.@code{works-like} @code{prototypical-cached-slot}.@refill  Note
;;.that @code{to-compute-value} is just such a slot; 

(define-unit prototypical-cached-slot
  (works-like 'prototypical-slot)
  (value-in-slot 'default-value)
  (accumulates-in-slot 'default-value))

(put-value 'to-compute-value 'works-like 'prototypical-cached-slot)
(put-value 'to-compute-value 'to-compute-value 'inherit-through-works-like)


;;;;.An Example

;;.Cached slots can be used for computing values in terms of other
;;.slots and updating those values when the required slots change.
;;.For instance, we might represent a circle's radius and
;;.circumference thus:
;;...@example
(define-internal-unit radius
  (works-like 'prototypical-slot)
  (must-be 'numberp))

(define-internal-unit circumference
  (works-like 'prototypical-cached-slot)
  (to-compute-value '2*pi*r))

(defun 2*pi*r (unit slot)
  (declare (ignore slot))
  (* 2 3.1415 (get-value unit 'radius)))
;;...@end example
;;.we might then say:
;;.@example
;;.(define-internal-unit circle-1
;;.  (radius 20))
;;.(get-value 'circle-1 'circumference)
;;.===> 125.66000000000001
;;.(assertion 'circle-1 'radius 40)
;;.(get-value 'circle-1 'circumference)
;;.==> 251.32000000000002
;;.@end example
;;.And note how the value changes as its elements change.



;;;;.Checking for values

(defun check-value (unit slot value)
  "This checks that a value holds for a slot, trying various methods."
  (or (find-value unit slot value)
      (computing (check-value unit slot value)
	 (let ((check-method (get-value slot 'to-check-value)))
	   (if (failurep check-method)
	       '()
	     (funcall check-method unit slot value))))))

(define-unit to-check-value
  (works-like 'prototypical-slot)
  (must-be 'function-namep)
  (makes-sense-for 'slotp)
  (value-in-slot 'default-value)
  (accumulates-in-slot 'default-value)
  (to-compute-value 'inherit-through-works-like))

(defun check-equal-value (unit slot value)
  "Returns the description of SLOT of UNIT, if its value is VALUE."
  (and (equal (get-value unit slot) value)
       (annotated-value unit slot)))
(put-value 'prototypical-slot 'to-check-value 'check-equal-value)

