(in-package :test-arlo)

(defvar *test-package* (find-package :test-arlo))

(defun replace-in-tree (froms tos tree)
   (cond ((atom tree)
              (if (member tree froms)  (nth (position tree froms) tos) tree))
            ((consp tree)
              (mapcar #'(lambda(e) (replace-in-tree froms tos e)) tree))))

(defvar *check* nil)
(defvar *test-body* nil)
(defvar *units* nil)
(defvar *replacement-units* nil)

(defvar *all-tests* nil)
(defvar *passed-tests* nil)

(defun run-tests (&optional rerun?)
  (when rerun? (setq *passed-tests* nil))
   (loop for test in (reverse *all-tests*)
            when (not (member test *passed-tests*))
            do 
            (print test)
            (when (funcall test)
                (pushnew test *passed-tests*))))

(defun clean-up ()
   (do-symbols (s 'test-arlo)
      (when (and (eq (symbol-package s) *test-package*)
                        (find #\_ (symbol-name s) :test 'char=))
          (setf (symbol-plist s) nil)
          (unintern s 'test-arlo))))

(defmacro define-test (name (&rest units) check &body body)
  `(progn 'compile
    (pushnew ',name *all-tests*)
    (defun ,name (&optional debug)
      (let* ((replacements (mapcar #'(lambda (s) (make-symbol (string s)))
				   ',units))
	     (new-form
	      (replace-in-tree (cons 'define-unit ',units) (cons 'define-internal-unit replacements)
			       (list* 'progn ',body)))
	     (check (replace-in-tree (cons 'define-unit ',units)  (cons 'define-internal-unit replacements) ',check)))
	(let ((*check* check) (*test-body* new-form) (*replacement-units* replacements) (*units* ',units))
	  (declare (special *check* *replacement-units* *test-body* *units*))
	  (if debug
	      (progn (eval new-form) (eval `(assert ,check)) t)
	    (progn
              (multiple-value-bind (result errorp)
                ;; This is to handle some bug in LUCID
                #+LUCID (eval `(ignore-errors (eval ',new-form)))
                #-LUCID (ignore-errors (eval new-form))
                (declare (ignore result))
                (when errorp
                  (format t "~&An error occured during the body of ~A: ~%~A~%Form was:" ',name errorp)
                  (pprint '(progn ,@body))
                  (return-from ,name (values ',name errorp))))
	      (multiple-value-bind (result errorp)
		  #+LUCID (eval `(ignore-errors (eval ',check)))
		  #-LUCID (ignore-errors (eval check))
		(declare (ignore result))
		(when errorp
		  (format nil "An error occured during the expectation test of ~A: ~A~%" ',name errorp)
		  (return-from ,name (values ',name errorp)))
		(if  (eval check)
		     t
		  (progn
		    (format t "~&The expectation wasn't satisfied in ~a. Form was" ',name)
		    (pprint '(progn ,@body))
		    (format t "~&Expected ~s" ',check)))))))))))

(defun tdu (name)
   (du
     (if (member name *units*)
        (nth (position name *units*) *replacement-units*)
        name)))
(defun tgv (name slot)
   (get-value
     (if (member name *units*)
        (nth (position name *units*) *replacement-units*)
        name)
     (if (member slot *units*)
        (nth (position slot *units*) *replacement-units*)
        slot)))

(defun set-equal (l1 l2)
  (null (set-exclusive-or l1 l2)))
;;****************************************************************
;; test of genl slots


(define-test test-genl-slots (s1 s2 s3 s4) 
   (set-equal (get-value 's4 's1)  '(foo bar quux ?))
   (define-unit s1 
      (works-like 'prototypical-set-slot))
   (define-unit s2
      (works-like 'prototypical-set-slot)
      (genl-slots 's1))
   (define-unit s3
      (works-like 'prototypical-set-slot)
      (genl-slots 's1))
   (define-unit s4
      (s2 'foo)
      (s2 'bar)
      (s3 'quux)
      (s3 '?)))

;;****************************************************************
;; test of specs slot
(define-test test-spec-slots (s1 s2 s3 s4)
   (set-equal (get-value 's4 's3) '(foo bar quux ?))
   (define-unit s1
      (works-like 'prototypical-set-slot))
   (define-unit s2
      (works-like 'prototypical-set-slot))
   (define-unit s3
      (works-like 'prototypical-set-slot)
      (spec-slots 's1)
      (spec-slots 's2))
   (define-unit s4
      (s1 'foo)
      (s1 'bar)
      (s2 'quux)
      (s2 '?)))

;;****************************************************************
;; test of order independant specs slots
(define-test test-spec-slots-out-of-order (s1 s2 s3 s4)
   (set-equal (get-value 's4 's3) '(foo bar quux ?))
   (define-unit s1
      (works-like 'prototypical-set-slot))
   (define-unit s2
      (works-like 'prototypical-set-slot))
   (define-unit s4
      (s1 'foo)
      (s1 'bar)
      (s2 'quux)
      (s2 '?))
   (define-unit s3
      (works-like 'prototypical-set-slot)
      (spec-slots 's1)
      (spec-slots 's2)))

;;****************************************************************
(define-test test-composition-alan (alan grandfather alans-grandfather father alans-dad)
   (get-value 'alan 'grandfather) 'alans-grandfather
   (define-unit father
      (works-like 'prototypical-slot))
   (define-unit alans-grandfather
      (member-of 'humans))
   (define-unit alans-dad 
      (member-of 'humans)
      (father 'alans-grandfather)) 
   (define-unit grandfather
      (works-like 'prototypical-slot)
      (composition-of '(father father)))
   (define-unit alan 
      (member-of 'humans)
      (father 'alans-dad)))

(define-test test-composition-kevin (bir christer eva joachim thomas birgitta nils children 
                                                              father fathers-children)
                   (null (set-difference  (get-value 'joachim 'fathers-children)
                                                   '(joachim thomas birgitta nils)))
   (define-unit father
      (member-of 'many-valued-slots)
      (must-be 'men))
   (define-unit children
      (member-of 'many-valued-slots))
   (define-unit fathers-children
      (member-of 'many-valued-slots)
      (composition-of '(children father)))
   (define-unit bir
      (children 'joachim))
   (define-unit eva
      (children 'thomas)
      (children 'birgitta)
      (children 'nils))
   (define-unit christer
      (member-of 'men))
   (define-unit thomas
      (father 'christer))
   (define-unit joachim
      (father 'christer))
   (define-unit nils
      (father 'christer))
   (define-unit birgitta
      (father 'christer))
   (define-unit christer
      (children 'joachim)
      (children 'thomas)
      (children 'nils)
      (children 'birgitta)))

;;****************************************************************
(define-test set-difference-constraint-method (s1 s2 s3 u)
                   (null (set-difference (get-value 'u 's3) '(d e)))
   (define-unit s1
      (works-like 'prototypical-set-slot))
   (define-unit s2
      (works-like 'prototypical-set-slot))
   (define-unit s3
      (works-like 'prototypical-set-slot)
      (constraint-methods '(set-difference s2 s1)))
   (define-unit u
      (s1 'a)
      (s1 'b)
      (s1 'c)
      (s2 'a)
      (s2 'b)
      (s2 'c)
      (s2 'd)
      (s2 'e)))


;;****************************************************************
  
(define-test test-unit-inferences (u1 u2 s1)
                   (satisfies? 'u1 'c1)
   (define-unit c1
      (member-of 'collections))
   (define-unit s1
      (works-like 'prototypical-slot)
      (unit-inferences '(member-of 'c1)))
   (define-unit u1
      (s1 'foo)))

;;****************************************************************
   
(define-test test-inverse (s1 s2 u1 u2)
                   (and (query 's1 'inverse-slot 's2) (query 'u2 's2 'u1))
   (define-unit s1
      (works-like 'prototypical-slot))
   (define-unit s2
      (works-like 'prototypical-slot)
      (inverse-slot 's1))
   (define-unit u1
      (s1 'u2)))

;;****************************************************************

(define-test test-pushes-through (owner parts car1 alan wheel1)
                   (query 'wheel1 'owner 'alan)
   (define-unit parts
      (works-like 'prototypical-set-slot))
   (define-unit owner
      (works-like 'prototypical-slot)
      (pushes-through 'parts))
   (define-unit car1
      (owner 'alan)
      (parts 'wheel1)))

;; ****************************************************************

(define-test test-pushes-through-out-of-order (owner parts car1 alan wheel1)
                   (query 'wheel1 'owner 'alan)
   (define-unit parts
      (works-like 'prototypical-set-slot))
   (define-unit owner
      (works-like 'prototypical-slot))
   (define-unit car1
      (owner 'alan)
      (parts 'wheel1))
   (assertion 'owner 'pushes-through 'parts))

;;****************************************************************

;; 1: associated-inferences-about-parts ->                                          inferences-about-parts (documentation bug)
;; 2: '(associated-inferences-about-parts (member-of red-things)) -> (inferences-about-parts '(member-of red-things)) (doc bug extra quote)

(define-test test-associate-inferences-plus-members-have (parts red-things car1 wheel1 nut1)
                   (query 'nut1 'member-of 'red-things)
   (define-unit parts
      (works-like 'prototypical-slot))
   (define-unit red-things
      (member-of 'collections)
      (members-have
        `(,(get-value 'parts 'associated-inferences) (member-of red-things))))
   (define-unit car1
      (member-of 'red-things)
      (parts 'wheel1))
   (define-unit wheel1
      (parts 'nut1)))

;;****************************************************************

(define-test test-associate-inferences
          (parts part-of things-which-are-part-of-something car1 wheel1 nut1)
                   (query 'nut1 'member-of 'things-which-are-part-of-something)
   (define-unit things-which-are-part-of-something
      (member-of 'collections))
   (define-unit parts
      (works-like 'prototypical-set-slot))
   (define-unit part-of
      (works-like 'prototypical-set-slot)
   (inverse-slot 'parts)
   (inferences
    '(member-of things-which-are-part-of-something)))
   (define-unit car1
      (parts 'wheel1))
   (define-unit wheel1
      (parts 'nut1)))

;; ****************************************************************
;; manual has a documentation bug (get-value 'block 'above) -> (get-value 'block-2 'above) 
(define-test test-structure-inference (block-1 block-2 top bottom tower1)
                   (query 'block-1 'above 'block-2)
   (define-unit block-1
      (member-of 'spatial-extents))
   (define-unit block-2
      (member-of 'spatial-extents))
   (define-unit top
      (works-like 'prototypical-slot))
   (define-unit bottom
      (works-like 'prototypical-slot)
      (structure '(below top)))
   (define-unit tower1
      (top 'block-1)
      (bottom 'block-2)))

;;****************************************************************
(define-test test-structure-consorting (partners children mary john bill jimmy consorting)
                   (query 'mary 'children 'bill)
   (define-unit children 
      (works-like 'prototypical-set-slot))
   (define-unit partners
      (works-like 'prototypical-set-slot)
      (structure '(children children)))
   (define-unit consorting
      (children 'bill)
      (children 'jimmy)
      (partners 'mary)
      (partners 'john)
      ))

;;****************************************************************

(define-test test-structure-consorting-funny-order (partners children mary john bill jimmy consorting)
                   (query 'mary 'children 'bill)
   (define-unit children 
      (works-like 'prototypical-set-slot))
   (define-unit partners
      (works-like 'prototypical-set-slot)
      (structure '(children children)))
   (define-unit consorting
      (partners 'mary)
      (partners 'john)
      (children 'bill)
      (children 'jimmy)
      ))


;; ****************************************************************
(define-test test-structure-inference-out-of-order (block-1 block-2 top bottom tower1)
                   (query 'block-1 'above 'block-2)
   (define-unit block-1
      (member-of 'spatial-extents))
   (define-unit block-2
      (member-of 'spatial-extents))
   (define-unit top
      (works-like 'prototypical-slot))
   (define-unit bottom
      (works-like 'prototypical-slot))
   (define-unit tower1
      (top 'block-1)
      (bottom 'block-2))
   (assertion 'bottom 'structure '(below top))
   )

;; ****************************************************************

(define-test test-structure-inference-out-of-order-redefine (block-1 block-2 top bottom tower1)
                   (query 'block-1 'above 'block-2)
   (define-unit block-1
      (member-of 'spatial-extents))
   (define-unit block-2
      (member-of 'spatial-extents))
   (define-unit top
      (works-like 'prototypical-slot))
   (define-unit bottom
      (works-like 'prototypical-slot))
   (define-unit tower1
      (top 'block-1)
      (bottom 'block-2))
   (assertion 'bottom 'structure '(below top))
   (aj::kill-changes 'tower1)
   (define-unit tower1
      (top 'block-1)
      (bottom 'block-2))
   )

;; ****************************************************************

;; documentation bug bottom of page 14 :value instead of %value%

(define-test put-demon-test (husband wife john mary inverse-slot)
                   (query 'james 'wife 'mary)
   (define-unit inverse-slot
      (english-description "The inverse of a slot.")
      (works-like 'prototypical-slot)
      (makes-sense-for 'slotp)
      (must-be         'slotp)
      (put-demons '(assert-value %unit% put-demons
                            `(assert-value %VALUE% ,%VALUE% %UNIT%)))
      (inverse-slot 'inverse-slot))
   (define-unit wife
      (works-like 'prototypical-slot))   
   (define-unit husband
      (works-like 'prototypical-slot)
      (inverse-slot 'wife))
   (define-unit mary
      (husband 'james)))

;; ****************************************************************

(define-test put-demon-test-2 (put-demon-put s1 u1)
                   (query 'u1 'put-demon-put '(u1 s1 foo))
   (define-unit put-demon-put
      (works-like 'prototypical-set-slot))
   (define-unit s1
      (works-like 'prototypical-set-slot)
      (put-demons '(assert-value %unit% 'put-demon-put `(,%unit% ,%slot% ,%value%))))
   (define-unit u1
      (s1 'foo)))

;; ****************************************************************
   
(define-test put-demon-test-2-truth-maintenance (put-demon-put s1 u1)
                   (null (query 'u1 'put-demon-put '(u1 s1 foo)))
   (define-unit put-demon-put
      (works-like 'prototypical-set-slot))
   (define-unit s1
      (works-like 'prototypical-set-slot)
      (put-demons '(assert-value %unit% 'put-demon-put `(,%unit% ,%slot% ,%value%))))
   (define-unit u1
      (s1 'foo))
   (retraction 's1 'put-demons '(assert-value %unit% 'put-demon-put `(,%unit% ,%slot% ,%value%))))

;;****************************************************************      
                   
(define-test test-constant-default (alan favourite-color pink)
                   (progn (get-value 'alan 'favourite-color) (query 'alan 'favourite-color 'pink))
   (define-unit favourite-color
      (works-like 'prototypical-slot)
      (value-defaults-to 'pink)))

;;****************************************************************

(define-test test-constant-default-query-only (alan favourite-color pink)
                    (query 'alan 'favourite-color 'pink)
   (define-unit favourite-color
      (works-like 'prototypical-slot)
      (value-defaults-to 'pink)))

;; ****************************************************************

(define-test test-constant-default-non-default (alan favourite-color pink)
                    (query 'alan 'favourite-color 'red)
   (define-unit favourite-color
      (works-like 'prototypical-slot)
      (value-defaults-to 'pink))
   (define-unit alan
      (favourite-color 'red)))

;;  ****************************************************************
;; same problem with get-value 'query
(define-test test-constant-default-non-default-truth-maintenance (alan favourite-color pink)
                    (progn (get-value 'alan 'favourite-color) (query 'alan 'favourite-color 'pink))
   (define-unit favourite-color
      (works-like 'prototypical-slot)
      (value-defaults-to 'pink))
   (define-unit alan
      (favourite-color 'red))
   (retraction 'alan 'favourite-color 'red))

;;****************************************************************

(define-test test-composition-ken (maternal-grandfather ken tineke john father mother)
                  (progn (get-value 'ken 'maternal-grandfather)
                             (query 'ken 'maternal-grandfather 'john))
   (define-unit father (works-like 'prototypical-slot))
   (define-unit mother (works-like 'prototypical-slot))
   (define-unit maternal-grandfather
      (works-like 'prototypical-slot)
      (composition-of '(father mother)))
   (define-unit ken
      (mother 'tineke))
   (define-unit tineke
      (father 'john)))

;;****************************************************************

(define-test test-composition-out-of-order (maternal-grandfather ken tineke john father mother)
                   (progn (get-value 'ken 'maternal-grandfather)
                             (query 'ken 'maternal-grandfather 'john))
   (define-unit father (works-like 'prototypical-slot))
   (define-unit mother (works-like 'prototypical-slot))
      (define-unit ken
      (mother 'tineke))
   (define-unit tineke
      (father 'john))
   (define-unit maternal-grandfather
      (works-like 'prototypical-slot)
      (composition-of '(father mother))))

;;  ****************************************************************

(define-test test-composition-retraction (maternal-grandfather ken tineke john father mother)
                   (failurep (get-value 'ken 'maternal-grandfather))
   (define-unit father (works-like 'prototypical-slot))
   (define-unit mother (works-like 'prototypical-slot))
   (define-unit ken
      (mother 'tineke))
   (define-unit tineke
      (father 'john)
      (mother 'mary))
   (define-unit maternal-grandfather
      (works-like 'prototypical-slot)
      (composition-of '(father mother)))
   (retraction 'maternal-grandfather 'composition-of '(father mother)))

;;****************************************************************

(define-test test-composition-retraction-2 (maternal-grandfather ken tineke john father mother)
                   (failurep (get-value 'ken 'maternal-grandfather))
   (define-unit father (works-like 'prototypical-slot))
   (define-unit mother (works-like 'prototypical-slot))
   (define-unit ken
      (mother 'tineke))
   (define-unit tineke
      (father 'john)
      (mother 'mary))
   (define-unit maternal-grandfather
      (works-like 'prototypical-slot)
      (composition-of '(father mother)))
   (retraction 'tineke 'father 'john))

;;****************************************************************

(define-test transitive-test (s1 u1 u2)
                   (query 'u2 's1 'a)
   (define-unit s1
      (member-of 'transitive-slots))
   (define-unit u1
      (s1 'a))
   (define-unit u2
      (s1 'u1)))
   
;;****************************************************************

(define-test test-kleene-star ()
                   (null (set-exclusive-or (get-value 'u4 'sup*) '(u1 u2 u3 u4)))
   (define-unit sup
      (works-like 'prototypical-set-slot))
   (define-unit sup*
      (works-like 'prototypical-slot)
      (kleene-star-of 'sup))
   (define-unit u1)
   (define-unit u2
      (sup 'u1))
   (define-unit u3
      (sup 'u2))
   (define-unit u4
      (sup 'u1)
      (sup 'u3)))

;;****************************************************************

(define-test test-kleene-plus ()
                   (null (set-exclusive-or (get-value 'u4 'sup+) '(u1 u2 u3)))
   (define-unit sup
      (works-like 'prototypical-set-slot))
   (define-unit sup+
      (works-like 'prototypical-slot)
      (kleene-plus-of 'sup))
   (define-unit u1)
   (define-unit u2
      (sup 'u1))
   (define-unit u3
      (sup 'u2))
   (define-unit u4
      (sup 'u1)
      (sup 'u3)))

;;****************************************************************
; fcn syntax is different than deffcn, fcn can't have arguments

(define-test inline-fcn-test  (c1 c2 u1 s1 )
                   (eq (get-value 'u1 's1) 'ok)
   (define-unit c1
      (member-of 'collections))
   (define-unit c2
      (member-of 'collections))
   (define-unit u1
      (member-of 'c1))
   (define-unit s1
      (works-like 'prototypical-set-slot)
      (makes-sense-for (fcn  test-union (union-of 'c1) (union-of 'c2)))
      (value-defaults-to 'ok)))

;;****************************************************************

(define-test value-defaults-makes-sense-for-interaction  (c1 c2 u1 u2 s1 )
                   (failurep (get-value 'u2 's1))
   (define-unit c1
      (member-of 'collections))
   (define-unit c2
      (member-of 'collections))
   (define-unit u1
      (member-of 'c1))
   (define-unit u2)
   (define-unit s1
      (works-like 'prototypical-set-slot)
      (makes-sense-for (fcn  test-union (union-of 'c1) (union-of 'c2)))
      (value-defaults-to 'ok)))

;;****************************************************************

(define-test test-restricted-slots (countries monarchies head-of-state monarch nobleagents queen 
                                                                     thatcher elizabeth england)
                   (and (query 'england 'queen 'elizabeth)
                           (eq (get-value 'england 'queen) 'elizabeth))
   (define-unit countries
      (member-of 'collections))
   (define-unit monarchies
      (member-of 'collections)
      (supersets 'countries))
   (define-unit head-of-state
      (works-like 'prototypical-set-slot)
      (makes-sense-for 'countries)
      (must-be 'agents))
   (define-unit monarch
      (works-like 'prototypical-set-slot)
      (restriction-of 'head-of-state)
      (makes-sense-for 'monarchies)
      (must-be 'nobleagents))
   (define-unit nobleagents
      (member-of 'collections)
      (supersets 'agents))
   (define-unit queen
      (works-like 'prototypical-slot)
      (restriction-of 'monarch)
      (makes-sense-for 'monarchies)
      (must-be (fcn test-intersection (intersection-of 'women) (intersection-of 'nobleagents))))
   (define-unit thatcher
      (member-of 'women))
   (define-unit elizabeth
      (member-of 'women)
      (member-of 'nobleagents))
   (define-unit england
      (member-of 'monarchies)
      (head-of-state 'elizabeth)
      (head-of-state 'thatcher)))
      
;; ****************************************************************   
   
(define-test mt-recursion-blowout (opposite compute-opposite truth-values truth falsehood)
             t
  (let ((aj::*recursion-checking* t))
    (define-unit opposite
      (works-like 'prototypical-slot)
      (makes-sense-for 'unitp)
      (english-description "Inverse of some property (male/female, black/white)")
      (to-compute-value 'compute-opposite)
      (inverse-slot 'opposite))
    
    ;;; Look for the first collection u is a member of that has exactly two elements,
    ;;; and return the other one.
    (deffcn compute-opposite (u slot)
      (declare (ignore slot))
      (dolist (collection (get-value u 'member-of) (fail u 'opposite))
        (when (= 2 (length (get-value collection 'members)))
          (return (car (remove u (get-value collection 'members)))))))
    
    ;Test
    (define-unit truth-values
      (member-of 'collections))
    
    (define-unit truth
      (member-of 'truth-values))
    (define-unit falsehood
      (member-of 'truth-values))
    
    ;;; This gets the error
    (get-value 'truth 'opposite)))

;****************************************************************

;; (get-value 'patty 'shared-traits) => nil
;; (get-value 'patty 'shared-gender) =>  aj::female
;; (get-value 'patty 'shared-traits) => aj::female
                             
;;			-Uri & Aaron

#|(define-test aaron-uri-composition (twin-human shared-gender shared-traits cathy patty)
             (query 'patty 'shared-traits 'female)
  (define-unit twin-human
    (member-of 'single-valued-slots)
    (makes-sense-for 'humans)
    (must-be 'humans))
  (define-unit shared-gender
    (member-of 'single-valued-slots)
    (composition-of '(gender twin-human)))
  (define-unit shared-traits
    (member-of 'many-valued-slots)
    (spec-slots 'shared-gender))
  (define-unit cathy
    (member-of 'women))
  (define-unit patty
    (member-of 'humans)
    (twin-human 'cathy)))|#

