;;;
;;; Iterator maker. An iterator is a procedure that carries on a
;;; (possibly recursive) computation. Whenever it produces a result, it
;;; returns it, and saves the current state of computation. In other
;;; words, it is a corutine. It is different from a general corutine
;;; in the sense that it will only release control when it has
;;; produced a result. To see how it is used, see below.
;;;

(define (make-iterator compute . initial-arguments)
  (define state #f)
  
  (define (go)
    (call-with-current-continuation
     (lambda (exit)
       (define (result val)
	 (cond ((call-with-current-continuation
		 (lambda (cont)
		   (set! state cont)
		   #t))
		(exit val))))
       (apply compute (cons result initial-arguments)))))
  
  (set! state (lambda ignore (go)))
  (lambda () (state #f)))

;;;
;;; Subset maker. If you do
;;; 
;;;   (define next (subset-maker '(1 2 3)))
;;;
;;; you can get the nonempty subsets of (1 2 3) by
;;;
;;; (next) => (1)
;;; (next) => (2)
;;; (next) => (3)
;;; (next) => (2 1)
;;;  ...
;;; (next) => (1 2 3)
;;; (next) => #f
;;; (next) => #f
;;;
;;; etc.
;;;

(define (subset-maker set)

  (define (subsets result set)

    (define (combinations suffix n set)
      (cond ((= (length set) n)
	     (result (append set suffix)))
	    ((= n 1)
	     (result (cons (car set)
			   suffix))
	     (combinations suffix n (cdr set)))
	    ((null? set) #f)
	    (else
	     (combinations (cons (car set) suffix)
			   (- n 1)
			   (cdr set))
	     (combinations suffix n (cdr set)))))

    ;; subsets
    (let ((len (length set)))
      (let loop ((n 1))
	(if (> n len)
	    #f
	    (begin
	      (combinations '() n set)
	      (loop (+ n 1)))))))

  (make-iterator subsets set))

