/* -*-C-*-

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: vms.c,v 9.29 87/11/04 20:04:23 GMT cph Rel $

   Contains operating system (VMS) dependent primitives and routines.

*/

#include <types.h>
#include <timeb.h>
#include <signal.h>
#include <time.h>
#include <descrip.h>
#include "vmsio.h"
#include ssdef
#include jpidef

/* The following is hackery for checking multiply overflow */

#if false

/*** Fix Macro_EMUL ***/

/* Macro_EMUL is currently broken, it should be fixed.  
 * The extra procedure call should be eliminated also.
 */

Mul(A, B)
fast long A, B;
{ fast long C;
  return Macro_EMUL(A,B);
}
#else

#include "mul.c"

#endif

/* Binary file I/O Operations */

extern long Load_Data(), Write_Data();
extern Boolean Open_Dump_File(), Close_Dump_File();

static FILE *File_Handle;

Boolean
Open_Dump_File( Name, flag)
     Pointer Name;
     char *flag;
{
  File_Handle = fopen(Scheme_String_To_C_String( Name), flag);
  return (File_Handle != ((FILE *) NULL));
}

Boolean
Close_Dump_File()
{
  return (fclose(File_Handle) == 0);
}

long
Load_Data(Count, To_Where)
     long Count;
     char *To_Where;
{
  extern int fread();

  return (fread(To_Where, sizeof(Pointer), Count, File_Handle));
}

long
Write_Data( Count, From_Where)
     long Count;
     char *From_Where;
{
  extern int fwrite();

  return (fwrite(From_Where, sizeof( Pointer), Count, File_Handle));
}

/* File I/O Operations */

FILE *
OS_file_open( name, output_p)
     char *name;
     Boolean output_p;
{
  return (fopen( name, (output_p ? "w" : "r")));
}

Boolean
OS_file_close( stream)
     FILE *stream;
{
  return ((fclose( stream)) == 0);
}

Boolean
OS_file_eof_p( stream)
     FILE *stream;
{
  return (feof( stream));
}

/*** Does this work? ***/

long
OS_file_length( stream)
     FILE *stream;
{
  long file_descriptor, current_position, result;
  extern long lseek();

  file_descriptor = fileno( stream);
  current_position = lseek( file_descriptor, 0, 1);
  result = lseek( file_descriptor, 0, 2);
  if (current_position != lseek( file_descriptor, current_position, 0))
    Primitive_Error( ERR_EXTERNAL_RETURN);
  return (result);
}

/* Three-valued; Lisp may need this smartness someday:

     -1   Does not exist.
      0   Don't know.
      1   Exists.

   Kludgerous implementation should be improved by someone
   who understands VMS file calls better. */

int
OS_file_existence_test (name)
     char *name;
{
  FILE *test_stream;

  test_stream = (OS_file_open (name, false));
  if (test_stream == NULL)
    return (-1);
  else
    {
      OS_file_close (test_stream);
      return (1);
    }
}

long
OS_file_read_chars( stream, buffer, nchars)
     FILE *stream;
     char *buffer;
     long nchars;
{
  return (fread( buffer, 1, nchars, stream));
}

Boolean
OS_file_write_chars( stream, buffer, nchars)
     FILE *stream;
     char *buffer;
     long nchars;
{
  return (fwrite( buffer, 1, nchars, stream) == nchars);
}

/* Working directory. */

/*** Does this work? ***/

Pointer
OS_working_dir_pathname()
{
  char *path;

  path = getenv("PATH");
  if (path == NULL)
    return (NIL);
  else
    return (C_String_To_Scheme_String( path));
}

Boolean
OS_set_working_dir_pathname( name)
     char *name;
{
  return (chdir( name) == 0);
}

/* File System Operations */

Boolean
OS_file_remove( name)
     char *name;
{
  return (delete( name) == 0);
}

/*** Implement the following 3 procedures if possible. ***/

Boolean
OS_file_link_physical( old_name, new_name)
     char *old_name, *new_name;
{
  return false;
}

Boolean
OS_file_link_symbolic( old_name, new_name)
     char *old_name, *new_name;
{
  return false;
}

Boolean
OS_file_rename( old_name, new_name)
     char *old_name, *new_name;
{
  return false;
}

/*** Does this work? ***/

#define file_copy_finish( result)				\
{								\
  if (!OS_file_close( source_file))				\
    Primitive_Error( ERR_EXTERNAL_RETURN);			\
  if (!OS_file_close( destination_file))			\
    Primitive_Error( ERR_EXTERNAL_RETURN);			\
  return (result);						\
}

#define file_copy_1( nchars)					\
{								\
  if (OS_file_read_chars( source_file, buffer, (nchars)) < (nchars)) \
    file_copy_finish( false);					\
  if (!OS_file_write_chars( destination_file, buffer, (nchars))) \
    file_copy_finish( false);					\
}

/* An arbitrary length -- could use `malloc' and compute

   file_copy_buffer_length = ((ulimit( 3, 0) - sbrk( 0)) / 2);

   but that is hairy and might not be easy to port to various unices.
   This should be adequate and hopefully will perform better than
   single-character buffering. */

#define file_copy_buffer_length 8192

Boolean
OS_file_copy( source_name, destination_name)
     char *source_name, *destination_name;
{
  FILE *source_file, *destination_file;
  long source_length, buffer_length;
  char buffer[file_copy_buffer_length];

  source_file = OS_file_open( source_name, false);
  if (source_file == NULL)
    return (false);
  destination_file = OS_file_open( destination_name, true);
  if (destination_file == NULL)
    {
      if (!OS_file_close( source_file))
	Primitive_Error( ERR_EXTERNAL_RETURN);
      return (false);
    }
  source_length = OS_file_length( source_file);
  buffer_length =
    ((source_length < file_copy_buffer_length)
     ? source_length : file_copy_buffer_length);
  while (source_length > buffer_length)
    {
      file_copy_1( buffer_length);
      source_length -= buffer_length;
    }
  file_copy_1( source_length);
  file_copy_finish( true);
}

/*** Implement the following three procedures if possible. */

Boolean
OS_directory_make( name)
     char *name;
{
  return false;
}

Pointer
OS_directory_open( name)
     char *name;
{
  return (NIL);
}

Pointer
OS_directory_read()
{
  Primitive_Error( ERR_EXTERNAL_RETURN);
}

/* Terminal hacking. */

forward int OS_tty_tyi();

#define define_OS_tty_read_char( name, immediate)		\
char								\
name()								\
{								\
  int chr;							\
								\
  while (true)							\
    {								\
      Boolean Interrupted;					\
								\
      chr = OS_tty_tyi( (immediate), &Interrupted);		\
      if (Interrupted)						\
	{							\
	  if ((IntEnb & IntCode) != 0)				\
	    Primitive_Interrupt();				\
	}							\
      else							\
	return ((char) chr);					\
    }								\
}

define_OS_tty_read_char( OS_tty_read_char, false)
define_OS_tty_read_char( OS_tty_read_char_immediate, true)

void
OS_tty_write_char (chr)
     char chr;
{
  putc (chr, stdout);
  return;
}

Boolean
OS_tty_write_chars (string, string_length)
     char *string;
     long string_length;
{
  Boolean result;

  result = ((fwrite (string, 1, string_length, stdout)) == string_length);
  return (result);
}

void
OS_Flush_Output_Buffer()
{
  fflush (stdout);
}

/*** All of the following should be looked at/implemented. ***/

Boolean
OS_Clear_Screen()
{
  putchar('\f');
  return (true);
}

Boolean
OS_tty_move_cursor( x, y)
     long x, y;
{
  return (false);
}

Boolean
OS_tty_beep()
{
  putchar(BELL);
  return true;
}

void
OS_tty_newline()
{
  OS_tty_write_char ('\n');
  return;
}

Boolean
OS_tty_get_cursor (x, y)
     long *x, *y;
{
  *x = 0;
  *y = 0;
  return (false);
}

long
NColumns ()
{
  return (79);
}

long
NLines ()
{
  return (24);
}

Boolean
OS_Clear_To_End_Of_Line ()
{
  return (false);
}

Boolean
OS_Under_Emacs ()
{
  return (false);
}

/* System Clock */

long start_time, start_milli_time, start_proc_time;

long
OS_real_time_clock ()
{
  struct timeb timeblock;
  long deltat, deltamillit, nowt, nowmillit;
  
  ftime(&timeblock);
  nowt = timeblock.time;
  nowmillit = timeblock.millitm;
  deltat = nowt - start_time;
  deltamillit = nowmillit - start_milli_time;
  return (deltat * 100) + (deltamillit / 10);
}

#ifndef HAVE_TBUFFER

/* Times are in centiseconds, as desired.
   The system times are always 0, according to the VAX-11 C book.
 */

struct tbuffer {
  int proc_user_time;
  int proc_system_time;
  int child_user_time;
  int child_system_time;
};  

#endif /* HAVE_TBUFFER */

long
System_Clock ()
{
  struct tbuffer buf;

  times(&buf);

  return( buf.proc_user_time - start_proc_time);
}

void 
Init_System_Clock ()
{
  struct timeb time;
  struct tbuffer buf;

  ftime(&time);
  start_time = time.time;
  start_milli_time = time.millitm;

  times(&buf);
  start_proc_time = buf.proc_user_time;

  return;
}

/* Time and dates. */

extern struct tm *(localtime());

#define Date_Part(C_Name, Which)					\
int									\
C_Name()								\
{									\
  struct tm *Time;							\
  long Ans, The_Time;							\
									\
  time(&The_Time);							\
  Time = localtime(&The_Time);						\
  return (Time->Which);							\
}

Date_Part(OS_Current_Year, tm_year);
Date_Part(OS_Current_Month, tm_mon + 1);
Date_Part(OS_Current_Day, tm_mday);
Date_Part(OS_Current_Hour, tm_hour);
Date_Part(OS_Current_Minute, tm_min);
Date_Part(OS_Current_Second, tm_sec);

/* Timers (for timer interrupts) */

/*** Implement this.
     The Vax-11 C book only lists alarm, with 1 second resolution.
     There is probably a more accurate way of doing it.
 ***/

void
Clear_Int_Timer()
{
  Primitive_Error(ERR_UNIMPLEMENTED_PRIMITIVE);
  /*NOTREACHED*/
}

void
Set_Int_Timer(days, centi_seconds)
     long days, centi_seconds;
{
  Primitive_Error(ERR_UNIMPLEMENTED_PRIMITIVE);
  /*NOTREACHED*/
}

/* Keyboard I/O and interrupts */

#define CONTROL_A	'A'
#define CONTROL_B       'B'
#define CONTROL_F	'F'
#define CONTROL_G       'G'
#define CONTROL_M	'M'
#define CONTROL_P       'P'
#define CONTROL_U	'U'
#define CONTROL_X       'X'
#define CONTROL_Z	'Z'

#define CONTROL_BIT	0100	/* Octal, control bit in ASCII */

#define C_A (CONTROL_A - CONTROL_BIT)
#define C_B (CONTROL_B - CONTROL_BIT)
#define C_F (CONTROL_F - CONTROL_BIT)
#define C_G (CONTROL_G - CONTROL_BIT)
#define C_M (CONTROL_M - CONTROL_BIT)
#define C_P (CONTROL_P - CONTROL_BIT)
#define C_X (CONTROL_X - CONTROL_BIT)
#define C_Z (CONTROL_Z - CONTROL_BIT)

#define Control_A_Bit (1 << C_A)
#define Control_B_Bit (1 << C_B)
#define Control_F_Bit (1 << C_F)
#define Control_G_Bit (1 << C_G)
#define Control_P_Bit (1 << C_P)
#define Control_X_Bit (1 << C_X)
#define Control_Z_Bit (1 << C_Z)
#define CR_Bit	      (1 << C_M)

#define MASK_SIZE 32	/* In bits */

#define Scheme_Interrupt_Mask (Control_G_Bit | Control_P_Bit | 		\
			       Control_F_Bit | Control_B_Bit |		\
			       Control_A_Bit | Control_Z_Bit )

#define Normal_Terminator_Mask CR_Bit

#define INTERRUPT	1
#define REDO		2

/* Interrupts & IO synchronization */

static long TT_Channel;

struct Block { short Status;
	       short Count;
	       long Ptr;
	     };
static struct Block
	Scheme_Interrupt_Descriptor = {0, 0, Scheme_Interrupt_Mask};

static struct Block 
	Scheme_Read_Terminator =
		{0, 0, (Scheme_Interrupt_Mask | Normal_Terminator_Mask)};

#define Is_Terminator(c)				\
 ((c >= MASK_SIZE) ? 					\
  false :						\
  (((1 << c) & Normal_Terminator_Mask) != 0))

void
Set_Interrupt_Mask(Mask, AST)
     struct Block *Mask;
     long *AST;
{ long Return_Status;
  struct Block IO_Status_Block = {0, 0, 0};
  Return_Status = SYS$QIO(0, TT_Channel, IO$_SETMODE|IO$M_OUTBAND,
                          &IO_Status_Block, 0, 0,
                          AST, Mask, 0xFFFFFFFF,
                          0, 0, 0);
      
  if (Return_Status != SS$_NORMAL)
    fprintf(stderr,
	    "\nStatus = %x Unable to Setmode",
	    Return_Status);
  return;
}

forward int Sift_Out_Interrupt();

#define Setup_Scheme_Interrupts()				\
  Set_Interrupt_Mask(&Scheme_Interrupt_Descriptor,		\
		     ((long *) Sift_Out_Interrupt))

#define Clear_Scheme_Interrupts()				\
  Set_Interrupt_Mask(((struct Block *) 0), ((long *) 0))

#define interrupt_start(signal_name, routine, restore_handler)
#define interrupt_end(action) return action

/* Keyboard input low level */

#define INPUT_BUF_SIZE 512	/* Over 6 full lines of text */
#define NO_INTERRUPTS   -1	/* Not a valid character */

#define Initialize_Input() 					\
  Input_Pointer = Input_End = &Input_Buffer[0]

static char Input_Buffer[INPUT_BUF_SIZE];
static char *Input_Pointer;
static char *Input_End;

int
fill_input_buffer(size, keep_type_ahead)
     int size;
     Boolean keep_type_ahead;
{
  struct Block IO_Status_Block = {0, 0, 0};
  long Return_Status;
  int terminator;

  Clear_Scheme_Interrupts();

  do
  { Return_Status =
      SYS$QIOW(0,TT_Channel,
	       (IO$_READVBLK | (keep_type_ahead ? 0 : IO$M_PURGE)),
	       &IO_Status_Block,0,0,
	       &Input_Buffer[0],size,0,
	       &Scheme_Read_Terminator,0,0);
  } while (Return_Status != SS$_NORMAL);

  Setup_Scheme_Interrupts();

  Input_Pointer = &Input_Buffer[0];
  Input_End  = &Input_Buffer[IO_Status_Block.Count];

  /* Interrupt? */

  terminator = (IO_Status_Block.Ptr & 0377);
  if (terminator == 0) return NO_INTERRUPTS;
  if (Is_Terminator(terminator))
  { *Input_End++ = ((terminator == C_M) ? '\n' : terminator);
    return NO_INTERRUPTS;
  }
  return terminator;
}

/* Keyboard input */

int OS_tty_tyi(Immediate, Interrupted)
     Boolean Immediate, *Interrupted;
{
  *Interrupted = false;
  while (Input_Pointer >= Input_End) /* Should be == ... */
  {
    int result;

    result = fill_input_buffer((Immediate ? 1 : INPUT_BUF_SIZE), true);
    if ((result != NO_INTERRUPTS) &&
        (Sift_Out_Interrupt(result) == INTERRUPT))
    {
      *Interrupted = true;
      return EOF;
    }
  }
  return (*Input_Pointer++ & 0377);
}

/* Flushes type ahead and ignores other interrupts */

char 
Interrupt_Getchar()
{
  int result;

  do
  {
    Initialize_Input();
    result = fill_input_buffer(1, false);

  } while (result != NO_INTERRUPTS);

  return *Input_Pointer++;
}  

/* The arguments to OS_Clean_Interrupt_Channel are ignored,
   The possible modes are:
 */

#define UNTIL_MOST_RECENT_INTERRUPT	0
#define MULTIPLE_COPIES_ONLY		1

Boolean
Os_Clean_Interrupt_Channel(mode, interrupt_char)
     int mode, interrupt_char;
{
  Initialize_Input();
  return true;
}

static char Int_Char;

int
OS_Get_Next_Interrupt_Character()
{
  int result;

  if (Int_Char == '\0')
    return -1;
  result = ((int) Int_Char);
  Int_Char = '\0';
  return result;
}

/*** Implement read_char_ready_p. ***/

Boolean
OS_read_char_ready_p( delay)
     long delay;
{
  return (false);
}

/* Interrupt handlers */

forward Suspend_Me();

long Ask_Me()
{
  char command;

  putchar('\007');
  putchar('\n');
Loop:
  printf("Interrupt character (? for help): ");
  command =  Interrupt_Getchar();
  switch (command)
  { case 'B':
    case 'b': Int_Char = CONTROL_B; break;

    case 'D':
    case 'd': putchar('\n');
	      Handle_Debug_Flags();
	      goto exit_gracefully;
    case 'T':
    case 't': putchar('\n');
	      Back_Trace();
	      goto exit_gracefully;

    case 'G':
    case 'g': Int_Char = CONTROL_G; break;

    case 'P':
    case 'p':
    case 'U':
    case 'u': Int_Char = CONTROL_U; break;

    case 'F':
    case 'f':
    case 'X':
    case 'x': Int_Char = CONTROL_X; break;

    case 'H':
    case 'h': printf("\nThe following control characters are available at\n");
              printf("any time in Scheme:\n\n");
              printf("^A: interactive choice of interrupt\n");
              printf("^B: create and enter a breakpoint REP loop\n");
              printf("^F: abort to current REP loop\n");
              printf("^G: abort to top level REP loop\n");
              printf("^P: abort to previous (lower number) REP loop\n");
	      printf("^Z: exit Scheme temporarily\n");
              printf("<returning to Scheme>\n\n");
              goto exit_gracefully;

    case 'Q':
    case 'q':
	      putchar('\n');
	      Microcode_Termination(TERM_HALT);

    case 'Z':
    case 'z':
    	      putchar('\n');
	      Suspend_Me(true);
	      goto exit_gracefully;

    case '\f': OS_Clear_Screen(); goto exit_gracefully;

    case 'I':
    case 'i': 	 printf("gnored.\n");
exit_gracefully: return REDO;

    default: putchar('\n');
             printf("B: Enter a breakpoint loop\n");
             printf("D: Debugging: change interpreter flags\n");
             printf("F or X: Abort to current REP loop\n");
             printf("G: Goto to top level read-eval-print (REP) loop\n");
	     printf("H: Print information about interrupt characters\n");
             printf("I: Ignore interrupt request\n");
             printf("P or U: Up to previous (lower numbered) REP loop\n");
             printf("Q: Quit instantly, killing Scheme\n");
             printf("T: Stack trace\n");
	     printf("Z: Exit Scheme temporarily\n");
             printf("^L: Clear the screen\n");
             goto Loop;
  }
  IntCode |= INT_Character;
  New_Compiler_MemTop();
  return INTERRUPT;
}

/* Interrupt handlers, continued */

int
Sift_Out_Interrupt(Interrupt_Char)
     char Interrupt_Char;
{
  interrupt_start(0, 0, false);
  switch(Interrupt_Char)
  { 
    case C_A:
      { long Result = Ask_Me();
        interrupt_end(Result);
      }
    case C_B:
      Int_Char = CONTROL_B; break;
    case C_G:
      Int_Char = CONTROL_G; break;
    case C_F:	/* You type ^F to get an ^X interrupt! */
      Int_Char = CONTROL_X; break;
    case C_P:	/* You type ^P to get an ^U interrupt! */
      Int_Char = CONTROL_U; break;
    case C_Z:
      { Suspend_Me(true);
	interrupt_end(REDO);
      }
    default:
      fprintf(stderr,
	      "\nAST error! Sift_Out_Interrupt %x",
	      Interrupt_Char);
      Int_Char = 0;
      interrupt_end(REDO);
  }
  printf("^%c", Int_Char);
  IntCode |= INT_Character;
  New_Compiler_MemTop();
  interrupt_end(INTERRUPT);
}

/* Temporary exit to parent process */

#define LAST_ITEM 0

forward void OS_Quit(), OS_Re_Init();

Suspend_Me(from_interrupt)
     Boolean from_interrupt;
{
  int result;
  short nbytes = 0;       
  long pid = -1;

  struct Block ignore_me;
  struct item_desc {
		     short length;
		     short code;
		     long *buffer;
                     short *ret_length;
		    } item_block[2];

  item_block[0].code = JPI$_OWNER;
  item_block[0].length = sizeof(long);
  item_block[0].buffer = &pid;
  item_block[0].ret_length = &nbytes;
  item_block[1].code = LAST_ITEM;
  item_block[1].length = 0;
  item_block[1].buffer = NULL;
  item_block[1].ret_length = NULL;
  result = SYS$GETJPI( 0, NULL, NULL, &item_block[0], &ignore_me, NULL, 0);
  if (result != SS$_NORMAL)
  { printf("SYS$GETJPI returned %d != SS$_NORMAL; Continuing\n", result);
    return;
  }
  if (result == 0)
  { printf("Scheme is running at top level, it cannot detach\n");
    return;
  }
  OS_Quit();
  result = LIB$ATTACH(&pid);
  if (result != SS$_NORMAL)
    printf("LIB$ATTACH returned %d; Continuing\n", result);
  OS_Re_Init();
  return;
}

Boolean
Restartable_Exit()
{
  Suspend_Me(false);
  return true;
}

/* Initializes OS dependent information for Scheme */

void
OS_Init()
{
  OS_Name = "vms";
  OS_Variant = NULL;
  printf("MIT Scheme, VMS version\n");
  printf("^AH (CTRL-A, then H) shows help on interrupt keys.\n");
  Int_Char = 0;
  TT_Channel = 0;
  Init_System_Clock();
  OS_Re_Init();
  return;
}

void
OS_Re_Init()
{
  static $DESCRIPTOR(Device_Name,"SYS$INPUT");

  if (TT_Channel == 0)
  {
    long Return_Status = SYS$ASSIGN(&Device_Name,&TT_Channel,3,0);

    if (Return_Status != SS$_NORMAL)
    { fprintf(stderr, "\nUnable to find Terminal: SYS$ASSIGN\n");
      Microcode_Termination(TERM_EOF);
    }
    else Setup_Scheme_Interrupts();
  }
  else Setup_Scheme_Interrupts();
  Initialize_Input();
  return;
}

void
OS_Quit()
{
  if (TT_Channel != 0)
    Clear_Scheme_Interrupts();
  return;
}
