;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus"))


;;;; PRINTOUT: An Expressional FORMAT command.

;;; PRINTOUT is a facility for producing formatted output from SCHEME.
;;; It's goal is essentially the same as Common LISP's FORMAT, but
;;; PRINTOUT attempts to retain the expressional syntax of SCHEME
;;; rather than sinking into the baroque mire of Teco-Like formatting
;;; strings.

;;; PRINTOUT is a procedure which takes an arbitrary number of `print
;;; tokens' as arguments; this list of tokens constitutes a PRINTOUT
;;; expression.  Each token is either an EXECUTE-TOKEN or an arbitrary
;;; object to be printed using the PRINTOUT-PRINTER procedure.
;;; Execute tokens, however, are not printed but rather provide a
;;; procedure and a set of arguments to apply the procedure to.  For
;;; instance, the identifier $NL is bound to an execute token which
;;; applies NEWLINE to no arguments; an expression like:
;;;   (printout $NL "Line One" $NL "Line Two" $NL "Line Three")
;;; produces output appropriately divided into lines:
;;;   Line One
;;;   Line Two
;;;   Line Three
;;; Execute tokens are commonly produced by procedures; these provide
;;; the expressional formatting commands PRINTOUT uses.  To define a
;;; new `command' for PRINTOUT, the user defines a procedure which
;;; returns an execute token producing the desired behaviour.

;;; This is the printer for non-executable tokens.  Strings are
;;; printed without quotes and symbols without slashification.  This
;;; is redefined (later in the file) to print procedures out using a
;;; mapping from procedures to names.
(define printout-printer display)

;;; Lists starting with this CONS node are executable tokens.
(define %%execute-token-tag%% (list 'PRINTOUT-EXECUTE))
;;; This returns #T for executable tokens.
(definline (execute-token? thing)
  (and (pair? thing) (eq? (car thing) %%execute-token-tag%%))) 
(define (make-execute-token function args)
  (cons %%execute-token-tag%% (cons function args)))
(declare (integrate execute-token-procedure execute-token-arguments))
(define execute-token-procedure cadr)
(define execute-token-arguments cddr)

(define (printout . tokens)
  (define (process-token token)
    (if (execute-token? token)
	(apply (execute-token-procedure token)
	       (execute-token-arguments token)) 
	(printout-printer token)))
  (define (process-tokens tokens)
    (if (not (null? tokens))
	(sequence (process-token (first tokens))
		  (process-tokens (rest tokens)))))
  (process-tokens tokens)
  #T)

(define (execute-token function . args)
  (make-execute-token function args)) 


;;;; Basic PRINTOUT commands

;;; Produces a newline.
(define $nl (execute-token (lambda () (newline))))
(define ($nls number)
  ;; Produces multiple newlines.
  (define (repeat-newlines n)
    (cond ((= n 0) #T) 
	  (else (newline) (repeat-newlines (-1+ n)))))
  (execute-token repeat-newlines number))

(define ($indent indent . printout-args)
  ;; Uses native indentation facilities to product indented output:
  (define (printout-indented . x)
    (with-left-margin indent (lambda () (apply printout x))))
  (make-execute-token printout-indented printout-args))

;;; WRITES (as opposed to DISPLAYING) its argument
(define ($literal x) (execute-token write x))
;;; Prints out a value which is computed at print-time.
(define ($delay fcn . args)
  (define (print-value . args) (printout (apply fcn args)))
  (make-execute-token print-value args))

(define (print-time t)
  (let ((hours (car t)) (minutes (cadr t)) (seconds (third t)))
    (display hours)
    (display (if (< minutes 10) ":0" ":")) (display minutes)
    (display (if (< seconds 10) ":0" ":")) (display seconds)))
;;; Prints out a time-description (a list of hours, minutes, and seconds).
(define ($time t) (execute-token print-time t))

(define (print-now) (print-time (hours-minutes-seconds)))
;;; Prints the current time.
(define $now (execute-token print-now))

;;; Prints a number and a quantity, attempting to pluralize the
;;; quantity depending on the number.
(define ($count number quantity . plural)
  (if (= number 1) (execute-token printout number " " quantity)
      (if plural (make-execute-token printout (cons* number " " plural))
	  (execute-token printout number " " quantity "s"))))
       

;;;; Pretty printing functions.

;;; This handles the printing of functions prettily.  The
;;; `pretty-name' of a function is either a list of print-tokens (a
;;; PRINTOUT specification), a procedure, or an object to be DISPLAYed.
(define procedure-pretty-name (make-mutable))
(define (name-procedure! procedure name)
  ;; Names a function: NAME is a symbol, a printing procedure, or a
  ;; PRINTOUT specification.
  ((modifier procedure-pretty-name) procedure name)
  procedure)

;;; This is a description of what a function does to be used by
;;; various utilities and PRINTOUT commands.
(define procedure-description (make-mutable))
(define set-procedure-description! (modifier procedure-description))
(define (procedure! procedure pretty-name . description)
  (name-procedure! procedure pretty-name)
  (if description
      (set-procedure-description! procedure description))
  procedure)

(define (print-procedure p)
  ;; This prints a procedure prettily if possible:
  (let ((name (procedure-pretty-name p)))
    (cond ((undefined? name) (display (procedure-name p)))
	  ((procedure? name) (name))
	  ((execute-token? name) (printout name))
	  ((pair? name) (apply printout name))
	  (ELSE (display name)))))

(define (describe-procedure p)
  (let ((description (procedure-description p)))
    (cond ((undefined? description) (display "undocumented"))
	  ((procedure? description) (description))
	  ((execute-token? description) (printout description))
	  ((pair? description) (apply printout description))
	  (ELSE (display description)))))

;;; Prints something as a procedure; not really neccessary since
;;; PRINTOUT-PRINTER (redefined below) prints procedures prettily as a
;;; special case.
(define ($procedure p) (execute-token print-procedure p))
(define $function $procedure)

(define ($procedure-description p) (execute-token describe-procedure p))
(define ($described-procedure p)
  (define (print-procedure-description)
    (print-procedure p)
    (display " [") (describe-procedure p) (display "] "))
  (execute-token print-procedure-description))

;;; This changes PRINTOUT-PRINTER to print procedures out prettily.
(set! printout-printer
      (named-lambda (printout-printer x)
	(if (procedure? x)
	    (let ((name (procedure-pretty-name x)))
	      (cond ((undefined? name) (display (procedure-name x)))
		    ((procedure? name) (name))
		    ((execute-token? name) (printout name))
		    ((pair? name) (apply printout name))
		    (ELSE (display name))))
	    (display x))))

;;;; PRINTOUT recursive invocations

;;; These are all commands which end up calling PRINTOUT recursively
;;; on some argument. 

(define ($nested . args) (make-execute-token printout args))
(define ($printout args) (make-execute-token printout args))
(define ($call proc) (make-execute-token proc ()))

(define ($for-each fcn list)
  (define (mapper l)
    (if (pair? l)
	(sequence (apply printout (fcn (first l)))
		  (mapper (rest l)))))
  (execute-token mapper list))

(define (printout-comma-list list-of-things print-format)
  (if (not (null? list-of-things))
      (if (null? print-format)
	  (printout-printer (car list-of-things))
	  (apply printout (print-format (car list-of-things)))))
  (cond ((null? (rest list-of-things)) #F)
	(ELSE (display ", ") (printout-comma-list (cdr list-of-things) print-format))))
(define ($comma-list list . map-fcn)
  (if (null? list) "Nothing"
      (execute-token printout-comma-list list (if map-fcn (car map-fcn) #F))))


;;;; Specialized PRINTOUT functions

(define (yes-or-no-p . args)
  ;; Prints out a question and then asks for YES or NO (in a variety
  ;; of languages) to return #T or #F.
  (apply printout args)
  (let ((input (read)))
    (cond ((memq input '(yes yup yeah ok sure oui ja yah hai)) #T)
	  ((memq input '(no nope nah  no! nyet non nein never)) #F)
	  (else (printout $nl "Please answer with YES or NO")
		(apply yes-or-no-p args)))))

;;; This accepts Y or N and asks for YES or NO if it gets neither.
(define (y-or-n-p . args)
  ;; Prints out a question and then accepts Y/N (or Space/Rubout) tor
  ;; return #T or #F.
  (apply printout args)
  (let ((input (read-char)))
    (cond ((memq input '(#\Y #\y #\Space)) #T)
	  ((memq input '(#\N #\n #\Rubout)) #F)
	  (else (printout $nl "Please answer with YES or NO")
		(apply yes-or-no-p args)))))
