;;; -*- Scheme -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))

;;; Extensions to C-Scheme.


;;;; DEFMACRO

(in-package system-global-environment
(define (add-syntax! name expander)
  (SYNTAX-TABLE-DEFINE SYSTEM-GLOBAL-SYNTAX-TABLE name expander))
)

(define-syntax define-macro
  (macro (pattern . body)
    (let ((the-macro `(macro ,(cdr pattern) ,@body)))
      `(sequence
	(define-syntax ,(car pattern) ,the-macro)	; Compile-time
	(add-syntax! ',(car pattern) ,the-macro)))))

(add-syntax! 'define-macro
	     (macro (pattern . body)
	       (let ((the-macro `(macro ,(cdr pattern) ,@body)))
		 `(sequence
		    (define-syntax ,(car pattern) ,the-macro); Compile-time
		    (add-syntax! ',(car pattern) ,the-macro)))))


;;;; Useful macros:

;;; This provides a hook for requesting inline coding:
(define-macro (definline call expression)
  (define (collect-vars params var-list)
    (cond ((null? params) var-list)
	  ((symbol? params) (cons params var-list))
	  ((pair? params)
	   (collect-vars (cdr params) (cons (car params) var-list)))))
  (if (pair? call)
      `(begin (declare (integrate-operator ,(first call)))
	      (define ,call
		(declare (integrate ,@(collect-vars (cdr call) ())))
		,expression))
      `(begin (declare (integrate ,call)) (define ,call ,expression))))
	    
;;; This defines a record struction with a given print-form.
(define-macro (define-structure name slots print-form . ignore)
  (define (symbol-append . args) (string->symbol (apply string-append (map symbol->string args))))
  (define (generate-accessors slots index accessors)
    (if (null? slots) accessors
	(generate-accessors
	 (rest slots) (1+ index)
	 `((definline (,(symbol-append name '- (first slots)) ,name) (vector-ref ,name ,index))
	   ,@accessors))))
  (define (generate-modifiers slots index accessors)
    (if (null? slots) accessors
	(generate-modifiers
	 (rest slots) (1+ index)
	 `((definline (,(symbol-append 'set- name '- (first slots) '!) ,name ,(first slots))
	     (vector-set! ,name ,index ,(first slots)))
	   ,@accessors))))
  `(begin (define (,(symbol-append 'CONS- name) ,@slots) (vector ',name ,@slots))
	  (definline (,(symbol-append name '?) x) (and (vector? x) (eq? (vector-ref x 0) ',name)))
	  ,@(generate-accessors slots 1 ())
	  ,@(generate-modifiers slots 1 ())
	  (define (,(symbol-append 'PRINT- name) ,name) ,print-form)
	  ((access add-unparser-special-object! unparser-package)
	   ',name ,(symbol-append 'PRINT- name))))
;;; In C-Scheme structures are just vectors whose first element is a symbol:
(definline (structure? x)
  (and (vector? x) (not (zero? (vector-length x))) (symbol? (vector-ref x 0))))

(define-macro (unwind-protect form . unwind-forms)
  `(dynamic-wind (lambda () #T) (lambda () ,form) (lambda () ,@unwind-forms)))


;;;; Random functions

(declare (integrate-operator rest first second))

(define rest cdr)
(define first car)
(define second cadr)

(define (procedure-name procedure)
  (cond ((primitive-procedure? procedure) (primitive-procedure-name procedure))
	((compound-procedure? procedure)
	 (let ((l  (procedure-lambda procedure))
	       (extractor (lambda (x . trash) x)))
	   (let ((name (lambda-components l extractor)))
	     (if (eq? name LAMBDA-TAG:UNNAMED) procedure name))))
	(else procedure)))

(define (string x)
  (if (string? x) x
      (if (symbol? x) (symbol->string x)
	  (error "Not a string:" x))))

(define (number->string number)
  (with-output-to-string (lambda () (display number))))

(define (identity x) x)


;;;; Random utilities.

(define binary-fasload (make-primitive-procedure 'binary-fasload))
(define (pure-load filename #!optional environment)
  (newline) (display ";;; Loading ") (write filename)
  (display " into ") (write environment) (display " ..... ")
  (let ((res (binary-fasload filename)))
    (display "read .....")
    (purify res #T)
    (scode-eval res (if (unassigned? environment)
			(rep-environment) environment))))

(define hours-minutes-seconds time)
(define systime (make-primitive-procedure 'system-clock))

(definline (gc-count) (access meter gc-statistics-package))

(define directory-history ())

(define (pwd) (working-directory-pathname))

(define (pushd! new-path)
  (set! directory-history (cons (working-directory-pathname) directory-history))
  (set-working-directory-pathname! new-path)
  (cons (working-directory-pathname) directory-history))
(define (popd!)
  (cond ((null? directory-history) "Nothing to pop!")
	(ELSE (set-working-directory-pathname! (first directory-history))
	      (set! directory-history (cdr directory-history)))))
(define (swapd!)
  (let ((top (car directory-history)))
    (set-car! directory-history (working-directory-pathname))
    (set-working-directory-pathname! top)
    (cons top directory-history)))
(define (cd! . arg) (if arg (pushd! (first arg)) (popd!)))


;;;; Weak conses

;;; Weak conses are conses whose CAR and CDR turn to #F if nobody else
;;; is pointing at them.  They're useful for a variety of data structures.

(define %%weak-cons-type%% (microcode-type 'weak-cons))

(definline (weak-cons x y)
  (system-pair-cons %%weak-cons-type%% x y))

(define weak-car system-pair-car)
(define weak-cdr system-pair-cdr)
(define set-weak-car! system-pair-set-car!)
(define set-weak-cdr! system-pair-set-cdr!)
(declare (integrate weak-car weak-cdr set-weak-car! set-weak-cdr!))

(define (weak-assq key a-list)
  (if (null? a-list) #F
      (if (eq? (weak-car (car a-list)) key)
	  (car a-list)
	  (weak-assq key (cdr a-list)))))


;;;; Bit strings

;;; Bit strings are used all over the place in TYPICAL.

(define make-bit-string (make-primitive-procedure 'make-bit-string))
(definline (make-empty-bit-string n) (make-bit-string n #F))
(define bit-string-length (make-primitive-procedure 'bit-string-length))
(define empty-bit-string (make-empty-bit-string 0))
(define (print-bits bitstring)
  (define (bit-print index to-go)
    (cond ((zero? to-go) (newline))
	  ((check-bit bitstring index) (write-char #\1) (bit-print (1+ index) (-1+ to-go)))
	  (else  (write-char #\0) (bit-print (1+ index) (-1+ to-go)))))
  (bit-print 0 (bit-string-length bitstring)))

;(define same-bit-string? (make-primitive-procedure 'same-bit-string?))
(define check-bit (make-primitive-procedure 'bit-string-check))
(define bit-string-modify (make-primitive-procedure 'bit-string-modify))
(define bit-string-modify! (make-primitive-procedure 'bit-string-modify!))
(define bit-string-and (make-primitive-procedure 'bit-string-and))
(define bit-string-or (make-primitive-procedure 'bit-string-or))

(definline (make-empty-type-cache size)
  (cons (make-empty-bit-string size) (make-empty-bit-string size)))


;;;; Union operations

(define (union . lists)
  (define (binary-union l1 l2)
    (if (null? l1) l2
	(if (memq (first l1) l2) (union (cdr l1) l2)
	    (binary-union (rest l1) (cons (first l1) l2)))))
  (cond ((null? lists) ())
	((null? (cdr lists)) (car lists))
	((null? (car lists)) (apply union (cdr lists)))
	(else (binary-union (apply union (cdr lists)) (car lists)))))


;;; Stuff for fancy indentation (who cares?)

;;; This is a list of the form ((port . indentation) (port . indentation) ...)
(define margin-procs ())
(define system-newline newline)

(define (indenting-newline . given-port)
  (let ((port (if (null? given-port) (current-output-port) (car given-port))))
    (define (print-margin margin)
      (if (not (null? margin))
	  (if (eq? (car (car margin)) port)
	      (sequence (print-margin (cdr margin)) ((cdr (car margin)) port))
	      (print-margin (cdr margin)))))
    (system-newline port)
    (print-margin margin-procs)))
(set! newline indenting-newline)

(define (with-left-margin-procedure margin proc . args)
  (fluid-let ((margin-procs
	       (cons (cons (current-output-port) margin)
		     margin-procs)))
    (apply proc args)))

(define (with-left-margin margin proc . args)
  (with-left-margin-procedure
   (lambda (port) (display margin port))
   (lambda () (apply proc args))))


;;;; Lookup table stuff.

;;; A lookup table consists of a value vector and a `weak list' of
;;; keys;  to look up a key, we go down the weak list of keys and get
;;; the position (the LOOKUP-ENTRY) of the key in this list; this is
;;; then used to index the vector of values.  Since the key-list is
;;; `weak,' 

;;; Each lookup starts with a value vector of this size.
(define lookup-start-size 10)
;;; When a lookup is expanded, the vector is grown by this much:
(define lookup-growth-factor 1.5)
;;; This is used for kludging storing #F in the lookup.
(define %%bogus-false%% '(FALSE))

;;; A `lookup' consists of a value table, a size factor, and a list of
;;; weak conses of keys.
(define (make-empty-lookup) (list (make-vector lookup-start-size) 0))

(define (lookup-entry key lookup)
  ;; Finds the index for a key in a lookup table, returning #F if it isn't there.
  ;; We keep the length of the key-list and number entries from the
  ;; end by subtracting depth in the list from the length of the key-list.
  (define (search-list list index)
    (if (null? list) #F
	(if (eq? (weak-car list) key) (- (cadr lookup) index)
	    (search-list (weak-cdr list) (1+ index)))))
  (search-list (cddr lookup) 1))
(define (get-empty-index key lookup)
  ;; Gets an empty index for the lookup which will reference KEY.
  ;; If no index is currently free, it allocates a new one.
  (define (search-list list index)
    (if (null? list) (allocate-new-index key lookup)
	(if (false? (weak-car list))
	    (sequence (set-weak-car! list key) (- (cadr lookup) index))
	    (search-list (weak-cdr list) (1+ index)))))
  (search-list (cddr lookup) 1))
(define (allocate-new-index for-item lookup)
  ;; Allocates a new index in the lookup for a particular item: 
  (let ((new-index (cadr lookup)))
    (if (> (vector-length (car lookup)) new-index)
	(sequence
	  (set-car! (cdr lookup) (1+ new-index))
	  (set-cdr! (cdr lookup) (weak-cons for-item (cddr lookup)))
	  new-index)
	(let* ((old-vector (car lookup))
	       (old-size (vector-length old-vector))
	       (new-size (ceiling (* (1+ old-size) lookup-growth-factor)))
	       (new-vector (make-vector new-size)))
	  (define (copy-entries from)
	    (if (>= from 0)
		(sequence
		  (vector-set! new-vector from (vector-ref old-vector from))
		  (copy-entries (-1+ from)))))
	  (copy-entries (-1+ old-size))
	  (set-car! lookup new-vector)
	  (set-car! (cdr lookup) (1+ new-index))
	  (set-cdr! (cdr lookup) (weak-cons for-item (cddr lookup)))
	  new-index))))

(definline (do-lookup item lookup fail)
  ;; Looks up an item in the lookup, returning FAIL if it isn't there:
  (let ((index (lookup-entry (if (false? item) %%bogus-false%% item)
			     lookup)))
    (if (false? index) fail (vector-ref (car lookup) index))))
(definline (mutate-lookup! item lookup modifier fail)
  ;; Changes the lookup value for an item by using the function MODIFIER
  (let ((index (lookup-entry (if (false? item) %%bogus-false%% item)
			     lookup)))
    (if (false? index)
	(let ((new-value (modifier fail)))
	  (let ((new-index
		 (get-empty-index (if (false? item) %%bogus-false%% item)
				  lookup)))
	    (vector-set! (car lookup) new-index new-value))
	  fail)
	(let* ((current-value (vector-ref (car lookup) index))
	       (new-value (modifier current-value)))
	  (vector-set! (car lookup) index new-value)))))
(definline (modify-lookup! item lookup value)
  (let ((index (lookup-entry (if (false? item) %%bogus-false%% item)
			     lookup)))
    ;; Sets the value for a key in a lookup.
    (if (false? index)
	(let ((new-index
	       (get-empty-index (if (false? item) %%bogus-false%% item)
				lookup)))
	  (vector-set! (car lookup) new-index value))
	(vector-set! (car lookup) index value))))
(define (over-lookup fcn lookup)
  ;; Maps over the valid key/value pairs in the lookup.
  (define (iterate over-keys index)
    (if (not (null? over-keys))
	(if (null? (weak-car over-keys))
	    (iterate (weak-cdr over-keys) (-1+ index))
	    (cond ((eq? (weak-car over-keys) %%bogus-false%%)
		   (fcn #F (vector-ref (car lookup) index))
		   (iterate (weak-cdr over-keys) (-1+ index)))
		  (ELSE (fcn (weak-car over-keys) (vector-ref (car lookup) index))
			(iterate (weak-cdr over-keys) (-1+ index)))))))
  (iterate (cddr lookup) (-1+ (cadr lookup))))
