;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Functions for collections:

;;; Collections have the property of enumerability, implemented by an
;;; enumerating function which returns a list of elements.
(define collection-enumerator (td-property 'elements-function))
(define (collection-elements type)
  (if (defined? (collection-enumerator type))
      ((collection-enumerator type))
      (ERROR "TYPE is not a collection!" type)))
;;; Being a collection is defined by having a collection enumerator.
(define (collection? type)
  (and (type-description? type) (defined? (collection-enumerator type))))

;;; Some collections or other types are modifiable in that new objects
;;; can be added to the type.  This addition is done by the type's
;;; MODIFY-FUNCTION. 
(define satisfaction-modifier   (td-property 'modify-function))
;;; Having a MODIFY-FUNCTION is the criterion of being a mutable type.
(define (mutable-type? x)
  (and (type-description? x) (defined? (satisfaction-modifier x))))

;;; A mutable collection is both enumerable and mutable.
(define (mutable-collection? type)
  (and (type-description? type)
       (defined? (collection-enumerator type))
       (defined? (collection-modifier type))))
(define collection-modifier satisfaction-modifier)

;;; Functions for modifying collections:

(define (modify-collection! element type flag)
  (if (defined? (collection-modifier type))
      ((collection-modifier type) element flag)
      (ERROR "TYPE is not a mutable collection!" type)))
(define (put-in-collection! x set)
  (modify-collection! x set #T))
(define (take-from-collection! x set)
  (modify-collection! x set #F))

;;; This returns the known elements of type; these are the declared
;;; elements of its specializations; to find these elements we search
;;; the tree above the type for collections and the extract those
;;; elements of the collection which satisfy the type.
(define (known-elements type)
  (let ((t (search-tree collection? (list type) td-generalizations)))
    (extract (lambda (x) (in? x type)) (collection-elements t))))


;;;----------------------------------------------------------------
;;;; SIMPLE-TYPE combinator
;;; This is the simplest synthetic type definition which simply
;;; specifies a predicate beneath an existing type.

(define simple-type
  (type-generator 
   (lambda (pred gen) pred)
   (lambda (pred gen) (if (null? gen) () (list gen)))
   (lambda (pred gen) ())))
(define (print-simple-type type)
  (let ((spec (td-specification type)))
    (printout ($literal (or (procedure-name (second spec)) (second spec))))))
(declare-combinator-printer simple-type print-simple-type)


;;;----------------------------------------------------------------
;;;;  PRIMITIVE-SET-OF combinator
;;; This combinator creates a new type which is satisfied by a
;;; fixed set of objects.

;;; In creating a primitive set, we first generate the type (by using
;;; the PRIMTIVE-SET-OF-TYPE combinator described below) and then
;;; annotating this type with the appropriate enumeration function.
;;; In calling the primitive combinator, we also standardize the list
;;; of elements into a canonical form; this could be done in a nicer
;;; fashion (for instance, we could put elements in canonical order so
;;; that set-identical lists would produce the same type) but nothing
;;; like this is currently done.
(define (primitive-set-of elements genzn)
  (let ((elements (primitive-set-standardizer elements)))
    (let ((set (primitive-set-of-type elements genzn)))
      ((modifier collection-enumerator) set  (lambda () elements))
      set)))
(define primitive-set-standardizer (make-standardizer))

;;; Simply does subsetting by predicates, using an user-declared
;;; generalization.
(define primitive-set-of-type
  (type-generator
   (lambda (elements g)
     (define (in-primitive-set? x) (memq x elements))
     in-primitive-set?)
   (lambda (elements g) (if g (list g) (list lattice-top)))
   (lambda (elements g) ())))
(define (print-explicit-set type)
  (let ((spec (td-specification type)))
    (printout ($comma-list (second spec)))))
(declare-combinator-printer primitive-set-of-type print-explicit-set)


;;;----------------------------------------------------------------
;;;;  QUERY-TYPE combinator
;;;  Specifies a type which requests user input:
;;;     E.G. (QUERY-TYPE "an interesting number" NUMBERS)

;;; This isn't used much because it leads to the asking of questions
;;; at random times.  If it is used, it is important to place it
;;; strictly beneath the right type; otherwise, you'll get questions
;;; ALL the time....
(define query-type
  (type-generator
   (lambda (query-class genzn)
     (named-lambda (query? x)
       (if (satisfies? x genzn)
	   (if dont-ask-stupid-questions (%undefined)
	       (yes-or-no-p $nl "Is " x " " query-class "? ")))))
   (lambda (ignore generalization) (list generalization))
   (lambda (ignore1 ignore2) ())))
(define (print-query-type type)
  (printout "Any object which is " (second (td-specification type))))
(declare-combinator-printer query-type print-query-type)

(define dont-ask-stupid-questions #F)



;;; ----------------------------------------------------------------
;;;;   DIVIDED-TYPE combinator
;;; Divided types are types which have in-tests and out-tests to
;;; determine satisfaction or failure; if neither passes, the type is
;;; undetermined. 

(define divided-type
  (type-generator
    (lambda (in-test out-test beneath)
      (lambda (x)
	(let ((possible? (satisfies? x beneath)))
	  (if (defined-true? possible?)
	      (if (out-test x) #F (if (in-test x) #T (%undefined)))
	      possible?))))
    (lambda (ignore1 ignore2 beneath) (list beneath))
    (lambda (ignore1 ignore2 ignore3) ())))



;;; ----------------------------------------------------------------
;;;;   GENERATED-COLLECTIONS combinator
;;; Generated collections are a special sort of collection which only
;;; have instances added on creation.  Thus, they can return
;;; definitively true or false, while not being fixed at any point.

;;; UNLOCKED-COLLECTIONS is used to enable adding to a generated collection.
(define unlocked-collections ())
;;; This is the list of all generated collections.
(define all-generated-collections ())
;;; Returns true if a type is a generatd collection.
(define (generated-collection? x) (memq x all-generated-collections))

(define (generated-collection beneath)
  (let ((in-set ()))
    (define (in-set? x) (memq x in-set))
    (let ((collection (simple-type in-set? beneath)))
      (define (new-element! x flag)
	(if (memq collection unlocked-collections)
	    (if flag (set! in-set (cons x in-set)))
	    (ERROR "Generated collection S hasn't been unlocked! " collection)))
      (set! all-generated-collections (cons collection all-generated-collections))
      ((modifier collection-modifier) collection new-element!)
      ((modifier collection-enumerator) collection (lambda () in-set))
      collection)))

(define (collection-generator for-collection generator-function)
  (define (generator . x)
    (let ((instance (apply generator-function x)))
      (fluid-let ((unlocked-collections (cons for-collection unlocked-collections)))
	(put-in-collection! instance for-collection))
      instance))
  (procedure! generator ($procedure generator-function)
	      "Generates instances of " for-collection " by "
	      ($procedure-description generator-function))
  generator)


;;; ----------------------------------------------------------------
;;;;  EMPIRICAL-COLLECTION combinator
;;; Empirical collections (this is a bad name) are collections which
;;; are satisfied for their members and undefined for everything else.

(define (make-empirical-collection beneath)
  (let ((members ()))
    (define (empirical-member? x)
      (if (memq x members) #T
	  (if (in? x beneath) (%undefined) #F)))
    (let ((type (simple-type empirical-member? beneath)))
      ((modifier collection-enumerator) type (lambda () members))
      ((modifier collection-modifier) type
       (lambda (x f)
	 (if f (if (not (memq x members))  (set! members (cons x members)))
	     (if (in? x beneath)
		 (error "Can't declare things out of the collection TYPE!" type)))))
      type)))


;;; ----------------------------------------------------------------
;;;;  DIVIDED-COLLECTION combinator
;;; Divided collections are like divided types; they are determined by
;;; an in-set and out-set and anything not in either is undetermined
;;; for the type.

(define (make-divided-collection beneath)
  (let ((in-set ()) (out-set ()) (now (hours-minutes-seconds)))
    (define (new-element! x flag)
      (if (in? x beneath)
	  (cond (flag
		 (if (memq x out-set) (set! out-set (delq x out-set)))
		 (if (not (memq x in-set)) (set! in-set (cons x in-set))))
		((not flag)
		 (if (memq x in-set) (set! in-set (delq x in-set)))
		 (if (not (memq x out-set)) (set! out-set (cons x out-set)))))
	  (error "Empirical membership of X only valid under BENEATH" (list x beneath))))
    (define (in-set? x) (memq x in-set))
    (define (out-set? x) (memq x out-set))
    (let* ((type (divided-type in-set? out-set? beneath))
	   (complement (complement type)))
      ((modifier collection-enumerator) type (lambda () in-set))
      ((modifier collection-enumerator) complement (lambda () out-set))
      ((modifier collection-modifier) type new-element!)
      ((modifier collection-modifier)
       complement (lambda (e f) (new-element! e (not f))))
      type)))
