;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))

;;;; Representing functions.

;;; Domain and Range (real useful)
(define function-domain (make-invertible))
(define function-range  (make-invertible))

(define (declared-function? x)
  (and (procedure? x) (defined? (function-domain x))))

;;; Declaring functions.  For simplicity, you can defined domain and
;;; range in the same expression as you set the name.  This is
;;; convenient and having nicely named functions is quite useful.
(define (declare-function! fcn from to . pretty-name)
  (if pretty-name (name-procedure! fcn pretty-name))
  ((modifier function-domain) fcn from)
  ((modifier function-range)  fcn to)
  fcn)
(define function! declare-function!)

;;; This converts an n-arg procedure into a procedure which takes one
;;; argument: a list of the n-args.
(define (%simplified-procedure from-procedure)
  (define (converted-procedure args) (apply from-procedure args))
  converted-procedure)
(define simplified-procedure (simple-cache %simplified-procedure))

;;; This extends the conversion above by making the converted
;;; procedure into a declared function with appropriate range and
;;; domain. 
(define (declare-procedure! original-procedure range . domain)
  (declare-function! (simplified-procedure original-procedure)
		     (apply cross-product domain) range))

;;; This is primarily for use by CYRANO.  It converts a procedure of
;;; n arguments into a procedure which takes a single argument of a
;;; TUPLE of the n arguments.  The new procedure is appropriately
;;; declared with domain and range. 
(define (tuple-procedure! original-procedure range . domain)
  (define (tuplized-procedure input)
    (apply original-procedure (tuple-elements input)))
  (declare-function! tuplized-procedure (apply tuple-product domain) range
		     ($procedure original-procedure) "-TUPLE"))

;;; Inverses for domain and range.
(define domain-for (inverse function-domain))
(define range-for  (inverse function-range))
;;; Things to copy when memoizing
(set! memoizer-properties-to-copy-from-fcn
      (union (list function-domain function-range)
	     memoizer-properties-to-copy-from-fcn))


;;;; Image Constraints

;;; ----------------------------------------------------------------
;;;;   IMAGE-CONSTRAINT combinator
;;; This combinator returns a type which restricts the domain of a
;;; mapping based on a restriction of its range.  An object satisfies
;;; the image constraint of F on T if F(object) satisfies T.

(define (image-constraint mapping image)
  ;; Do type checking.
  (if (not (defined? (function-domain mapping)))
      (error "FCN does not have domain and range declared!" mapping))
  ;; Check for overlap problems. 
  (let ((range (function-range mapping)))
    (cond ((eq? range image)
	   (let ((type (function-domain mapping)))
	     ((modifier (td-properties image)) mapping type)
	     type))
	  ((<< range image)
	   (error "Image IMAGE is too big to restrict by MAPPING!" image))
	  ((disjoint? range image)
	   (error "Image IMAGE disjoint from range of MAPPING!" image))
	  (ELSE (let ((type (constraint-on-image mapping image)))
		  ((modifier (td-properties image)) mapping type)
		  type)))))

;;; The predicate function just calls SATISFIES? on the result of the mapping.
(define (image-constraint-function mapping image)
  (let ((predicate (td-predicate image))
	(domainp (td-predicate (function-domain mapping))))
    (define (maps-into-image? x)
      (and (domainp x) (satisfies? (mapping x) image)))
    maps-into-image?))

;;; Finding generalizations, we climb looking for other constraints
;;; placed on the image.
(define (find-superior-constraints mapping image)
  (let ((superiors ()) (range (function-range mapping)))
    (define (check-possible-superior type)
      (let ((props (td-properties type)))
	(cond ((defined? (props mapping))
	       (set! superiors (cons (props mapping) superiors))
	       #F)
	      ((eq? type range) 
	       (set! superiors (cons (function-domain mapping) superiors))
	       #F)
	      (else #T))))
    (maptree check-possible-superior (list image) td-generalizations)
    superiors))

;;; Finding specializations, we descend looking for other constraints
;;; placed on the image.
(define (find-inferior-constraints mapping image)
  (let ((inferiors ()))
    (define (check-possible-inferior type)
      (cond ((defined? ((td-properties type) mapping))
	     (set! inferiors (cons ((td-properties type) mapping) inferiors))
	     #F)
	    (else #T)))
    (maptree check-possible-inferior (list image) td-specializations)
    inferiors))

;;; This is the actual combinator definition.
(define constraint-on-image
  (type-generator image-constraint-function
		  find-superior-constraints
		  find-inferior-constraints))

;;; Printing image constraints.
(define (print-image-constraint type)
  (let ((spec (td-specification type)))
    (printout ($procedure (second spec)) "(x)=" (third spec))))
(declare-combinator-printer constraint-on-image print-image-constraint)


;;;; Dismantling composite image constraints

;;; We may wish to extract image constraint information from composite
;;; types; the functions in this section implement that.

(define (determine-image-constraints mapping type)
  ;; This climbs the lattice trying to find image constraints
  ;; contributing to the type.
  (let ((types ()))
    (define (check-type type)
      (let ((specification (td-specification type)))
	(if (and (eq? (first specification) constraint-on-image)
		 (eq? (second specification) mapping))
	    (set! types (cons (third specification) types)))
	#T))
    (maptree check-type (list type) td-generalizations)
    (or (minimal-type-set types) (list (function-range mapping)))))

(define (mapping-constraint mapping type)
  ;; This constructs a new type by intersection the results of an image
  ;; constraint search.
  (let ((constraints (determine-image-constraints mapping type)))
    (if constraints (apply <AND> constraints) #F)))

(define (all-image-constraints type)
  ;; This returns all the constraints which define a type.
  (let ((properties (td-properties type))
	(constraints ()))
    (define (process-mapping-range mapping range)
      (if (and (defined? range) (<<? type range))
	  (if (defined? (properties mapping))
	      (set! constraints
		    (cons (properties mapping) constraints)))))
    ((enumerator function-range) process-mapping-range)
    constraints))

;;; This is for prettying printing type descriptions (primarily by DF).
(define (images-constrained x) (or (all-image-constraints x) (%undefined)))
(define ($constraint x)
  ($nested ($procedure (image-constraint-mapping x)) " constrained: " x))


;;;----------------------------------------------------------------
;;;; SUBTYPES-OF combinator
;;; This is a meta-type combinator which produces a type of types; a
;;; SUBTYPES-OF type is satisfied solely by the specializations of a type.

(define subtypes-of
  (type-generator
   (lambda (type)
     (define (subtype? x) (and (type-description? x) (td-<<? x type)))
     subtype?)
   (lambda (type)
     (cons types (get-fringe type td-generalizations subtypes-of-cache)))
   (lambda (type) (get-fringe type td-specializations subtypes-of-cache))))

(define (print-subtypes-of-type type)
  (printout "SubTypes(" (second (td-specification type)) ")"))
(declare-combinator-printer subtypes-of print-subtypes-of-type)

(define subtypes-of-cache (memoizer-cache subtypes-of))

(define (get-fringe root direction property)
  (let ((fringe ()))
    (define (check-node node)
      (let ((prop (property node)))
	(if (defined? prop)
	    (begin (set! fringe (cons prop fringe)) #F)
	    #T)))
    (maptree check-node (list root) direction)
    fringe))


;;;----------------------------------------------------------------
;;;; SUPERTYPES-OF combinator
;;; This is a meta-type combinator which produces a type of types; a
;;; SUPERTYPES-OF type is satisfied solely by the generalizations of a type.

(define supertypes-of
  (type-generator
   (lambda (type)
     (define (supertype? x) (and (type-description? x) (td-<<? type x)))
     supertype?)
   (lambda (type)
     (cons types (get-fringe type td-specializations supertypes-of-cache)))
   (lambda (type) (get-fringe type td-generalizations supertypes-of-cache))))

(define (print-supertypes-of-type type)
  (printout "SuperTypes(" (second (td-specification type)) ")"))
(declare-combinator-printer supertypes-of print-supertypes-of-type)

(define supertypes-of-cache (memoizer-cache supertypes-of))
