;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Tracing confirmation

(define trace-confirmation #T)
(define (confirmation-message . args)
  (if trace-confirmation (apply message args))) 

;;; This file implements the confirmation and disconfirmation of
;;; empirical regularities in CYRANO.  The ideas are detailed in the
;;; TYPICAL technical report, but the key notion is that regularities
;;; are represented by `empirical classes' which are divided types
;;; annotated with SAMPLE and EVIDENCE functions.  These functions ---
;;; applied to an object --- return two types in TYPICALs lattice and
;;; if the first subsumes the second (neccessarily or empirically),
;;; the regularity is considered satisfied.  This file implements the
;;; machinery for discovering this relationahip.


;;;; Empirical classes

;;; Defining empirical classes.

(define sample-space-generator (td-property 'sample-space-generator))
(define evidence-space-generator (td-property 'evidence-space-generator))
(define given-instances (td-property 'given-instances))
(define empirical-specialization-of (td-property 'empirical-specialization-of))

;;; An empirical class has a sample generator function, an
;;; evidence generator function, a set of given instances and a set of
;;; given non-instances.  The sample and evidence generators generate
;;; types which correspond to samles and evidence for a given
;;; regularity.  The EMPIRICAL-CLASS procedure creates an empirical
;;; class appropirately annotated with these properties.
(define (empirical-class beneath sample-fcn evidence-fcn)
  (let* ((property (divided-collection beneath))
	 (given-examples (type (list "Given examples of " property)
			       generated-collection property))
	 (given-non-examples (type (list "Given non-examples of " property)
				   generated-collection (complement property))))
    ((modifier sample-space-generator) property (simple-cache sample-fcn))
    ((modifier evidence-space-generator) property (simple-cache evidence-fcn))
    ((modifier given-instances) property given-examples)
    ((modifier given-instances) (complement property) given-non-examples)
    ((modifier empirical-specialization-of) property beneath)
    ((modifier empirical-specialization-of) (complement property) beneath)
    property))

;;; An empirical class is defined by having a sample space generator
;;; (and an evidence space generator too, but we only have to check
;;; for one.)
(define (empirical-class? x)
  (and (type-description? x) (defined? (sample-space-generator x))))
(define empirical-classes
  (type "Empirical Classes" simple-type empirical-class? mutable-collections))

;;; Creating sample spaces 
(define (sample-space object property)
  (let ((space ((sample-space-generator property) object)))
    (if (or (eq? space object) (eq? space property)) space
	(sequence (name-type! space (list "Samples for " property " on " object))
		  space))))

;;; Creating evidence spaces 
(define (evidence-space object property)
  (let ((space ((evidence-space-generator property) object)))
    (if (or (eq? space object) (eq? space property)) space
	(sequence (name-type! space (list "Evidence for " property " on " object))
		  space))))

;;; Properties for DF and EF
(add-property! sample-space-generator   empirical-classes "Sample Space Generator")
(add-property! evidence-space-generator empirical-classes "Evidence Space Generator")


;;;; Given properties

;;; An object can be declared GIVEN for an empirical class which will
;;; both forestall experiments and cancel any inherent interestingness
;;; arising from the property.
(define (given! x empirical-class)
  (put-in-collection! x empirical-class)
  (let ((givens (given-instances empirical-class)))
    (fluid-let ((unlocked-collections (cons givens unlocked-collections)))
      (put-in-collection! x givens))
    x))
(define (given? x empirical-class) (in? x (given-instances empirical-class)))

(define (infer-given! property from to)
  (if (in? from property) (given! to property)))

;;; This is for making inferences which are not expressed strictly in
;;; the lattice.  
(define (add-inference! from to)
  (define (make-inference about)
    (if (not (in? about to))
	(sequence 
	  (confirmation-message $NL "!!! Inferring that " about " is in " to)
	  (confirmation-message $NL "!!!  since it is in " from)
	  (inhibit-daemon! make-inference about)
	  (assert! about to))))
  (procedure! make-inference 'INFERENCE "Infer that instances of "
	      from " are also instances of " to)
  (add-daemon! make-inference from))


;;;; Examples and Counterexamples

(define (create-examples-space subject property)
  (<AND> (sample-space subject property)
	 (evidence-space subject property)))
(define examples-space (canonical-cache create-examples-space))
(name-procedure! examples-space 'EXAMPLES-SPACE)

(define (create-counterexamples-space subject property)
  (<AND> (sample-space subject property)
	 (complement (evidence-space subject property))))
(define counterexamples-space (canonical-cache create-counterexamples-space))
(name-procedure! counterexamples-space 'COUNTEREXAMPLES-SPACE)


;;;; Creating experiments.

;;; This should be done in a more intelligent fashion.
(define (example-threshold subject property) 10)

;;; This starts the process of determining if SUBJECT does (or does
;;; not) satisfy PROPERTY.
(define (hypothesis subject property)
  (let ((sample (sample-space subject property))
	(evidence (evidence-space subject property)))
    ;; We begin by computing the sample and evidence spaces; we then
    ;; check for preemptive assertions, or neccessary satisfaction.
    (cond ((defined? (satisfies? subject property)) #F) ; Catch preemptive assertions.
	  ;; If the sample space is within the evidence space, the
	  ;; property is trivially satisfied.
	  ((subsumed-by? sample evidence)
	   (confirmation-message $NL "!!! By definition, " subject " is in " property)
	   (confirmation-message $NL "!!! Instances of " sample " are always in " evidence)
	   (assert! subject property))
	  ;; If the sample space is disjoint from the evidence space, the
	  ;; property trivially fails.
	  ((disjoint? sample evidence)
	   (confirmation-message $NL "!!! By definition, " subject "cannot be shown in " property)
	   (confirmation-message $NL "!!! Instances of " sample " are never in " evidence)
	   (assert! subject (complement property)))
	  ;; Otherwise, set up daemons for catching counterexamples
	  ;; and counting examples; then begin the task of generating
	  ;; samples for the property you wish to show.
	  (ELSE (add-daemon! (counterexamples-daemon subject property)
			     (counterexamples-space subject property))
		(add-daemon! (examples-daemon subject property)
			     (examples-space subject property))
		(task generate-samples subject property)))))

;;; A daemon generator for hypothesizing.
(define (hypothesize property)
  (procedure! (lambda (subject) (hypothesis subject property))
	      'HYPOTHESIZE "Test for membership in " property))


;;;; Example and Counterexample Daemons

;;; This is the counterexample identified in a counterexample class.
(define identified-counterexample (td-property 'identified-counterexample))

;;; This constructs a counterexamples daemon which notes the occurence
;;; of counterexamples to a property for a subject.  The usual version
;;; of this to use is the cached version, COUNTEREXAMPLES-DAEMON.
(define (make-counterexamples-daemon subject property)
  (define (notice-counterexample x)
    ;; If you find a counterexample, announce it
    (confirmation-message $NL "!!! Found a counterexample excluding " subject " from " property)
    ((modifier identified-counterexample) (counterexamples-space subject property) x)
    (confirmation-message $NL "!!! Declaring " property " unsatisfied for " subject)
    ;; And assert its membership in the appropriate empirical class.
    (assert! subject (complement property))
    ;; Remove the apparatus for noticing counterexamples.
    (let ((counterexamples (counterexamples-space subject property)))
      (remove-daemon! notice-counterexample counterexamples)
      (add-generator! counterexamples
		      (make-combiner counterexamples (lambda () x))))
    property)
  (procedure! notice-counterexample 'NOTICE-COUNTEREXAMPLE
	      "Note evidence that " subject " is not in " property))
(define counterexamples-daemon (canonical-cache make-counterexamples-daemon))

;;; This constructs an examples daemon which counts unique occurences
;;; of examples and --- at some threshold --- asserts an empirical
;;; property satisfied for some subject.  The usual version to use is
;;; the cached definition of this, EXAMPLES-DAEMON.
(define (make-examples-daemon subject property)
  (let ((threshold (example-threshold subject property))
	(examples-seen ()) (examples-count 0))
    ;; The THRESHOLD is the number of examples required for confirmation.
    (define (notice-example x)
      (if (not (memq x examples-seen))
	  (begin (set! examples-seen (cons x examples-seen))
		 (set! examples-count (+ examples-count 1))))
      (if (defined? (satisfies? subject property))
	  (remove-daemon! notice-example (examples-space subject property))
	  (if (> examples-count threshold) 
	      ;; If there are `enough' examples, announce your discovery
	      (begin
		(confirmation-message
		 $NL "!!! Found " ($count examples-count "example") " of " property " for " subject)
		(confirmation-message $NL "!!! Declaring " property " tentatively satisfied for " subject)
		;; And finally, assert it into the appropriate empirical class.
		(assert! subject property)
		property))))
    (procedure! notice-example 'NOTICE-EXAMPLE
		"Note evidence showing that " subject " is in " property)))
(define examples-daemon (canonical-cache make-examples-daemon))


;;;; Generating samples:
;;; This initiates a sample generation task.

(define (generate-samples subject property)
  (let* ((sample-space (sample-space subject property))
	 (source (get-instance-generator sample-space)))
    (define (test-if-done at-count samples-seen)
      (if (undefined? (satisfies? subject property))
	  (go get-next-sample at-count samples-seen)
	  (if (satisfies? subject property)
	      (finished! "Concluded after generating " ($count at-count "sample")
			 " that " subject " is in " property)
	      (finished! "Discovered after generating " ($count at-count "sample")
			 " that " subject " is not in " property))))
    (define (index-sample sample count samples-seen)
      (progress-report! "Generated " ($count count "sample") " to determine if "
			subject " satisfies " property)
      (index sample)
      (test-if-done count (cons sample samples-seen)))
    (define (get-next-sample count samples-seen)
      (let ((sample (source)))
	(if (and (defined? sample) (not (memq sample samples-seen)))
	    (index-sample sample (1+ count) samples-seen)
	    (test-if-done count samples-seen))))
    (progress-report! "Trying to generate samples determining if "
		      subject " is in " property)
    (go get-next-sample 0 ())))
(name-procedure! generate-samples 'generate-samples)




