;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel"))



;;;; Tracing example generation

(define trace-example-generation #F)
(define (examples-message . printout-args)
  (if trace-example-generation (apply message printout-args)))

;;; Example generator is critically important in CYRANO because it is
;;; the only way it gets any data to make conclusions about; the
;;; problem solving of CYRANO consists in its construction of examples
;;; generators.  The key idea in example generation is that we
;;; construct a network of example generators which call each other to
;;; produce examples at any given point.  The nodes in this network of
;;; genrators are types, and the links are INSTANCE-GENERATORS.  Every
;;; type has a set of instance generators; these are procedures which
;;; are either primitive (and particular to the type) or (more
;;; commonly) combine examples generated by other types.


;;;; Instance Generators

;;; Each type has a list of GENERATORS; the generator of a type calls
;;;  a random generator on its list.  Generators are of three sorts:
;;;  primitive generators, type generators, and combining generators.
;;;  Primitive generators are magic functions which generate instances
;;;  of the type; for instance, by picking a random member of an
;;;  enumerated collection.  Type generators are simply functions
;;;  which do the generation for a type.  And combining generators
;;;  combine the results of other generators in some way.

;;; Instance Generator methods.

;;; Instance generators are stored here:
(define instance-generators (td-property 'instance-generators))
(define get-instance-generators
  (generate-memoizer instance-generators (lambda (ignore) (list 0))))

;;; Stuff for DF and EF.
(add-type-property! instance-generators types)
(define (print-generators g)
  (printout ($list-field "Example Generators" (map $procedure (cdr g)))))
(set-property-printer! instance-generators print-generators)

;;; This adds a generator to a type.
(define (add-generator! type g)
  (let ((generators (get-instance-generators type)))
    (set-car! generators (+ (car generators) 1))
    (set-cdr! generators (cons g (cdr generators)))
    type))

;;; This makes a daemon for adding a particular type of generator.
(define (make-generation-daemon generator-generator)
  (define (add-generator-to-type type)
    (add-generator! type (generator-generator type)))
  (procedure! add-generator-to-type (list "USE-" ($procedure generator-generator))
	      "Use generating procedure " ($described-procedure generator-generator))
  add-generator-to-type)
(define use-generator (simple-cache make-generation-daemon))

(define (make-combiner for-type combiner . source-types)
  (let ((sources-reversed
	 (reverse (map get-instance-generator source-types))))
    (define (generate-from-sources sources result)
      (if (null? sources) (apply combiner result)
	  (let ((instance ((first sources))))
	    (if (defined? instance)
		(generate-from-sources (rest sources) (cons instance result))
		instance))))
    (define (combination-method)
      (generate-from-sources sources-reversed ()))
    (procedure! combination-method ($procedure combiner)
		"Apply " ($procedure combiner) " to "
		($comma-list source-types))
    combination-method))


;;;; Instance Generators

(define instance-sources
  (type "Instance Sources" empirical-collection types))

;;; This tries to avoid infinite recursion on generators.  When a
;;; geneator is applied, its result is undefined if its application is
;;; recursive.  Note that we can ask for examples of a TYPE
;;; recursively, but we just can't use the same method.  While we
;;; might pre-filter the generators applied by this criterion, it
;;; would be hairy and probably not terribly useful (this is a
;;; purely intuitive judgement; no arguments given).
(define active-instance-generators ())
(define (apply-generator generator for-type)
  (if (memq generator active-instance-generators) (%undefined)
      (fluid-let ((active-instance-generators
		   (cons generator active-instance-generators)))
	(if trace-example-generation
	    (message $NL ">|||> Using generator " ($procedure generator)
		     " to generate instance of " for-type))
	(generator))))

;;; An instance generator keeps a cache of instances generated and
;;; accesses the side-effectable table of instance generators.  When
;;; the instance generator is called, it calls one of these at random;
;;; if a `defined' result is returned, it is stored in the cache and
;;; returned as a result from the generator.  If no defined result is
;;; returned, a random instance from the cache is returned.
(define (construct-instance-generator type)
  (let ((generators (get-instance-generators type))
	(cache (make-vector 10 (%undefined)))
	(index 0))
    (define (guess)
      ;; This is called to try and pull examples out of the instance cache.
      (let ((second-try (vector-ref cache (random 10))))
	(if (and trace-example-generation (defined? second-try))
	    (message $NL "<|||< Using recorded example: " second-try))
	second-try))
    (define (process-attempt x)
      ;; This processes a generated example:
      (cond ((undefined? x)
	     ;; If the example is `undefined', generation failed....
	     (if trace-example-generation (message $NL "<?|||?< Generation failed...."))
	     (guess))
	    ((in? x type)
	     ;; If the example is defined and in the type, generation succeeded....
	     (if trace-example-generation
		 (message $NL "<|||< Successfully generated: " x))
	     (vector-set! cache index x)
	     (if (= index 9) (set! index 0) (set! index (1+ index)))
	     x)
	    (ELSE
	     ;; But we may have produced a bogus example...
	     (if trace-example-generation (message $NL "<?|||?< Bogus example generated: " x))
	     (guess))))
    (define (instance-generator)
      ;; The actual instance generator procedure chooses a generator
      ;; at random and processes its result.
      (let ((generator-count (car generators)))
	(if (zero? generator-count)
	    (sequence (if trace-example-generation
			  (message $NL "<?|||?> No generators for " type))
		      (%undefined))
	    (process-attempt
	     (apply-generator (list-ref (cdr generators) (random generator-count))
			      type)))))
    (procedure! instance-generator
		(list "GENERATE-INSTANCES-OF-TYPE-" (td-id type))
		(list "Generate instances of " type))
    instance-generator))

;;; Instance generators are cached.
(define instance-generator (td-property 'instance-generator))
(define make-instance-generator
  (generate-memoizer instance-generator construct-instance-generator))
(add-type-property! instance-generator instance-sources)
(set-property-name+format! instance-generator "Instance Generator" $function)

;;; This forces an instance generator to be created AND indexes the
;;; associated type.
(define (get-instance-generator type)
  (let ((generator (make-instance-generator type)))
    (assert! type instance-sources)
    generator))
;;; This makes an instance generator for a type but returns the type.
(define (declare-instance-generator! type)
  (get-instance-generator type)
  type)


;;;; Using generalizations and specializations as sources.
;;; To keep from spending all of its time climbing up and down the
;;; lattice in search of examples, this is encapsulated into a single
;;; generator procedure which tries a generalization of specialization
;;; at random.

(define related-sources (td-property 'related-sources))

(define (try-related-types to-type)
  (define (use-related-types)
    (define (try-related-type from-types)
      (let ((choice  (list-ref from-types (random (length from-types)))))
	(examples-message $NL ">|||> Using relation :" choice)
	((get-instance-generator choice))))
    (apply-if-defined try-related-type (related-sources to-type)))
  (add-generator! to-type use-related-types))
(procedure! try-related-types 'TRY-RELATED-TYPES
	    "Use the generalizations and specializations as example sources.")
(add-daemon! try-related-types (<AND> instance-sources synthetic-types))

(define (collect-related-sources of-type)
  (let ((related-types ()))
    (define (check-related-type t)
      (if (in? t instance-sources)
	  (sequence (if (not (memq t related-types))
			(set! related-types (cons t related-types)))
		    #F)
	  #T))
    (maptree check-related-type (genzns of-type) td-generalizations)
    (maptree check-related-type (speczns of-type) td-specializations)
    (if (not (null? related-types))
	((modifier related-sources) of-type related-types))))
(procedure! collect-related-sources 'COLLECT-RELATED-SOURCES
	    "Collect the sources related to this source.")
(add-daemon! collect-related-sources (<AND> instance-sources synthetic-types))

(define (use-as-related-source of-type)
  (let ((mutate! (mutator related-sources)))
    (define (add-related-source v)
      (if (defined? v)
	  (if (memq of-type v) v (cons of-type v))
	  (list of-type)))
    (define (update-type t)
      (if (and (in? t synthetic-types) (in? t instance-sources))
	  (sequence (mutate! t add-related-source) #F)
	  #T))
    (maptree update-type (genzns of-type) td-generalizations)
    (maptree update-type (speczns of-type) td-specializations)))
(procedure! use-as-related-source 'USE-AS-RELATED-SOURCE
	    "Use this type as a source for its generalizations and specializations.")
(add-daemon! use-as-related-source instance-sources)


;;;; Particular generation methods.

;;; This generator picks an element at random from the declared
;;; elements of a collection.
(define (collection-elements-generator type)
  (define (random-element)
    (let ((size (collection-size type)))
      (if (zero? size) (%undefined)
	  (list-ref (collection-elements type) (random size)))))
  (procedure! random-element
	      (list "RANDOM-ELEMENT-OF-TYPE-" (td-id type))
	      "Select a random element from the collection " type)
  random-element)
(add-daemon! (use-generator collection-elements-generator)
	     (<AND> collections instance-sources))

;;; Applicable to SUBTYPES-OF types, this keeps mapping over the
;;; sublattice beneath a type.
(define (subtypes-generator type)
  (let ((nodes ()))
    (define (enumerate-subtypes)
      (cond ((null? nodes) (set! nodes (list type)) type)
	    (ELSE (let ((head (car nodes)))
		    (set! nodes (append (td-specializations head) (cdr nodes)))
		    head))))
    (add-generator! type enumerate-subtypes)))
(add-daemon! (use-generator subtypes-generator) (<AND> instance-sources subtype-types))

;;; Applicable to SUPERTYPES-OF types, this keeps mapping over the
;;; sublattice above a type.
(define (supertypes-generator type)
  (let ((nodes ()))
    (define (enumerate-supertypes)
      (cond ((null? nodes) (set! nodes (list type)) type)
	    (ELSE (let ((head (car nodes)))
		    (set! nodes (append (td-generalizations head) (cdr nodes)))
		    head))))
    (add-generator! type enumerate-supertypes)))
(add-daemon! (use-generator supertypes-generator) (<AND> instance-sources supertype-types))


;;;; Filling in tuple constraints.
;;; These randomly instantiate tuples.

(define (get-list-constraints type)
  (define (collect-constraints type constraints)
    (if (<<? type conses)
	(collect-constraints (mapping-constraint cdr type)
			     (cons (or (mapping-constraint car type)
				       lisp-objects)
				   constraints))
	constraints))
  (reverse (collect-constraints type ())))
(define list-constraints
  (generate-memoizer (td-property 'list-constraints) get-list-constraints))
(name-procedure! list-constraints "LIST-CONSTRAINTS")

(define (tuple-constraints tuple-type)
  (list-constraints (mapping-constraint tuple-elements tuple-type)))

(define (tuple-constraints-as-tuple type) (make-tuple (tuple-constraints type)))
(define tuple-type->type-tuple
  (generate-memoizer (td-property 'tuple-constraints-as-tuple)
		     tuple-constraints-as-tuple))
(declare-function! tuple-type->type-tuple (subtypes-of tuples) (tuple-of types))
(name-procedure! tuple-type->type-tuple "TUPLE-TYPE->TYPE-TUPLE")
(add-property! (td-property 'tuple-constraints-as-tuple)
	       (subtypes-of tuples) "Element Constraints")

(define (random-tuple-generator for-type)
  (let ((element-types (tuple-constraints for-type)))
    (if (null? element-types) (%undefined)
	(apply make-combiner for-type tuple element-types))))

(define constrained-tuples
  (type "Constrained Tuples"
	simple-type (lambda (x) (not (null? (tuple-constraints x))))
	(subtypes-of tuples)))
(add-daemon! (use-generator random-tuple-generator)
	     (<AND> instance-sources constrained-tuples))


;;;; This replaces `collect examples'
;;; This establishes a daemon which `notices' examples into a
;;; collection which can be enumerated.

(define (make-collection-daemon collection)
  (define (record-in-collection! x)
    (assert! x collection))
  record-in-collection!)
(define collect-into (simple-cache make-collection-daemon))

(define (map-examples fcn type . number-spec)
  (let ((generator (get-instance-generator type)))
    (for-range (lambda (ignore) (fcn (generator)))
	       0 (if (null? number-spec) 20 (first number-spec)))))
(define (sample-examples number type . fcn-spec)
  (let ((generator (get-instance-generator type)))
    (for-range (if (null? fcn-spec)
		   (lambda (ignore) (printout $NL ">> " (generator)))
		   (let ((fcn (first fcn-spec)))
		     (lambda (ignore) (printout $NL ">> " (fcn (generator))))))
	       0 number)))



