#include <varargs.h>

#include "kernel.h"

static WEAK_NODE *first;

void Call_Terminators();

Init_Terminate () {
    Register_After_GC (Call_Terminators);
}

/* Register an object with the given group and termination function;
 * object can be marked as LEADER.
 */
void Register_Object (obj, group, term, leader_flag) Object obj; GENERIC group;
	PFO term; {
    WEAK_NODE *p;

    p = (WEAK_NODE *)Safe_Malloc (sizeof (*p));
    p->obj = obj;
    p->group = group;
    p->term = term;
    p->flags = leader_flag? WK_LEADER : 0;
    p->next = first;
    first = p;
}

void Deregister_Object (obj) Object obj; {
    WEAK_NODE *p, **pp;

    Disable_Interrupts;
    for (pp = &first; (p = *pp); ) {
	if (EQ(p->obj, obj)) {
	    *pp = p->next;
	    free ((char *)p);
	} else pp = &p->next;
    }
    Enable_Interrupts;
}

/* Search for an object of a given type (arg 1) and group (arg 2).
 * Use the given match function (arg 3); it is called with an object
 * and the remaining arguments of Find_Object() (a va_list).
 * Null is returned when the object has not been found.
 */
/*VARARGS*/
Object Find_Object (va_alist) va_dcl {
    WEAK_NODE *p;
    int type;
    GENERIC group;
    MATCHFUN match;
    va_list args;

    va_start (args);
    type = va_arg (args, int);
    group = va_arg (args, GENERIC);
    match = va_arg (args, MATCHFUN);
    for (p = first; p; p = p->next) {
	if (TYPE(p->obj) != type || p->group != group)
	    continue;
	if (match (p->obj, args)) {
	    va_end (args);
	    REVIVE_OBJ(p->obj);
	    return p->obj;
	}
    }
    va_end (args);
    return Null;
}

/* Terminate all objects belonging to the given group except leaders.
 */
void Terminate_Group (group) GENERIC group; {
    WEAK_NODE *p, **pp;

    Disable_Interrupts;
    for (pp = &first; (p = *pp); ) {
	if (p->group == group && !(p->flags & WK_LEADER)) {
	    if (p->term)
		(void)p->term (p->obj);
	    *pp = p->next;
	    free ((char *)p);
	} else pp = &p->next;
    }
    Enable_Interrupts;
}

/* Terminate all objects of a given type.
 */
void Terminate_Type (type) int type; {
    WEAK_NODE *p, **pp;

    Disable_Interrupts;
    for (pp = &first; (p = *pp); ) {
	if (TYPE(p->obj) == type) {
	    if (p->term)
		(void)p->term (p->obj);
	    *pp = p->next;
	    free ((char *)p);
	} else pp = &p->next;
    }
    Enable_Interrupts;
}

/* The after-GC function.  Leaders are terminated in a second pass.
 */
void Call_Terminators () {
    WEAK_NODE *p, **pp;

    Disable_Interrupts;
    for (pp = &first; (p = *pp); ) {
	if (IS_ALIVE(p->obj)) {
	    if (WAS_FORWARDED(p->obj))
		UPDATE_OBJ(p->obj);
	    pp = &p->next;
	} else if (p->flags & WK_LEADER) {
	    p->flags |= WK_MARK;
	    pp = &p->next;
	} else {
	    if (p->term)
		(void)p->term (p->obj);
	    *pp = p->next;
	    free ((char *)p);
	}
    }
    for (pp = &first; (p = *pp); ) {
	if (p->flags & WK_MARK) {
	    if (p->term)
		(void)p->term (p->obj);
	    *pp = p->next;
	    free ((char *)p);
	} else pp = &p->next;
    }
    Enable_Interrupts;
}
