;;;; This is the file SILVERADO.SCM

;;;    The Silverado bank is famous for having been defrauded of
;;;  zillions of dollars by a combination of stupidity and greediness,
;;;  and for having passed the cost of this fraud onto us taxpayers.

;;; Accounts in the Silverado have a name, a list of owners (who may
;;; jointly hold an account) and an initial balance.  Only account 
;;; owners may withdraw from an account any amount less than the
;;; balance.  The request to withdraw returns to the depositor the
;;; amount that he actually obtained from the bank.  Anyone may
;;; deposit in any account, but of course one may only withdraw or
;;; deposit positive amounts.

(define (make-account account-name owners balance)
  (let ((serialized (make-serializer)))
    (define (account mess)
      (cond ((account-queried account mess))
	    ((eq? mess 'account-name)
	     account-name)
	    ((eq? mess 'owners)
	     owners)
	    ((eq? mess 'balance)
	     balance)
            ((eq? mess 'withdraw)
	     (lambda (person amount)
	       (cond ((withdrawal-requested account person amount))
		     ((not (memq person owners)) 0)
		     ((< amount 0) 0)
		     ((< balance amount) 0)
		     (else
		      ((serialized
			(lambda ()
			  (set! balance (- balance amount)))))
		      amount))))
            ((eq? mess 'deposit)
	     (lambda (person amount)	       
	       (cond ((deposit-requested account person amount))
		     ((< amount 0) 0)
		     (else
		      ((serialized
			(lambda ()
			  (set! balance (+ balance amount))
			  amount)))))))	    
            (else (error "Unknown message -- ACCOUNT" mess))))
    account))


;;; The accounting system at the Silverado is rather crude.  No 
;;;  records are kept of transactions, and no questions are asked.
;;;  The following "stubs" are places where it is appropriate to add
;;;  code to perhaps make audit trails or other checks.  These
;;;  procedures may invisibly examine the transactions by returning
;;;  FALSE.  If they return any other value than false they capture
;;;  the flow of control.

(define (account-queried account message)
  false)
(define (withdrawal-requested account person amount)
  false)
(define (deposit-requested account person amount)
  false)

;;; The auditor for the Silverado is the infamous Poopers-and-Liebrand
;;;  company which does no more than print out the sum of the balances
;;;  of all the accounts in the bank.  But they sure charge lots for
;;;  that service!

(define (total-balances accounts)
 (apply + (map (lambda (account) (account 'balance)) accounts)))

(define (do-audit accounts)
  (write-line (list 'audit (total-balances accounts))))




;;; You may need the following utility procedure in your auditors

(define (union set1 set2)
  (define (loop s2 ans)
    (if (null? s2)
	ans
	(loop (cdr s2)
	      (if (memq (car s2) ans)
		  ans
		  (cons (car s2) ans)))))
  (loop set2 set1))


#|
;;; Most depositors are honest ... They work like this:

(define (make-honest-depositor name authorization laziness accounts) 
  (let ((num-accounts (length accounts)))
    (define (buzz)
      (let ((from (random num-accounts))
            (to (random num-accounts))
            (amount (random authorization)))
        (let ((pocket (((list-ref accounts from) 'withdraw) name amount)))
	  (((list-ref accounts to) 'deposit) name pocket)))
      (sleep-current-thread (random laziness))
      (buzz))
    buzz))


;;; Some depositors like to kite funds ... They work like this:

(define (make-kiting-depositor name authorization laziness kite-time accounts) 
  (let ((num-accounts (length accounts)))
    (define (buzz)
      (let ((from (random num-accounts))
            (to (random num-accounts))
            (amount (random authorization)))
        (let ((pocket (((list-ref accounts from) 'withdraw) name amount)))
	  (sleep-current-thread (random kite-time))
	  (((list-ref accounts to) 'deposit) name pocket)))
      (sleep-current-thread (random laziness))
      (buzz))
    buzz))


;;; Some depositors skim the "cream" off of their accounts:

(define (make-skimming-depositor name authorization cream laziness accounts) 
  (let ((num-accounts (length accounts)))
    (define (buzz)
      (let ((from (random num-accounts))
            (to (random num-accounts))
            (amount (random authorization)))
        (let ((pocket (((list-ref accounts from) 'withdraw) name amount)))
	  (((list-ref accounts to) 'deposit)
	   name
	   (if (< cream pocket)
	       (- pocket cream)
	       pocket))))
      (sleep-current-thread (random laziness))
      (buzz))
    buzz))
|#
