;;;;This file contains the data structure definitions for maintaining
;;;;census and geographic data.

(declare (usual-integrations))


(define (make-dataset geographic census)
  (list geographic census))

(define dataset-geographic car)
(define dataset-census cadr)


;;;; CENSUS DATA

#|
These data structures contain data from the 1990 US Census TIGER data
disks.  

The POP-DATA structure specifies a census block.  
Census blocks are small areas bounded on all sides by visible features
such as streets, roads, streams, and railroad tracks, and by invisible
boundaries such as city, township, and county limits, property lines,
and short, imaginary extensions of streets and roads.  
|#

(define-structure
  (pop-data (conc-name pop-data/)
	    (type vector)
	    (named (string->symbol "#[pop-data]"))
	    (constructor make-pop-data))
  (state false read-only true)		;codes for state, county, census tract
  (cnty false read-only true)
  (tract false read-only true)
  (blkgrp false read-only true)		;code for this block group
  (race-sex false read-only true)	;sub-structures ...
  (persons-per-unit false read-only true)
  (house-value false read-only true)
  (rent false read-only true)
  (blkgrp-boundary false)) ; this field is used to display the block group on a map

;;;each slot of a RACE-SEX structure contains an AGE structure, which
;;;specifies the breakdown by age of people of that racew and sex

(define-structure
  (race-sex (conc-name race-sex/)
	    (type vector)
	    (named (string->symbol "#[race-sex]"))
	    (constructor make-race-sex))
  (white-male false read-only true)
  (white-female false read-only true)
  (black-male false read-only true)
  (black-female false read-only true)
  (amerind-male false read-only true)  ; American Indian, Eskimo, or Aleut
  (amerind-female false read-only true)
  (asian-male false read-only true)
  (asian-female false read-only true)
  (other-male false read-only true)
  (other-female false read-only true))

(define race-sex-desc-list
  '((white male)
    (white female)
    (black male)
    (black female)
    (amerind male)
    (amerind female)
    (asian male)
    (asian female)
    (other male)
    (other female)))

;;;AGE
;;;each slot contains the number of people in the given age range

(define-structure
  (age (conc-name age/)
       (type vector)
       (named (string->symbol "#[age]"))
       (constructor make-age-data))
  (=<1 false read-only true)
  (=1-2 false read-only true)
  (=3-4 false read-only true)
  (=5 false read-only true)
  (=6 false read-only true)
  (=7-9 false read-only true)
  (=10-11 false read-only true)
  (=12-13 false read-only true)
  (=14 false read-only true)
  (=15 false read-only true)
  (=16 false read-only true)
  (=17 false read-only true)
  (=18 false read-only true)
  (=19 false read-only true)
  (=20 false read-only true)
  (=21 false read-only true)
  (=22-24 false read-only true)
  (=25-29 false read-only true)
  (=30-34 false read-only true)
  (=35-39 false read-only true)
  (=40-44 false read-only true)
  (=45-49 false read-only true)
  (=50-54 false read-only true)
  (=55-59 false read-only true)
  (=60-61 false read-only true)
  (=62-64 false read-only true)
  (=65-69 false read-only true)
  (=70-74 false read-only true)
  (=75-79 false read-only true)
  (=80-84 false read-only true)
  (=>85 false read-only true))

;;;the DESC-LIST information is used by the accessor functions for
;;;selecting items in the correct range

(define age-desc-list
  (list 0 1 3 5 6 7 10 12 14 15 16 17 18 19 20 21 22 25 30 35 40
	45 50 55 60 62 65 70 75 80 85))

;;;the AVERAGE-PER-CATEGORY list is used to compute averages

(define average-age-per-category
  (list .5 1.5 3.5 5. 6. 8. 10.5 12.5 14. 15. 16. 17. 18. 19. 20. 21. 23. 27.
	32. 37. 42. 47. 52. 57. 60.5 63. 67. 72. 77. 82. 87.))

;;; PERSONS-PER-UNIT
;;; each slot contains the number of housing units with that number of

(define-structure
  (persons-per-unit (conc-name persons-per-unit/)
		    (type vector)
		    (named (string->symbol "#[persons-per-unit]"))
		    (constructor make-persons-per-unit))
  (=1 false read-only true)
  (=2 false read-only true)
  (=3 false read-only true)
  (=4 false read-only true)
  (=5 false read-only true)
  (=6 false read-only true)
  (=>7 false read-only true))

(define persons-per-unit-desc-list
  (list 1 2 3 4 5 6 7))

(define average-persons-per-unit-per-category
  (list 1. 2. 3. 4. 5. 6. 8.))


;;;number of houses with a given value (in thousands of dollars)

(define-structure
  (house-value (conc-name house-value/)
	       (type vector)
	       (named (string->symbol "#[house-value]"))
	       (constructor make-house-data))
  (=<15 false read-only true)
  (=15-19 false read-only true)
  (=20-24 false read-only true)
  (=25-29 false read-only true)
  (=30-34 false read-only true)
  (=35-39 false read-only true)
  (=40-44 false read-only true)
  (=45-49 false read-only true)
  (=50-59 false read-only true)
  (=60-74 false read-only true)
  (=75-99 false read-only true)
  (=100-124 false read-only true)
  (=125-150 false read-only true)
  (=150-174 false read-only true)
  (=175-199 false read-only true)
  (=200-249 false read-only true)
  (=250-299 false read-only true)
  (=300-399 false read-only true)
  (=400-499 false read-only true)
  (=>500 false read-only true))

;;;house values are in thousands of dollars
(define house-value-desc-list
  (list 0 15 20 25 30 35 40 45 50
	60 75 100 125 150 175 200 250
	300 400 500))

(define average-house-value-per-category
  (list 0 7.5 17.5 22.5 27.5 32.5 37.5 42.5 47.5 55. 67.5
	87.5 112.5 137.5 162.5 187.5 225. 275. 350. 450. 750.))

(define-structure
  (rent (conc-name rent/)
	(type vector)
	(named (string->symbol "#[rent]"))
	(constructor make-rent-data))
  (=<100 false read-only true)
  (=100-149 false read-only true)
  (=150-200 false read-only true)
  (=200-249 false read-only true)
  (=250-300 false read-only true)
  (=300-349 false read-only true)
  (=350-400 false read-only true)
  (=400-449 false read-only true)
  (=450-500 false read-only true)
  (=500-549 false read-only true)
  (=550-600 false read-only true)
  (=600-649 false read-only true)
  (=650-700 false read-only true)
  (=700-749 false read-only true)
  (=750-999 false read-only true)
  (=>1000 false read-only true)
  (none false read-only true))

(define rent-desc-list
  (list 0 100 150 200 250 300 350 400 450 500 550 600 650 700 750 1000))

(define average-rent-per-category
  (list 0 50. 125. 175. 225. 275. 325. 375. 425. 475. 
	525. 575. 625. 675. 725. 857. 1125. 0.))


;;;; GEOGRAPHIC DATA

;;; a TOWN data structure contains information need to draw a map of a city

(define-structure
  (town (conc-name town/)
	(type vector)
	(named (string->symbol "#[town]"))
	(constructor make-town))
  (min-long false read-only true)
  (min-lat false read-only true)
  (max-long false read-only true)
  (max-lat false read-only true)
  (streets false read-only true))

;;;each STREET structure specifies from and to points (latitude and
;;;longitude) street name, and other stuff


(define-structure
  (street (conc-name street/)
	  (type vector)
	  (named (string->symbol "#[street]"))
	  (constructor make-street))
  (from-long false read-only true)
  (from-lat false read-only true)
  (to-long false read-only true)
  (to-lat false read-only true)
  (cfcc false read-only true)		;street classification
  (name false read-only true)
  (suffix false read-only true)
  (from-addr-l false read-only true)
  (to-addr-l false read-only true)
  (from-addr-r false read-only true)
  (to-addr-r false read-only true)
  (zip-l false read-only true)
  (zip-r false read-only true)
  (tract-l false read-only true)
  (tract-r false read-only true)
  (blkgrp-l false read-only true)	;block-groups to right and to left
  (blkgrp-r false read-only true))

;;; These functions decode the TIGER street classification information 

(define (really-major-road? street)
  (and (char=? #\A (string-ref (street/cfcc street) 0))
       (let ((foo (string->number (substring (street/cfcc street) 1 2))))
	 (< foo 3))))

(define (major-road? street)
  (and (char=? #\A (string-ref (street/cfcc street) 0))
       (let ((foo (string->number (substring (street/cfcc street) 1 2))))
	 (= foo 3))))

(define (neighborhood-road? street)
  (and (char=? #\A (string-ref (street/cfcc street) 0))
       (let ((foo (string->number (substring (street/cfcc street) 1 2))))
	 (>= foo 4))))

(define (road? street)
  (char=? #\A (string-ref (street/cfcc street) 0)))

(define (railroad? street)
  (char=? #\B (string-ref (street/cfcc street) 0)))

(define (water? street)
  (char=? #\H (string-ref (street/cfcc street) 0)))

(define (boundary? street)
  (char=? #\F (string-ref (street/cfcc street) 0)))

(define (blkgrp-boundary? street)
  (not (and (= (street/tract-l street) (street/tract-r street))
	    (= (street/blkgrp-l street) (street/blkgrp-r street)))))

