;;; Rishiyur Nikhil, August 23, 1988
;;;
;;; ``defabs'', a tool for defining abstractions in Scheme.
;;;
;;; Usage:
;;;    (defabs 'foo)
;;;    (defabs '(baz x1 .. xN))            where N >= 1
;;;
;;; The first form defines a constant called ``foo'' and a predicate called ``foo?''
;;;
;;; The second form defines
;;;    a constructor called ``baz''      (a function of N arguments)
;;;    a predicate   called ``baz?''     (a function of 1 argument)
;;;    a selector    called ``baz-x1''   (a function of 1 argument)
;;;    ...
;;;    a selector    called ``baz-xN''   (a function of 1 argument)
;;;
;;; Example: binary trees can be defined as follows:
;;;
;;;    (defabs 'empty)
;;;    (defabs '(node val left right))
;;;
;;; A function to add 1 to each number in a binary tree can now be written:
;;;
;;;    (define (add1 tree)
;;;       (cond
;;;         ( (empty? tree) empty )
;;;         ( (node?  tree) (node (+ 1 (node-val tree))
;;;                               (add1 (node-left tree))
;;;                               (add1 (node-right tree))) ))
;;;         ( else          (error "add1: argument is not a tree") ) ))

(define *defabs-debug* #f)

(define (defabs spec)
    (enter-defs (spec->defs spec)))

(define (enter-defs def-and-msg-list)
    (if (null? def-and-msg-list)
       'defabs-done
        (let*
            ((def-and-msg (car def-and-msg-list))
             (def (car def-and-msg))
             (msg (cdr def-and-msg)))
          (eval def user-initial-environment)
          (newline)
          (display msg)
          (if *defabs-debug*
              (begin (display " : ") (write def)))
          (enter-defs (cdr def-and-msg-list))
         )))

(define (spec->defs spec)
    (cond
       ( (symbol? spec)            (atomic-spec->def1 spec) )
       ( (and (pair? spec)
              (> (length spec) 1)) (structured-spec->def1 spec) )
       ( else                      (error "defabs specification syntax error") ) ))

(define (atomic-spec->def1 cons-symbol)
    (let*
        ((cons-name   (symbol->string cons-symbol))
         (pred-name   (string-append cons-name "?"))
         (pred-symbol (string->symbol pred-name)))

      (list (cons `(define ,cons-symbol ',cons-symbol)
                  (string-append "Constructor " cons-name))
            (cons `(define ,pred-symbol (lambda (x) (eq? x ',cons-symbol)))
                  (string-append "Predicate   " pred-name)))))

(define (structured-spec->def1 spec)
    (let*
        ((cons-symbol (car spec))
         (cons-name   (symbol->string cons-symbol)))

      (cons (make-cons-def cons-symbol cons-name (length (cdr spec)))
          (cons (make-pred-def cons-symbol cons-name)
              (make-sel-defs cons-name (cdr spec) 1)))))

(define (make-cons-def cons-symbol cons-name number-of-slots)
    (let
        ((xs (make-arg-list 0 number-of-slots)))
      (cons `(define ,cons-symbol
                 (lambda ,xs
                     ,(cons 'list (cons `',cons-symbol xs))))
             (string-append "Constructor " cons-name))))

(define (make-arg-list j number-of-slots)
    (if (>= j number-of-slots)
        '()
        (cons (string->symbol (string-append "x" (number->string j)))
              (make-arg-list (+ j 1) number-of-slots))))

(define (make-pred-def cons-symbol cons-name)
    (let*
        ((pred-name (string-append cons-name "?"))
         (pred-symbol   (string->symbol pred-name)))
      (cons
        `(define ,pred-symbol
            (lambda (x) (and (pair? x) (eq? (car x) ',cons-symbol))))
        (string-append "Predicate   " pred-name))))

(define (make-sel-defs cons-name slot-list position)
    (if (null? slot-list)
        '()
        (cons
          (make-sel-def  cons-name (car slot-list) position)
          (make-sel-defs cons-name (cdr slot-list) (+ position 1)))))

(define (make-sel-def cons-name bare-slot-symbol position)
    (let*
        ((sel-name (string-append cons-name "-" (symbol->string bare-slot-symbol)))
         (sel-symbol (string->symbol sel-name)))
      (cons
        `(define ,sel-symbol
              (lambda (x) (list-ref x ,position)))
        (string-append "Selector    " sel-name))))
