/*    File:	 term.h  (~bevemyr/KAM/Emulator/term.h)
 *    Author:	 Johan Bevemyr
 *    Created:	 Mon Mar 23 19:29:31 1992
 *    Purpose:   Example term definition file.
 */ 

/**********************************************************************
 *
 * Tagged objects
 *
 * A tagged object consists of a value field (usually a pointer) and
 * a tag field. 
 *
 *         00000000 00000000 00000000 000000   00 
 *         ---------------------------------   --
 *            Value field                      Tag
 *             
 */

#define WORDSIZE 32
#define TAGSIZE 2


typedef unsigned long TAGGED;

typedef enum {
    VAR = 0,            /* 000...000 */
    CON,                /* 000...001 */
    STR                 /* 000...010 */
} tagvalue;

#define TAGMASK 3       /* 000...011 */

#define Tag(P,T)         ((TAGGED) (((unsigned long) (P)) | (T)))
#define TagOf(P)         ((tagvalue) (((unsigned long) (P)) & (TAGMASK)))
#define TagToPointer(P)  ((TAGGED *) ((((unsigned long) (P))|TAGMASK)-TAGMASK))
#define RemoveTag(P,T)   ((TAGGED *) (((unsigned long) (P)) - (T)))
#define PointerToTerm(P) Tag(P,CON)
#define TermToPointer(T) RemoveTag(T,CON)

/**********************************************************************
 *
 * Variables
 *
 */

#define IsVAR(T)        TagOf(T) == VAR
#define RefVAR(To,T)    To = *((TAGGED *) (T))

/**********************************************************************
 *
 * Constants
 *
 * Constants are tagged with an CON tag. A subtag is used to
 * distinguish between atoms and integers. This only has to
 * be done when printing them! On integers the subtag is set.
 * 
 * 1 000000...000 01
 * = ------------ ==
 * ^         ^     ^
 * subtag    |     |
 *         value   |
 *              atom tag
 *
 *
 * Atoms:
 * The atom table starts at atombase. To get smaller atom identifiers
 * we substract this address from the atom pointer, and add it when we
 * need the actual address (only when we need the print name).
 *
 * Integers:
 * An integer value is leftshifted two positions in order to make
 * space for the tag. An additional bit is used for the subtag. This
 * allows for 29 bit integers (in the range of 536870912).
 *
 */

#define IsCON(T)         (TagOf(T) == CON)

#define SUBTAG           (((long) 0x1L) << (WORDSIZE-1))   /* 10...0 */

#define GetIntSubtag(T)  (((unsigned long) (T)) >> (WORDSIZE -1))
#define Remove_Subtag(T) (((long) (T)) & ~SUBTAG)
#define Add_Subtag(T)    (((long) (T)) | SUBTAG)

/*****************************************
 * Atoms
 */

struct atom {
    char *pname;
};

typedef struct atom *atom;

#define IsAtom(T)        ((IsCON(T) && (GetIntSubtag(T) == 0)))
#define AddAtomBase(A)   ((atom) (((long) (A)) + ((long) atombase)))
#define RemAtomBase(A)   ((atom) (((long) (A)) - ((long) atombase)))
#define GetString(T)     AddAtomBase(RemoveTag(T,CON))->pname

/*****************************************
 * Integers
 */


#define IsInt(I)         (IsCON(I) && (GetIntSubtag(I) == 1))
#define MakeInteger(I)   Add_Subtag(Tag((((long) (I))<<2),CON))
#define GetInteger(I)    (((long) Remove_Subtag(RemoveTag(I,CON))) >> 2)

/**********************************************************************
 *
 * Functors
 *
 * We store the arity in the topmost 8 bits of an atom 
 *
 *         00000000  00000000 00000000 000000  00 
 *         --------  ------------------------  --
 *         Arity          Atom identifier      Tag
 *
 * This means that integers cannot be used as functornames since
 * they have a subtag.
 *
 */

#define ARITYSIZE        8
#define ARITYOFFSET	 (32 - ARITYSIZE)
#define ARITYMASK        0xff              /* 000...0011111111 */

#define MakeFunctor(T,A) ((TAGGED) (((unsigned long) (T)) |\
				    ((A) << ARITYOFFSET)))
#define FunctorToAtom(F) ((TAGGED) ((((unsigned long) (F)) << ARITYSIZE)\
				    >> ARITYSIZE))
#define ArityOf(F)	 ((int) ((((unsigned long) (F)) >> ARITYOFFSET) &\
				 ARITYMASK))

/**********************************************************************
 *
 * Structures
 *
 */

struct structure {
    TAGGED functor;
    TAGGED arg[ANY];
};
			 
typedef struct structure *structure;

#define IsSTR(T)	TagOf(T) == STR
#define GetArg(T,A)     &(((structure) RemoveTag(T,STR))->arg[A])
#define GetFunctor(T)   ((structure) RemoveTag(T,STR))->functor	 
#define GetArity(T)     ArityOf(GetFunctor(T))

/**********************************************************************
 *
 * Other
 *
 */

typedef TAGGED code;

typedef enum {
    FALSE = 0,
    TRUE = 1
} BOOL;

extern TAGGED store_atom PROTO((char *));
