/*
 * PCN Abstract Machine Emulator
 * Authors:     Arun Nanda    (07/17/91)
 *              Steve Tuecke  (Modifed for better integration with PCN)
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * usc.c	-  Microsecond clock package.
 */

#include "pcn.h"

#ifdef UPSHOT

#include "usc.h"

usc_time_t _p_usc_MD_rollover_val = 0;

#ifdef encore
#   include <parallel.h>
#   define usc_MD_timer_size  (sizeof(unsigned)*8)
    unsigned *_p_usc_encore_timer;
#endif


#if defined(balance) || defined(symmetry)
#   define usc_MD_timer_size  (sizeof(usclk_t)*8)
#endif


#ifdef tc2000
#   define usc_MD_timer_size  (sizeof(unsigned long)*8)
#endif


#if defined(ipsc860) || defined(delta)
/* These are included from gauge.h *
#   if defined(delta)
#       include <mesh.h>
#   else
#       include <cube.h>
#   endif
*/
#   define usc_MD_timer_size ((sizeof(long)*8)+3)
#   define usc_MD_ticks_per_usec (HWHZ/1000000)
#endif


#ifdef PCN_USE_BSD_TIME
#   include <sys/time.h>
    usc_time_t _p_usc_MD_reference_time = 0;
#endif


/*
 * _p_usc_init()
 *
 * Initialize the usc timers
 */
void _p_usc_init()
{
#if defined(encore)
    _p_usc_encore_timer = timer_init();
    _p_usc_MD_rollover_val = (usc_time_t) ((1<<usc_MD_timer_size)-1);
#endif

#if defined(balance) || defined(symmetry)
    unsigned long roll;

    usclk_init();

    roll = 1 << (usc_MD_timer_size-1);
    _p_usc_MD_rollover_val = (usc_time_t) (roll + roll - 1);
#endif

#ifdef tc2000
    unsigned long roll;

    roll = 1 << (usc_MD_timer_size-1);
    _p_usc_MD_rollover_val = (usc_time_t) (roll + roll - 1);
#endif

#if defined(ipsc860) || defined(delta)
    esize_t hwtime;
    double ustime;
    
    hwtime.shigh = hwtime.slow = ~0x0;
    hwtime.shigh = (hwtime.shigh & 0x7) << (sizeof(long)*8-3);
    hwtime.slow = ((hwtime.slow >> 3) & ~(0x7 << (sizeof(long)*8-3)))
	| hwtime.shigh;
    ustime = (unsigned long)hwtime.slow * 0.8;
    _p_usc_MD_rollover_val = (usc_time_t) ustime; 
#endif

#ifdef PCN_USE_BSD_TIME
    struct timeval tp;
    struct timezone tzp;
    unsigned long roll;
    
    gettimeofday(&tp,&tzp);
    _p_usc_MD_reference_time = (usc_time_t) tp.tv_sec;
    
    roll = 1 << ((sizeof(usc_time_t)*8)-1);
    roll = roll + roll - 1;
    _p_usc_MD_rollover_val = (usc_time_t) (roll / 1000000);
#endif
} /* _p_usc_init() */


/*
 * _p_usc_MD_clock()
 *
 * Return the current clock value.
 */
usc_time_t _p_usc_MD_clock()
{
#ifdef tc2000
    struct {
	unsigned long hi;
	unsigned long low;
    } usclock;
    
    get64bitclock(&usclock);
    return((usc_time_t)usclock.low);
#endif

#if defined(ipsc860) || defined(delta)
    esize_t hwtime;
    double ustime;
    
    hwclock(&hwtime);
    hwtime.shigh = (hwtime.shigh & 0x7) << (sizeof(long)*8-3);
    hwtime.slow = ((hwtime.slow >> 3) & ~(0x7 << (sizeof(long)*8-3)))
	| hwtime.shigh;
    ustime = (unsigned long)hwtime.slow * 0.8;
    return((usc_time_t)ustime);
#endif

#ifdef PCN_USE_BSD_TIME
    unsigned long ustime;
    struct timeval tp;
    struct timezone tzp;
    
    gettimeofday(&tp,&tzp);
    ustime = (unsigned long) (tp.tv_sec - _p_usc_MD_reference_time);
    ustime = ustime % _p_usc_MD_rollover_val;
    ustime = (ustime * 1000000) + (unsigned long) tp.tv_usec;
    
    return((usc_time_t) ustime);
#endif
} /* _p_usc_MD_clock() */

#endif /* UPSHOT */
