/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	util.C
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <new.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "hash-index.h"
#include "externs.h"
#include "globals.h"
#include "interp.h"
#include "parser.h"
#include "config.h"

#define COR_DEFAULT_DB "default_ws"
#define COR_EXODUS_WS "exodus_ws"

char	coral_name[200];     // full pathname of the executing program.
                             // probably 'coral' or 'a.out'
		             // used by incremental loader

#ifdef UNIX_HP
typedef void (*COR_SIGNAL_HANDLER) (int);
#include <sys/syscall.h>
#define getrusage(a, b)  syscall(SYS_GETRUSAGE, a, b)
#else
#ifdef __GNUC__
#define COR_SIGNAL_HANDLER 
#else
/* typedef SIG_TYP COR_SIGNAL_HANDLER ; */
/* #define COR_SIGNAL_HANDLER SIG_TYP */
#define COR_SIGNAL_HANDLER SIG_PF
#endif

#endif

int startMemUsage;
struct rusage startRUsage;
struct rusage SystemUsage;
int beginMem;

extern void createFunctorTable();

extern DatabaseStruct BuiltinDB;
extern struct SymbolTable *DatabaseTable;

DatabaseStruct ExodusDB(EnterSymbol(COR_EXODUS_WS));

extern addDBStruct(DatabaseStruct *);
extern DatabaseStruct *createDB(Name dbname);
extern void clearDatabases();
extern void clearSymbols();
extern int consultFile(char *, char *);
EXTERN int fulldir ARGS((char *));

#ifdef WITH_PERSISTENCE
extern void abortTrans();
extern void shutdown_server() ;
extern void sm_init();
#endif

// these are initialized in int_coral() and used in interp.C
TableStack T_Stack(COR_MAX_RECURSION_DEPTH);

extern void initBuiltins();
extern Arg *TheParameterArg;

/*------------------------------------------------------------------
 Function Behaviour :: Reads in a specified file and dumps it on
  exEnv.error_file. If there is any problem opening the file, 1 is returned,
  else 0 is returned.

 Oddities/Quirks ::

 -------------------------------------------------------------------*/
int DisplayFile(char *file_name)
{
  char file_buffer[1000] ;
  
  ASSERT((strlen(file_name)<1000));
  sprintf(file_buffer, "more %s", file_name);
  return system(file_buffer);
}
/*------------------------------------------------------------------
 Function Behaviour :: Signal Handler

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void exceptionHandler(int recvd_signal)
{
  CORAL_error(COR_SIGNAL_RECEIVED,NULL,NULL) ;

  fprintf(exEnv.error_file, "Received signal no: %d\n", recvd_signal);
  /*
   * Code for recovering from abnormal situation
   */

  if (recvd_signal != SIGINT) {
    fprintf(exEnv.error_file, "Cannot recover from received signal\n");
    fprintf(exEnv.error_file, "Initiating shutdown procedures\n");

#ifdef WITH_PERSISTENCE
  abortTrans();
#endif
    exit_coral();
    exit(1);
  }	

  if (exEnv.C_interrupt_raised == 1) {
    fprintf(exEnv.error_file, "Received second interrupt\n");
    fprintf(exEnv.error_file, "Initiating shutdown procedures\n");

#ifdef WITH_PERSISTENCE
  abortTrans();
#endif
    exit_coral();
    exit(1);
  }

  exEnv.C_interrupt_raised = 1;

}

/*------------------------------------------------------------------
 Function Behaviour :: Exception handler for the new() system call

 Oddities/Quirks ::

 -------------------------------------------------------------------*/
void freeStoreException()
{
	CORAL_error(COR_NO_MEMORY, NULL, "freeStoreException");
	fflush(stdout);fflush(exEnv.error_file);
	exit(1) ;
}

/*------------------------------------------------------------------
 Function Behaviour :: Initializes CORAL. Called from main routine
 as first step before using CORAL. 

 Oddities/Quirks ::

 -------------------------------------------------------------------*/
void init_coral(char *pgm)
{

#ifndef DEBUG
  // Catch signals that might otherwise cause termination
  for (int i = SIGINT; i < SIGPIPE; i++)
    if (i != SIGKILL)
#ifdef __GNUC__
      signal(i, (exceptionHandler)) ;
#else
  signal(i, (COR_SIGNAL_HANDLER)(exceptionHandler)) ;
#endif

#else

  // for debugging purposes, catch only SIGINT (CTRL-C)
#ifdef __GNUC__
  signal(SIGINT, (exceptionHandler)) ;
#else
  signal(SIGINT, (COR_SIGNAL_HANDLER)(exceptionHandler)) ;
#endif
#endif

  // Provide exception handler for new()
  set_new_handler( freeStoreException) ;

  // create the CurrModuleInfo structure
  CurrModuleInfo = new ModuleInfo();

  // create the functor table
  createFunctorTable();

  strcpy(coral_name, (pgm ? pgm : "coral"));

#ifdef DO_EXPLAIN
  if (exEnv.C_exec_mode == COR_DOING_INTERPRET) {
    /** Tarun **/
    /* Create debug directory */
    if (fulldir("dump_directory") != -1)
      system("rm -rf dump_directory");   /* hack cause unlink */
    mkdir("dump_directory", 00755 ); /* set permission to 755 */
  }
#endif

  if ((exEnv.C_exec_mode == COR_DOING_INTERPRET) ||
      (exEnv.C_exec_mode == COR_DOING_IMPERATIVE)){
    
    // initialize global stats for CPU and memory usage
#ifndef UNIX_HP
    getrusage(RUSAGE_SELF, &startRUsage);
#endif
    startMemUsage = (int)sbrk(0);

#ifdef WITH_PERSISTENCE

    if (exEnv.C_connect_exodus_default) {
      // Create exodus workspace
      CurDB = &ExodusDB;	
      CurDB->RelationTable = AllocTab(COR_RELATION_TABLE_INCR, NULL);    
      // Add exodus database to the Database Table
      if (!addDBStruct(CurDB)) {
	CORAL_error(COR_MULTIPLE_DB_ERR, "unable to add exodus DB",
		    "init-coral");
       }
      else 
	sm_init();
     }    
#endif

    // initialize TheParameterArg used for C++ host variables
    TheParameterArg = &TheDontCareArg;

    // remove the stdin.M file
    unlink("stdin.M");

    // Initialize builtin database.
    BuiltinDB.name = EnterSymbol("builtin_ws");
    initBuiltins();
    CurDB = &BuiltinDB;

    // Add builtin database to the Database Table
    if (!addDBStruct(CurDB)) {
      CORAL_error(COR_MULTIPLE_DB_ERR, "unable to add builtin DB",
		  "init-coral");
      exit(1);
    }

    // Create default database
    DatabaseStruct *db = createDB(EnterSymbol(COR_DEFAULT_DB));
    if (!db) {
      CORAL_error(COR_MULTIPLE_DB_ERR, "unable to add default DB",
		  "init-coral");
      exit(1);
    }
    CurDB = db ;

    if (exEnv.C_read_coralrc_default) {
      // Read .coralrc if it exists, else read input from $(HOME)/.coralrc
      if (!consultFile(".coralrc", ".coralrc")) {
	char *home_path = getenv("HOME") ;
	char *tmp_name = new char[strlen(home_path) + 15] ;
	strcpy(tmp_name, home_path);
	strcat(tmp_name, "/.coralrc");
	consultFile(tmp_name, tmp_name) ;
	delete [] tmp_name ;
      }
    }

   }
   else {
     exEnv.C_do_history_default = 0;
    }
#ifndef UNIX_HP
  getrusage(RUSAGE_SELF, &SystemUsage);
#endif
  beginMem = (int)sbrk(0);
}

/*------------------------------------------------------------------
 Function Behaviour :: Called when quitting coral

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void exit_coral()
{
#ifdef DO_EXPLAIN
    if (exEnv.C_exec_mode == COR_DOING_INTERPRET) {
      /** Tarun **/
      /* delete dump directory if empty */
      if (fulldir("dump_directory") == 2)
	system("rm -rf dump_directory"); /* hack cause unlink */
    }
#endif
    
#ifdef WITH_PERSISTENCE
    if (exEnv.C_exec_mode == COR_DOING_INTERPRET) {
	shutdown_server();
    }
#endif

    clearDatabases();
    clearSymbols();
    exEnv.flush_all_outputs();

}

/***** Methods for classes declared in globals.h **********/

void ExecutionEnv::flush_all_outputs()
{
  if ((trace_file) && (trace_file != stdout)
      && (trace_file != stderr))
    fclose((trace_file)) ;
  
  if ((print_file) && (print_file != stdout)
      && (print_file != stderr))
    fclose((print_file)) ;
  
  if ((error_file) && (error_file != stdout)
      && (error_file != stderr))
    fclose((error_file)) ;
}

/*------------------------------------------------------------------
 ExecutionEnv::display(FILE *outf)
 -------------------------------------------------------------------*/
void ExecutionEnv::display(FILE *outf)
{

  fprintf(outf, "\tmode :\t\t\t");
  switch(C_exec_mode) {
  case COR_DOING_INTERPRET : fprintf(outf, "INTERPRETER\n"); break ;
  case COR_DOING_COMPILE : fprintf(outf, "COMPILER\n"); break ;
  case COR_DOING_MAGIC : fprintf(outf, "MAGIC Rewriting\n"); break ;
  case COR_DOING_FACTOR : fprintf(outf, "FACTOR Rewriting\n"); break ;
  case COR_DOING_EXIST : fprintf(outf, "Existential Query Optimization\n"); break ;
  case COR_DOING_IMPERATIVE : fprintf(outf, "Imperative module\n"); break;
  }

  fprintf(outf, "\tinsert_mode : \t\t%s\n",
	  (C_insert_mode_default) ? "on" : "off");
  fprintf(outf, "\tquiet_mode : \t\t%s\n",
	  (C_quiet_mode_default) ? "on" : "off");

  fprintf(outf,"--- Defaults for Rewriting ---\n\n");

  fprintf(outf, "\trewriting : \t\t%s\n",
	  (C_rewriting_default) ? "on" : "off");
  fprintf(outf, "\tsup_magic :\t\t%s\n",
	  (C_use_supplementary_magic_default) ? "on" : "off");
  fprintf(outf, "\tmagic :\t\t\t%s\n",
	  (C_use_magic_default) ? "on" : "off");
  fprintf(outf, "\tsup_magic_indexing :\t%s\n",
	  (C_sup_magic_indexing_default) ? "on" : "off");
  fprintf(outf, "\texistential : \t\t%s\n",
	  (C_use_exist_opt_default) ? "on" : "off");
  fprintf(outf, "\tfactoring : \t\t%s\n\n",
	  (C_use_factoring_default) ? "on" : "off");

  fprintf(outf,"--- Defaults for Execution ---\n\n");

  fprintf(outf, "\tinteractive_mode :\t%s\n",
	  (C_interactive_mode_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t\t%s\n", SymbolString(PredicateSNSymbol),
	  (C_use_predicate_sn_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(UsePipeliningSymbol),
	  (C_use_pipelining_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(UseOrdSearchSymbol),
	  (C_use_ordsearch_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(EagerEvalSymbol),
	  (C_use_eager_eval_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(SaveModuleSymbol),
	  (C_save_module_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(MonotonicSymbol),
	  (C_use_monotonic_default) ? "on" : "off");

  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(MultisetSymbol),
	  (C_multisets_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(CheckSubsumSymbol),
	  (C_check_subsum_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(IndexDeltaSymbol),
	  (C_index_deltas_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(ReturnUnifySymbol),
	  (C_use_return_unify_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(NonGroundFactsSymbol),
	  (C_non_ground_facts_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(DiskResidentSymbol),
	  (C_diskrel_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(EagerIndexingSymbol),
	  (C_eager_indexing_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(SipReorderSymbol),
	  (C_sip_reorder_default) ? "on" : "off");

  fprintf(outf, "\n--- Defaults for Output ---\n");
  fprintf(outf, "\tprint_file :\t\t%s\n", print_filename);
  fprintf(outf, "\ttrace_file :\t\t%s\n", trace_filename);
  fprintf(outf, "\ttracing :\t\t%s\n",
	 (exEnv.GlobalRelationOptions & REL_DISPLAY_INSERTIONS) ? "on" : "off");
  fprintf(outf, "\tprofiling :\t\t%s\n",
	 (exEnv.GlobalRelationOptions & REL_PROFILE_STATS) ? "on" : "off");
  fprintf(outf, "\texplain :\t\t%s\n",
	 (exEnv.GlobalRelationOptions & REL_EXPLAIN) ? "on" : "off");
  fprintf(outf, "\tanswer_style :\t%s\n",
	  (C_answer_style_default) ? "binding" : "tuple");
  fprintf(outf, "\tverbose_answer :\t%s\n",
	  (C_verbose_answer_default) ? "on" : "off");
  fprintf(outf, "\tprint_prompt :\t%s\n",
	  (C_print_prompt_default) ? "on" : "off");
  fprintf(outf, "\tcreate_magic :\t%s\n",
	  (C_create_magic_default) ? "on" : "off");
  fprintf(outf, "\thistory :\t%s\n",
	  (C_do_history_default) ? "on" : "off");

  fprintf(outf, "\n--- Defaults for Profiling ---\n");
  fprintf(outf, "\tprofile_module :\t%s\n",
	  (profile_module) ? "on" : "off");
  fprintf(outf, "\tprofile_scc :\t\t%s\n",
	  (profile_scc) ? "on" : "off");
  fprintf(outf, "\tprofile_sn :\t\t%s\n",
	  (profile_sn) ? "on" : "off");

#ifdef DEBUG
  fprintf(outf, "\n--- Defaults at System Level ---\n");

  fprintf(outf, "\tconnect_exodus: \t%s\n",
	  (C_connect_exodus_default) ? "on" : "off");
  fprintf(outf, "\tpreprocessing : \t%s\n",
	  (C_preprocessing_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t\t%s\n", SymbolString(SingleSccSymbol),
	  (C_single_scc_default) ? "on" : "off");
  fprintf(outf, "\t%s :\t%s\n", SymbolString(ConvertFunctionsSymbol),
	  (C_convert_functions_default) ? "on" : "off");

  fprintf(outf, "\n--- Defaults for System Debugging ---\n");
  fprintf(outf, "\tdebug_unify :\t\t%s\n",
	  (dbg_unify) ? "on" : "off");
  fprintf(outf, "\tdebug_subsumes :\t%s\n",
	  (dbg_subsumes) ? "on" : "off");
  fprintf(outf, "\tdebug_indexing :\t%s\n",
	  (dbg_indexing) ? "on" : "off");
  fprintf(outf, "\tdebug_aggregates :\t%s\n",
	  (dbg_aggregates) ? "on" : "off");
  fprintf(outf, "\tdebug_sets :\t\t%s\n",
	  (dbg_sets) ? "on" : "off");
  fprintf(outf, "\tdebug_get_next :\t%s\n",
	  (dbg_get_next) ? "on" : "off");
  fprintf(outf, "\tdebug_ordered_search :\t%s\n",
	  (dbg_ordsearch) ? "on" : "off");
  fprintf(outf, "\tdebug_pipelining :\t%s\n",
	  (dbg_pipelining) ? "on" : "off");
  fprintf(outf, "\tdebug_backtrack :\t%s\n",
	  (dbg_backtrack) ? "on" : "off");


  fprintf(outf, "\n--- Defaults for System Tuning ---\n");
  fprintf(outf, "\thash_buckets :\t\t%d\n",C_hash_buckets_default);
  fprintf(outf, "\tdelta_buckets :\t\t%d\n", C_delta_buckets_default);
  fprintf(outf, "\tincr_ratio :\t\t%f\n",C_incr_ratio_default);
  fprintf(outf, "\tmax_occupancy :\t\t%f\n",C_max_occupancy_default);
  fprintf(outf, "\trecursion_depth :\t%d\n", C_max_recursion_depth_default);
  fprintf(outf, "\trelation_type :\t\t%d\n", C_relation_type_default);
#endif 
}

void ExecutionEnv::init()
{
  SM_initialized = 0;
  C_exec_mode = COR_DOING_INTERPRET ;
  C_insert_mode_default = COR_INSERT_MODE_DEFAULT;
  C_quiet_mode_default = 0;
  GlobalRelationOptions = 0;
  dbg_unify= dbg_subsumes= dbg_indexing= dbg_aggregates= dbg_get_next= 0;
  dbg_sets=0;
  dbg_ordsearch=0;
  dbg_pipelining = 0;
  dbg_backtrack = 0;
  C_hash_buckets_default = COR_DEFAULT_INDEX_BUCKETS;
  C_delta_buckets_default = COR_DEFAULT_DELTA_BUCKETS;
  C_incr_ratio_default = COR_INDEX_INCR_RATIO;
  C_max_occupancy_default = COR_INDEX_OCCUPANCY;
  C_max_recursion_depth_default = COR_MAX_RECURSION_DEPTH;
  C_relation_type_default = COR_RELATION_TYPE_DEFAULT;
  C_connect_exodus_default = COR_CONNECT_EXODUS_DEFAULT;
  
  profile_sn = COR_PROFILE_SN_DEFAULT;
  profile_scc = COR_PROFILE_SN_DEFAULT;
  profile_module = COR_PROFILE_MODULE_DEFAULT;
  profile_file = stderr;
  trace_file = stderr;
  trace_filename = "stderr";
  print_file = stdout;
  print_filename = "stdout";
  
  error_file = stderr;
  error_filename = "stderr";
  
  C_answer_style_default = COR_ANSWER_STYLE_DEFAULT;
  C_create_magic_default = COR_CREATE_MAGIC_DEFAULT;
  C_read_coralrc_default = COR_READ_CORALRC_DEFAULT;
  C_verbose_answer_default = COR_VERBOSE_ANSWER_DEFAULT;
  C_print_prompt_default = COR_PRINT_PROMPT_DEFAULT;
  C_do_history_default = COR_DO_HISTORY_DEFAULT;

  C_preprocessing_default = COR_PREPROCESSING_DEFAULT;
  C_rewriting_default = COR_REWRITING_DEFAULT;
  C_save_module_default = COR_SAVE_MODULE_DEFAULT;
  C_use_supplementary_magic_default = COR_USE_SUPPLEMENTARY_MAGIC_DEFAULT ;
  C_use_magic_default = COR_USE_MAGIC_DEFAULT;
  C_sup_magic_indexing_default = COR_USE_SUP_MAGIC_INDEXING_DEFAULT;
  C_check_subsum_default = COR_CHECK_SUBSUM_DEFAULT;
  C_single_scc_default = COR_SINGLE_SCC_DEFAULT;
  C_single_answer_default = COR_SINGLE_ANSWER_DEFAULT;
  C_convert_functions_default = COR_CONVERT_FUNCTIONS_DEFAULT;
  C_index_deltas_default = COR_INDEX_DELTA_DEFAULT;
  C_use_factor_magic_default = COR_USE_FACTOR_MAGIC_DEFAULT;
  C_use_return_unify_default = COR_USE_RETURN_UNIFY_DEFAULT;
  C_use_pipelining_default = COR_USE_PIPELINING_DEFAULT;
  C_use_ordsearch_default = COR_USE_ORDSEARCH_DEFAULT;
  C_use_eager_eval_default = COR_USE_EAGER_EVAL_DEFAULT;
  C_use_monotonic_default = COR_USE_MONOTONIC_DEFAULT;
  C_interactive_mode_default = COR_INTERACTIVE_MODE_DEFAULT;
  C_use_exist_opt_default = COR_USE_EXIST_OPT_DEFAULT;
  C_use_factoring_default = COR_USE_FACTORING_DEFAULT;
  C_use_predicate_sn_default = COR_USE_PREDICATE_SEMINAIVE_DEFAULT;
  C_non_ground_facts_default = COR_NON_GROUND_FACTS_DEFAULT;
  C_diskrel_default = COR_DISKREL_DEFAULT;
  C_eager_indexing_default = COR_EAGER_INDEXING_DEFAULT;
  C_sip_reorder_default = COR_SIP_REORDER_DEFAULT;
  
  stackmark = new StackMark ;
  exEnv.coral_path = getenv("CORALROOT");
  if (!exEnv.coral_path) {
#ifdef CORALROOT
    exEnv.coral_path = CORALROOT;
#else
    fprintf(error_file, " Warning ! CORALROOT not set in environment !\n");
    exEnv.coral_path = DEFAULT_CORAL_PATH;
    fprintf(error_file, " Using %s\n", DEFAULT_CORAL_PATH);
#endif

  }

  C_interrupt_raised = 0;
}

void ParserStruct::reset(int flag)
{
 switch (flag) {
  case 0:{
   if (cur_rule){
     if (cur_rule->head) {
       if (cur_rule->head->predicate) delete cur_rule->head->predicate;
       delete cur_rule->head;
      }
     delete cur_rule;
    }
   cur_rule = NULL;
   if (PredicateTable) delete PredicateTable;
   if (snrule) {
     delete snrule;
     delete table;
     delete rInfo_arr;
    }
   if (parameters) delete parameters;
   break;
  }
  case 1: {
   // How Much of this should be commented out ?? : PRAVEEN
   if (cur_rule){
   /*****
     if (cur_rule->head) {
       if (cur_rule->head->predicate) delete cur_rule->head->predicate;
       delete cur_rule->head;
      }
    *****/
     delete cur_rule;
    }
   cur_rule = NULL;
   /****
    if (PredicateTable) delete PredicateTable;
   if (snrule) {
     delete snrule;
     delete table;
     delete rInfo_arr;
    }
   ****/
   if (parameters) delete parameters;
   break;
  }
   default :
     break;

  }
 this->init();
}

void ParserStruct::init()
{
   // Many of these may need to have destructors called on them
   AllPredicates = NULL ;
   Predicates = NULL ;
   rule_list = NULL ;
   make_index_annotations = NULL ;
   agg_sel_annotations = NULL;
   prioritize_annotations = NULL;
   AllowedAdornList = NULL ;
   DisallowedAdornList = NULL ;
   MultisetAnnotations = NULL ;
   diskrel_annotations = NULL ;
   PredicateTable = NULL ;
   CurModule.init();
   func_setup = 0;
   cur_arg_chain = NULL;
   parameters = new parameter_chain;
   cur_set = NULL;
   lhs_flag = 0;
   nested_index = 0;
   cur_rule = NULL;
   snrule = NULL;
   table = NULL;
   rInfo_arr = NULL;
}

History::History(int size)
{
  commands = new char* [size];
  counts   = new int [size];
  for (int i = 0; i < size; i++) {
    commands[i] = NULL;
    counts[i] = 0;
  }
  array_size = size;
  cur_pos = 0;
}

History::History()
{
  array_size = MAX_HISTORY_DEPTH;
  commands = new char* [array_size];
  counts   = new int [array_size];
  for (int i = 0; i < array_size; i++) {
    commands[i] = NULL;
    counts[i] = 0;
  }
  cur_pos = 0;
}

void History::addCommand(char *newcommand, int current_count)
{
  if (commands[cur_pos]) delete (commands[cur_pos]);

  commands[cur_pos] = new char[strlen(newcommand) + 1];
  strcpy(commands[cur_pos], newcommand);
  counts[cur_pos] = current_count;
  if ((++cur_pos) >= array_size) cur_pos  = 0;
}

char * History::getCommand(int desired_count)
{
  for (int i = 0; i < array_size; i++) {
    if (counts[i] == desired_count)
      return commands[i];
  }
  return NULL;
}

void History::display(FILE *outf)
{
  int i;
  fprintf(outf, "  Command History\n");
  for (i = cur_pos; i < array_size; i++)
    {
      if (commands[i]) fprintf(outf, "  %d : %s\n", counts[i], commands[i]);
    }
  for (i = 0; i < cur_pos; i++)
    {
      if (commands[i]) fprintf(outf, "  %d : %s\n", counts[i], commands[i]);
    }
  fprintf(outf, "\n");
}

/*------------------------------------------------------------------
 Function Behaviour :: Initializes a collection of rules, and provides
 it with a hash table

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void init_collection(struct collection *collection, int hashable)
{
 collection->chain = NULL;
 collection->lastptr = &collection->chain;
 collection->count = 0;
 collection->hashable = hashable;
 if (hashable)
   InitTab(collection->tab, 10, NULL);
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void alloc_preds(struct rule *rule, int count)
{
    rule->num_preds = count;
    rule->preds = (Literal**)malloc(count * sizeof(Literal*));
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
#ifdef DEBUG
static void check_collection(struct collection *collection)
{
    Link *link, **lptr;
    Name name = NULL;
    int count = 0, diff_count = 0;
    lptr = &collection->chain;
    for ( ; *lptr != NULL; lptr = &(*lptr)->next) {
	if (lptr != (*lptr)->prevptr) abort();
	count++;
	if ((*lptr)->name != name) diff_count++;
	name = (*lptr)->name;
	if (collection->hashable) {
	    Association *hptr = SymbolLookup(collection->tab, (*lptr)->name);
	    if (HashNone(hptr->arg)) abort();
	    for (link= (Link*)hptr->val; link != *lptr; link = link->next)
		if (link == NULL || link->name != (*lptr)->name) abort();
	}
    }
    if (lptr != collection->lastptr) abort();

    if (count != collection->count) abort();
    if (collection->hashable && collection->tab->cur_size != diff_count)
	abort();
}
#endif

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void add_member(Link *member, struct collection *collection)
{
    if (collection->hashable) {
	Association *ptr = SymbolLookup(collection->tab, member->name);
	if (HashNone(ptr->arg)) {
	    ptr->arg = (Pointer)member->name;
	    ptr->val = (Pointer)member;
	    TabInserted(collection->tab, ptr);
	    /* link into linked list */
	    member->next = NULL;
	    member->prevptr = collection->lastptr;
	    *collection->lastptr = member;
	    collection->lastptr = &member->next;
	} else {
	    Link *old_member = (Link*)ptr->val;
	    Link *next_member = old_member->next;
	    /* keep members ordered, for no real reason */
	    while ( next_member && next_member->name == member->name)
		old_member = next_member, next_member = old_member->next;
	    member->next = old_member->next;
	    member->prevptr = &old_member->next;
	    if (old_member->next) old_member->next->prevptr = &member->next;
	    else collection->lastptr = &member->next;
	    old_member->next = member;
	}
    }
    else {
	/* link into linked list */
	member->next = NULL;
	member->prevptr = collection->lastptr;
	*collection->lastptr = member;
	collection->lastptr = &member->next;
    }
    collection->count++;
#ifdef DEBUG
    if (collection->hashable)
      check_collection(collection);
#endif
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void remove_member(Link *member, struct collection *collection)
{
    if (collection->hashable) abort();
    *member->prevptr = member->next;
    if (member->next) member->next->prevptr = member->prevptr;
    else collection->lastptr = member->prevptr;
    collection->count--;
}

/*------------------------------------------------------------------
 Function Behaviour :: Select some member from 'collection', remove it 
 from the 'collection', and return the removed element.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Link * pop_member(struct collection *collection)
{
    Link *member = collection->chain;
    if (member == NULL) return NULL;
    if (collection->hashable) {
	Association *ptr = SymbolLookup(collection->tab, member->name);
	if (HashNone(ptr->arg) || (Link*)ptr->val != member) abort();
	if (member->next && member->next->name == member->name)
	    ptr->val = (Pointer)member->next;
	else {
	    collection->tab->cur_size--;
	    ptr->arg = HashDeleted;
	}
    }
    *member->prevptr = member->next;
    if (member->next) member->next->prevptr = member->prevptr;
    else collection->lastptr = member->prevptr;
    collection->count--;

#ifdef DEBUG
    check_collection(collection);
#endif
    return member;
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void add_rule(struct rule *rule, struct collection *collection)
{
    rule->link.name = collection->hashable ? rule->head->pred : NULL;
    add_member((Link*)rule, collection);
}

/*------------------------------------------------------------------
 Function Behaviour ::  add element x to array a 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void add_to_array(struct array *a, int x)
{
    if (a->length >= a->data_size) {
	if (a->data_size == 0) {
	    a->data_size = 30;
	    a->data = (int *)malloc(a->data_size * sizeof(int));
	}
	else {
	    a->data_size += a->data_size;
	    a->data = (int*)realloc(a->data, a->data_size * sizeof(int));
	}
    }
    a->data[a->length++] = x;
}

/*------------------------------------------------------------------
 Function Behaviour :: Makes a copy of the header fields of the rule, 
 but not the head or body predicates.  Should be called  copy_proto_rule 
 probably.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
struct rule *copy_rule(struct rule *prototype)
{
    register i;
    register struct rule *rule;
    //rule = (struct rule *)malloc(sizeof(struct rule));
    rule = new struct rule;
    rule->link.next = NULL ;
    rule->link.prevptr = NULL ;
    rule->num_var_names = 0;
    rule->var_names = NULL;
    rule->head = NULL ;
    rule->num_preds = 0 ;
    rule->head_deletes_count = 0;
    rule->head_deletes = NULL;
    rule->preds = NULL ;
    rule->sips = NULL ;
    rule->line_number = -1 ;
    rule->type = COR_NORMALRULE ;
    if (prototype) {
	rule->clause_names = prototype->clause_names;
	if (prototype->num_var_names > 0) {
	    i = prototype->num_var_names;
	    rule->num_var_names = i;
	    rule->var_names = (Name*)malloc(i * sizeof(Name));
	    while (--i >= 0) rule->var_names[i] = prototype->var_names[i];
	    rule->type = prototype->type ;
	}
	rule->line_number = prototype->line_number;
    }
    else {
	rule->clause_names = NULL;
    }
    return rule;
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Literal *copy_pred(Literal *prototype)
{
    int size = sizeof(Literal) + prototype->args.count() * sizeof(ArgPtr);
    Literal *pred = (Literal*)malloc(size);
    //bcopy(prototype, pred, size);
    //memmove(pred, prototype, size); :: NOTE :: this is non-standard !!
    memcpy(pred, prototype, size); 
    pred->adorn.init(prototype->adorn); // Fake copy constructor.
    return pred;
}

int pred_arity(Literal *pred)
{
     return pred->arity();
}

/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 
//NOTE: This should be the constructor for Predicate -- PRAVEEN
 -------------------------------------------------------------------*/
Predicate *AllocPredicate(ParserStruct& parserStruct, Name name, int arity)
{
    Predicate *pred = new Predicate;
    pred->name = name;
    pred->arity = arity;
    pred->clauses = NULL;
    pred->clause_tail = &pred->clauses;
    pred->work = NULL;
    pred->allprednext = parserStruct.AllPredicates;
    parserStruct.AllPredicates = pred;
    return pred;
}

/*------------------------------------------------------------------
 Function Behaviour :: 
 * Find the (globally unique) Predicate with the given 'name' and 'arity'.
 * Allocate a new one if none is found.

 Oddities/Quirks :: LookupPredicate() is similar, except that it does 
 NOT allocate a new predicate if it is not found.

 -------------------------------------------------------------------*/
Predicate *FindPredicate(ParserStruct& parserStruct, Name name, int arity)
{
    Predicate *pred;
    Association *ptr;
    if (parserStruct.PredicateTable == NULL) {
	parserStruct.PredicateTable = AllocTab(COR_PREDICATE_TABLE_INCR, NULL);
    }

    ptr = SymbolLookup(parserStruct.PredicateTable, name);
    if (HashNone(ptr->arg)) {
	/* No previous Predicate with that name */
	pred = AllocPredicate(parserStruct, name, arity);
	ptr->arg = (Name)name;
	ptr->val = pred;
	TabInserted(parserStruct.PredicateTable, ptr);
	pred->symtabnext = parserStruct.Predicates;
	parserStruct.Predicates = pred;
	return pred;
    }
    /* Check all existing Predicates with same 'name' for given 'arity' */
    pred = (Predicate*)ptr->val;
    for (; ; pred = pred->symtabnext)
	if (pred->arity == arity)
	    return pred;
	else if (pred->symtabnext == NULL || pred->symtabnext->name != name) {
	    /* No such Predicate. Allocate and link in a new one. */
	    Predicate *npred = AllocPredicate(parserStruct, name, arity);
	    npred->symtabnext = pred->symtabnext;
	    pred->symtabnext = npred;
	    return npred;
	}
}

/*------------------------------------------------------------------
 Function Behaviour :: Same as FindPredicate, except that if it is
 not found, a new predicate is NOT created. Instead NULL is returned.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Predicate *LookupPredicate(ParserStruct& parserStruct, Name name, int arity)
{
    Predicate *pred;
    Association *ptr;
    if (parserStruct.PredicateTable == NULL) {
	parserStruct.PredicateTable = AllocTab(COR_PREDICATE_TABLE_INCR, NULL);
    }
    ptr = SymbolLookup(parserStruct.PredicateTable, name);
    if (HashNone(ptr->arg)) {
	return NULL;
    }
    /* Check all existing Predicates with same 'name' for given 'arity' */
    pred = (Predicate*)ptr->val;
    for (; ; pred = pred->symtabnext)
	if (pred->arity == arity)
	    return pred;
	else if (pred->symtabnext == NULL || pred->symtabnext->name != name) {
	    /* No such Predicate. */
	    return NULL;
	}
}

/*------------------------------------------------------------------
 Function Behaviour :: Creates a Literal  of the
 specifed predicate

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Literal * AllocLiteral(ParserStruct& parserStruct, Name name, int arg_count)
{
    Literal *lit =
	(Literal*)malloc(sizeof(Literal) + arg_count * sizeof(ArgPtr));
    lit->predicate = FindPredicate(parserStruct, name, arg_count);
    lit->kind = COR_REGULAR;
    lit->pred = name;
    lit->base_pred = 0;
    lit->negated = 0;
    lit->args.set_count(arg_count);
    lit->adorn.init(arg_count);
    return lit;
}

/*------------------------------------------------------------------
 Function Behaviour :: NOTE: No predicate is specified. Otherwise
 similar to previous function.

 Oddities/Quirks :: Warning : predicate field is NULL

 -------------------------------------------------------------------*/
Literal * AllocLiteral(int arg_count)
{
    Literal *pred =
	(Literal*)malloc(sizeof(Literal) + arg_count * sizeof(ArgPtr));
    pred->predicate = NULL;
    pred->kind = COR_REGULAR;
    pred->pred = NULL;
    pred->base_pred = 0;
    pred->negated = 0;
    pred->args.set_count(arg_count);
    pred->adorn.init(arg_count);
    return pred;
}

/*------------------------------------------------------------------
 Function Behaviour :: Adds a rule(a.k.a. clause) to the list of rules
 defining a predicate.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void AddClauseToPredicate(Predicate *predicate, Clause *clause)
{
    clause->pnext = *predicate->clause_tail;
    *predicate->clause_tail = clause;
    predicate->clause_tail = &clause->pnext;
}


/*------------------------------------------------------------------
 Function Behaviour :: 

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
StorageRelation * AllocateRelation(Name name, int arity, int delta_indexed)
{
  StorageRelation *rel;
  rel = new HashSimpleRelation(arity, delta_indexed);
  rel->name = name;
  rel->r_kind = find_r_kind(name);
  return rel;

}

/*------------------------------------------------------------------
 Function Behaviour :: 
   Convert an adornment of the form "bbf" to a BitVector 110.
   Return 0 if ok, -1 otherwise.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int ConvertAdornment(BitVector& bits, char *adorn_string, int adorn_length)
{
    for (int i = 0; i < adorn_length; i++) {
	char adorn_char = *adorn_string++;
	switch (adorn_char) {
	  case 'b':
	    bits.setlen(i+1);
	    bits.set(i, 1);
	    break;
	  case 'f':
	    bits.setlen(i+1);
	    bits.set(i, 0);
	    break;
	  default:
	    return -1;
	}
    }
    return 0;
}

/*------------------------------------------------------------------
 Function Behaviour :: Used to determine if the predicate is a magic
    predicate.

 Oddities/Quirks :: This may not be the right way to do it. Perhaps
 a subclass called MagicPredicate should be defined --- PS

 -------------------------------------------------------------------*/

/** conventions for magic predicates, supplementary predicates, and
  * mp_done predicates below.
**/

int is_magic(Name name)
{
    char *str_name = name->string();
    int len = strlen(str_name);
    if (len > 1) {
        return str_name[0] == 'm' && str_name[1] == '_';
    }
    else return 0;
}

int is_mp_done(Name name)
{
    char *str_name = name->string();
    int len = strlen(str_name);
    if (len > 4) {
        return (str_name[0] == 'd' && str_name[1] == 'o' && str_name[2] == 'n'
			&& str_name[3] == 'e' && str_name[4] == '_') ;
    }
    else return 0;
}

int is_supp(Name name)
{
    char *str_name = name->string();
    int len = strlen(str_name);
    if (len > 3) {
	return (str_name[0] == 's' && str_name[1] == 'u' &&
			str_name[2] == 'p' && str_name[3] == '_') ;
    }
    else return 0;
}

char *make_done_prefix()
{
    return "done_";
}


/* A collecting of various string hashing functions : originally in hash.c */

int StrHash(const char *s, int n)
{
  /* simpler hash function : Praveen */
  int hash = 0;
  for (n -= 1; n >= 0; n--) {
    hash += *s++;
  }
  return hash;
}

int hash_string(const char *text, int len)
{
    return StrHash(text, len);
}
