/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	user-builtin.C

	Contains declarations for built-in relations that form
	the user interface.

 ***********************************************************************/
#include <stdio.h>
#include <strings.h>

#include "config.h"

#ifndef UNIX_HP
#include <math.h>	
#endif

#include "arg.h"
#include "builtin-rel.h"
#include "unify.h"
#include "hash.h"
#include "externs.h"
#include "parser.h"
#include "interface.h"
#include "globals.h"
#include "interp.h"

// PRAVEEN : /lib/cpp predefines certain names, and replaces them. In order to avoid this,
// put any name you think might be a predefined name into this list. Look up the man
// pages for cpp on each host machine. This is a very partial list so far.
#ifdef UNIX_HP
#define COR_CPP_OPTS   "-P -Umips -Usparc -Uhp -Uultrix -Usun -Uunix -Uhpux"
#else
#define COR_CPP_OPTS   "-P -B -Umips -Usparc -Uhp -Uultrix -Usun -Uunix -Uhpux"
#endif


extern char *strip_quotes(char *);
extern Symbol * TraceSymbol;
extern Symbol * UntraceSymbol;
extern Symbol * SetSymbol;
extern Symbol * AssignSymbol;
extern Symbol * ClearSymbol;
extern Symbol * ExplainOffSymbol;
extern Symbol * ExplainOnSymbol;


extern void print_Annotations(ModuleInfo&, FILE *, PredAnnotations *,
			      PredAnnotations *, PredAnnotations *,
			      PredAnnotations *, PredAnnotations *,
			      PredAnnotations *);

extern BindEnv* readTable(const char*,TupleIterator &);	// in user-builtin2.C
extern void loadIncrFile(char *, TupleIterator &); // in incr-load.C

extern int SimpleCopyArgs(ArgList *new_arglist, ArgList *old_arglist,
		BindEnv *rule_env, BindEnv *dont_rename_env,
		TermLink *&renamed_vars);

/* declared in derived-rel.h */
extern int andOptionToRelation(Relation *,
			       char *adorn_form, RelationOptions opt);
extern int orOptionToRelation(Relation *,
			       char *adorn_form, RelationOptions opt);

extern void ResetCounts(), DumpCounts(FILE *);

// This struct is used for all global variables that the user might want
// to set, clear or assign from the CORAL prompt. It is accessed by the
// Solve_Procs of BuiltinDebugRelations

// change this when changing the GlobalVarTable !!
#define COR_GLOBAL_VAR_TABLE_SIZE 51

struct {
  char	*var_name ;
  void	*var_ptr  ;
} GlobalVarTable[COR_GLOBAL_VAR_TABLE_SIZE] =
  { {"debug_unify",  (void *)&(exEnv.dbg_unify)},                    
    {"debug_subsumes",  (void *)&(exEnv.dbg_subsumes)},       
    {"debug_indexing", (void *)&(exEnv.dbg_indexing)}, 
    {"debug_get_next", (void *)&(exEnv.dbg_get_next)}, 
    {"debug_aggregates",(void *)&(exEnv.dbg_aggregates)},
    {"debug_sets", (void *)&(exEnv.dbg_sets)},
    {"debug_ordered_search", (void *)&(exEnv.dbg_ordsearch)},
    {"debug_return_unify", (void *)&(exEnv.dbg_return_unify)},
    {"debug_pipelining", (void *)&(exEnv.dbg_pipelining)},
    {"debug_backtrack", (void *)&(exEnv.dbg_backtrack)},
    {"hash_buckets", (void *)&(exEnv.C_hash_buckets_default)},
    {"delta_buckets", (void *)&(exEnv.C_delta_buckets_default)},
    {"incr_ratio", (void *)&(exEnv.C_incr_ratio_default)},
    {"max_occupancy", (void *)&(exEnv.C_max_occupancy_default)},
    {"recursion_depth", (void *)&(exEnv.C_max_recursion_depth_default)},
    {"relation_type", (void *)&(exEnv.C_relation_type_default)},
    {"profile_sn", (void *)&(exEnv.profile_sn)},
    {"profile_scc", (void *)&(exEnv.profile_scc)},
    {"profile_module", (void *)&(exEnv.profile_module)},
    {"sup_magic", (void *)&(exEnv.C_use_supplementary_magic_default)},
    {"magic", (void *)&(exEnv.C_use_magic_default)},
    {"factor_magic", (void *)&(exEnv.C_use_factor_magic_default)},
    {"existential", (void *)&(exEnv.C_use_exist_opt_default)},
    {"factoring", (void *)&(exEnv.C_use_factoring_default)},
    {"sup_magic_indexing", (void *)&(exEnv.C_sup_magic_indexing_default)},
    {"pipelining", (void *)&(exEnv.C_use_pipelining_default)},
    {"save_module", (void *)&(exEnv.C_save_module_default)},
    {"ordered_search", (void *)&(exEnv.C_use_ordsearch_default)},
    {"lazy_eval", (void *)&(exEnv.C_use_eager_eval_default)},
    {"monotonic", (void *)&(exEnv.C_use_monotonic_default)},
    {"interactive_mode", (void *)&(exEnv.C_interactive_mode_default)},
    {"psn", (void *)&(exEnv.C_use_predicate_sn_default)},
    {"check_subsumption", (void *)&(exEnv.C_check_subsum_default)},
    {"multiset", (void *)&(exEnv.C_multisets_default)},
    {"single_scc", (void *)&(exEnv.C_single_scc_default)},
    {"single_answer", (void *)&(exEnv.C_single_answer_default)},
    {"convert_functions", (void *)&(exEnv.C_convert_functions_default)},
    {"index_deltas", (void *)&(exEnv.C_index_deltas_default)},
    {"return_unify", (void *)&(exEnv.C_use_return_unify_default)},
    {"rewriting", (void *)&(exEnv.C_rewriting_default)},
    {"preprocessing", (void *)&(exEnv.C_preprocessing_default)},
    {"non_ground_facts", (void *)&(exEnv.C_non_ground_facts_default)},
    {"insert_mode", (void *)&(exEnv.C_insert_mode_default)},
    {"quiet_mode", (void *)&(exEnv.C_quiet_mode_default)},
    {"answer_style", (void *)&(exEnv.C_answer_style_default)},
    {"read_coralrc", (void *)&(exEnv.C_read_coralrc_default)},
    {"verbose_answer", (void *)&(exEnv.C_verbose_answer_default)},
    {"create_magic", (void *)&(exEnv.C_create_magic_default)},
    {"print_prompt", (void *)&(exEnv.C_print_prompt_default)},
    {"history", (void *)&(exEnv.C_do_history_default)},
    {NULL}
  } ;

// Used by start_timer() and display_timer() commands.
// getrusage() is called with this structure as a parameter.

extern int startMemUsage;
extern struct rusage startRUsage;
extern struct rusage SystemUsage;
extern int beginMem;
#ifdef UNIX_HP
#include <sys/syscall.h>
#define getrusage(a, b)  syscall(SYS_GETRUSAGE, a, b)
#endif

BuiltinUserTupleRelation::BuiltinUserTupleRelation(int arity, Name n, 
	UserTupleProc user_op) : BuiltinRelation(arity, n, NULL)
{
    op = user_op;
}

/* **************************************************************** **/

BindEnv * BuiltinUserTupleRelation::get_next(TupleIterator& iterator)
{
    Tuple *tuple = get_next_tuple(iterator);
    if (iterator.no_match()) return NULL;
    else return iterator.bindenv;
}


Tuple * BuiltinUserTupleRelation::get_next_tuple(TupleIterator& iterator)
{

    BindEnv *env = iterator.bindenv;

    if (iterator.arg_list.arity() != arity()) {
      iterator.set_no_match(); 
      return NULL;
    }

    if (iterator.ipos > 0) { 
      iterator.stack_mark.pop_to();
      iterator.set_no_match(); return NULL;
    }
    iterator.ipos++;
    iterator.stack_mark.get_mark();

    // first create the query tuple from the bindenv
    ArgList *alist = ArgList::New(arity());
    BindEnv *queryenv = NULL;

    for (int i = 0; i < arity(); i++) {
	Term term;
	term.expr = iterator.arg_list[i];
	term.bindenv = env;
	//FULL_DEREFERENCE_TERM(term);
	DEEP_DEREFERENCE_TERM(term);
	if (!(((term.kindof() == COR_VARIABLE) && (term.bindenv == env)) 
						|| (term.isConstant()))) {
	  term.expr = iterator.arg_list[i];  // Dereferencing lead to a
		// different bindenv -- cannot use dereferenced term.expr
		// with iterator.bindenv, so revert to original arg_list[i]
	  queryenv = iterator.bindenv;	  
	}
	(*alist)[i] = term.expr;
    }

    Tuple query(alist, queryenv); 


    // now call the user routine to compute the (single) answer to the query
    Tuple *result = (Tuple *) (*op)(&query); 

    if (result == NULL) { // no answer to query
		iterator.stack_mark.pop_to();
	iterator.set_no_match(); 
	return NULL;
    }

    if (unify_arg_lists(iterator.arg_list,
	    result->args(), env, result->bindenv)) {
	iterator.reset_no_match();
	return result;
    }
    else {
	iterator.stack_mark.pop_to();
	iterator.set_no_match();
	return NULL;
    }

}

/***************************************************************************/

BuiltinUserRelRelation::BuiltinUserRelRelation(int arity, Name n, 
	UserRelProc user_op) : BuiltinRelation(arity, n, NULL)
{
    op = user_op;
}

/**********************************************************
*  BuiltinUserRelRelation::get_next  is called only the first time.
*  This computes the desired relation, and then destructively
*  updates the tuple iterator, so that the `relation' field points
*  to the user returned relation.  It then directly calls the 
*  get_next on the tuple iterator, which should work fine since
*  the built-in relation has been replaced by a fixed relation.
*  Subsequent calls to get_next, also work with the modified 
*  relation field.
**********************************************************/

BindEnv * BuiltinUserRelRelation::get_next(TupleIterator& iterator)
{
    Tuple *tuple = get_next_tuple(iterator);
    if (iterator.no_match())
	return NULL;
    return iterator.bindenv;
}

Tuple * BuiltinUserRelRelation::get_next_tuple(TupleIterator& iterator)
{
    BindEnv *env = iterator.bindenv;
    BindEnv *queryenv = NULL;

    if (iterator.arg_list.arity() != arity()) {
      iterator.set_no_match(); 
      return NULL;
    }


    // first create the query tuple from the bindenv
    ArgList *alist = ArgList::New(arity());

    for (int i = 0; i < arity(); i++) {
	Term term(iterator.arg_list[i], env);
	FULL_DEREFERENCE_TERM(term);
	if (!(((term.kindof() == COR_VARIABLE) && (term.bindenv == env)) 
						|| (term.isConstant()))) {
	  term.expr = iterator.arg_list[i];  // Dereferencing lead to a
		// different bindenv -- cannot use dereferenced term.expr
		// with iterator.bindenv, so revert to original arg_list[i]
	  queryenv = iterator.bindenv;	  
	}
	(*alist)[i] = term.dereference().expr;
    }

    Tuple query(alist, queryenv); 

    // now call the user routine to compute the (single) answer to the query
    Relation *result = (Relation *) (*op)(&query); 

    if (result == NULL) { // no answer to query
	iterator.set_no_match();
	return NULL;
    }

    // now destructively modify the tuple iterator to point to the 
    // computed relation, and call the iterator's get_next_tuple which
    // will now work on the computed relation.

    iterator.relation = result;
    return iterator.get_next_tuple();

}

/***************************************************************************/

/* #define MAKE_SUPP_INDEX */

static GoalIdCount = 0;

extern int subsumes_arg_lists(ArgList& left, ArgList& right,
    BindEnv *left_bindings, BindEnv *right_bindings);

BindEnv *GoalIdOp(BuiltinRelation& , TupleIterator& iter)
{
    StackMark stack_mark0;

    if (iter.arg_list.arity() != 1) { // should not happen
	CORAL_error(COR_BAD_ARITY, "of GoalID", "GoalIdOp");
	stack_mark0.pop_to();
	iter.set_no_match();
	return NULL;
    }

    Term term(iter.arg_list.arg(0), iter.bindenv);
    FULL_DEREFERENCE_TERM(term);

    Arg *query_arg = term.expr;
    BindEnv *query_env = term.bindenv;

    if (!is_functor(query_arg)) { 
	CORAL_error(COR_BAD_ARG_TYPE, "arg of GoalID", "GoalIdOp") ;
	stack_mark0.pop_to();
	iter.set_no_match();
	return NULL;
    }

    // find out all info about the magic predicate in GoalId
    Name mpred_name = ((FuncArg *) query_arg)->functor();
    ArgList *mpred_args = functor_args(query_arg);
    int mpred_arity = mpred_args->arity();

    // check if id is indeed a variable
    Arg *mpred_id = mpred_args->arg(0); 
    if (!is_var(mpred_id)) { // should not happen
	fprintf(exEnv.error_file, "Id of GoalId should not be bound\n");
	stack_mark0.pop_to();
	iter.set_no_match();
	return NULL;
    }

    /** we need to access the relation off the frame because 
	of recursive calls. **/
    Relation *mreln = find_relation(mpred_name,mpred_arity); 
    if (!mreln) mreln = make_relation(mpred_name, mpred_arity);

#ifdef MAKE_SUPP_INDEX

    if (mreln->for_supp_index == NULL) {
	// make a new relation and add all appropriate indices to it
	
	mreln->for_supp_index = new HashSimpleRelation(mreln->arity());
	mreln->for_supp_index->check_subsum = mreln->check_subsum;
	mreln->add_all_indices(mreln->for_supp_index);
	// mreln->for_supp_index inherits all indices from mreln
    }

#endif

    if (mreln->check_subsum==1) { 
        // set up a tuple iterator on relation and find matching tuple
    	StackMark stack_mark1;
	
	// searching main relation
    	TupleIterator gid_iter(mreln, *mpred_args, query_env);
    
    	for (;;) { 
	    StackMark stack_mark2;
	    Tuple *tuple1 = gid_iter.get_next_tuple();
	    stack_mark2.pop_to(); // undo bindings to query_env
	    if (tuple1 == NULL) break; // search for_supp_index now

	    // found a matching tuple; check for subsumption

	    Arg *tuple1_id = tuple1->arg(0);

	    // this will work only if the tuple in the relation has
	    // no bindings.

	    Term term;
	    term.expr = tuple1_id;
	    unify_binding(query_env, ((VarArg *) mpred_id)->var, term);
	    ArrayBindEnv tuple1_env(tuple1->env_size);
	    if (subsumes_arg_lists(tuple1->args(), *mpred_args,
			&tuple1_env, query_env)) {
		// found the right id value too
		stack_mark1.pop_to(); // undo bindings because of subsumes
		unify_binding(query_env, ((VarArg *) mpred_id)->var, term);
		    // get back the id into the BindEnv
		iter.reset_no_match();
		return query_env;
	    }
	    stack_mark2.pop_to();
        }

	// now searching for_supp_index relation
        stack_mark1.pop_to();


#ifdef MAKE_SUPP_INDEX

        TupleIterator gid_iter1(mreln->for_supp_index, *mpred_args, query_env);

        for (;;) { 
	    StackMark stack_mark3;
	    Tuple *tuple2 = gid_iter1.get_next_tuple();
	    stack_mark3.pop_to(); // undo bindings
	    if (tuple2 == NULL) break; // need to get brand new id value 

	    // found a matching tuple; check for subsumption

	    Arg *tuple2_id = tuple2->arg(0);

	    // this will work only if the tuple in the relation has
	    // no bindings.

	    /**** why is this part necessary ? isnt the fact that
	          the iterator returned a tuple sufficient to
	          show that there is already a subsuming tuple  -- PRAVEEN ???
	    ****/

	    Term term;
	    term.expr = tuple2_id;
	    unify_binding(query_env, ((VarArg *) mpred_id)->var, term);
	    ArrayBindEnv tuple2_env(tuple2->env_size);
	    if (subsumes_arg_lists(tuple2->args(), *mpred_args,
			&tuple2_env, query_env)) {
		// found the right id value too
		stack_mark3.pop_to(); // undo bindings because of subsumes
		unify_binding(query_env, ((VarArg *) mpred_id)->var, term);
		    // get back the id into the BindEnv
		iter.reset_no_match();
		return query_env;
	    }
	    stack_mark3.pop_to();
        }
#endif

    }

    // else can immediately insert it in mreln->for_supp_index with NEW id
    Term term1;
    Arg *new_id = make_arg(GoalIdCount++);
    term1.expr = new_id;
    unify_binding(query_env, ((VarArg *) mpred_id)->var, term1);

/**************************************
// WARNING::  ReturnUnify optimizations needed here!!!
**************************************/

/****
#define INSERT_GOALID 

If this is included, the magic relation gets inserted into, without a parent goal
being added to the magic fact. This results in the return_unify optimization
getting screwed up, since that expects that every magic fact has a pointer to
a supplementary magic fact as a parent goal. : Praveen.
*****/

#ifdef INSERT_GOALID
#ifdef MAKE_SUPP_INDEX
        mreln->for_supp_index->insert_new(*mpred_args, query_env);
#else
        mreln->insert_new(*mpred_args, query_env);
#endif
#endif

    iter.reset_no_match();
    return query_env;
}

#undef FAIL_RETURN
#define FAIL_RETURN  {iterator.set_no_match(); return NULL;} 

int set_print_file(char *file_name)
{
    if ((exEnv.print_file) && (exEnv.print_file != stdout)
	  && (strcmp(exEnv.print_filename, exEnv.trace_filename))) {
    fclose(exEnv.print_file);
    delete [] exEnv.print_filename ;
  }
    
  if (!strcmp(file_name, "stdout")) {
    exEnv.print_file = stdout;
    exEnv.print_filename = new char[7];
    strcpy(exEnv.print_filename, "stdout");
  }
  else if (!strcmp(file_name, "stderr")) {
    exEnv.print_file = stderr;
    exEnv.print_filename = new char[7];
    strcpy(exEnv.print_filename, "stderr");
  }
  else if (!strcmp(file_name, exEnv.trace_filename)) {
    exEnv.print_file = exEnv.trace_file;
    exEnv.print_filename = exEnv.trace_filename;
  }
  else if (exEnv.print_file = fopen(file_name, "a"))
    exEnv.print_filename = file_name ;
  else {
    exEnv.print_file = stdout;
    exEnv.print_filename = new char[7];
    strcpy(exEnv.print_filename, "stdout");
    CORAL_error(COR_CANNOT_OPEN_FILE, file_name, "set_print_file");
    return 0;
  }
  return 1;
}

static int set_trace_file(char *file_name)
{
  if ((exEnv.trace_file) && (exEnv.trace_file != exEnv.error_file)
	  && (strcmp(exEnv.print_filename, exEnv.trace_filename))) {
    fclose(exEnv.trace_file);
    delete [] exEnv.trace_filename ;
  }
  
  if (!strcmp(file_name, "stdout")) {
    exEnv.print_file = stdout;
    exEnv.print_filename = new char[7];
    strcpy(exEnv.print_filename, "stdout");
  }
  else if (!strcmp(file_name, "stderr")) {
    exEnv.print_file = stderr;
    exEnv.print_filename = new char[7];
    strcpy(exEnv.print_filename, "stderr");
  }
  else if (!strcmp(file_name, exEnv.error_filename)) {
    exEnv.trace_file = exEnv.error_file;
    exEnv.trace_filename = new char[7];
    strcpy(exEnv.trace_filename, exEnv.error_filename);
  }
  else if (!strcmp(file_name, exEnv.print_filename)) {
    exEnv.trace_file = exEnv.print_file;
    exEnv.trace_filename = exEnv.print_filename;
  }
  else if (exEnv.trace_file = fopen(file_name, "a"))
    exEnv.trace_filename = file_name ;
  else {
    exEnv.trace_file = exEnv.error_file;
    exEnv.trace_filename = new char[7];
    strcpy(exEnv.trace_filename, "exEnv.error_file");
    CORAL_error(COR_CANNOT_OPEN_FILE, file_name, "set_trace_file");
    return 0;
  }
  return 1;
}

BindEnv * RelOptionsSolver(BuiltinRelation& rel, TupleIterator& iterator)
{
  if (iterator.arg_list.count() == 0)
    {
      /** Tarun **/
      if (rel.name == ExplainOnSymbol)
	{
	  exEnv.GlobalRelationOptions |= REL_EXPLAIN ;
	}
      else if (rel.name == ExplainOffSymbol)
	{
	  exEnv.GlobalRelationOptions &= ~REL_EXPLAIN ;
	}
      else if (rel.name == TraceSymbol) 
	{
	  exEnv.GlobalRelationOptions |= REL_DISPLAY_INSERTIONS ;
	}
      else if (rel.name == UntraceSymbol) 
	{
	  exEnv.GlobalRelationOptions &= ~REL_DISPLAY_INSERTIONS;
	}
      else if (rel.name == ProfileOnSymbol) 
	{
	  exEnv.GlobalRelationOptions |= REL_PROFILE_STATS;
	}
      else if (rel.name == ProfileOffSymbol) 
	{
	  exEnv.GlobalRelationOptions &= ~REL_PROFILE_STATS;
	}
      
      return iterator.bindenv;
    }

  FOR_EACH_ARG(arg, iterator.arg_list) {

    Term term1 (arg, iterator.bindenv);
    //FULL_DEREFERENCE_TERM(term1);
    DEEP_DEREFERENCE_TERM(term1);
    if ( term1.expr->kindof() != COR_FUNCTOR) {
	term1.expr->printon(exEnv.error_file) ;
	CORAL_error(COR_BAD_ARGLIST, NULL, "RelOptionsSolver");
	FAIL_RETURN ;
    }
    
    FuncArg * fa = (FuncArg *)term1.expr;
    if (fa->arity() != 1 || fa->args[0]->kindof() != COR_SYMBOL_CONST) {
	term1.expr->printon(exEnv.error_file) ;
	CORAL_error(COR_BAD_ARGLIST, NULL, "RelOptionsSolver");
	FAIL_RETURN;
    }

    char *adorn = SymbolString(fa->args[0]);

    Relation *newrel = find_external_relation(fa->functor(), strlen(adorn));
    if (! newrel) {
      CORAL_error(COR_BAD_REL_NAME, SymbolString(term1.expr),
		  "RelOptionsSolver");
      FAIL_RETURN;
    }
    
    if (rel.name == TraceSymbol) 
      {
	orOptionToRelation(newrel, adorn, REL_DISPLAY_INSERTIONS);
      }
    else if (rel.name == UntraceSymbol)
      {
	andOptionToRelation(newrel, adorn, ~REL_DISPLAY_INSERTIONS);
      }
    else if (rel.name == ProfileOnSymbol)
      {
	orOptionToRelation(newrel, adorn, REL_PROFILE_STATS);
      }
    else if (rel.name == ProfileOffSymbol)
      {
	andOptionToRelation(newrel, adorn, ~REL_PROFILE_STATS);
      }
    else if (rel.name == ExplainOnSymbol)
      {
	orOptionToRelation(newrel, adorn, REL_EXPLAIN);
      }
    else if (rel.name == ExplainOffSymbol)
      {
	andOptionToRelation(newrel, adorn, ~REL_EXPLAIN);
      }

  } END_EACH_ARG;

  return iterator.bindenv;
}

/**************************************************************** */

//This function is used to assign values to global variables that are accessed
//using the 'set','clear' and 'assign' builtin debug relations

void AssignValueToGlobal(Arg *arg, int value)
{
  int index ;

  if (!strcmp(SymbolString(arg), "recursion_depth")) {
    // Do some appropriate action here !! -- PRAVEEN
    if (value <= T_Stack.count) {
      fprintf(exEnv.error_file,
   "CORAL Error:: New value should be greater than total stack depth %d !\n",
	      T_Stack.count);
      return;
    }
    exEnv.C_max_recursion_depth_default = value;
    Table* *tmp = T_Stack.tables;
    T_Stack.tables  = new Table*[value];
    for (int i = 0; i < T_Stack.count; i++)
      T_Stack.tables[i] = tmp[i];
    delete tmp;
    return;
  }

  for (index=0; GlobalVarTable[index].var_name ;index++)
     if (!strcmp(GlobalVarTable[index].var_name, SymbolString((Symbol *)arg))){
	*((int *)(GlobalVarTable[index].var_ptr)) = value ;
	if (!exEnv.C_quiet_mode_default)
	  fprintf(exEnv.error_file, "Value of %s is %d\n",
		  SymbolString((Symbol *)arg), value);
	return ;
     }

  CORAL_error(COR_BAD_GLOBAL_VAR, NULL, "AssignValueToGlobal");
}

void AssignValueToGlobal(Arg *arg, double value)
{
  int index ;

  if (!strcmp(SymbolString(arg), "recursion_depth")) {
    fprintf(exEnv.error_file, "The recursion depth should be an integer !!\n");
    return;
  }

  for (index=0; GlobalVarTable[index].var_name ;index++)
     if (!strcmp(GlobalVarTable[index].var_name, SymbolString((Symbol *)arg))){
	*((double *)(GlobalVarTable[index].var_ptr)) = value ;
	if (!exEnv.C_quiet_mode_default)
	  fprintf(exEnv.error_file, "Value of %s is %f\n",
			SymbolString((Symbol *)arg), value);
	return ;
     }
  CORAL_error(COR_BAD_GLOBAL_VAR, NULL, "AssignValueToGlobal");
}

BindEnv * DebugSolver(BuiltinRelation& rel, TupleIterator& iterator)
{
	if (iterator.arg_list.count() == 0)
	{
		CORAL_error(COR_EMPTY_ARGLIST, NULL, "DebugSolver");
		return iterator.bindenv;
	}

	if (rel.name == SetSymbol) 
	{
	  //Set each global variable specified in the argument list

	  FOR_EACH_ARG(arg, iterator.arg_list){
	    Term term1 (arg, iterator.bindenv);
	    //FULL_DEREFERENCE_TERM(term1);
	    DEEP_DEREFERENCE_TERM(term1);
	    if (term1.expr->kindof() != COR_SYMBOL_CONST){
	      CORAL_error(COR_BAD_ARG_TYPE, NULL, "DebugSolver");
	      term1.printon(exEnv.error_file);
	      fprintf(exEnv.error_file,"\n");
	      iterator.set_no_match();
	      return iterator.bindenv;
	    }
	    AssignValueToGlobal(term1.expr, 1);
	  }END_EACH_ARG;
	}

        else if (rel.name == ClearSymbol) 
	{
	  //Clear each global variable specified in the argument list

	  FOR_EACH_ARG(arg, iterator.arg_list){
	    Term term1 (arg, iterator.bindenv);
	    //FULL_DEREFERENCE_TERM(term1);
	    DEEP_DEREFERENCE_TERM(term1);
	    if (term1.expr->kindof() != COR_SYMBOL_CONST){
	      CORAL_error(COR_BAD_ARG_TYPE, NULL, "DebugSolver");
	      term1.printon(exEnv.error_file);
	      fprintf(exEnv.error_file,"\n");
	      iterator.set_no_match();
	      return iterator.bindenv;
	    }
	    AssignValueToGlobal(term1.expr, 0);
	  }END_EACH_ARG;
	}

        else if (rel.name == AssignSymbol)
	{
	  //Assign the value in the second argument to the global variable 
	  //specifed inthe first argument

	  if (iterator.arg_list.count() != 2)
	    {
	      CORAL_error(COR_BAD_ARITY, "2 arguments needed","DebugSolver");
	      return iterator.bindenv;
	    }

	  Term term1 (iterator.arg_list[0], iterator.bindenv);
	  //FULL_DEREFERENCE_TERM(term1);
	  DEEP_DEREFERENCE_TERM(term1);
	  if (term1.expr->kindof() != COR_SYMBOL_CONST){
	    CORAL_error(COR_BAD_ARG_TYPE, NULL, "DebugSolver");
	    term1.printon(exEnv.error_file);
	    fprintf(exEnv.error_file,"\n");
	    iterator.set_no_match();
	    return iterator.bindenv;
	  }

	  Term term2 (iterator.arg_list[1], iterator.bindenv);
	  //FULL_DEREFERENCE_TERM(term2);
	  DEEP_DEREFERENCE_TERM(term2);
	  if (term2.expr->kindof() == COR_NUM_CONST){

	    switch(((NumArg *)(iterator.arg_list[1]))->num_kindof()) {
	    case COR_INTEGER :
	      AssignValueToGlobal(iterator.arg_list[0],
				  ((NumArg *)(iterator.arg_list[1]))->i_val);
	      break;
	      
	    case COR_DOUBLE :
	      AssignValueToGlobal(iterator.arg_list[0],
				  ((NumArg *)(iterator.arg_list[1]))->d_val);
	      break;
	      
	    }
	  }
	  else if (term2.expr->kindof() == COR_SYMBOL_CONST){
	    char *file_name = strip_quotes(SymbolString(term1.expr));
	    if (!strcmp(file_name, "trace_file")) {
	      char *new_file = strip_quotes(SymbolString(term2.expr));
	      if (!set_trace_file(new_file)) {
		delete new_file;
		iterator.set_no_match();
		return iterator.bindenv;
	      }
	    }

	    else if (!strcmp(file_name, "print_file")) {
	      char *new_file = strip_quotes(SymbolString(term2.expr));
	      if (!set_print_file(new_file)) {
		delete new_file;
		iterator.set_no_match();
		return iterator.bindenv;
	      }
	    }

	    else {
	      CORAL_error(COR_BAD_GLOBAL_VAR, NULL, "AssignValueToGlobal");
	      delete file_name;
	      iterator.set_no_match();
	      return iterator.bindenv;
	    }
	    delete file_name;
	  }

	  else {
	    CORAL_error(COR_BAD_ARG_TYPE, NULL, "DebugSolver");
	    term2.printon(exEnv.error_file);
	    fprintf(exEnv.error_file,"\n");
	    iterator.set_no_match();
	    return iterator.bindenv;
	  }

	}
	else
	  // Should never reach here. Safety check.
	  abort();

	return iterator.bindenv;

 }


/*********************************************************** */

BindEnv *HelpSolver(BuiltinRelation& , TupleIterator& iterator)
{
    if (iterator.arg_list.count() == 0) {
      char name_buffer[1000] ;
      sprintf(name_buffer, "%s%s/general.help", exEnv.coral_path, HELP_PATH);
      if (DisplayFile(name_buffer))
	CORAL_error(COR_FILE_NOT_FOUND, "Sorry, no help available", NULL);

    }
    else {
	  Term term1 (iterator.arg_list[0], iterator.bindenv);
	  //FULL_DEREFERENCE_TERM (term1);
	  DEEP_DEREFERENCE_TERM (term1);
	  if (term1.expr->kindof() != COR_SYMBOL_CONST ) {
		CORAL_error(COR_BAD_ARG_TYPE, NULL, "HelpSolver");
	        term1.printon(exEnv.error_file);
		fprintf(exEnv.error_file,"\n");
		iterator.set_no_match();
		return iterator.bindenv;
	  }

	  char *help_name = strip_quotes(SymbolString(term1.expr));
	  char name_buffer[1000] ;
	  sprintf(name_buffer, "%s%s/%s.help",
		  exEnv.coral_path, HELP_PATH, help_name);
	  if (DisplayFile(name_buffer))
	     CORAL_error(COR_FILE_NOT_FOUND, "Sorry, no help available", NULL);
	  delete [] help_name ;
    }

    return iterator.bindenv ;	  
}

int consultFile(char *filename, char *stack_filename)
{
  FILE *tmp_fd = fopen(filename,"r") ;

  if (tmp_fd == NULL)
    return 0 ;
  else {
    parseEnv.handle_new_file(tmp_fd, stack_filename);
    if (strcmp(filename, stack_filename)) {
      char temp_str[200];
      // Remove the <filename>.M file
      sprintf(temp_str, "%s.M", stack_filename);
      unlink(temp_str);
    }
  }
  return 1 ;
}

/*
 * executes the magic program using the system() call. The input file
 * to magic is <infile_name>, and the output is <outfile_name>.M
 */
static int callMagic(char *infile_name, char* outfile_name)
{
  char term_str[1000];

  sprintf(term_str, "cat %s | magic ", infile_name);
  sprintf(term_str +strlen(term_str), " > %s.M", outfile_name);

  if (system(term_str)) return 0 ;
  
  return 1 ;

}

/*
 * executes the exist program using the system() call. The input file
 * to magic is <infile_name>, and the output is <outfile_name>.M
 */
static int callEQO(char *infile_name, char* outfile_name)
{
  char term_str[1000];

  sprintf(term_str, "cat %s | exist ", infile_name);
  sprintf(term_str +strlen(term_str), " > %s.E", outfile_name);

  if (system(term_str)) return 0 ;
  
  return 1 ;

}

#define TMP_CONSULT_FILE "/tmp/__consult.CORAL"
/*
 * ".T" ending specifies a table file.
 */
int tableFile(const char* name)
{
  int l = strlen(name);
  if(name[l-2] == '.' && name[l-1] == 'T')
    return 1;
  return 0;
}

/*
 * files ending in .S, .C and .o are to be incrementally loaded
 */
int incrLoadFile(const char* name)
{
  int l = strlen(name);
  if(name[l-2] == '.' && 
     ((name[l-1] == 'S') || (name[l-1] == 'C') || (name[l-1] == 'o')))
    return 1;
  return 0;
}

/*
 * The solver for the consult builtin.
 */
BindEnv *ConsultSolver(BuiltinRelation& , TupleIterator& iterator)
{

#ifdef DO_TRACE
    if (exEnv.dbg_get_next) {
        fprintf(exEnv.trace_file, "Entering %s\n", "consult");
    }
#endif

  if (iterator.arg_list.count() != 1) {
    CORAL_error(COR_BAD_ARITY, "1 argument to consult needed", "ConsultSolver");
    iterator.set_no_match();
    return iterator.bindenv;
  }

  Term term1 (iterator.arg_list[0], iterator.bindenv);
  //FULL_DEREFERENCE_TERM(term1);
  DEEP_DEREFERENCE_TERM(term1);

  if (term1.expr->kindof() != COR_SYMBOL_CONST ) {
	CORAL_error(COR_BAD_ARG_TYPE, NULL, "ConsultSolver");
        term1.printon(exEnv.error_file);
	fprintf(exEnv.error_file,"\n");
	iterator.set_no_match();
	return iterator.bindenv;
  }

  char *file_name = strip_quotes(SymbolString(term1.expr));

  if (tableFile(file_name)) {
    readTable(file_name,iterator);
    delete [] file_name;
    return iterator.bindenv;
  } 
  else if (incrLoadFile(file_name)) {
    char input[1000];

    if (exEnv.C_interactive_mode_default) {

      fprintf(exEnv.error_file, 
	      "\t... sure you want to incrementally load code ? (y/n)[n]");
      gets(input);
      if ((input[0] != 'y') && (input[0] != 'Y')) {
	delete [] file_name;
	iterator.set_no_match();
	return iterator.bindenv;
      }
    }
    loadIncrFile(file_name,iterator);
    delete [] file_name;
    return iterator.bindenv;
  } 
  else {
    char term_str[200];
    sprintf(term_str, "/lib/cpp %s %s > %s", COR_CPP_OPTS, file_name, TMP_CONSULT_FILE);
    if (system(term_str)) {
      fprintf(exEnv.error_file, "CORAL::error : Unable to execute the shell command\n");
      fprintf(exEnv.error_file, "%s\n", term_str);
      fprintf(exEnv.error_file, "Aborting attempt to consult file\n");
      unlink(TMP_CONSULT_FILE);
      iterator.set_no_match();
      return iterator.bindenv;
    }
    
    
    if (!consultFile(TMP_CONSULT_FILE, file_name)) {
      CORAL_error(COR_FILE_NOT_FOUND, file_name, "ConsultSolver");
      iterator.set_no_match();
      unlink(TMP_CONSULT_FILE);    
      delete [] file_name;
      return iterator.bindenv;
    }
    
    unlink(TMP_CONSULT_FILE);
  }
  delete [] file_name;
  return iterator.bindenv ;
}


/*
 * called from yyparse() at the end of a module. For each exported
 * predicate, all the rules are written out to a temporary file,
 * magic is called on it, and the magic output file is placed on
 * the scanner input stack.
 */

#define TMP_REWRITE_FILE "/tmp/__rewrite.CORAL"
#define TMP_M_REWRITE_FILE "/tmp/__rewrite.CORAL.M"
#define TMP_E_REWRITE_FILE "/tmp/__rewrite.CORAL.E"

void perform_rewriting(ParserStruct& parserStruct)
{
  ExportInfo *export_info;
  
  if (parserStruct.CurModule.ExportedPreds == NULL) {
    fprintf(exEnv.error_file, "CORAL: Error - No exported query form.\n");
    return;
  }

  // create output file
  FILE *outf;
  int i;

  if (!(outf = fopen(TMP_REWRITE_FILE, "w"))) {
    fprintf(exEnv.error_file, "CORAL::error : unable to open temporary file\n");
    fprintf(exEnv.error_file, "Aborting attempt to process module\n");
    return ;
  }

  // Process each exported query form independently.
  for (export_info = parserStruct.CurModule.ExportedPreds, i = 0;
       export_info != NULL; export_info = export_info->next, i++) {

    if (parserStruct.CurModule.name)
      fprintf(outf, "\nmodule %s%d.\n", 
	      SymbolString(parserStruct.CurModule.name), i);
    fprintf(outf, "export %s(", export_info->pred_name->string());
    if (export_info->adorn_symbol)
      fputs(export_info->adorn_symbol->string(), outf);
    fprintf(outf, ").\n");

    parserStruct.CurModule.ConvertFunctions = 0;
    print_Annotations(parserStruct.CurModule, outf,
		      parserStruct.make_index_annotations ,
		      parserStruct.agg_sel_annotations ,
		      parserStruct.AllowedAdornList ,
		      parserStruct.MultisetAnnotations ,
		      parserStruct.prioritize_annotations,
		      parserStruct.diskrel_annotations);

    FOR_EACH_RULE(rule, parserStruct.rules) {
      print_rule(rule, outf);
    } END_EACH_RULE

    fprintf(outf, "end_module.\n\n") ;
  }

  // close output file
  fclose(outf);

  char magic_input_file[200];
  char term_str_E[200];
  char term_str_M[200];

  if (parserStruct.CurModule.UseExistOpt) {
    if (parseEnv.cur_file_name())
      sprintf(term_str_E, "%s.E", parseEnv.cur_file_name());
    else 
      sprintf(term_str_E, "%s.E", "stdin");

    if (exEnv.C_create_magic_default) {
    // check to see if the magic output file for the currently
    // consulted file already exists, else create it.
    if (!(outf = fopen(term_str_E, "r"))) { 
      if (!(outf = fopen(term_str_E, "w"))) { 
	fprintf(exEnv.error_file,
		"CORAL::error : unable to create temporary file\n");
	fprintf(exEnv.error_file,
		"Aborting attempt to process module\n");
	unlink(TMP_REWRITE_FILE);
	return ;
      }
      else fclose(outf);
    }
    else fclose(outf);
   }

  
    // perform existential query optimization rewriting on the file
    if (!callEQO(TMP_REWRITE_FILE, TMP_REWRITE_FILE)) {
      fprintf(exEnv.error_file, "CORAL::error : unable to call eqo\n");
      fprintf(exEnv.error_file, "Aborting attempt to process module\n");
      unlink(TMP_REWRITE_FILE);
      return;
    }

    // delete TMP_REWRITE_FILE
    unlink(TMP_REWRITE_FILE);

    if (exEnv.C_create_magic_default) {
    // concatenate TMP_E_REWRITE_FILE to the magic output file
    char buf[300];
    sprintf(buf, "cat %s >> %s", TMP_E_REWRITE_FILE, term_str_E);
    if (system(buf))
      fprintf(exEnv.error_file,
	      "Unable to save eqo output in %s\n", term_str_E);
    
    // delete TMP_E_REWRITE_FILE
    unlink(TMP_E_REWRITE_FILE);

    strcpy(magic_input_file, term_str_E);
   }
    else
        strcpy(magic_input_file, TMP_E_REWRITE_FILE);
  }
  else
    strcpy(magic_input_file, TMP_REWRITE_FILE);


  // Now handle magic rewriting
  if (parseEnv.cur_file_name())
    sprintf(term_str_M, "%s.M", parseEnv.cur_file_name());
  else 
    sprintf(term_str_M, "%s.M", "stdin");


  if (exEnv.C_create_magic_default) {
  // check to see if the magic output file for the currently
  // consulted file already exists, else create it.
  if (!(outf = fopen(term_str_M, "r"))) { 
    if (!(outf = fopen(term_str_M, "w"))) { 
      fprintf(exEnv.error_file, 
	      "CORAL::error : unable to create temporary file\n");
      fprintf(exEnv.error_file,
	      "Aborting attempt to process module\n");
      unlink(magic_input_file);
      return ;
    }
    else fclose(outf);
  }
  else fclose(outf);
 }
  
  // perform magic rewriting on the file
  if (!callMagic(magic_input_file, TMP_REWRITE_FILE)) {
    fprintf(exEnv.error_file, "CORAL::error : unable to call magic\n");
    fprintf(exEnv.error_file, "Aborting attempt to process module\n");
    unlink(magic_input_file);
    return ;
  }

  // shouldnt this be unlink(magic_input_file) ??
  //unlink(TMP_REWRITE_FILE); 
  unlink(magic_input_file); 

  // open temporary magic output file and put the descriptor
  // on the scanner input stack
  if (!consultFile(TMP_M_REWRITE_FILE, TMP_M_REWRITE_FILE)) {
    CORAL_error(COR_FILE_NOT_FOUND, TMP_M_REWRITE_FILE, "perform_rewriting");
    fprintf(exEnv.error_file, "Aborting attempt to process module\n");
  }

  if (exEnv.C_create_magic_default) {
  // concatenate TMP_M_REWRITE_FILE to the magic output file
  char buf[300];
  sprintf(buf, "cat %s >> %s", TMP_M_REWRITE_FILE, term_str_M);
  if (system(buf))
     fprintf(exEnv.error_file, 
	     "Unable to save magic output in %s\n", term_str_M);

 }

  // delete TMP_M_REWRITE_FILE
  unlink(TMP_M_REWRITE_FILE);
}


static void user_continue()
{
  char input[100];

  fprintf(exEnv.error_file, "\t... next answer ? (y/n/all)[y]");
  gets(input);
  if ((input[0] == 'n') || (input[0] == 'N'))
    exEnv.C_interrupt_raised = -1; // prevents any further computation
  else if ((input[0] == 'a') || (input[0] == 'A'))
    exEnv.C_interactive_mode_current = 0;   // prevents further answer prompting
}

BindEnv *TuplePrintSolver(BuiltinRelation& , TupleIterator& iterator)
{
  if (!exEnv.C_verbose_answer_default) return iterator.bindenv;

  int count = iterator.arg_list.count();

  if (count > 0) {
    fprintf(exEnv.print_file, " (");
    ArgList *copy_args = ArgList::New(count);
    int env_size = CopyArgs(copy_args->first(), iterator.arg_list.first(),
			    iterator.bindenv, count, NULL);

    FOR_EACH_ARG(arg, ((ArgList)(*copy_args))) {
      arg->printon(exEnv.print_file) ;
      if (--count) fprintf(exEnv.print_file,", ");
     } END_EACH_ARG;

    delete copy_args;
    fprintf(exEnv.print_file,")\n");
   }
  else
    if (exEnv.C_verbose_answer_default)
      fprintf(exEnv.print_file, "yes.\n");

  fflush(exEnv.print_file);

  if (exEnv.C_interactive_mode_current) user_continue();

  return iterator.bindenv;
}

BindEnv *BindingsPrintSolver(BuiltinRelation& , TupleIterator& iterator)
{
  int count = iterator.arg_list.count();

  if (count > 0) {
    ArgList *copy_args = ArgList::New(count);
    int env_size = CopyArgs(copy_args->first(), iterator.arg_list.first(),
			    iterator.bindenv, count, NULL);

    for (int i = 0; i < count; ) {
      iterator.arg_list[i]->printon(exEnv.print_file);
      fprintf(exEnv.print_file, "=");
      (*copy_args)[i]->printon(exEnv.print_file);
      /***
      if ((*copy_args)[i]->kindof() == COR_NUM_CONST)
	  delete (*copy_args)[i];
      ****/

      if (++i < count) fprintf(exEnv.print_file, ", ");
     }

    delete copy_args;
    fprintf(exEnv.print_file,".\n");
   }
  else
    if (exEnv.C_verbose_answer_default)
      fprintf(exEnv.print_file, "yes.\n");

  fflush(exEnv.print_file);

  if (exEnv.C_interactive_mode_current) user_continue();

  return iterator.bindenv;
}

BindEnv *PrintSolver(BuiltinRelation& , TupleIterator& iterator)
{
  int count = iterator.arg_list.count();

  if (count > 0) {
    ArgList *copy_args = ArgList::New(count);
    int env_size = CopyArgs(copy_args->first(), iterator.arg_list.first(),
			    iterator.bindenv, count, NULL);

    FOR_EACH_ARG(arg, (*copy_args)) {
      arg->printon(exEnv.print_file) ;
      if (--count) fprintf(exEnv.print_file," ");
     } END_EACH_ARG;

    delete copy_args;
   }

  fflush(exEnv.print_file);

  return iterator.bindenv;

}

BindEnv * PrintfSolver(BuiltinRelation&, TupleIterator& iterator)
{
  int argcount = iterator.arg_list.count();

  if (argcount <= 0)
    {
      CORAL_error(COR_EMPTY_ARGLIST, NULL, "PrintSolver");
      return iterator.bindenv ;
    }

  ArgList *copy_args = ArgList::New(argcount);
  int env_size = CopyArgs(copy_args->first(), iterator.arg_list.first(),
			  iterator.bindenv, argcount, NULL);

  Term term1(copy_args->arg(0), NULL);
  if (term1.expr->kindof() != COR_SYMBOL_CONST )
    {
    CORAL_error(COR_BAD_ARG_TYPE,
                  "arg 1 isn't a format string","PrintSolver");
    term1.printon(exEnv.error_file);
    fprintf(exEnv.error_file,"\n");
    iterator.set_no_match();
    return iterator.bindenv;
   }

  if (*(((Symbol *)(term1.expr))->string()) != '\"')
    {
      CORAL_error(COR_BAD_ARG_TYPE, "arg 1 isn't a format string",
          "PrintSolver");
      return iterator.bindenv ;
    }

  char * format_str = new char[200];
  char * temp1;
  char * temp2 = SymbolString(term1.expr)+1 ;
  int count = 1;
  for (;;)
    {
        if (*temp2 == '\"')
error:          break ;
        if (*temp2 == '\0')
          {
            fprintf(exEnv.error_file, "Unterminated \".\n");
            break;
          }
        if (*temp2 == '\\')
          {
        temp2++ ;
        switch(*temp2) {
            case 'n'  : fputc('\n', exEnv.print_file); temp2++; break;
            case 't'  : fputc('\t', exEnv.print_file); temp2++; break;
            case 'v'  : fputc('\v', exEnv.print_file); temp2++; break;
            case 'b'  : fputc('\b', exEnv.print_file); temp2++; break;
            case 'r'  : fputc('\r', exEnv.print_file); temp2++; break;
            case 'f'  : fputc('\f', exEnv.print_file); temp2++; break;
            case '\"' : fputc('\"', exEnv.print_file); temp2++; break;
            case '\\' : fputc('\\', exEnv.print_file); temp2++; break;
            case '?' : fputc('?', exEnv.print_file); temp2++; break;
            case '\'' : fputc('\'', exEnv.print_file); temp2++; break;
            }
        }
       else if (*temp2 == '%')
         {
            temp1 = format_str;
            *temp1 = '%';
            temp1++;

            temp2++;
            if (*temp2 == '-')
              {
                *temp1 = '-';
                temp1++;
                temp2++;
              }
            while  ((*temp2 >= '0') && (*temp2 <= '9'))
              {
                *temp1 = *temp2;
                temp1++;
                temp2++;
              }
            if (*temp2 == '.')
              {
                *temp1 = '.';
                temp1++;
                temp2++;
              }
            while  ((*temp2 >= '0') && (*temp2 <= '9'))
              {
                *temp1 = *temp2;
                temp1++;
                temp2++;
              }
            if ((*temp2 == 'h') || (*temp2 == 'l'))
             {
               *temp1 = *temp2;
               temp1++;
               temp2++;
            }
            switch(*temp2) {
                case 'd': case 'i': case 'u': case 's': case 'f':
                case 'e': case 'E': case 'g': case 'G':
                         *temp1 = *temp2;
                         temp1++;
                         temp2++;
                         *temp1 = '\0';
                         if (count >= argcount)
                           {
                              fprintf(exEnv.error_file,
                               "Not enough arguments to print.\n");
                                goto error;
                            }
                          {
                          Term term(copy_args->arg(count), NULL);
                          count++;
                          term.expr->print(NULL, exEnv.print_file, format_str);
                          temp1 = format_str;
                          break;
                          }
                case '%': fputc('%', exEnv.print_file); break;
                }
        }
    else
      {
        fputc( *temp2, exEnv.print_file);
        temp2++;
      }

  }

  //fprintf(exEnv.print_file, "\n");
  fflush(exEnv.print_file);

  delete copy_args;
  return iterator.bindenv ;

}


BindEnv * StartTimeSolver(BuiltinRelation&, TupleIterator& iterator)
{
#ifndef UNIX_HP
    getrusage(RUSAGE_SELF, &SystemUsage) ;
#endif
    beginMem = (int)sbrk(0); // save the top of the execution stack.

#ifdef DEBUG
    ResetCounts();
#endif
    return iterator.bindenv ;
}

#define COR_U_SEC 1000000

BindEnv * EndTimeSolver(BuiltinRelation& , TupleIterator& iterator)
{
  FILE *fd;

  if (iterator.arg_list.count() == 1) {
    Term term1 (iterator.arg_list[0], iterator.bindenv);
    //FULL_DEREFERENCE_TERM(term1);
    DEEP_DEREFERENCE_TERM(term1);
    if (term1.expr->kindof() != COR_SYMBOL_CONST ) {
      fprintf(exEnv.error_file,"CORAL :: Error -- bad argument to display_timer =");
      term1.printon(exEnv.error_file);
      fprintf(exEnv.error_file,"\n");
      iterator.set_no_match();
      return iterator.bindenv;
    }
    char *file_name = strip_quotes(SymbolString(term1.expr));

    if (!strcmp(file_name, "stdout"))
      fd = stdout;
    else if (!strcmp(file_name, "stderr")) 
      fd = stderr;
    else if (!strcmp(file_name, exEnv.print_filename))
      fd = exEnv.print_file;
    else if (!strcmp(file_name, exEnv.trace_filename))
      fd = exEnv.trace_file;
    else if (!(fd = fopen(file_name, "a"))) {
      CORAL_error(COR_CANNOT_OPEN_FILE, file_name, "EndTimeSolver");
      delete [] file_name ;
      FAIL_RETURN ;
    }
  }
  else fd = exEnv.error_file ;


  long t1 , t2;
  double t3 , t4;
  struct rusage tmp_usage ;
  int mem_use;

  
#ifndef UNIX_HP
    getrusage(RUSAGE_SELF, &tmp_usage) ;
#endif

    t1 = tmp_usage.ru_utime.tv_sec - SystemUsage.ru_utime.tv_sec;
    t2 = tmp_usage.ru_utime.tv_usec - SystemUsage.ru_utime.tv_usec;
    t3 = ((double) t1) + ((double) t2) / COR_U_SEC;

    fprintf(fd, "User CPU Usage = %f seconds\n", t3) ;

    t1 = tmp_usage.ru_stime.tv_sec - SystemUsage.ru_stime.tv_sec;
    t2 = tmp_usage.ru_stime.tv_usec - SystemUsage.ru_stime.tv_usec;
    t4 = ((double) t1) + ((double) t2) / COR_U_SEC;

  fprintf(fd, "System CPU Usage = %f seconds\n", t4) ;

#ifdef DEBUG 
  /*** this is an incorrect memory metric !!! -- PRAVEEN *****/
  mem_use = (int)sbrk(0);
  fprintf(fd,"Memory Usage = %d bytes\n",mem_use - beginMem );
#endif

#ifdef DEBUG
  fprintf(fd,"Page Faults = %d\n",(tmp_usage.ru_minflt + tmp_usage.ru_majflt) -
	  (SystemUsage.ru_minflt + SystemUsage.ru_majflt));

  fprintf(fd,"Number of Context Switches = %d\n", (tmp_usage.ru_nvcsw + tmp_usage.ru_nivcsw) - 
	  (SystemUsage.ru_nvcsw + SystemUsage.ru_nivcsw));
#endif


  t1 = tmp_usage.ru_utime.tv_sec - startRUsage.ru_utime.tv_sec;
  t2 = tmp_usage.ru_utime.tv_usec - startRUsage.ru_utime.tv_usec;
  t3 = ((double) t1) + ((double) t2) / COR_U_SEC;
  fprintf(fd, "Total CPU Usage of session = %f seconds\n", t3); 

#ifdef DEBUG 
  /*** this is an incorrect memory metric !!! -- PRAVEEN *****/
  fprintf(fd, "Total Memory Usage = %d bytes\n", mem_use - startMemUsage);
#endif

#ifdef DEBUG
  if (!exEnv.C_quiet_mode_default)
    DumpCounts(fd);
#endif

  if ((fd != exEnv.error_file) && (fd != stdout) &&
      (fd != exEnv.print_file))
    fclose(fd);
  
  return iterator.bindenv ;
}

Tuple *CputimeSolver(Tuple *query)
{
  if (query == NULL) {
    fprintf(exEnv.error_file,"Null argument passed to cputime \n");
    return NULL; 
  }

  int arity;
  if ((arity = query->arity()) != 1) {
    CORAL_error(COR_BAD_ARITY, "1 argument to cputime needed", NULL);
    return NULL;
  }

  if (((query->args())[0])->kindof() != COR_VARIABLE) {
    fprintf(exEnv.error_file,"CORAL :: Error -- first argument to cputime should be free");
    (query->args())[0]->printon(exEnv.error_file);
    fprintf(exEnv.error_file,"\n");
    return NULL;
  }

  long t1 , t2;
  double t3 ;
  struct rusage tmp_usage ;

#ifndef UNIX_HP
  getrusage(RUSAGE_SELF, &tmp_usage) ;
#endif

  t1 = tmp_usage.ru_utime.tv_sec - startRUsage.ru_utime.tv_sec;
  t2 = tmp_usage.ru_utime.tv_usec - startRUsage.ru_utime.tv_usec;
  t3 = ((double) t1) + ((double) t2) / COR_U_SEC;

  ArgList *alist = ArgList::New(arity);
  (*alist)[0] = make_arg(t3);

  Tuple *return_tuple = new Tuple(alist);
  return return_tuple;
}
#undef COR_U_SEC

/*---------------------------------------------------------------- */
/* Here's an example of how to declare a builtin relation 	   */

BindEnv * ShellSolver(BuiltinRelation& , TupleIterator& iterator)
{
  Term term1 (iterator.arg_list[0], iterator.bindenv);
  //FULL_DEREFERENCE_TERM(term1);
  DEEP_DEREFERENCE_TERM(term1);
  if (term1.expr->kindof() != COR_SYMBOL_CONST ) {
	fprintf(exEnv.error_file,"CORAL :: Error -- bad argument to shell =");
        term1.printon(exEnv.error_file);
	fprintf(exEnv.error_file,"\n");
	iterator.set_no_match();
	return iterator.bindenv;
  }

  char *term_str = strip_quotes(SymbolString(term1.expr)) ;
  if (system(term_str)) {
	CORAL_error(COR_FAILED_COMMAND, term_str, "ShellSolver");
	iterator.set_no_match() ;
  }

  delete [] term_str ;
  return iterator.bindenv ;
}

Tuple * ExtractSolver(Tuple *query)
{
  if (query == NULL) {
    fprintf(stderr,"Null argument passed to extractsolver \n"); 
    return NULL; 
   } 
  int arity = query->arity(); 
  ArgList *alist = ArgList::New(arity);
  (*alist)[0] = (query->args())[0];
  (*alist)[1] = (query->args())[1];
  char* arg0 ; 
  arg0 = strip_quotes((char *)make_string((query->args())[0])); 
  int  arg1 ; 
  arg1 = (int)make_int((query->args())[1]); 
  char* retvalue ; 
  if (arg1 >= strlen(arg0)) {
    delete alist;
    return NULL;
   }
  retvalue = new char[2];
  retvalue[0] = arg0[arg1];
  retvalue[1] = '\0';
  
  (*alist)[2] = make_arg(retvalue);
  Tuple *return_tuple = new Tuple(alist);
  return return_tuple;
}

Tuple * LengthSolver(Tuple *query)
{
  if (query == NULL) {
    fprintf(stderr,"Null argument passed to lengthsolver \n"); 
    return NULL; 
   } 
  int arity = query->arity(); 
  ArgList *alist = ArgList::New(arity);
  (*alist)[0] = (query->args())[0];
  char* arg0 ; 
  arg0 = (char *)make_string((query->args())[0]); 
  int  retvalue ; 
  retvalue = strlen(arg0);
  (*alist)[1] = make_arg(retvalue);
  Tuple *return_tuple = new Tuple(alist);
  return return_tuple;
}

Tuple * SubstringSolver(Tuple *query)
{
  if (query == NULL) {
    fprintf(stderr,"Null argument passed to SubstringSolver \n"); 
    return NULL; 
   } 
  int arity = query->arity(); 
  ArgList *alist = ArgList::New(arity);
  (*alist)[0] = (query->args())[0];
  (*alist)[1] = (query->args())[1];
  char* arg0 ;
  arg0 = strip_quotes((char *)make_string((query->args())[0])); 
  char* arg1 ;
  arg1 = strip_quotes((char *)make_string((query->args())[1])); 

  char *tmpstr = strstr(arg0, arg1);
  if (!tmpstr) tmpstr = "";

  (*alist)[2] = make_arg(tmpstr);
  Tuple *return_tuple = new Tuple(alist);
  return return_tuple;
}
