/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

#ifdef WITH_PERSISTENCE

#include "persistent-rel.h"
#include <strings.h>
#include "interface.h"
#include "cor_error.h"
#include "interp.h"

extern int unify_literal(TupleIterator&, Tuple *, int) ;
extern int SM_initialized;
extern DatabaseStruct ExodusDB;

#define DELETE_INDEX_ENTRY 1
#define DELETE_OBJ_AND_INDEX_ENTRY 2


/*------------------------------------------------------------------
  PersistentRel() : Constructor that initializes the persistent-rel
  structure from the catalog if it exists in the catalog. Otherwise,
  an error is raised.

 Oddities/Quirks :: Note that the file names are only guarenteed to be
 unique upto COR_MAX_FILE_NAME_LEN

 -------------------------------------------------------------------*/
PersistentRel::PersistentRel(Name name, int arity, int& error)
  : StorageRelation(arity, -1), index_set(this)
{
  if (!SM_initialized) return ;

  OID oid ;

  if (!SM.lookupFile(name, arity, &oid)) {
    fprintf(exEnv.error_file, "SORRY ! Unknown persistent relation referenced !\n");
    error = 1 ;
    return ;
  }

  USERDESC *u_desc ;
  if (!SM.retrieveObject(&oid, &u_desc)) {
    fprintf(exEnv.error_file, "SORRY ! Unknown persistent relation referenced !\n");
    fprintf(exEnv.error_file, "Error in catalogs !!\n");
    error = 1 ;
    return ;
  }

  fd.copyDesc((FileDescStruct *)(u_desc->basePtr));

  this->name = name ;
  this->oid  = oid ;
  ref_count = 1 ;            // This is to prevent it from being emptied
                             // when it is closed in a database other than
			     // db_rels.

  lddesc= NULL;
  count = fd.count;
  check_subsum = !fd.duplicates;
  r_kind = find_r_kind(name);
  SM.releaseObject(u_desc) ;

  int index_offset = -1 ;
  char attrbuf[COR_MAX_ARGLIST_SIZE+1];

  // create appropriate in-memory index structures
  for (int i = 0; i < fd.num_indexes; i++) {
    if ((index_offset = fd.nextIndex(index_offset)) < 0) {
      CORAL_error(COR_INDEX_ERR, NULL, "PersistentRel::Ct");
      return ;
    }
    for (int i = 0; i < arity; i++)
      attrbuf[i] = ((fd.indexes[index_offset].attrs[i]) ? '1' : '0');

    attrbuf[arity] = '\0';

    if (this->add_index(attrbuf) == COR_I_FAIL) {
      fprintf(exEnv.error_file, "Failed to add index to persistent relation %s\n",
	      SymbolString(this->name));
     }
  }

  // enter into the exodus workspace
  if (!ExodusDB.RelationTable)
    ExodusDB.RelationTable = AllocTab(COR_RELATION_TABLE_INCR, NULL);
  Insert(ExodusDB.RelationTable, name, this);

}

PersistentRel::PersistentRel(Name name, int arity, int& error, OID* oidptr)
  : StorageRelation(arity, -1), index_set(this)
{
  if (!SM_initialized) return ;

  oid = *oidptr;

  USERDESC *u_desc ;
  if (!SM.retrieveObject(&oid, &u_desc)) {
    fprintf(exEnv.error_file, "SORRY ! Unknown persistent relation referenced !\n");
    fprintf(exEnv.error_file, "Error in catalogs !!\n");
    error = 1 ;
    return ;
  }

  fd.copyDesc((FileDescStruct *)(u_desc->basePtr));

  this->name = name ;
  this->oid  = oid ;
  ref_count = 1 ;            // This is to prevent it from being emptied
                             // when it is closed in a database other than
			     // db_rels.

  lddesc = NULL;
  r_kind = find_r_kind(name);
  count = fd.count;
  SM.releaseObject(u_desc) ;

  int index_offset = -1 ;
  char attrbuf[COR_MAX_ARGLIST_SIZE+1];

  // create appropriate in-memory index structures
  for (int i = 0; i < fd.num_indexes; i++) {
    if ((index_offset = fd.nextIndex(index_offset)) < 0) {
      CORAL_error(COR_INDEX_ERR, NULL, "PersistentRel::Ct");
      return ;
    }
    for (int i = 0; i < arity; i++)
      attrbuf[i] = ((fd.indexes[index_offset].attrs[i]) ? '1' : '0');

    attrbuf[arity] = '\0';

    if (this->add_index(attrbuf) == COR_I_FAIL) {
      fprintf(exEnv.error_file, "Failed to add index to persistent relation %s\n",
	      SymbolString(this->name));
     }
  }

  // enter into the exodus workspace
  if (!ExodusDB.RelationTable)
    ExodusDB.RelationTable = AllocTab(COR_RELATION_TABLE_INCR, NULL);
  Insert(ExodusDB.RelationTable, name, this);

}


// This version of the constructor is solely used for temporary relations
// used during module computation, and does not insert the relation into
// the persistente workspace table
PersistentRel::PersistentRel(Name name, int arity, int& error, 
			     FileDescStruct *fdptr)
  : StorageRelation(arity, -1), index_set(this)
{
  if (!SM_initialized) return ;

  fd.copyDesc(fdptr);

  this->name = name ;
  INVALIDATE_OID((this->oid));
  ref_count = 1 ;            // This is to prevent it from being emptied
                             // when it is closed in a database other than
			     // db_rels.
  lddesc = NULL;
  count = fd.count;
  r_kind = find_r_kind(name);

  int index_offset = -1 ;
  char attrbuf[COR_MAX_ARGLIST_SIZE+1];

  // create appropriate in-memory index structures
  for (int i = 0; i < fd.num_indexes; i++) {
    if ((index_offset = fd.nextIndex(index_offset)) < 0) {
      CORAL_error(COR_INDEX_ERR, NULL, "PersistentRel::Ct");
      return ;
    }
    for (int i = 0; i < arity; i++)
      attrbuf[i] = ((fd.indexes[index_offset].attrs[i]) ? '1' : '0');

    attrbuf[arity] = '\0';

    if (this->add_index(attrbuf) == COR_I_FAIL) {
      fprintf(exEnv.error_file, "Failed to add index to persistent relation %s\n",
	      SymbolString(this->name));
     }
  }

}

typedef Association (*Func)(Pointer, Association*);
extern void ForEachTabElement(struct SymbolTable *, Func, Pointer) ;
Association flush_to_disk(Pointer dummy, Association *ass)
{
  ((PersistentRel *)(ass->val))->flush_to_disk();

  return *ass ;
}

void propagateChangesToDisk()
{
  if (!SM_initialized) return ;

  ForEachTabElement(ExodusDB.RelationTable, (Func)flush_to_disk, NULL);
}

// send the info in the PersistentRel back on disk
void PersistentRel::flush_to_disk()
{
  fd.count = count;

  USERDESC *u_desc ;
  if (!SM.retrieveObject(&oid, &u_desc)) {
    fprintf(exEnv.error_file, "SORRY ! Unknown persistent relation referenced !\n");
    fprintf(exEnv.error_file, "Error in catalogs !!\n");
    return ;
  }

  // write out the new file descriptors
  int e = sm_WriteObject(SM.bufGroup(), 0, sizeof(FileDescStruct),
			 (char *)(&fd), u_desc, TRUE ) ;
  ErrorCheck(e, "sm_WriteObject") ;
  
}

void PersistentRel::print_name(FILE *outf)
{
  fprintf(outf, "%15s/%d  : (exodus)  ", SymbolString(name), arity()) ;
  index_set.print_adornments(outf);
//  fd.print_index_adornments(outf);
  fprintf(outf, ": count %d\n", count);
  fprintf(outf, "\n");
}

void PersistentRel::print_facts(FILE *outf, ArgList *) const
{
  fprintf(outf, "rel_open(%s(", SymbolString(name));
  for (int i = 0; i < arity() -1 ; i++) {
    fprintf(outf, "x,");
  }

  if (arity()) fprintf(outf, "x), db_rels).\n") ;
  else fprintf(outf, "), db_rels).\n");

}


/*------------------------------------------------------------------
  insertObject() : Inserts the tuple object specified by the arglist
  into the relation.

 Oddities/Quirks :: The schema describes the structure of the data.
 The entire object is first constructed in memory, and a persistent
 object is created out of it at one shot. In the case
 of strings, the length of the string(an int) is first written to the
 file, followed by the actual string. The object is also added to the
 appropriate existing indexes.

 -------------------------------------------------------------------*/
void PersistentRel::insertObject(ArgList &arglist)
{
  if (!SM_initialized) return  ;

  int e;

  // Check that arity of relation and arglist match - defensive programming
  if (arglist.arity() != arity()) {
    CORAL_error(COR_BAD_ARGLIST, "this error shouldn't happen",
		"PersistentRel::insert");
    return  ;
  }

  int obj_len = 0;

  char **data_ptr = new char* [arity()];  // array of pointers to data items
  int  *data_size = new int [arity()];

  // determine size of each field of the object
  for (int i = 0; i < arity(); i++) {
    switch(fd.schema[i]) {

    case COR_STRING_DBTYPE :
      data_size[i] = strlen(data_ptr[i] = make_string(arglist[i])) + 1 ;

      // to store the length of the string along with the string
      obj_len += data_size[i] * sizeof(char) + sizeof(int); 
	
      break ;

    case COR_INT_DBTYPE :
      obj_len += sizeof(int) ;
      break ;

    case COR_DOUBLE_DBTYPE :
      obj_len += sizeof(double) ;
      break ;

    default :
      CORAL_error(COR_BAD_SCHEMA, "INVALID field reached", "insertObject");
      return  ;
      // This should not happen !!
    }
  }

  char *obj_data = new char[obj_len];
  int obj_offset = 0;
  int tmp_int ;
  double tmp_double ;

  // create an in-memory copy of the object data
  for (i = 0; i < arity(); i++) {
    switch(fd.schema[i]) {

    case COR_STRING_DBTYPE :
      memcpy(obj_data+obj_offset,(char *)(data_size + i), sizeof(int));
      obj_offset += sizeof(int);
      memcpy(obj_data+obj_offset,(char *)(data_ptr[i]), 
	     data_size[i]*sizeof(char));
      obj_offset += data_size[i]*sizeof(char);
      delete data_ptr[i];
      break ;

    case COR_INT_DBTYPE :
      tmp_int = make_int(arglist[i]);
      data_ptr[i] = (char *)(&tmp_int) ;
      memcpy((char *)(obj_data)+obj_offset,(char *)(data_ptr[i]), sizeof(int));
      /**
      memcpy((char *)(obj_data)+obj_offset,(char *)(&tmp_int), sizeof(int));
      **/
      obj_offset += sizeof(int);
      break ;

    case COR_DOUBLE_DBTYPE :
      tmp_double = make_double(arglist[i]);
      data_ptr[i] = (char *)(&tmp_double) ;
      memcpy(obj_data+obj_offset,(char *)(data_ptr[i]), sizeof(double));
      obj_offset += sizeof(double) ;
      break ;

    default :
      CORAL_error(COR_BAD_SCHEMA, "INVALID field reached", "insertObject");
      return  ;
      // This should not happen !!
    }
    ASSERT((obj_offset <= obj_len));

  }

  delete data_ptr;
  delete data_size;


  // now create the new object using the buffer group of the Storage Manager!
  // This is probably not optimal, but its better than creating a separate 
  // buffer pool just for this
  OID oid ;

  if (lddesc) {
    e = sm_LoadNextObject(lddesc, obj_len, (void *)obj_data, &oid);
    if (ErrorCheck(e, "sm_LoadNextObject")) {
      delete obj_data;
      return;
    };
  }
  else {
    e = sm_CreateObject(SM.bufGroup(), &(fd.fid), NEAR_LAST,
			NULL, NULL, obj_len,
			(void *)obj_data, &oid) ;
    if (ErrorCheck(e, "sm_CreateObject")) {
      delete obj_data;
      return;
    };
  }

  delete obj_data;

  USERDESC *u_desc ;
  int index_offset = -1 ;

  // Add object to all valid indexes

  if (fd.num_indexes) {
    if (!SM.retrieveObject(&oid, &u_desc)) {
      fprintf(exEnv.error_file, "SORRY ! Unknown persistent relation referenced !\n");
      fprintf(exEnv.error_file, "Error in catalogs !!\n");
      return;
    }
    for (i = 0; i < fd.num_indexes; i++) {
      if ((index_offset = fd.nextIndex(index_offset)) < 0) {
	CORAL_error(COR_INDEX_ERR, NULL, "PersistentRel::insertObject");
	SM.releaseObject(u_desc) ;
	return ;
      }
      fd.addObjectToIndex(index_offset, u_desc) ;
    }
  }
  (fd.count)++;
  
  if (fd.num_indexes)
    SM.releaseObject(u_desc) ;

}

/*------------------------------------------------------------------
  PersistentRel::checkTuple(USERDESC *, TupleIterator &)

   Determine if the retrieved tuple satisfies the bindings so far.
   If so, add to the bindings(change the bindenv) and return
   To do this, create an arglist that reflects the tuple
   and call unify_arglists(iterator.arglist, new_arglist,
  			     iterator.bindenv, NULL_BINDENV);
   This automatically checks for validity of the tuple and if valid
   also pushes the new environment on to the stack.

 Oddities/Quirks :: 
  In general, the tuple may contain variable length strings as fields.
  So the length of a string field is stored as an integer in front of it.

 -------------------------------------------------------------------*/
Tuple * PersistentRel::checkTuple(USERDESC *u_desc, TupleIterator &iterator)
{ 
  if (!SM_initialized) return NULL ;

  ArgList *new_arglist ;
  StackMark  stackmark0;
  int success = 0 ;
  int offset = 0 ;
  int str_len = 0 ;
  int data_len = 0 ;

  // Create an arglist corresponding to the retrieved tuple.
    new_arglist = ArgList::New(this->arity()) ;

    for (int i=0; i < arity(); i++) {
      switch(fd.schema[i]) {

      case COR_STRING_DBTYPE :
	memcpy((char *)&str_len ,(char *)(u_desc->basePtr+offset),sizeof(int));
	offset = offset + sizeof(int) ;
	data_len = str_len * sizeof(char) ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	    (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol))
	  (*new_arglist)[i] = make_arg((char *)(u_desc->basePtr + offset)) ;
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	  
	break ;
	
      case COR_INT_DBTYPE :
	data_len = sizeof(int) ;
	int itmp ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	   (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol)){
	  memcpy((char *)&itmp, (char *)(u_desc->basePtr+offset), sizeof(int));
	  (*new_arglist)[i] = make_arg(itmp) ;
	}
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	break ;
	
      case COR_DOUBLE_DBTYPE :
	data_len = sizeof(double) ;
	double dtmp ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	   (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol)){
	  memcpy((char *)&dtmp, (char *)(u_desc->basePtr+offset), sizeof(double));
	  (*new_arglist)[i] = make_arg(dtmp) ;
	}
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	break ;
	
      default :
	CORAL_error(COR_BAD_SCHEMA, "INVALID field reached", "checkTuple");
	return 0 ;
      }

      offset += data_len ;
    }

  // Create a Tuple corresponding to the retrieved tuple.
    Tuple *tuple = new Tuple(new_arglist) ;

  // Call unify_literal
  if (unify_literal(iterator, tuple, exEnv.dbg_indexing))
    return tuple ;

  for (int ti = 0; ti < tuple->arity(); ti++)
    if (tuple->arg(ti)->kindof() == COR_NUM_CONST)
      delete tuple->arg(ti);
  delete new_arglist;
  delete tuple;

  stackmark0.pop_to();
   return NULL ;
}

#undef FAIL_RETURN
#define FAIL_RETURN      {	iterator.scanDesc = NULL ;\
                                iterator.ipos = Scanning_Done ;\
                                iterator.set_no_match() ; \
	                        return NULL; }

/*------------------------------------------------------------------
  fileScanGetNext(iterator) : Gets the next tuple from the persistent
  relation using a file scan. The tuple should satisfy the selection
  implicit in the arglist of the iterator. The new bindings due to
  the selected tuple are (automatically) pshed onto the global
  trail stack.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Tuple * PersistentRel::fileScanGetNext(TupleIterator &iterator, int delete_flag)
{
  if (!SM_initialized) FAIL_RETURN ;

  int e ;
  USERDESC *u_desc ;
  BOOL     eof ;

  if (exEnv.C_interrupt_raised) FAIL_RETURN;

  // If this is the first time get_next is being called, the scan
  // may have to be initialized. However, subsequent get_next() calls
  // should continue from where they left off.

    if (!iterator.scanDesc) {
      e = sm_OpenScanWithGroup(&(fd.fid), PHYSICAL_SCAN, SM.bufGroup(),
			       (SCANDESC **)&(iterator.scanDesc), NULL);
      if (ErrorCheck(e, "sm_OpenScanWithGroup"))
	FAIL_RETURN ;
    }

    for (;;) {
      e = sm_ScanNextObject((SCANDESC *)iterator.scanDesc, 0,
			    READ_ALL, &u_desc, &eof) ;
      if (ErrorCheck(e, "sm_ScanNextObject") || eof) {
      // If there are no more tuples, return.
	e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
	ErrorCheck(e, "sm_CloseScan") ;
	FAIL_RETURN ;
      }

      // Determine if the retrieved tuple satisfies the bindings so far.
      // If so, add to the bindings(change the bindenv) and return
      Tuple *tuple ;
      if (tuple = checkTuple(u_desc, iterator)) {
	if (delete_flag) {
	  SM.destroyObject(&(u_desc->oid), SM.bufGroup());
	}
	return tuple;
      }
      // Else, get next tuple (return to loop )

    }

}

#undef FAIL_RETURN
#define FAIL_RETURN   { INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor)) ;\
			iterator.ipos = Scanning_Done ;\
			iterator.set_no_match() ;\
			return NULL ;	}

/*------------------------------------------------------------------
  indexScanGetNext(TupleIterator &iterator, IID iid) :
  Gets the next tuple from the persistent relation using an index scan
  using the given index id. The tuple should satisfy the selection
  implicit in the arglist of the iterator. The new bindings due to
  the selected tuple are (automatically) pushed onto the global
  trail stack.

 Oddities/Quirks ::

 -------------------------------------------------------------------*/
Tuple * PersistentRel::indexScanGetNext(TupleIterator &iterator, int index,
					int delete_flag)
{
  if (!SM_initialized)  FAIL_RETURN ;

  USERDESC *u_desc ;
  BOOL     eof ;
  KEY key ;
  OID oid ;
  int tmp_int ;
  double tmp_double ;
  char value_buff[COR_MAX_KEY_LEN] ;
  int buff_offset = 0 ;
  char *data_ptr = NULL;
  int data_len ;
  int str_len ;
  int e ;

  if (exEnv.C_interrupt_raised) FAIL_RETURN;

  // If this is the first time get_next is being called, the scan
  // may have to be initialized. However, subsequent get_next() calls
  // should continue from where they left off.

  if (CURSOR_IS_INVALID(*((SMCURSOR *)iterator.cursor))) {

  // Fill value_buff with the key value.
  // This is done by copying the appropriate attributes into value_buff

    ASSERT(iterator.arg_list.count() == arity()) ;

    for (int i = 0; i < arity(); i++) {

      Term term(iterator.arg_list[i], iterator.bindenv);
      FULL_DEREFERENCE_TERM(term);

      if (!(term.expr->isConstant())) continue ;

      switch(fd.schema[i]) {
	
      case COR_INT_DBTYPE :
	if (term.expr->kindof() != COR_NUM_CONST) {
	  fprintf(exEnv.error_file, "Error in scan of relation %s\n",
		  SymbolString(name));
	  CORAL_error(COR_BAD_ARGLIST, "int expected", "indexScanGetNext");
	  term.expr->printon(exEnv.error_file);	  
	  FAIL_RETURN
	}
	tmp_int = make_int(term.expr) ;
	data_ptr = (char *)&tmp_int ;
	data_len = sizeof(int) ;
	break ;

      case COR_STRING_DBTYPE  :
	if (term.expr->kindof() != COR_SYMBOL_CONST) {
	  fprintf(exEnv.error_file, "Error in scan of relation %s\n",
		  SymbolString(name));
	  CORAL_error(COR_BAD_ARGLIST, "string expected", "indexScanGetNext");
	  term.expr->printon(exEnv.error_file);
	  FAIL_RETURN
	}
	data_ptr = make_string(term.expr) ;
	str_len = strlen(data_ptr)+1 ;
	data_len = str_len * sizeof(char) ;
	break ;

      case COR_DOUBLE_DBTYPE :
	if (term.expr->kindof() != COR_NUM_CONST) {
	  fprintf(exEnv.error_file, "Error in scan of relation %s\n",
		  SymbolString(name));
	  CORAL_error(COR_BAD_ARGLIST, "double expected", "indexScanGetNext");
	  term.expr->printon(exEnv.error_file);	  
	  FAIL_RETURN
	}
	tmp_double = make_double(term.expr) ;
	data_ptr = (char *)&tmp_double ;
	data_len = sizeof(double) ;
	break ;

      default :
	CORAL_error(COR_BAD_SCHEMA, "INVALID field reached",
		      "indexScanGetNext");
	FAIL_RETURN;
	// This should not happen !!
      }

      if ((buff_offset+data_len < COR_MAX_KEY_LEN) && 
	  fd.indexes[index].attrs[i]){
	memcpy((char *)(value_buff + buff_offset),(char *)(data_ptr),
	       data_len ) ;
	buff_offset += data_len ;
      }
    }

 
    key.length = (TWO)(buff_offset) ;
    key.valuePtr = (void *)(value_buff) ;

    // Initialize the index scan
    e = sm_FetchInit(&(fd.indexes[index].iid), SM.bufGroup(),
		     &key, SM_EQ, &key, SM_EQ, (SMCURSOR *)iterator.cursor) ;
    
    if (ErrorCheck(e, "sm_FetchInit"))
      FAIL_RETURN ;
  }


  // At this stage, the scan has already been initialized either during
  // this function invocation, or in some previous invocation.
  for (;;) {
    // Get the next OID from the index
    e = sm_FetchNext((SMCURSOR *)iterator.cursor, NULL, &oid, &eof) ;
    // If there are no more index entries, return.
    if (ErrorCheck(e, "sm_FetchNext") || eof) FAIL_RETURN;

    verboseError = 0;
    //if (!SM.retrieveObject(&oid, &u_desc)) FAIL_RETURN ;
    if (!SM.retrieveObject(&oid, &u_desc)) continue;
    verboseError = !exEnv.C_quiet_mode_default;
    
    Tuple *tuple ;
    // Determine if the retrieved tuple satisfies the bindings so far.
    // If so, if delete_flag is set, delete the object, else
    // add to the bindings(change the bindenv) and return
    if (tuple = checkTuple(u_desc, iterator)) {
      if (delete_flag)  {
	// remove the index entry for the object
	(fd.indexes[index]).delete_entry(u_desc, fd.schema);
	if (delete_flag == DELETE_OBJ_AND_INDEX_ENTRY)
	  SM.destroyObject(&(u_desc->oid), SM.bufGroup());
      }
      SM.releaseObject(u_desc) ;
      return tuple;
    }

    // Else, get next tuple (return to loop )
    SM.releaseObject(u_desc) ;
  }

#undef FAIL_RETURN
}

void PersistentRel::initialize_iterator(TupleIterator& iterator)
{
  if (!SM_initialized) return ;

#ifdef DEBUG
  if (exEnv.dbg_get_next) {
    fprintf(exEnv.trace_file, "\nPersistentRelation::initialize_iterator: ");
    iterator.relation->name->print(NULL, exEnv.trace_file);
    fprintf(exEnv.trace_file, " ( ");
    iterator.arg_list.print(iterator.bindenv, exEnv.trace_file);
    fprintf(exEnv.trace_file, " )\n");
  }
#endif
  iterator.stack_mark.get_mark();

  if (! delta_indexed) {
    if (iterator.start_mark) { /* don't need to go to disk  */
      iterator.ipos = Scanning_New;
      iterator.ppos = (iterator.start_mark)->lastptr;
    }
    else {
      // need to scan from disk
      // find the appropriate index to be used
      iterator.ipos = Scanning_Known;
      iterator.index = index_set.find_index( iterator );
    }
  }
  else {
    if ( !iterator.in_delta()) { /* Starting on main relation */
      if (iterator.start_mark) { /* go to the delta relations */
	iterator.ipos = Scanning_Init;
	iterator.delta_rel_pos =  (iterator.start_mark)->rel_p;
      }
      else {
	// need to scan from disk
	// find the appropriate index to be used
        iterator.ipos = Scanning_Known;
	iterator.index = index_set.find_index( iterator );
      }
    }
    else {  /* In a delta relation */
      iterator.index = index_set.find_index( iterator );
    }
  }

  if (iterator.index) {

#ifdef DEBUG
    if (exEnv.dbg_indexing) {
      fprintf(exEnv.trace_file, "Index scan chosen");
      fprintf(exEnv.trace_file, "[ btree index %d ]\n",
	      ((PersistentIndex *)(iterator.index))->chosen_index);
    }
#endif
    // Index scan cursor needs to be created
    iterator.cursor = (void *) new SMCURSOR ;
    INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor));
  }
}

BindEnv *PersistentRel::get_next(TupleIterator& iterator)
{
  if (!SM_initialized) return NULL ;

  this->get_next_tuple(iterator) ;
  if (iterator.no_match()) return NULL;
  return(iterator.bindenv) ;
}


#define FAIL_RETURN  {iterator.ipos = Scanning_Done; 	\
			iterator.set_no_match();	\
			return NULL;			\
		     }

Tuple *PersistentRel::get_next_tuple(TupleIterator& iterator)
{
#ifdef DO_PROFILE
  if (T_Stack.count > 0)
    ((T_Stack.top()->moduledata->module_info.GlobalProfileInfo.getnexts)++;
  else
    (GlobalProfileInfo.getnexts)++ ;
#endif
  
  BindEnv *result_env = iterator.bindenv;
  iterator.tuple_env = NULL;
  
  Tuple *result = NULL;

  if (exEnv.C_interrupt_raised) FAIL_RETURN;
  if (!SM_initialized) {
    fprintf(exEnv.error_file, "Exodus not initialized !\n");
    FAIL_RETURN ;
  }

  // initialize
  if (iterator.ipos == Scanning_Init)
    initialize_iterator(iterator);
  else iterator.stack_mark.pop_to();

  if (!delta_indexed) { 
    if (iterator.ipos != Scanning_New) { /* need to scan from disk */

      // If no index is available, use a full file scan.
      if (!iterator.index)
	// This sets iterator.scanDesc
	result = fileScanGetNext(iterator) ;
      else 
	// iterator.index is set
	result = (iterator.index)->get_next_tuple(iterator);

      if (result != NULL) {
	// found something in main disk relation
	delete result->_args;
	delete result;
	return NULL; 
       }

      else {
	iterator.ipos = Scanning_New;
	iterator.ppos = &chain;
      }
    }
    StackMark stack_mark;
    // need to scan from the chain now
      
      for (;;) { 
	TupleLink **next = (TupleLink**)iterator.ppos;
	TupleLink *temp;
	if (*next == NULL) {
	  iterator.ipos = Scanning_Done;
	  iterator.set_no_match();
	  result = NULL;
	  stack_mark.pop_to();
	  return result;
	}
	if (iterator.end_mark && (iterator.end_mark)->lastptr == next) {
	  iterator.ipos = Scanning_Done;
	  iterator.set_no_match();
	  result = NULL;
	  stack_mark.pop_to();
	  return result;
	}
	Tuple *tuple = (*next)->tuple;
	
	iterator.ppos = &(*next)->next;
	
	if (tuple->is_deleted()) { /* the tuple has been logically deleted */
				     
	  temp = *next;
	  *(next) = (*next)->next;
	  delete temp;
	  
#ifdef DO_PROFILE
	  if (T_Stack.count > 0)
	    (T_Stack.top()->moduledata->module_info.
		GlobalProfileInfo.deleted_tuples)++; 
	  else
	    (GlobalProfileInfo.deleted_tuples)++; 
#endif
	  continue;
	} 
	
	if (unify_literal(iterator, tuple, exEnv.dbg_get_next))
	  return (tuple);
	stack_mark.pop_to();
      }
  }
  else { /* relation is delta-indexed */

    if (!iterator.in_delta()) { /* Scanning from main relation on disk */
      // If no index is available, use a full file scan.
      if (!iterator.index)
	// This sets iterator.scanDesc
        result = fileScanGetNext(iterator) ;
      else 
	// iterator.index is set
        result = (iterator.index)->get_next_tuple(iterator);

      if (result != NULL) {
	delete result->_args;
	delete result;
	return NULL;
       }
      if (!delta_relations) FAIL_RETURN;

      // now go into the delta relations
      iterator.ipos = Scanning_Init;	// Scanning_Init for delta reln
      iterator.delta_rel_pos = delta_relations->reln_chain;
    }

    // delta_rel_pos has to have been set by this point.
    if (delta_relations != NULL) { 
      // iterator scanning deltas, this == main relation
      for(;;) { /* Loop till successful delta is found */
	if (iterator.ipos == Scanning_Init) {
	  if (iterator.delta_rel_pos == NULL)
	    FAIL_RETURN;
	  if (iterator.end_mark && (iterator.delta_rel_pos == 
				    (iterator.end_mark)->rel_p)) 
	    FAIL_RETURN;
	}
	result = iterator.delta_rel_pos->reln->get_next_tuple(iterator);
	if (result != NULL) return result;
	// Done with current delta relation.  Got to go to next delta.
	iterator.delta_rel_pos = iterator.delta_rel_pos->next;
	iterator.ipos = Scanning_Init;  	
	// To start on new delta reln.
      } 
    }
    else { 
      // iterator scanning deltas, this == delta relation
      return iterator.index->get_next_tuple( iterator);
    }
  }
}

void PersistentRel::freeMark( RMark *mark, int hide_em )
{
  if (mark == NULL) return;
  RMark *this_mark = mark;

/**
  char **data_ptr = new char* [2];
**/

  // Unlink mark
  this_mark->next->prev = this_mark->prev;
  this_mark->prev->next = this_mark->next;

  // add it to the free marks list
  this_mark->next = free_marks;
  free_marks = this_mark;

  // check if this was the first mark in the list of marks
  if (this_mark->prev != &headMark) return ;
      // thus, if an "intermediate" mark is deleted, nothing is done!
      // this is fine for (!delta_indexed) relations, but with delta_indexed
      // relations, an alternative is to merge the delta relations 
      // corresponding to this mark with the "previous" relation.

  if (!delta_indexed) {
    TupleLink *limit;

    // Move all linked tuples upto new first mark to hash table.
    if (there_are_marks()) {
      limit = *headMark.next->lastptr;
      for (RMark *hmark = headMark.next; ; hmark = hmark->next) {
	if (hmark == &headMark) break;
	if (*hmark->lastptr != limit) break;
	hmark->lastptr = &chain;  
      }
    }
    else
      limit = *lastptr;
	
    TupleLink *old;
    while (chain && chain != limit) {
      Tuple *tuple = chain->tuple;
      old = chain;
      chain = chain->next;
      if (!hide_em) {
	count ++;
	/**
	fprintf(exEnv.error_file, "inserting this tuple ::");
	tuple->printon(exEnv.error_file);
	**/
	insert_raw(tuple);
      }
      else insert_in_hidden_relation(tuple);

      delete old;

      // make sure this calls the index updates also.
    }
    if (chain == NULL)
      lastptr = &chain;
  }
  else{ 
    RelationLink *limit;

    ASSERT((delta_relations!= NULL));
    ASSERT((delta_relations->reln_lastptr != NULL));
    
    if (there_are_marks()) 
      limit = headMark.next->rel_p;
    else limit = NULL;

    RelationLink *old;
    while(delta_relations->reln_chain!=limit) {
      StorageRelation *reln = delta_relations->reln_chain->reln;
      old = delta_relations->reln_chain;
      delta_relations->reln_chain = old->next;

      reln->add_all_tuples(this, /*marks_into_account=*/ 0,
                               /* check_dupl=*/ 0,
                               hide_em);
      reln->empty_relation();	// Empty tuples out of relation

      old->next = free_reln_list;
      free_reln_list = old;
    }
    if (delta_relations->reln_chain==NULL)
      delta_relations->reln_lastptr = NULL;
  }
}

void PersistentRel::free_allMarks()
{
    RMark *hmark;
    while (headMark.next != &headMark) {
	hmark = headMark.next;
	freeMark(hmark);
    }
    ASSERT((headMark.prev == headMark.next));
}

static void start_iterator(TupleIterator& iterator)
{
  if (!SM_initialized) return ;

#ifdef DEBUG
  if (exEnv.dbg_get_next) {
    fprintf(exEnv.error_file, "\nPersistentRelation::initialize_iterator: ");
    iterator.relation->name->print(NULL, exEnv.error_file);
    fprintf(exEnv.error_file, " ( ");
    iterator.arg_list.print(iterator.bindenv, exEnv.error_file);
    fprintf(exEnv.error_file, " )\n");
  }
#endif

  iterator.stack_mark.get_mark();
  iterator.ipos = Scanning_Persistent;
}

int PersistentRel::tuple_delete(Tuple *tuple)
{
  if (!SM_initialized) return 0;
  
  Tuple *return_tuple;
  int object_deleted = 0;

  // set up index scan cursor on all available indexes
  for (int i = 0; i < COR_MAX_INDEXES; i++)
    if (fd.indexes[i].valid()) {
      TupleIterator iterator(this, tuple);

      start_iterator(iterator);
      iterator.index = (GenericIndex *)i;
      iterator.cursor = (void *) new SMCURSOR ;
      INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor));

      if (object_deleted)
	return_tuple = indexScanGetNext(iterator, (int)(iterator.index), DELETE_INDEX_ENTRY);
      else {
	return_tuple = indexScanGetNext(iterator, (int)(iterator.index),
					DELETE_OBJ_AND_INDEX_ENTRY);
	object_deleted = 1;
      }
      iterator.stack_mark.pop_to();
      INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor)) ;
    }

  // if the object has not been deleted yet (because no index is present),
  // it is deleted using a full file scan

  if (object_deleted) {
    (fd.count)--;
    return 1;
  }

  TupleIterator iterator(this, tuple);
  start_iterator(iterator);
  iterator.index = (GenericIndex *)(-1);

  return_tuple = fileScanGetNext(iterator, DELETE_OBJ_AND_INDEX_ENTRY) ;
  iterator.stack_mark.pop_to();
  int e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
  ErrorCheck(e, "sm_CloseScan") ;
  iterator.scanDesc = NULL ;
  (fd.count)--;
  return 1;
}

int PersistentRel::is_subsumed(Tuple *tuple, RMark *, RMark *)
{
  if (!SM_initialized) return 0 ;
  StackMark stackmark ;

  TupleIterator tuple_its(this, tuple);
  if (tuple_its.get_next_tuple()) {
    stackmark.pop_to();
    return 1;
  }
  stackmark.pop_to();
  return 0;
}

int PersistentRel::insert_raw(Tuple *tuple)
{
  if (!SM_initialized) return 0 ;

  // Only flat tuples can be persistent !
  if (tuple->env_size) {
    CORAL_error(COR_BAD_TUPLE, "Only flat tuples can be persistent",
		"PersistentRel::insert");
    tuple->printon(exEnv.error_file);
    return 0 ;
  }

#ifdef DO_TRACE
  if (exEnv.dbg_indexing) {
    fprintf(exEnv.trace_file, "PersistentRel::insert_raw: ");
    name->print(NULL, exEnv.trace_file);
    fprintf(exEnv.trace_file,"(");
    tuple->printon(exEnv.trace_file);
    fprintf(exEnv.trace_file, ") ");
  }
#endif

  // Materialize a tuple in the desired format as an object to be inserted
  // insert it into the file
  insertObject(tuple->args()) ;

  delete &(tuple->args());
  delete tuple;
  return 1 ;
}

int PersistentRel::insert_tuple(Tuple *tuple)
{
  if (there_are_marks()) {
    if (delta_indexed) { delta_relations->insert_tuple(tuple); }
    else {
      this->tlist_insert_tuple(tuple);

      count++;
    }
  }
  else {
    count++;
    insert_raw(tuple);
  }

  return 1;
}

StorageRelation *make_pers_relation(Name name, int arity, int delta_index,
				  int set_subsum, ArgList *a_list = NULL)
{
  if (!SM_initialized) {
    fprintf(exEnv.error_file, "Not connected to EXODUS Server!\n");
    fprintf(exEnv.error_file, "Unable to create disk relation!\n");
    return make_local_relation(name, arity, delta_index, set_subsum);
  }
  int error = 0 ;

  // Create a file descriptor for the new relation

  if (!a_list) {
    a_list = ArgList::New(arity);
    Arg *intsym = EnterSymbol("int");
    for (int i = 0; i < arity; i++)
      (*a_list)[i] = intsym;
   }
  FuncArg *f_arg = (FuncArg *)make_arg(SymbolString(name), a_list);
  
  FileDescStruct *fd = new FileDescStruct(f_arg, error, 1 /*temp*/);

  PersistentRel *p_rel = new PersistentRel(name, arity,
					   error, fd);

  p_rel->delta_indexed = delta_index;
  p_rel->check_subsum = set_subsum;

  delete fd;

  if (error){
    delete p_rel ;
    return NULL ;
  }
  
  else return (StorageRelation *)p_rel ;
}

void deletePersistentRelation(Name name, int arity)
{
  if (!SM_initialized) return  ;
  int input = 'Y';

  if (exEnv.C_interactive_mode_default) {
    fprintf(exEnv.error_file, "\nAre you sure you want to delete persistent relation ");
    fprintf(exEnv.error_file, "%s arity %d ? (Y/N)\n", SymbolString(name), arity);
    input = getchar() ;
   }
  if ((input == 'Y')||(input == 'y'))
    SM.deleteFile(name, arity);
  else
    fprintf(exEnv.error_file, "Delete of persistent relation aborted by user !\n");
}

void PersistentRel::empty_relation(int delete_tuples)
{
  if (!SM_initialized) return ;

  int e ;
  OID oid, nextoid ;
  OBJHDR objhdr ;
  BOOL     eof ;

  if (!delete_tuples) return ;

  e = sm_GetFirstOid(SM.bufGroup(), &(fd.fid), &oid, &objhdr, &eof);
  if (ErrorCheck(e, "sm_GetFirstOid") || eof) return;

  for (;;) {
    e = sm_GetNextOid(SM.bufGroup(), &oid, &nextoid, &objhdr, &eof) ;
    // If there are no more tuples, return.
    if (ErrorCheck(e, "sm_GetNextOid") || eof) return;

    sm_DestroyObject(SM.bufGroup(), &oid) ;
    oid = nextoid ;

  }

}

void PersistentRel::release(TupleIterator& iterator)
{
  int e ;

  if (((int)(iterator.index)) < 0) {
    // Full file scan 
    if (iterator.scanDesc) {
      e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
      ErrorCheck(e, "sm_CloseScan") ;
      iterator.scanDesc = NULL ;
    }
  }
  else {
    // index scan
    if (iterator.cursor) {
      //delete (SMCURSOR *)(iterator.cursor);  CAUSING CORE DUMP on SPARKY !!
      iterator.cursor = NULL;
    }
  }

}

int PersistentRel::start_bulk_load()
{
  if (!SM_initialized) return 0 ;

  if (lddesc) {
    fprintf(exEnv.error_file, "previous bulk load not closed !\n");
    return 0;
  }

  int e = sm_OpenLoad(&(fd.fid), LOGICAL_LOAD, 8, 1.0, &lddesc);
  fprintf(exEnv.error_file, "lddesc is %x\n", lddesc);

  if (ErrorCheck(e, "sm_OpenLoad")) return 0;
  else return 1;
}

int PersistentRel::end_bulk_load()
{
  if (!SM_initialized) return 0 ;

  if (!lddesc) {
    fprintf(exEnv.error_file, "bulk load not opened !\n");
    return 0;
  }

  fprintf(exEnv.error_file, "lddesc is %x\n", lddesc);

  int e = sm_CloseLoad(lddesc);

  lddesc = NULL;

  if (ErrorCheck(e, "sm_CloseLoad")) return 0;
  else return 1;
}

#endif
