/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

#include "globals.h"
#include "ordsearch.h"
#include "interp.h"

extern int subsumes_tuple(Tuple *tuple, Tuple *newtuple, int display);
extern int subsumes_arg_lists(ArgList& left, ArgList& right,
    BindEnv *left_bindings, BindEnv *right_bindings);
extern Relation *find_relation(char *name, int arity);

int StorageRelation::insert_in_context(GoalNode *g_node, GoalNode *parent_goal)
{

    // first insert tuple in the for_ordsearch relation to index context
    for_ordsearch->insert_tuple(g_node);

    ASSERT((T_Stack.count != 0));
    ASSERT((T_Stack.top()->context != NULL));

    T_Stack.top()->context->add_new_goal(g_node, parent_goal);

    if (check_subsum != 1) return 1;

    ContextNode *g_c_node = g_node->find_context_node();
    ASSERT((g_c_node != NULL));
    int g_node_number = g_c_node->node_number;
    GoalNode *lowest_marked = NULL;
    int lowest_marked_number = g_node_number + 1;
    
    ArrayBindEnv *g_node_env = NULL;
    if (g_node->bindenv == NULL) {
	// this is a hack because of some restrictions in 
	// subsumes_arg_lists
	g_node_env = new ArrayBindEnv(g_node->env_size);
    }
    else g_node_env = (ArrayBindEnv *) g_node->bindenv;

    // now the code to find matching tuples from for_ordsearch relation
    // for subsumption purposes
    StackMark stack_mark10;
    TupleIterator iter10(for_ordsearch, g_node->args(), g_node_env);

    for (;;) {
        StackMark stackmark20;
        Tuple *tuple20 = iter10.get_next_tuple();
        stackmark20.pop_to();
    	// Undo any bindings done by get_next_tuple();
        if (iter10.no_match()) break;

        ASSERT((tuple20 != NULL));
        if (tuple20 == g_node) continue; 
    	// this can happen because g_node has already been 
    	// inserted in for_ordsearch relation 

	ContextNode *tuple20_c_node = 
		((GoalNode *) tuple20)->find_context_node();
	ASSERT((tuple20_c_node != NULL));
        int tuple20_node_number = tuple20_c_node->node_number;
        if (subsumes_arg_lists(tuple20->args(), g_node->args(),
    		iter10.tuple_env, g_node->bindenv)) {
    	    // we know that tuple20 subsumes g_node
    	    stackmark20.pop_to();
    	        // Undo bindings done by subsumes_arg_lists
    	
    	    if (g_node_number < tuple20_node_number) {
    	        // can delete the new inserted goal node 
    	        T_Stack.top()->context->delete_goal(g_node);
		stack_mark10.pop_to();
    	        return 0; 	// not inserted
    	    }
        }
        stackmark20.pop_to();

        if (subsumes_arg_lists(g_node->args(), tuple20->args(),
    		g_node_env, iter10.tuple_env)) {
    	    stackmark20.pop_to();
    	        // Undo bindings done by subsumes_arg_lists

    	    if ((tuple20_node_number <= g_node_number) && 
    	            (!(((GoalNode *) tuple20)->is_marked())) ) {
    	        // tuple20 is below g_node on context and can be deleted
    	        T_Stack.top()->context->delete_goal((GoalNode *) tuple20);

    	        // NOTE: might need to delete goal from the 
    	        // for_ordsearch relation 
    	        continue;
    	    }
    	    else if ((tuple20_node_number <= g_node_number) &&
    		    (((GoalNode *) tuple20)->is_marked())) {
    	        // have to find the lowest marked node subsumed by g_node
    	        if (tuple20_node_number < lowest_marked_number) {
    		    lowest_marked_number = tuple20_node_number ;
    		    lowest_marked = (GoalNode *) tuple20;
    	        }
    	        continue;
    	    }
        }
        stackmark20.pop_to();
    }
    stack_mark10.pop_to();

    // have removed all lower unmarked goals subsumed by g_node and
    // found the lowest marked goal subsumed by g_node

    if (lowest_marked_number <= g_node_number) {
        // g_node subsumes something marked below it
        // hence, nodes need to be collapsed

	ASSERT((lowest_marked != NULL));
        T_Stack.top()->context->collapse_nodes(
    		lowest_marked, parent_goal, g_node);
    }

    if (g_node->bindenv == NULL) {
        delete g_node_env;
    }
    return 1; // has been added to context but not subsumed
}

/****************************************************************/
/* ContextNode member functions */

ContextNode::ContextNode(GoalNode *new_goal, Context *c) 
{
    // assuming that a context node is created with only one unmarked goal

    goals = new_goal;

    // set the ContextNode field of GoalNode 
    new_goal->c_node = this;

    // link in the new goal in the unmarked list
    GoalNodeLink *g_link = new GoalNodeLink(new_goal);
    first_unmarked_goal = g_link;
    last_unmarked_goal = g_link;

    first_marked_goal = NULL;
    last_marked_goal = NULL;

    mark = 0;	// unmarked initially
    node_number = 0;

    negated = new_goal->negated;	// CHECK IF NEEDED??

    next = NULL;
    prev = NULL;
    next_marked = NULL;
    prev_marked = NULL;

    context = c;
}

int ContextNode::any_unmarked_goals()
{
    // NOTE: we also have to check for deleted subgoals, which will be 
    // retained in the goals if the context node is
    // non-singleton because deleting nodes from a set is difficult

    // this also tries and moves nodes from the unmarked goals list to
    // the marked goals list so that space can be reclaimed later
    // NOTE: should not affect any asymptotics

    if (first_unmarked_goal == NULL) return 0;	// failure
    for (GoalNodeLink *g_ptr = first_unmarked_goal; g_ptr != NULL; ) {
        if (!(g_ptr->g_node->is_deleted())) return 1;	// success	
	else { // the node pointed to by g_ptr is deleted
	    // deallocate the node and update first_unmarked_goal
	    first_unmarked_goal = g_ptr->next;
	    move_to_marked_list(g_ptr);
	    // NOTE: should not do a "delete g_ptr"
	    // because that would result in unreclaimable space
	}
    }
    update_goalnode_info();
    return 0;
}

GoalNode *ContextNode::get_unmarked_goal()
{
    if (!any_unmarked_goals()) { 
	return NULL;
    }

    // choose the first goal node from unmarked list, mark it, 
    // update unmarked and marked lists
    GoalNodeLink *g_ptr = first_unmarked_goal;
    first_unmarked_goal = first_unmarked_goal->next;
    move_to_marked_list(g_ptr);
    update_goalnode_info();

    GoalNode *g = g_ptr->g_node;
    g->mark_goal();
        // marking context node and updating Context done from caller site
	// as is updating the node_number field of this node

    return g;
}

void ContextNode::move_goals_to_done()
{
    ASSERT((is_marked()));
    ASSERT((first_unmarked_goal == NULL));

    // potentially there are two types of goals in the marked list
    // (1) magic facts: these have to be moved to the done relation
    // (2) supplementary facts: probably nothing has to be done to these

    for (GoalNodeLink *g_tmp = first_marked_goal; g_tmp != NULL; 
			g_tmp = g_tmp->next) {
	if ((g_tmp->g_node->is_magic_goal())&&(!(g_tmp->g_node->is_deleted()))){
	    StorageRelation *mp_done = g_tmp->g_node->find_done_relation();
#ifdef DO_TRACE
            if (exEnv.dbg_ordsearch) {
	        fprintf(exEnv.trace_file,
			"[Inserting %s(", SymbolString(mp_done->name));
	        g_tmp->g_node->args().print(NULL,exEnv.trace_file);
	        if (g_tmp->g_node->env_size == 0) 
		    fprintf(exEnv.trace_file, ") - done (ground)]\n");
	        else 
		    fprintf(exEnv.trace_file, ") - done (env-size:%d)]\n",
			    g_tmp->g_node->env_size);
	        context->print_all(exEnv.trace_file);
	    }
#endif

	    // NOTE !! This is not a copy of the tuple, but the very same tuple.
	    // Consequently, while emptying the done relation, the tuples should NOT
	    // be deleted.!!
	    mp_done->insert_tuple(g_tmp->g_node);
		g_tmp->g_node->owner_rel = mp_done;
	}
    }
}

void ContextNode::move_to_marked_list(GoalNodeLink *g_ptr)
{
    if (first_marked_goal == NULL) { // create a new list
	first_marked_goal = g_ptr;
	last_marked_goal = g_ptr;
    }
    else { // insert at end of list
	last_marked_goal->next = g_ptr;
	last_marked_goal = g_ptr;
    }
    last_marked_goal->next = NULL;
}

void ContextNode::merge_nodes(ContextNode *tbm)
{
    // the fields of tbm have to be merged into "this"

    // merge goals
    goals = goals->link_goals(tbm->goals);

    // merge unmarked goal lists
    if (first_unmarked_goal == NULL) { // nothing unmarked in "this"
	first_unmarked_goal = tbm->first_unmarked_goal;
	last_unmarked_goal = tbm->last_unmarked_goal;
    }
    else {
	if (tbm->first_unmarked_goal != NULL) { // something unmarked in tbm
	    last_unmarked_goal->next = tbm->first_unmarked_goal;
	    last_unmarked_goal = tbm->last_unmarked_goal;
	}
    }

    // merge marked goal lists
    ASSERT((first_marked_goal != NULL)); 
    if (tbm->first_marked_goal != NULL) { // something marked in tbm
	last_marked_goal->next = tbm->first_marked_goal;
	last_marked_goal = tbm->last_marked_goal;
    }

    // update node_number of the merged node to the higher of the two
    ASSERT((tbm->node_number > node_number));
    node_number = tbm->node_number;

    // NOTE: make sure caller in Context updates the first, last, etc. fields

    // NOTE: freeing up space of merged in node should be done from caller
}

void ContextNode::update_goalnode_info()
{
    if (first_unmarked_goal == NULL) last_unmarked_goal = NULL;
    if (first_marked_goal == NULL) last_marked_goal = NULL;
}

void ContextNode::free_all()
{
    // frees up the space of GoalNodeLink's but not the GoalNode's 
    // themselves because they are in a separate index and freeing
    // them would result in dangling pointers

    goals = NULL;

    GoalNodeLink *g_ptr;
    for (g_ptr = first_unmarked_goal; g_ptr != NULL; ) {
	first_unmarked_goal = g_ptr->next;
	delete g_ptr;
	g_ptr = first_unmarked_goal;
    }
    last_unmarked_goal = NULL;

    for (g_ptr = first_marked_goal; g_ptr != NULL; ) {
	first_marked_goal = g_ptr->next;
	delete g_ptr;
	g_ptr = first_marked_goal;
    }
    last_marked_goal = NULL;

    mark = 0;
    node_number = 0;

    next = NULL;
    prev = NULL;
    next_marked = NULL;
    prev_marked = NULL;

    context = NULL;

}

void ContextNode::print_all(FILE *file)
{
    fprintf(file, "\tPrinting ContextNode:\n");     
    fprintf(file, "\t\tMark = %d;  Node number = %d.\n", mark, node_number);

    GoalNodeLink *temp;

    if (first_unmarked_goal == NULL)
        fprintf(file, "\t\tNo unmarked goals.\n");
    else { // there are unmarked goals
	fprintf(file, "\t\tList of unmarked goals =\n");
	for (temp = first_unmarked_goal; temp != last_unmarked_goal->next;
					temp = temp->next) {
	    temp->g_node->print_all(file);
	}
    }

    if (first_marked_goal == NULL)
	fprintf(file, "\t\tNo marked goals.\n");
    else { // there are marked goals
	fprintf(file, "\t\tList of marked goals =\n");
	for (temp = first_marked_goal; temp != last_marked_goal->next;
					temp = temp->next) {
	    temp->g_node->print_all(file);
	}
    }
    fprintf(file, "\n");
}

// ContextNode::~ContextNode()
// {
//     free_all();
// }

/****************************************************************/
/* Context member functions */

Context::Context()
{
    first = NULL;
    last = NULL;
    first_marked = NULL;
    last_marked = NULL;
}

void Context::add_new_goal(GoalNode *g_node, GoalNode *par)
{
    // NOTE: this does not do the duplicate checking on the Context.
    // that HAS to be done elsewhere

    g_node->create_goalnode_info(par);
    ContextNode *c_node = new ContextNode(g_node, this);

    // the new goal and context are unmarked

    // first, we handle the extreme case of seed facts on Context
    // NOTE: all unmarked seed facts have a node_number of 0

    if (par == NULL) { // insert at end of Context
        insert_last(c_node);
	return;
    }

    // with a non-null "par" field, have to insert it just before
    // the next marked context node from context node of par
    // or at end of context if context node of par is last marked node

    ASSERT((par->is_marked()));
    ContextNode *par_c_node = par->find_context_node();
    ASSERT((par_c_node != NULL));
    ASSERT((par_c_node->is_marked()));

    // node number of newly inserted node is same as that of its parent's node
    c_node->node_number = par_c_node->node_number;

    ContextNode *par_next_c_node = par_c_node->next_marked;

    if (par_next_c_node == NULL) { // insert at end of Context
	insert_last(c_node);
    }
    else {
	insert_before(c_node, par_next_c_node);
    }

}

void Context::insert_last(ContextNode *c_node)
{
    if (first == NULL) { // this is the first context node in Context
	first = c_node;
	last = c_node;
    }
    else { 
	c_node->prev = last;
	last->next = c_node;
	last = c_node;
    }
}

void Context::insert_before(ContextNode *c_node, ContextNode *par_next_c_node)
{
    c_node->next = par_next_c_node;
    c_node->prev = par_next_c_node->prev;
    par_next_c_node->prev = c_node;
    c_node->prev->next = c_node;
}

// NOTE: deleting goal nodes from a context node in the non-ground case
// can only be done by setting a delete flag.  how does that
// affect the running time??

// In the all ground case, any node which is not a singleton node
// should have no unmarked goals in it.  so for the ground case,
// we have to worry about deletion only from singleton nodes,
// which should not mess anything up.

void Context::delete_goal(GoalNode *g)
{
    // NOTE: ordered search needs to delete only unmarked goals 
    //       during subsumption checks

    ASSERT((!(g->is_marked())));

    ContextNode *g_c_node = g->find_context_node();
    ASSERT((g_c_node != NULL));

    /******** the following is a hack ************
    if ((g_c_node->first_marked_goal == NULL) && 
		(g_c_node->first_unmarked_goal == NULL)) 
	// something shady happened
	return ;
    ******** the above is a hack ************/

#ifdef DO_TRACE
    if (exEnv.dbg_ordsearch) {
	fprintf(exEnv.trace_file, "\n\tTrying to delete GoalNode:  \n");
	g->print_all(exEnv.trace_file);
	fprintf(exEnv.trace_file, "\tCorresponding ContextNode is:  \n");
	g_c_node->print_all(exEnv.trace_file);
    }
#endif

    if (g_c_node->is_marked()) {  	// unmarked goal in marked node
	g->do_delete();	// mark goal as deleted, do nothing else
	g->c_node = NULL;
	g->parent_goal = NULL;
	g->parent_in_set = NULL;

	// NOTE: this is safe since unmarked goal nodes in marked
	//       context nodes are guaranteed to be leaf nodes because
	//       of the set manipulation operations
    }
    else { // unmarked goal in singleton unmarked node -- can delete everything
	ASSERT((g_c_node->first_marked_goal == NULL));
	ASSERT((g_c_node->first_unmarked_goal == g_c_node->last_unmarked_goal));

	g->do_delete();	// marking the goal itself as deleted
	g->c_node = NULL;
	g->parent_goal = NULL;
	g->parent_in_set = NULL;

	// unlink the context node
	unlink_node(g_c_node);

	g_c_node->free_all();
	delete g_c_node;
    }	 
}

void Context::unlink_node(ContextNode *g_c_node)
{

    if (g_c_node->next == NULL) { // last node in Context
	if (g_c_node->prev != NULL) { // not first node in Context
	    g_c_node->prev->next = NULL;
	}
	last = g_c_node->prev;
    }
    else { // not last node in Context
	g_c_node->next->prev = g_c_node->prev;
	if (g_c_node->prev != NULL) { // not first node in Context
	    g_c_node->prev->next = g_c_node->next;
	}
    }

    if (g_c_node->is_marked()) {
	if (g_c_node->next_marked == NULL) { // last marked node
	    if (g_c_node->prev_marked != NULL) { // not first marked node
		g_c_node->prev_marked->next_marked = NULL;
	    }
	    last_marked = g_c_node->prev_marked;
	}
	else { // not last marked node
	    g_c_node->next_marked->prev_marked = g_c_node->prev_marked;
	    if (g_c_node->prev_marked != NULL) { // not first marked node
		g_c_node->prev_marked->next_marked = g_c_node->next_marked;
	    }
	}
    }

    // update context info
    if (last == NULL) first = NULL;
    if (last_marked == NULL) first_marked = NULL;

}

GoalNode *Context::get_unmarked_goal(int &any_new_fact_in_done)
{
    any_new_fact_in_done = 0;
    if (last == NULL) { // empty context

#ifdef DO_TRACE
	if (exEnv.dbg_ordsearch) {
	    fprintf(exEnv.trace_file, "        OrdSearch: Empty context.\n");
	}
#endif

	return NULL;
    }

    GoalNode *fac;
    ContextNode *c = last; 
    fac = c->get_unmarked_goal();
    if (fac == NULL) { // node contains no unmarked subgoal
	ASSERT((c->is_marked()));
#ifdef DO_TRACE
	if (exEnv.dbg_ordsearch) {
	    fprintf(exEnv.trace_file, "OrdSearch: moving goals to done.\n"); 
	}
#endif
	c->move_goals_to_done();
	any_new_fact_in_done = 1;
	unlink_node(c);

	c->free_all();
        // ERROR!!  This causes major memory overwrite problems. If c is deleted here, there
        // are still pointers to it from the c_node of some GoalNode. This gets accessed later
        // via find_context_node, and we get screwed. : PRAVEEN
	//delete c;
	    // since we have moved facts to the done relation new inferences
	    // can possibly be made in Ordered Search
    }
    else { // have an unmarked goal
#ifdef DO_TRACE
	if (exEnv.dbg_ordsearch) {
            fprintf(exEnv.trace_file, "OrdSearch: found unmarked goal: ");
	    fprintf(exEnv.trace_file, 
		    "    %s(", SymbolString(fac->owner_rel->name));
	    fac->args().print(NULL,exEnv.trace_file);
	    fprintf(exEnv.trace_file, ") \n");
	    print_all(exEnv.trace_file);
	}

#endif
        if (!(c->is_marked())) { 
	    c->mark_node();
	    update_marked_info(c);
	    
	    // set node number of newly marked context node
	    if (c->prev_marked == NULL) {
		c->node_number = 1;
	    }
	    else {
		c->node_number = c->prev_marked->node_number + 1;
	    }
	}
    }
    return fac;
}

void Context::update_marked_info(ContextNode *c_node)
{
    // we are assuming that only the highest unmarked node on
    // Context ever gets marked, which is fine for ordered search

    if (first_marked == NULL) { // no marked nodes in Context yet
	first_marked = c_node;
	last_marked = c_node;
    }
    else {
	// only marked context nodes are linked up in this way
	c_node->prev_marked = last_marked;
	c_node->next_marked = NULL;
	last_marked->next_marked = c_node;
	last_marked = c_node;
    }
}

void Context::collapse_nodes(GoalNode *from, GoalNode *par, GoalNode *to)
{
    if (from == to) return;	// no collapsing required
    
    ASSERT((from->is_marked()));
    ASSERT((par->is_marked()));
    ASSERT((!(to->is_marked())));

#ifdef DO_TRACE
    if (exEnv.dbg_ordsearch) {
        fprintf(exEnv.trace_file, "OrdSearch: collapsing nodes\n");
        fprintf(exEnv.trace_file, "           from:  ");
        from->args().print(from->bindenv, exEnv.trace_file);
        fprintf(exEnv.trace_file, "\n           par :  ");
        par->args().print(par->bindenv, exEnv.trace_file);
        fprintf(exEnv.trace_file, "\n           to  :  ");
        to->args().print(to->bindenv, exEnv.trace_file);
    }
#endif 

    ContextNode *from_c_node = from->find_context_node();
    ContextNode *par_c_node = par->find_context_node();

    if (from_c_node != par_c_node) { // collapse marked nodes into from_c_node
        ContextNode *c_tmp;
	ContextNode *pnxt_tmp = par_c_node->next_marked;

	for (c_tmp = from_c_node->next_marked; 
		c_tmp && c_tmp != pnxt_tmp; ) {

	    // First check if any cycle of negative dependencies exist.
	    // Only the c_tmp nodes need to be checked, not the 
	    // from_c_node, since that could be negatively dependent on
	    // a previous goal, without any problems

	    // ASSERT((!(c_tmp->negated)));
	    if (c_tmp->negated) {
		fprintf(exEnv.error_file, "Cycle of negative dependencies detected!\n");
	    }

	    from_c_node->merge_nodes(c_tmp);
	    unlink_node(c_tmp);

	    // NOTE: cannot call c_tmp->free_all() since the goal node
	    //       links have also been linked up
            // TMP!!
	    delete c_tmp;
	    c_tmp = from_c_node->next_marked;
	}
    }
    
    // now collapse "to" into "from"
    ContextNode *to_c_node = to->find_context_node();

    // ASSERT((!(to_c_node->negated)));
    if (to_c_node->negated) {
	fprintf(exEnv.error_file, "Cycle of negative dependencies detected!\n");
    }
    if (!(subsumes_tuple(from, to, exEnv.dbg_ordsearch))) {
	from_c_node->merge_nodes(to_c_node);	
	unlink_node(to_c_node);

        // NOTE: cannot call to_c_node->free_all() since the goal node
	//       links have also been linked up
        // TMP!!
	delete to_c_node;
    }
    else { // no merging takes place
	to->do_delete();  // to make sure get_next_tuple does not get this
        unlink_node(to_c_node);
	to_c_node->free_all();
        // TMP!!
        delete to_c_node;
    }

#ifdef DO_TRACE
    if (exEnv.dbg_ordsearch) {
	print_all(exEnv.trace_file);
    }
#endif
}


void Context::print_all(FILE *file) 
{ 
    ASSERT((last->next == NULL)); 
    fprintf(file, "OrdSearch: printing state of Context.\n"); 
    fprintf(file, "\t---------------------------\n");
    for (ContextNode *temp = first; temp != NULL; temp = temp->next) {
	temp->print_all(file);
    }
    fprintf(file, "\t---------------------------\n");
}

void Context::free_all()
{
    // should also go through the context nodes and free all of those 
    first = NULL;
    last = NULL;
    first_marked = NULL;
    last_marked = NULL;
}

/****************************************************************/
/* GoalNode member functions */

extern int C_NewTupleCount, C_NewTupleSize;

GoalNode::GoalNode(StorageRelation *rel, int neg) : Tuple()
{
    set_defaults(rel, neg);
}

GoalNode::GoalNode(int arity, StorageRelation *rel, int neg): Tuple(arity)
{
    set_defaults(rel, neg);
}

GoalNode::GoalNode(ArgList *arglist, BindEnv *env, StorageRelation *rel, int neg) 
		: Tuple(arglist, env)
{
    set_defaults(rel, neg);
}

GoalNode::GoalNode(ArgList&a, StorageRelation *rel, int neg) : Tuple(a)
{
    set_defaults(rel, neg);
}

GoalNode::GoalNode(ArgList *arglist, StorageRelation *rel, int neg) : Tuple(arglist)
{
    set_defaults(rel, neg);
}

void GoalNode::set_defaults(StorageRelation *rel, int neg)
{
    mark = 0;
    parent_goal = NULL;
    c_node = NULL;
    owner_rel = rel;

    negated = neg;

    // now set the fields related to makeset() of union-find
    parent_in_set = this;
    rank_in_set = 0;
#ifdef DEBUG
	C_NewTupleSize += sizeof(GoalNode) - sizeof(Tuple);
#endif
}

GoalNode *GoalNode::find_canonical_elem ()
{
    // this is the find() operation in tarjan's union-find operation
    // it automatically does path compression, while doing a single
    // pass through the parent_in_set chain

    if (this != parent_in_set) {
	parent_in_set = parent_in_set->find_canonical_elem();
    }

    return parent_in_set;
}

ContextNode *GoalNode::find_context_node()
{
    // GoalNode maintains a c_node field, but for reasons of 
    // asymptotics, only the c_node field in the canonical GoalNode
    // of the set is the CORRECT one.

    GoalNode *can_g_node = find_canonical_elem();

    return can_g_node->c_node;
}

// NOTE: unmarked goal nodes in marked context nodes better be 
//       guaranteed to be leaf nodes because
//       this assumption is used elsewhere

GoalNode *GoalNode::link_goals(GoalNode *g)
{
    // this is the link() operation of tarjan's union-find
    if (rank_in_set < g->rank_in_set) {
	parent_in_set = g;
	g->c_node = c_node; // since g is the root of merged set
	return g;
    }
    if (rank_in_set = g->rank_in_set) {
	// going to make "this" as g's parent 
	rank_in_set++;
    }
    g->parent_in_set = this;
    g->c_node = NULL;		// for safety only
    return this;
}

int GoalNode::is_magic_goal()
{
    return (owner_rel->r_kind == COR_R_MAGIC);
}

extern char *make_done_prefix();

StorageRelation *GoalNode::find_done_relation()
{

    ASSERT((is_magic_goal()));
    if (!(owner_rel->mp_done_reln)) {
        char *rel_str = owner_rel->name->string();
        int rel_str_len = strlen(rel_str);

	char *done_str = make_done_prefix();
	int done_str_len = strlen(done_str);

        char *rel_done_str = new char [rel_str_len+done_str_len+1];
	    // 1 for the extra '\0' at end

	strcpy(rel_done_str, done_str);
        strcpy(rel_done_str+done_str_len, rel_str);
        rel_done_str[done_str_len+rel_str_len] = '\0';
	
	Name rel_done_name = EnterSymbol(rel_done_str);
	owner_rel->mp_done_reln = (StorageRelation *)
		find_local_relation(rel_done_name, owner_rel->arity());
	    // this only searches on the current frame
	if (!owner_rel->mp_done_reln) { // not on frame
	    owner_rel->mp_done_reln = (StorageRelation *)
	      make_local_relation(rel_done_name, owner_rel->arity(), 0);
		// NOTE: where can this relation be deleted???
	}

        delete rel_done_str;	
		// free up some space here
    }
    return owner_rel->mp_done_reln;
}

void GoalNode::print_all(FILE *file)
{
    fprintf(file, "\t\tPrinting GoalNode information.\n");
    fprintf(file, "\t\tMark = %d;    ", mark);
    fprintf(file, "%s (", SymbolString(owner_rel->name));
    Tuple::printon(file);
    fprintf(file, ")\n");
}

/****************************************************************/
/* GoalNodeLink member functions */

GoalNodeLink::GoalNodeLink(GoalNode *goal, GoalNodeLink *ptr)
{
    g_node = goal;
    next = ptr;
}

