/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/** 
	interface.C:
	This file implements several functions that can be used to create 
	(internal representations of) terms, arguments etc.  These
	may be used when coding in the imperative language part of CORAL.
**/

#include "interface.h"
#include "parse-utils.h"
#include "globals.h"
#include "parser.h"

extern "C" NumArg *MakeFixInt (int i);

/********************************************/

Arg *make_arg (int i)
{
    return (Arg *) MakeFixInt(i);
}

Arg *make_arg(short s)
{
    return (Arg *) MakeFixInt((int) s);
}

Arg *make_arg (long l)
{
    return (Arg *) MakeFixInt((int)l);
}

Arg *make_arg (float f)
{
    double doub = f;
    Arg *d = new NumArg(doub);
    return d;
}

Arg *make_arg (double n)
{
    Arg *d = new NumArg(n);
    return d;
}

Arg *make_arg (char *name)
{
    return (Arg *) EnterSymbol(name);
}

Arg *make_arg (Name name)
{
    return (Arg *) name;
}

/********************************************/

Arg *make_var (int i)
{
    char buf[100];
    sprintf(buf, "V_%d", i);
    Arg *v = new VarArg(EnterSymbol(buf), i);

    /********
    fprintf(stderr, "make_var (num=%d):",i);
    v->print(NULL,stderr);
    fprintf(stderr,"\n");
    ********/

    return v;
}

Arg *make_var (char *name, int i)
{
    Arg *v = new VarArg(EnterSymbol(name), i);
    return v;
}

Arg *make_var (Name name, int i)
{
    Arg *v = new VarArg(name,i);
    return v;
}

/********************************************/

Arg *make_arg (char *func_name, ArgList *args)
{
    FuncArg *fa = FuncArg::New(EnterSymbol(func_name), args->count());

    for (int i=0; i < args->count(); i++) {
	fa->args[i] = (*args)[i];
	ASSERT((fa->args[i] != NULL));
    }
    return fa;
}

Arg *make_cons (Arg *a, Arg *b)
{
    FuncArg *fa = FuncArg::New(ConsSymbol, 2);

    fa->args[0] = a;
    fa->args[1] = b;

    return fa;
}

Arg *make_nil ()
{
    return ((Arg *) NilArg);
    // this is defined in externs.h
}

/********************************************/

ArgList *make_arglist (int len ...)
{
    int i=0;
    va_list ap;
    va_start(ap,len);

    ArgList *al = ArgList::New(len);

    for(i =0; i < len; i++) {
    	(*al)[i] = ( (Arg **) ap)[i];
	ASSERT(((*al)[i] != NULL));
    }
    va_end(ap);
    return al;
}

ArgList *make_arglist1 (Arg* a1)
{

    ArgList *alist = ArgList::New(1);
    (*alist)[0] = a1;
    return alist;
}

ArgList *make_arglist2 (Arg* a1, Arg *a2)
{
    ArgList *alist = ArgList::New(2);
    (*alist)[0] = a1;
    (*alist)[1] = a2;
    return alist;
}

ArgList *make_arglist3 (Arg* a1, Arg *a2, Arg *a3)
{
    ArgList *alist = ArgList::New(3);
    (*alist)[0] = a1;
    (*alist)[1] = a2;
    (*alist)[2] = a3;
    return alist;
}

ArgList *make_arglist4 (Arg* a1, Arg *a2, Arg *a3, Arg*a4)
{
    ArgList *alist = ArgList::New(4);
    (*alist)[0] = a1;
    (*alist)[1] = a2;
    (*alist)[2] = a3;
    (*alist)[3] = a4;
    return alist;
}

ArgList *concat_arglists (ArgList *al1, ArgList *al2)
{
    int c1 = al1->count();
    int c2 = al2->count();
    ArgList *alist = ArgList::New(c1+c2);

    for (int i=0; i < c1; i++){
	(*alist)[i] = (*al1)[i];
    }

    for (int j=0; j < c2; j++){
	(*alist)[c1+j] = (*al2)[j];
    }
    return alist;
}

ArgList *make_vararglist (int n)
{
    ArgList *alist = ArgList::New(n);
    for (int i=0; i<n; i++)
	(*alist)[i] = make_var(i);
    return(alist);
}

/********************************************/

/** Tuple tuple_name (ArgList *alist)   and 
    Tuple *tuple_name_ptr = new Tuple (ArgList *alist)   are in term.h **/

/** Arg *Tuple::arg (int arg_pos)   is in term.C **/

/********************************************/

int is_int (Arg *a)
{
    return (is_num(a) && (((NumArg *) a)->num_kindof() == COR_INTEGER));
}

int is_short (Arg *a)
{
    return (is_num(a) && (((NumArg *) a)->num_kindof() == COR_INTEGER));
}

int is_long (Arg *a)
{
    return (is_num(a) && (((NumArg *) a)->num_kindof() == COR_INTEGER));
}

int is_float (Arg *a)
{
    return (is_num(a) && (((NumArg *) a)->num_kindof() == COR_DOUBLE));
}

int is_double (Arg *a)
{
    return (is_num(a) && (((NumArg *) a)->num_kindof() == COR_DOUBLE));
}

int is_string (Arg *a)
{
    return (a->kindof() == COR_SYMBOL_CONST);
}

int is_num (Arg *a)
{
    return (a->kindof() == COR_NUM_CONST);
}

int is_constant (Arg *a)
{
    return (is_string(a) || is_num(a));
	// assuming only two types of constants
}

int is_var (Arg *a)
{
    return (a->kindof() == COR_VARIABLE);
}

int is_functor (Arg *a)
{
    return (a->kindof() == COR_FUNCTOR);
}

int is_list (Arg *a)
{
    return ((is_functor(a)) && (((FuncArg *) a)->functor() == ConsSymbol));
}

int make_int(Arg *a)
{ 
    return ((NumArg *)a)->i_val; 
}

short make_short(Arg *a)
{ 
    return (short)(((NumArg *)a)->i_val);
}

long make_long(Arg *a)
{
    return (long)(((NumArg *)a)->i_val);
}

float make_float(Arg *a)
{
    double d = ((NumArg *) a)->d_val;
    return (float)d;
}

double make_double(Arg *a)
{
    double d = ((NumArg *) a)->d_val;
    return d;
}

char *make_string(Arg *a)
{
    char *result = new char[((Symbol *) a)->length() + 1];
  //bcopy(((Symbol *) a)->string(), result, (int) ((Symbol *) a)->length()+1);
   // memmove(result, ((Symbol *)a)->string(), (int)((Symbol *)a)->length()+1);
    memcpy(result, ((Symbol *)a)->string(), (int)((Symbol *)a)->length()+1);
    return result;
}

char *functor_name (Arg *a)
{
    Symbol *f_n = ((FuncArg *) a)->functor();
    char *result = new char[f_n->length() + 1];
    //bcopy(f_n->string(), result, (int) f_n->length() +1);
    //memmove(result, f_n->string(), (int) f_n->length() +1);
    memcpy(result, f_n->string(), (int) f_n->length() +1);
    return result;
}

ArgList *functor_args (Arg *a)
{
    return (& ((FuncArg *) a)->args);
}

Arg *make_car (Arg *a)
{
    return ((FuncArg *) a)->args[0]; 
}

Arg *make_cdr (Arg *a)
{
    return ((FuncArg *) a)->args[1];
}

Tuple *make_tuple(int arity)
{
  return new Tuple(arity);
}

Tuple *make_tuple(ArgList *arglist)
{
  return new Tuple(arglist);
}

Relation *make_rel(char *rel_name, int arity)
{
  return make_relation(EnterSymbol(rel_name), arity);
}

/********************************************/

void delete_id_tuple (Relation *rel, Tuple *tuple)
{
   /* we just mark the tuple as deleted. the actual deletion from the
    * the relation and indexes is done in a lazy fasion, when the tuple
    * is encountered in the course of a scan
    */
    if (tuple->is_deleted()) {
        fprintf(stderr, "Error: delete_tuple: tuple previously deleted\n");
    }
    else {
        tuple->do_delete();
	(rel->count)--;

#ifdef DO_TRACE
	if (exEnv.GlobalRelationOptions & REL_DISPLAY_INSERTIONS) {
	    fprintf(exEnv.trace_file, "[Deleting %s(", SymbolString(rel->name));
	    tuple->args().print(tuple->bindenv,exEnv.trace_file);
	    fprintf(exEnv.trace_file, ")]\n");
	}
#endif
  }
}

extern int subsumes_arg_lists(ArgList& left, ArgList& right,
    BindEnv *left_bindings, BindEnv *right_bindings);

/*
 * This is a value based delete, where every tuple
 * subsumed by tuple is deleted from the relation. 
 */
void delete_tuple (Relation *rel, Tuple *tuple)
{
  TupleIterator it(rel, tuple);
  StackMark stack_mark;

  for (;;) {
    Tuple *new_tuple = it.get_next_tuple();
    if (it.no_match())
      break;
    if (subsumes_arg_lists(tuple->args(), new_tuple->args(), 
				tuple->bindenv, it.tuple_env)) {
        delete_id_tuple(rel, new_tuple);
    }
    stack_mark.pop_to();
  }
  stack_mark.pop_to();

}

/**  update_tuple does not update indices yet.  fix this bug sometime.  **/

void update_tuple (Tuple *tuple, int arg_pos, Arg *new_val)
{
    if ((arg_pos < 0) || (arg_pos >= tuple->args().count())) {
	fprintf(stderr, "Error: update_tuple, argument position %d ", arg_pos);
	fprintf(stderr, "does not exist.\n");
	return /* 0 */ ;
    }
    else { // the argument position exists
	(tuple->args())[arg_pos] = new_val;
	return /* 1 */ ;
    }
}

void update_tuple (Relation *rel, Tuple *old_tuple, Tuple *new_tuple)
{
    delete_tuple(rel, old_tuple);
    rel->insert_new(new_tuple);
}

/********************************************/

int call_coral(char *exp_pred_name, Tuple *query, Relation *&result)
{

    Name pred = EnterSymbol(exp_pred_name);
    Relation *rel = find_external_relation(pred, query->args().count());
    if (!rel) {
	fprintf(stderr,"Error: call_coral: Relation %s not found\n", 
		exp_pred_name);
	return -1;
    }

    if (!result) result = new HashSimpleRelation(rel->arity());
    else
      if (rel->arity() != result->arity()) {
	fprintf(stderr, "Error: call_coral: arity mismatch between \
                         query and result relation\n");
	return -1;
      }
	

    ArrayBindEnv env(query->args().max_vars());

    TupleIterator it(rel, query->args(), &env, NULL, NULL);
    int answer_count = 0;
#ifdef DO_TRAIL
    StackMark stack_mark;
#endif
    for (;;) {
	it.get_next_tuple();
	if (it.no_match()) break;
	else {
	    result->insert_new(query->args(), &env);
	    answer_count ++;
	}
#ifdef DO_TRAIL
	stack_mark.pop_to();
#endif
    }
    return answer_count;
}

/* this function is probably outdated --- PRAVEEN */
int call_coral(char *exp_pred_name, Relation *queries, Relation *&result)
{

    Name pred = EnterSymbol(exp_pred_name);
    Relation *rel = find_external_relation(pred, queries->arity());
    if (!rel) {
	fprintf(stderr,"Error: call_coral: Relation %s not found\n", 
		exp_pred_name);
	return -1;
    }

    if (!result) result = new HashSimpleRelation(rel->arity());
    else
      if (rel->arity() != result->arity()) {
	fprintf(stderr, "Error: call_coral: arity mismatch between \
                         query and result relation\n");
	return -1;
      }

    int answer_count = 0;

    ArrayBindEnv env0(queries->arity());
    ArgList *arglist0 = make_vararglist(queries->arity());
       		// create an arglist with vars 0 through (arity-1) for it.

    TupleIterator it0(queries,*arglist0, &env0, NULL, NULL);
    	/** to iterate over tuples in queries **/

#ifdef DO_TRAIL
    StackMark stack_mark0;
#endif
    for (;;) {
	it0.get_next();
	if (it0.no_match())
	    break;
#ifdef DO_TRAIL
	StackMark stack_mark1;
#endif
        TupleIterator it1(rel, *arglist0, &env0, NULL, NULL);

	for(;;) {
	    it1.get_next();
	    if (it1.no_match()) break;
	    else {
	        result->insert_new(*arglist0, &env0);
	        answer_count ++;
	    }
#ifdef DO_TRAIL
	    stack_mark1.pop_to();
#endif
	}
#ifdef DO_TRAIL
	stack_mark0.pop_to();
#endif
    }
#ifdef DO_TRAIL
    stack_mark0.pop_to();
#endif
    return answer_count;
}

/********************************************/

/* add_builtin is only used from builtin-rel.C. It is NOT part of the
 * C++ interface
 */
int add_builtin (char *pred, int arity, UserRelProc pred_solver)
{

    BuiltinUserRelRelation *dummy = new BuiltinUserRelRelation (arity, 
		EnterSymbol(pred), pred_solver);
    return 1; // Success

}

int add_builtin (char *pred, int arity, UserTupleProc pred_solver)
{

    BuiltinUserTupleRelation *dummy = new BuiltinUserTupleRelation (arity, 
		EnterSymbol(pred), pred_solver);
    return 1; // Success
   
}

static int fill_in_params(ParserStruct *pstruct, ParameterStruct *params,
			   int& index)
{
  if (!params) return 0;
  struct parameter_link *temp_param = (pstruct->parameters)->last;
  Arg * temp_arg;
  
  while (temp_param) {
    switch (params[index].kind) {

    case COR_INT :
      temp_arg = make_arg(*((int *)(params[index].value_ptr)));
      break;

    case COR_SHORT :
      temp_arg = make_arg(*((short *)(params[index].value_ptr)));
      break;

    case COR_LONG :
      temp_arg = make_arg(*((long *)(params[index].value_ptr)));
      break;

    case COR_FLOAT :
      temp_arg = make_arg(*((float *)(params[index].value_ptr)));
      break;

    case COR_DOUB :
      temp_arg = make_arg(*((double *)(params[index].value_ptr)));
      break;

    case COR_STR :
      temp_arg = make_arg(*((char **)(params[index].value_ptr)));
      break;
      
      default :
	fprintf(stderr, "CORAL::error : erroneous parameter type\n");
        return 1;
    }
    *(temp_param->parameter_slot) = temp_arg;
    temp_param = temp_param->prev;
    index++;
  }
  return 0;
}

// execute the i'th ParserStruct in the ExecutionList
int ExecutionList::execute(int i, int& param_index)
{
  if (i >= actual_no_of_terms)
    return 1;
  

  if ((param_index < no_of_params) && 
	fill_in_params(exec_terms[i], parameters, param_index)) {
    fprintf(stderr, "fill_in_params fails !\n");
    return 1;
  }

  ASSERT((param_index <= no_of_params));
  exec_terms[i]->execute();

  return 0;  
}

int ExecutionList::execute()
{
  int param_index = 0;

  for (int i = 0; (i < actual_no_of_terms);i++){
    execute(i, param_index);
  }	

  return 0;
}

void ExecutionList::add(ParserStruct *pstruct)
{
  if (actual_no_of_terms >= max_no_of_terms) {
    ParserStruct **temp = exec_terms;
    max_no_of_terms = max_no_of_terms + step_size;
    exec_terms = new ParserStruct*[max_no_of_terms];
    for (int i = 0; i < actual_no_of_terms; i++)
      exec_terms[i] = temp[i];    
  }
  exec_terms[actual_no_of_terms++] = pstruct;
}

/*
 * copies contents of array arr2 to array arr1
 */
static void copy_array(struct array& arr1, struct array& arr2)
{
  arr1.length = arr2.length;
  arr1.data_size = arr2.data_size;
  arr1.data = (int *)malloc(arr1.data_size * sizeof(int));
  for (int i = 0; i < arr1.length; i++)
    arr1.data[i] = arr2.data[i];
}

void copy_structs(ParserStruct* pstruct, ParserStruct& parserStruct)
{
  *pstruct = parserStruct;
  pstruct->rules->lastptr = &(pstruct->rules->chain);
  pstruct->rule_list = (struct rule*)(pstruct->rules->chain);  

  copy_array(pstruct->rule_var_names, parserStruct.rule_var_names);
  copy_array(pstruct->sip_var_names, parserStruct.sip_var_names);
  copy_array(pstruct->rhspreds, parserStruct.rhspreds);
}

//process input stack until there are no entries on the stack
void processInputStack()
{
  exEnv.C_exec_mode = COR_DOING_INTERPRET;
  if (yyparse()) {
    fprintf(stderr, "yyparse error !\n");
  //  exit(1);
  }
  exEnv.C_exec_mode = COR_DOING_IMPERATIVE;
}

C_ScanDesc::C_ScanDesc(Relation *rel)
{
  stackmark = NULL;
  env = new ArrayBindEnv(rel->arity());
  arg_list = make_vararglist(rel->arity());
  it = new TupleIterator(rel, *arg_list, env);
}

C_ScanDesc::C_ScanDesc(Relation *rel, Tuple *match_tuple)
{
  stackmark = NULL;
  env = new ArrayBindEnv(rel->arity());
  arg_list = NULL;
  it = new TupleIterator(rel, match_tuple->args(), env);
}

Tuple *C_ScanDesc::next_tuple()
{
  if (stackmark) {
    stackmark->pop_to();
    /***
    // IS THIS NECESSARY ??
      delete env;
      env = new ArrayBindEnv(it->relation->arity());
    ***/
  }

  Tuple *ret_tuple = it->get_next_tuple();

  // Note :: pipelining does not return an actual tuple. instead it merely
  //         provides bindings for variables. Since the interface expects
  //         a handle on a real tuple, we have to materialize it
  if (!it->no_match() && !ret_tuple) {
    for (int i = 0; i < it->arg_list.arity(); i++) {
      TermLink *rv = NULL;
      it->arg_list[i]->simplify(it->bindenv, rv, it->bindenv, NULL);
    }    
    ret_tuple = new Tuple(it->arg_list);
  }
  return ret_tuple;
}

C_ScanDesc::~C_ScanDesc()
{
  delete stackmark;
  if (arg_list) delete arg_list;
  /***
    delete env;
  ***/
  delete it;
}
