/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

#include "externs.h"
#include "hash-index.h"
#include "profile.h"
#include "globals.h"
#include "ordsearch.h"
#include "interp.h"

int C_NewHashIndex = 0, C_NewHashIndexSize = 0;
int C_DeleteHashIndex = 0, C_DeleteHashIndexSize = 0;

extern int subsumes_arg_lists(ArgList& left, ArgList& right,
    BindEnv *left_bindings, BindEnv *right_bindings);
extern int subsumes_tuple(Tuple *tuple, Tuple *newtuple, int display);
int unify_literal(TupleIterator& iterator, Tuple *tuple, int display);

#define BACKBONESIZE 16

HashIndex::HashIndex( StorageRelation* rel, BitVector* aset, int buckets,
	double incr_ratio1, double max_occupancy1)
    : GenericIndex( rel, aset )
{

    // for the 0-arity reln, or for the all-free index
    if ((rel->arity() == 0) || (!aset->any())) {
	buckets = 1;  // Ought to be 0, but too much code probably depends on 
		      // a non-empty hash table.
	incr_ratio1 = 1.0;
	max_occupancy1 = 1000000.0;
    }

    if (buckets < 0) buckets = exEnv.C_hash_buckets_default;
    if (incr_ratio1 < 0.0) incr_ratio1 = exEnv.C_incr_ratio_default;
    if (max_occupancy1 < 0.0) max_occupancy1 = exEnv.C_max_occupancy_default;

    hash_size = buckets;
    hash_count = 0;
    incr_ratio = incr_ratio1;
    max_occupancy = max_occupancy1;
    limit = (int) (hash_size *max_occupancy);

    hash_table = new HashNodePtr[hash_size];

    register int i;
    for (i = 0; i < hash_size; i++)
      hash_table[i] = NULL;
    var_chain = NULL;
    C_NewHashIndex ++;
    C_NewHashIndexSize += hash_size * sizeof(HashNode *);

    non_empty_buckets = new int[hash_size];
    for (i = 0; i < hash_size; i++)
	non_empty_buckets[i] = -1;
    non_empty_bucket_head = -1 ;
    prev_bucket_ptrs = new int*[hash_size];

}


HashIndex::HashIndex( StorageRelation* rel, ArgList* pat, int n_var, 
	ArgList* v_nm, int buckets, double incr_ratio1, double max_occupancy1)
    : GenericIndex( rel, pat, n_var, v_nm )
{

    if (rel->arity() == 0)
	buckets = 1;  // Ought to be 0, but too much code probably depends on 
		      // a non-empty hash table.

    if (buckets < 0) buckets = exEnv.C_hash_buckets_default;
    if (incr_ratio1 < 0.0) incr_ratio1 = exEnv.C_incr_ratio_default;
    if (max_occupancy1 < 0.0) max_occupancy1 = exEnv.C_max_occupancy_default;

    hash_size = buckets;
    hash_count = 0;
    incr_ratio = incr_ratio1;
    max_occupancy = max_occupancy1;
    limit = (int) (hash_size *max_occupancy);

    hash_table = new HashNodePtr[hash_size];

    int i;
    for (i = 0; i < hash_size; i++)
	hash_table[i] = NULL;
    var_chain = NULL;
    C_NewHashIndex ++;
    C_NewHashIndexSize += hash_size * sizeof(HashNode *);

    non_empty_buckets = new int[buckets];
    for (i = 0; i < hash_size; i++)
	non_empty_buckets[i] = -1;
    non_empty_bucket_head = -1 ;
    prev_bucket_ptrs = new int*[buckets];

}

void HashIndex::reset_index()
{
    int i;

    non_empty_bucket_head = -1 ;
    var_chain = NULL;
    hash_count = 0;
    limit = (int) (hash_size *max_occupancy);
    for (i = 0; i < hash_size; i++)
	hash_table[i] = NULL;
    for (i = 0; i < hash_size; i++)
	non_empty_buckets[i] = -1;

}

void HashIndex::free_all_entries()
{
  register int j = 0;
    // delete nodes in the var_chain list
    HashNodePtr next;
    for (HashNodePtr h = var_chain; h; h = next) {
    	next = h->next;
	delete h;
    }
    var_chain = NULL;

    hash_count = 0;

    // delete nodes in the hash buckets

    for (j=non_empty_bucket_head; j >= 0 ; j=non_empty_buckets[j]) {
      for (h = hash_table[j]; h; h = next) {
	next = h->next;
	delete h;
      }
      hash_table[j] = NULL;
    }

    non_empty_bucket_head = -1 ;
}

void HashIndex::move_all_entries(GenericIndex *ind)
{

    if  (index_type() != ind->index_type()) {
	/***** WARNING - should also check for matching of indices: 
		With the current implementation the indices will match,
		but this is something that must be checked.  Insert a call :
	    if (!matches(ind)) error ....
		It has been left out for efficiency reasons, and because 
		matches is not fully implemented yet.
	******/
	fprintf(exEnv.error_file, "CORAL:: Internal Error - move_all_entries type ");
	fprintf(exEnv.error_file,"mismatch between indices.\n");
	return ;
    }

    register HashIndex *hindex = (HashIndex *) ind;
    register HashNodePtr next, h;

    // Move nodes in the var_chain list
    for (h = var_chain; h; h = next) {
	next = h->next;
	hindex->insert_entry(h);
    }
    var_chain = NULL;

    hash_count = 0;

    // Move nodes in the hash buckets

    register int j;

    for (j=non_empty_bucket_head; j >= 0 ; j=non_empty_buckets[j]) {
      for (h = hash_table[j]; h; h = next) {
	next = h->next;
	hindex->insert_entry(h);
      }
      hash_table[j] = NULL;
    }
    non_empty_bucket_head = -1 ;

}

void HashIndex::delete_all_tuples()
{
#ifdef DEBUG
if (exEnv.dbg_deletes) {
    fprintf(exEnv.trace_file, "Deleting %d tuples from index of relation", hash_count );
    relation->name->print(NULL, exEnv.trace_file);
    fprintf(exEnv.trace_file, "\n");
   }
#endif

    // delete nodes in the var_chain list
    HashNodePtr next;
    for (HashNodePtr h = var_chain; h; h = next) {
    	next = h->next;
	delete h->tuple();
	// h->value = NULL;
	delete h;
    }
    var_chain = NULL;

    hash_count = 0;

    // delete nodes in the hash buckets

    register int j ;
    for (j=non_empty_bucket_head; j >= 0 ; j=non_empty_buckets[j]) {
	for (h = hash_table[j]; h; h = next) {
	    next = h->next;
           /*******
	    for (int ti = 0; ti < h->tuple()->arity(); ti++)
	    // NUM_CONSTs can be safely deleted since they are never shared
	    if (h->tuple()->arg(ti)->kindof() == COR_NUM_CONST)
	      delete h->tuple()->arg(ti);
           *****/
	    delete h->tuple()->_args;
	    delete h->tuple();
	    delete h;
	}
	hash_table[j] = NULL;
    }
    non_empty_bucket_head = -1 ;
}

HashIndex::~HashIndex()
{
    free_all_entries();
    C_DeleteHashIndex ++;
    C_DeleteHashIndexSize += hash_size * sizeof(HashNode *);

    delete non_empty_buckets ;
    delete prev_bucket_ptrs;
  
   delete hash_table ;
}

/*****************************************************************************/

/** The following is an optimization to let a static environment to be used in
	reentrant code.  We assume no parallelism however.
***/
#define DONT_USE_STATIC_ENV   1
#define MAX_VARS 64
#define VAR_INCR 8
static ArrayBindEnv static_env1(MAX_VARS,VAR_INCR);
static int static_env1_inuse=DONT_USE_STATIC_ENV;


HashVal HashIndex::hash(ArgList *args, BindEnv *bindenv)
{
    if (args->count() == 0)
    /*************************************
      Normally this doesn't really mean that arity==0.
      Rather, it means that the TupleIterator was
      created with an ArgList parameter, which is a
      convention meaning essentially (VAR1, VAR2, ..., VARn).
      However, in pathological cases, it might really
      be a nullary Relation. Do the same thing in either case.
      *************************************/
      return VarHashValue;

    if (argset != NULL) { // the argument form index will be used
	return args->hash(*argset, bindenv);
    }

    StackMark stackmark0;
    HashVal result;

    // else the pattern form index will be used 

    ArrayBindEnv* pat_env;
    if (static_env1_inuse++ == 0 && num_vars_pat <= MAX_VARS)
        pat_env = &static_env1;
    else {
	pat_env = new ArrayBindEnv (num_vars_pat);
    }


    //if (subsumes_arg_lists(*pattern, *args, pat_env, bindenv)) {
    /*
     * NOTE: a subsumption check alone is insufficient ! In the presence of
     * non-ground facts, the tuple to insert may be more general than the
     * pattern of the index, and unification is needed to ensure that
     * all desired inserts occur --- PRAVEEN
     */
    if (unify_arg_lists(*pattern, *args, pat_env, bindenv) == COR_U_SUCCEED) {
	BitVector aset(var_names->count());
	for (int i = 0; i < var_names->count(); i++)
	    aset.set(i);
        
	result = var_names->hash(aset, pat_env);
	    /* hash on all arguments */
	stackmark0.pop_to();
	if( --static_env1_inuse != 0 || num_vars_pat > MAX_VARS) {
	    delete pat_env;
	}

	return result;
    } 
    /* else the tuple and the pattern don't match */
    stackmark0.pop_to();
    if( --static_env1_inuse != 0 || num_vars_pat > MAX_VARS) {
	delete pat_env;
    }
    return UnmatchedHashValue;
}


void HashIndex::insert_tuple( Tuple *tuple )
{
    if (tuple->is_deleted()) return;
	// tuple has been logically deleted

    HashVal h_val = hash(&tuple->args(), tuple->bindenv);

#ifdef DO_TRACE
    if (exEnv.dbg_indexing) {
        fprintf(exEnv.trace_file, "HashIndex::insert_tuple: ");
	relation->name->print(NULL, exEnv.trace_file);
	fprintf(exEnv.trace_file,"(");
	tuple->printon(exEnv.trace_file);
	fprintf(exEnv.trace_file, ") ");

	if (argset != NULL) { // argument form index
	    fprintf(exEnv.trace_file, "[hash index on ");
	    argset->printon(exEnv.trace_file);
	    fprintf(exEnv.trace_file, " ]\n");
	}
	else { // pattern form index
	    fprintf(exEnv.trace_file, "[pattern matched ");
	    pattern->print(NULL, exEnv.trace_file);
	    fprintf(exEnv.trace_file, "] ");
            fprintf(exEnv.trace_file, "[indexed on ");
	    var_names->print(NULL, exEnv.trace_file);
	    fprintf(exEnv.trace_file, " ]\n");
	}
	fprintf(exEnv.trace_file, "		hash value is %ld\n", h_val);
    }
#endif

    HashNode *new_node = new HashNode( h_val, tuple ); 
    tuple->ref_count++;
    if (h_val == VarHashValue) { // link it to the beginning of var_chain
	new_node->next = var_chain;
	var_chain = new_node;

#ifdef DO_TRACE
	if (exEnv.dbg_indexing) {
	    fprintf(exEnv.trace_file, "HashIndex::insert_tuple: ");
	    fprintf(exEnv.trace_file, "inserting in var_chain\n");
	}
#endif
    }
    else if (h_val != UnmatchedHashValue) { 
		// link it to the beginning of the appropriate hash bucket

	hash_count ++;
	int tmp_hash_index = (int) (h_val % hash_size);
	new_node->next = hash_table[tmp_hash_index];
	hash_table[tmp_hash_index] = new_node;
        if (!new_node->next) {
	  if (non_empty_bucket_head >= 0) 
	    prev_bucket_ptrs[non_empty_bucket_head] = &(non_empty_buckets[tmp_hash_index]);
	  non_empty_buckets[tmp_hash_index] = non_empty_bucket_head ;
	  non_empty_bucket_head = tmp_hash_index ;
	  prev_bucket_ptrs[tmp_hash_index] = &(non_empty_bucket_head);
	}

#ifdef DO_TRACE
	if (exEnv.dbg_indexing) {
	    fprintf(exEnv.trace_file, "HashIndex::insert_tuple: ");
	    fprintf(exEnv.trace_file, "inserting in %s bucket %d \n",
                    (new_node->next ? "non-empty" : "empty"), tmp_hash_index);
	}
#endif
    }
    /****** else (h_val == UnmatchedHashValue) { } ******/
    /*		i.e, pattern does not match iterator    */

    if ( hash_count > limit)
	rehash_bigger();
}

void HashIndex::insert_entry( HashNodePtr new_node )
{

    Tuple *tuple = new_node->tuple();
    HashVal h_val = new_node->hash_val;

    if (tuple->is_deleted()) return;
	// tuple has been logically deleted

#ifdef DO_TRACE
    if (exEnv.dbg_indexing) {
        fprintf(exEnv.trace_file, "HashIndex::insert_entry: ");
	relation->name->print(NULL, exEnv.trace_file);
	fprintf(exEnv.trace_file,"(");
	tuple->printon(exEnv.trace_file);
	fprintf(exEnv.trace_file, ") ");

	if (argset != NULL) { // argument form index
	    fprintf(exEnv.trace_file, "[hash index on ");
	    argset->printon(exEnv.trace_file);
	    fprintf(exEnv.trace_file, " ]\n");
	}
	else { // pattern form index
	    fprintf(exEnv.trace_file, "[pattern matched ");
	    pattern->print(NULL, exEnv.trace_file);
	    fprintf(exEnv.trace_file, "] ");
            fprintf(exEnv.trace_file, "[indexed on ");
	    var_names->print(NULL, exEnv.trace_file);
	    fprintf(exEnv.trace_file, " ]\n");
	}
	fprintf(exEnv.trace_file, "		hash value is %ld\n", h_val);
    }
#endif

    tuple->ref_count++;
    if (h_val == VarHashValue) { // link it to the beginning of var_chain
	new_node->next = var_chain;
	var_chain = new_node;

#ifdef DO_TRACE
	if (exEnv.dbg_indexing) {
	    fprintf(exEnv.trace_file, "HashIndex::insert_tuple: ");
	    fprintf(exEnv.trace_file, "inserting in var_chain\n");
	}
#endif
    }
    else if (h_val != UnmatchedHashValue) { 
		// link it to the beginning of the appropriate hash bucket
	hash_count ++;
	int tmp_hash_index = (int)(h_val % hash_size);
	new_node->next = hash_table[tmp_hash_index];
	hash_table[tmp_hash_index] = new_node;

        if (!new_node->next) {
	  if (non_empty_bucket_head >= 0) 
	    prev_bucket_ptrs[non_empty_bucket_head] = &(non_empty_buckets[tmp_hash_index]);
	  non_empty_buckets[tmp_hash_index] = non_empty_bucket_head ;
	  non_empty_bucket_head = tmp_hash_index ;
	  prev_bucket_ptrs[tmp_hash_index] = &(non_empty_bucket_head);
	}

#ifdef DO_TRACE
	if (exEnv.dbg_indexing) {
	    fprintf(exEnv.trace_file, "HashIndex::insert_tuple: ");
	    fprintf(exEnv.trace_file, "inserting in %s bucket %d \n",
                    (new_node->next ? "non-empty" : "empty"), tmp_hash_index);
	}
#endif
    }
    /****** else (h_val == UnmatchedHashValue) { } ******/
    /*		i.e, pattern does not match iterator    */

    if ( hash_count > limit)
	rehash_bigger();
}

void HashIndex::rehash_bigger( )
{
    int i;
    HashNodePtr cur, next, *prevptr;

    limit     = (int) (limit *incr_ratio);
    int old_hash_size = hash_size;
    hash_size = (int) (hash_size *incr_ratio);

    delete non_empty_buckets ;
    non_empty_buckets = new int[hash_size];
    for (i = 0; i < hash_size; i++)
	non_empty_buckets[i] = -1;
    non_empty_bucket_head = -1;
    delete prev_bucket_ptrs;
    prev_bucket_ptrs = new int*[hash_size];

    limit = (int) (hash_size * max_occupancy); 
			// Recalculated to avoid roundoff errors

    HashNodePtr *old_table = hash_table ;
    hash_table = new HashNodePtr[hash_size];
    for (i=0; i < hash_size; i++) {
	hash_table[i] = NULL;
    }

    for(i=0; i < old_hash_size; i++)  {
	prevptr = &(old_table[i]);
	for(cur=*prevptr; cur != NULL;) {
	    next = cur->next;
	    int tmp_hash_index = (int)(cur->hash_val % hash_size);
	    cur->next = hash_table[tmp_hash_index];
	    hash_table[tmp_hash_index] = cur;
	    if (!cur->next) {
	      if (non_empty_bucket_head >= 0) 
		prev_bucket_ptrs[non_empty_bucket_head] = 
		  &(non_empty_buckets[tmp_hash_index]);
	      non_empty_buckets[tmp_hash_index] = non_empty_bucket_head ;
	      non_empty_bucket_head = tmp_hash_index ;
	      prev_bucket_ptrs[tmp_hash_index] = &(non_empty_bucket_head);
	    }
	    cur = next;
	}
    }

    delete old_table ;

    C_NewHashIndexSize += (hash_size - old_hash_size) * sizeof(HashNode *);
}


/*
 iterator.ipos is used to keep track of where were are in the scan.
 The possible values are:


#define Scanning_Init 0 	 Initial state 
#define Scanning_Known 1 	 scanning known bucket 
#define Scanning_Vars 2 	 scanning from var_chain 
#define Scanning_New 3 	 scanning from chain
#define Scanning_Done 4 	 done
#define Scanning_Hash 5
*/

/** For i in [Scanning_Hash .. Scanning_Hash+hash_size-1] means
    that we are scanning all hash buckets, and are currently at
    bucket i-Scanning_Hash **/

void HashIndex::initialize_iterator( TupleIterator& iterator )
{

  iterator.stack_mark.get_mark();	

  // hash() retruns VarHashValue if iterator.arg_list.count == 0
  iterator.hash = hash(&iterator.arg_list, iterator.bindenv); 
  // changed to hashing on the appropriate set of arguments 

#ifdef DO_TRACE
  if (exEnv.dbg_indexing) {
    fprintf(exEnv.trace_file, "\nHashIndex::initialize_iterator: indexing ");
    iterator.relation->name->print(NULL, exEnv.trace_file);
    fprintf(exEnv.trace_file, " using ");
    iterator.arg_list.print(iterator.bindenv, exEnv.trace_file);
    
    if (argset != NULL) { 
      // argument form index
      fprintf(exEnv.trace_file, "[hash index on ");
      argset->printon(exEnv.trace_file);
      fprintf(exEnv.trace_file, " ]\n");
    }
    else { 
      // pattern form index
      fprintf(exEnv.trace_file, "[pattern matched ");
      pattern->print(NULL, exEnv.trace_file);
      fprintf(exEnv.trace_file, "] ");
      fprintf(exEnv.trace_file, "[indexed on ");
      var_names->print(NULL, exEnv.trace_file);
      fprintf(exEnv.trace_file, " ]\n");
    }
    fprintf(exEnv.trace_file, "	       hash value is %d\n", iterator.hash);
  }
#endif
  
  
  if (iterator.hash == UnmatchedHashValue) {
    iterator.ipos = Scanning_Done;
  }
  else if (iterator.hash != VarHashValue) {
    iterator.ipos = Scanning_Known;
    iterator.pindex = (int)(iterator.hash) % hash_size;
    iterator.ppos = &hash_table[iterator.pindex];
  } 
  else {
    // the entire relation has to be searched
    if (non_empty_bucket_head < 0) {
      iterator.ipos = Scanning_Vars ;
      iterator.ppos = &var_chain;
      iterator.pindex = -1;
    }
    else {
      iterator.ipos = Scanning_Hash + non_empty_bucket_head;
      iterator.pindex = non_empty_bucket_head;
      iterator.ppos = &hash_table[iterator.pindex]; 
    }
  }
}

/******************
*   HashIndex::get_next_tuple which returns a Tuple* and 
*   HashIndex::get_next which returns a BindEnv* are extremely similar
*   in their code.  In fact, get_next_tuple duplicates the entire
*   code of get_next.  If trailing is used (i.e. DO_TRAIL is defined)
*   get_next modifies the iterator.bindenv AND returns a pointer to 
*   the BindEnv.  One way of making it consistent is to make it explicit
*   that iterator.bindenv will contain the modified BindEnv, and 
*   return a Tuple* consistently across the board.
******************/

Tuple *HashIndex::get_next_tuple(TupleIterator& iterator)
{
  HashVal h_val;
  
  BindEnv *result_env = iterator.bindenv;
  StackMark  stackmark0;
  
  Tuple *result = NULL;
  iterator.tuple_env = NULL;
  
  if (iterator.ipos == Scanning_Init) // initialize iterator
    initialize_iterator( iterator );
  else iterator.stack_mark.pop_to();
  
  for (;;) { /* Continue until we find a match or hash-table exhausted */
	       
    HashNodePtr *next = (HashNodePtr*) iterator.ppos;
    if ((iterator.ipos == Scanning_Done) || (*next == NULL)) {
      switch (iterator.ipos) {
      case Scanning_Known:
	iterator.ipos = Scanning_Vars;
	iterator.ppos = &var_chain;
	continue;
	
      case Scanning_Vars:
      case Scanning_New: // this should not occur
      case Scanning_Done: 
	result = NULL;
	return result;
	
      default: // iterator.ipos >= Scanning_Hash
	
	h_val = iterator.ipos - Scanning_Hash;
	if (non_empty_buckets[(int)h_val] < 0) {
	  iterator.ipos = Scanning_Vars;
	  iterator.ppos = &var_chain;
	}
	else {
	  iterator.pindex = non_empty_buckets[(int)h_val] ;
	  iterator.ipos = Scanning_Hash + iterator.pindex ;
	  iterator.ppos = &hash_table[iterator.pindex] ;
	}
	continue;
      }
    }

    HashNodePtr tuple_index = *next;
    
    if ((tuple_index->tuple())->is_deleted()) {
      /* the tuple has been logically deleted
       * catch the case where iterator.ipos == ScanningKnown, and so
       * iterator.pindex becomes -4 (a wrong value)
       */
      if (iterator.pindex < 0) {
	iterator.ppos = &tuple_index->next;
	continue;
      }

      if (relation->scan_count != 1) {
	iterator.ppos = &tuple_index->next;
	continue;
       }

      *(next) = (*next)->next;
      
      /* The if check is needed because we might have a rule of the form
	 try :- p(X), delete(p(X)), print(X), p(Y), print(Y).
	 The delete() builtin merely marks the tuples as deleted.
	 The later p iterator must not reclaim the space, because this 
	 will corrupt the first iterator.
	 */

      // PRAVEEN:: I am doubtful about adding this here. Will this break the system ?
      //     Looks like it is okay, except for ordered search. For some reason, in
      //     Ordered Search, a tuple is deleted from within insert_in_context
      //     and then crashes later here !!
      if (CurrModuleInfo && !CurrModuleInfo->UseOrdSearch && 
	  !(--(tuple_index->tuple()->ref_count))) {
	/***
          for (int ti = 0; ti < tuple_index->tuple()->arity(); ti++)
	  // NUM_CONSTs can be safely deleted since they are never shared
	  if (tuple_index->tuple()->arg(ti)->kindof() == COR_NUM_CONST)
	    delete tuple_index->tuple()->arg(ti);
	****/
	//if (tuple_index->tuple()->bindenv) 
	//delete tuple_index->tuple()->bindenv;
	delete tuple_index->tuple()->_args;
	delete tuple_index->tuple();
	delete tuple_index;
       }

      if ((next == &hash_table[iterator.pindex]) && (*next == NULL)) {
	*(prev_bucket_ptrs[iterator.pindex]) = 
	  non_empty_buckets[iterator.pindex];
	if (non_empty_buckets[iterator.pindex] >= 0) {
	  prev_bucket_ptrs[non_empty_buckets[iterator.pindex]] = 
	    prev_bucket_ptrs[iterator.pindex];
	 }
       }
      hash_count--;
      
#ifdef DO_PROFILE
      if (T_Stack.count > 0)
	(T_Stack.top()->moduledata->module_info.
	 GlobalProfileInfo.deleted_tuples)++; 
      else
	(GlobalProfileInfo.deleted_tuples)++; 
#endif
      continue;
     }
    
    iterator.ppos = &tuple_index->next;
    
    if (iterator.ipos == Scanning_Known && 
	iterator.hash != tuple_index->hash_val) continue;
    /** since many hash values could be mapped to the same
      hash bucket, this preliminary check is more efficient. **/
    

    if (!CurrModuleInfo->NonGroundFacts) {
      int t_cnt = relation->arity();
      Term left_term, right_term;
      for (int l = 0; l < t_cnt; l++) {
	left_term.expr = iterator.arg_list[l];
	right_term.expr = tuple_index->tuple()->args()[l];
	left_term.bindenv = iterator.bindenv;
	right_term.bindenv = NULL;
	if (unify_args(left_term, right_term) == COR_U_FAIL)
	  break;
      }
      if (l == t_cnt) return tuple_index->tuple();
      stackmark0.pop_to();
    }
    else {
      
      if (unify_literal(iterator, tuple_index->tuple(), exEnv.dbg_indexing))
	return tuple_index->tuple();
      
      stackmark0.pop_to();
    }
  } 
}

BindEnv * HashIndex::get_next( TupleIterator& iterator )
{
    Tuple* tuple = get_next_tuple(iterator);
    if (iterator.no_match())
	return NULL;
    else return iterator.bindenv;
}


/**************
*   WARNING::
*   problem in HashIndexSet::find_exact
*    	re-orderings of var_names is not taken into account while
*	checking whether two indexes are the same.
*	thus, [V0|V1],V2 ; V1,V2 is treated differently from
*	      [V0|V1],V2 ; V2,V1 though the two should really 
*	be the same.
*	merely sorting the var_names will not do the trick since
*	      [V0|V1],V2 ; V1,V2 subsumes and is subsumed by 
*	      [V2|V1],V0 ; V1,V0 but sorting var_names would mess that up.
**************/

int HashIndexSet::find_exact( HashIndex *index )
{
  for (int i = 0; i < num_in_array; i++) {
    if (index->argset != NULL) { /* argument form indexing */
      if ( ((HashIndexPtr) Indices[i])->argset != NULL && 
	  *index->argset == *((HashIndexPtr) Indices[i])->argset) {
	return COR_I_SUCCESS;
      }
    }
    else { /* pattern form indexing */
      if ( ((HashIndexPtr) Indices[i])->argset == NULL &&
	  (index->num_vars_pat == 
	   ((HashIndexPtr) Indices[i])->num_vars_pat) && 
	  (index->var_names->count() == 
	   ((HashIndexPtr) Indices[i])->var_names->count())) {
	
	StackMark stackmark;
	
	ArrayBindEnv pat_env1(index->num_vars_pat);
	ArrayBindEnv pat_env2(index->num_vars_pat);
	ArrayBindEnv pat_env3(index->num_vars_pat);
	ArrayBindEnv pat_env4(index->num_vars_pat);
	    
	/** check if pattern and the var_names of each subsume 
	  that of the other **/
	
	if ((subsumes_arg_lists(*concat_arglists(index->pattern, 
						 index->var_names), 
		      *concat_arglists(((HashIndexPtr) Indices[i])->pattern,
				       ((HashIndexPtr) Indices[i])->var_names),
				&pat_env1, &pat_env2) == COR_U_SUCCEED) &&
	    (subsumes_arg_lists(*concat_arglists(
				       ((HashIndexPtr) Indices[i])->pattern,
				       ((HashIndexPtr) Indices[i])->var_names),
		      *concat_arglists(index->pattern, index->var_names),
				&pat_env3, &pat_env4) == COR_U_SUCCEED)) {
	  stackmark.pop_to();
	  return COR_I_SUCCESS;
	}
	else stackmark.pop_to();
	
      }
    }
  }
  return COR_I_FAIL;
}

/**
  *This is an optimization to let a static environment to be used in 
  *reentrant code.  We assume no parallelism however.
  ***/
static ArrayBindEnv static_env(MAX_VARS,VAR_INCR);
static int static_env_inuse=DONT_USE_STATIC_ENV;


GenericIndex *HashIndexSet::find_index( ArgList &arg_list, BindEnv *bindenv,
			int *tryindexp /* = NULL*/)
{
  /** find an index that best matches the pattern contained arg_list, bind_env **/
  // WARNING:: The following code should at least check if the index is
  //	appropriate.  What it does is rather dangerous.
  // this is inappropriate for programs with non-ground terms. what is an appropriate
  // index for one invocation may not be appropriate for subsequent invocations : PRAVEEN

    if ( (tryindexp) && ((*tryindexp) >= 0) &&  ((*tryindexp) < num_in_array) ) {
	// PRAVEEN: i checked that this gets entered, and yes it does !
	return ((GenericIndex *) Indices[*tryindexp]);
	}	

    if (!num_in_array) {
	fprintf(exEnv.error_file, "There are no indices: Quitting\n");
	return NULL;
    }

    // find bitvector of pattern 
    int arity = arg_list.count();

    if (arity == 0) {  // propositional case.
	if (rel->arity() != 0)
	    fprintf(exEnv.error_file,"Error: HashIndexSet::find_index: arity mismatch\n");
	if(tryindexp) *tryindexp = 0;
	return  (GenericIndex *) Indices[0];
	/**
	 *  arbitrarily choose the 1st index and do a sequential scan.
	 *  need to ensure that the 1st index is an argument-type index.
	 **/
    }

    BitVector adornment(arity);
    int num_bound = 0; // number of bound arg positions in pattern
    for (int i = 0; i < arity; i++) {
	int bound = 1;
	// check which arguments are ground and set those bits to 1; rest 0
	if (arg_list[i]->hash(bindenv) == VarHashValue)
	    bound = 0;
	if (bound == 1) num_bound++;
	adornment.set(i, bound);
    }


    /****************
    *  need to find "best" match 
    *  for argument form indices: only those with FEWER ground arguments
    *  		considered: i.e. index pattern subsumes pattern
    *  for pattern form indices: index pattern should subsume pattern.
    *		if all hash variables bound: give appropriate high weight
    *		if only some bound: give a small weight say 0.5 because of
    *		    greater selectivity than other trivial indices.
    ****************/

    GenericIndex *return_index = NULL;
    double max_weight = -1.0;

    for (i = 0; i < num_in_array; i++) {
	double cur_weight = 0.0;
	if (((HashIndexPtr) Indices[i])->argset != NULL) { // argument index
	    if (arity != ((HashIndexPtr) Indices[i])->argset->len()) continue;

	    if ( (num_bound == 0) && 
		!(((HashIndexPtr) Indices[i])->argset->any())) {
	      // all free index
		if (tryindexp) *tryindexp = i;
		return ((GenericIndex *)Indices[i]);	      
	    }

	    if (    (num_bound == arity) && 
		    (adornment == *(((HashIndexPtr) Indices[i])->argset))   
	          ) { // All arguments are bound - no better match can be found.
		if (tryindexp) *tryindexp = i;
		return ((GenericIndex *)Indices[i]) ;
	    }
	    for (int j = 0; j < arity; j++) {
	        if (adornment.test(j) && 
		        ((HashIndexPtr) Indices[i])->argset->test(j)) {
		    cur_weight += 1.0;	// both are bound
		    continue;
	        }
	        if (((HashIndexPtr) Indices[i])->argset->test(j)) { 
		        // and !adornment.test(j)
		        // the index has more bound arguments; it is useless
		    cur_weight = 0.0; 
		    break;
	        }
	    }
	}

#define CLEANUP  { stack_mark.pop_to();					\
	if( --static_env_inuse != 0 ||  				\
		((HashIndexPtr) Indices[i])->num_vars_pat > MAX_VARS)	\
	    { delete patenv1; } }


	else { /* it is a pattern index */
	  StackMark stack_mark;
	  ArrayBindEnv *patenv1;

	  if (static_env_inuse++ == 0 && 
	      ((HashIndexPtr) Indices[i])->num_vars_pat <= MAX_VARS)
	    patenv1 = &static_env;
	  else {
	    patenv1 = new ArrayBindEnv 
	      (((HashIndexPtr) Indices[i])->num_vars_pat);
	  }

	  if (subsumes_arg_lists(*((HashIndexPtr) Indices[i])->pattern,
				 arg_list, patenv1, bindenv) ==COR_U_SUCCEED){
	    for (int j = 0; 
		 j < ((HashIndexPtr) Indices[i])->var_names->count(); 
		 j++) {
	      if ((*((HashIndexPtr) Indices[i])->var_names)[j]->hash(
					       patenv1) == VarHashValue) {
		cur_weight = 0.5;
		CLEANUP
		  break;
	      }
	      cur_weight += 1.0;
	      if (cur_weight > arity) {
		cur_weight = arity;
		CLEANUP
		  break;
	      }
	      /**********
	       *  the above is a heuristic for deciding what weight
	       *  to assign to pattern form matches.
	       *  this will need to be updated as and when need be.
	       **********/
	    }
	  }
	  else { /* the pattern form index does not subsume pattern */
	    cur_weight = 0.0;
	    CLEANUP
	  }
	}
	if (cur_weight > max_weight) { /* need to update information */
	  max_weight = cur_weight;
	  if (tryindexp) *tryindexp = i;
	  return_index = (GenericIndex *) Indices[i];
	}
      }
    
    if (!return_index) { /* have not been able to find any match */
      if (tryindexp) *tryindexp = 0;
      return_index = (GenericIndex *) Indices[0];
      /************
       *  arbitrarily choose the 1st index and do a sequential scan.
       *  need to ensure that the 1st index is an argument-type index.
       ************/
    }

  return return_index;
}

int HashIndexSet::insert_index( GenericIndex *index )
{

    HashIndex *h_index = (HashIndex *) index;
    if (find_exact( h_index )) {
#if DO_TRACE
	if (exEnv.GlobalRelationOptions & REL_DISPLAY_INDEXOPS) {
	    fprintf(exEnv.trace_file, "Duplicate index: not inserted\n");
	}
#endif
	return(COR_I_FAIL);
    }
    else { /* new index to be inserted in list */
      if (num_in_array == max_in_array) {
#ifdef DEBUG
	  fprintf(exEnv.error_file, "Too many indices: not inserted\n");
#endif
	  return COR_I_FAIL;
      }
      Indices[num_in_array++] = (GenericIndex *) h_index;
      // this is fine since array insertion starts from zero.
      return COR_I_SUCCESS;
    }
}

void HashIndexSet::fill_index( GenericIndex *index )
{
    HashIndex *h_index = (HashIndex *) index;
    HashIndex *f_index = (HashIndex *) Indices[0];
	/**********
	*  this is the index that will be used to fill up h_index. 
	*  it should be an argument form index or a pattern form
	*  index that indexes ALL the tuples in the relation, else
	*  the filling up will not be correct.
	**********/
    ASSERT(h_index != f_index);
	/* don't want to fill up an index using itself */

    /* now sequentially scan through f_index */
    Tuple *tuple = NULL;
    HashNode *cur_pos = NULL;
    for (int i = 0; i < f_index->hash_size; i++) {
	for (cur_pos = f_index->hash_table[i]; cur_pos != NULL; 
		cur_pos=cur_pos->next) {
	    h_index->insert_tuple(cur_pos->tuple());
	}
    }
    for (cur_pos = f_index->var_chain; cur_pos != NULL; cur_pos = cur_pos->next)
	h_index->insert_tuple(cur_pos->tuple());
}

int HashIndexSet::add_all_tuples(StorageRelation *_reln, int marks_into_account,
				 int check_dupl, int hide_em)
{
    /* WARNING: duplicate checking is done only when marks 
       are taken into account. */

    HashIndex *f_index = (HashIndex *) Indices[0];
    int dupl_check = check_dupl && _reln->check_subsum;

    if (!f_index->is_argument_form()) {
	fprintf(exEnv.error_file, "ERROR: add_all_tuples does not have proper index.\n");
	return COR_I_FAIL;
    }

    Tuple *tuple = NULL;
    HashNode *cur_pos = NULL;

    register int j;

    if (! marks_into_account) {
      // Move nodes in the hash buckets

      for (j=f_index->non_empty_bucket_head; j >= 0 ;
	   j=f_index->non_empty_buckets[j])
	for (cur_pos = f_index->hash_table[j]; cur_pos; cur_pos = cur_pos->next)
	  if (!hide_em)
	    _reln->insert_raw(cur_pos->tuple());
	  else _reln->insert_in_hidden_relation(cur_pos->tuple());

      for (cur_pos = f_index->var_chain; cur_pos != NULL; 
	   cur_pos = cur_pos->next)
	if (!hide_em)
	  _reln->insert_raw(cur_pos->tuple());
	else _reln->insert_in_hidden_relation(cur_pos->tuple());
    }
    else {
      // same as above code except that insert_tuple has to be used

      for (j=f_index->non_empty_bucket_head; j >= 0 ;
	   j=f_index->non_empty_buckets[j])
	for (cur_pos = f_index->hash_table[j];cur_pos; cur_pos=cur_pos->next)
	  if ((!dupl_check) || (!_reln->is_subsumed(cur_pos->tuple())))
	    if (!hide_em)
	      _reln->insert_tuple(cur_pos->tuple());
	    else _reln->insert_in_hidden_relation(cur_pos->tuple());
      
      for (cur_pos = f_index->var_chain; cur_pos != NULL; 
	   cur_pos = cur_pos->next)
	if ((!dupl_check) || (! _reln->is_subsumed(cur_pos->tuple())))
	  if (!hide_em)
	    _reln->insert_tuple(cur_pos->tuple());
	  else _reln->insert_in_hidden_relation(cur_pos->tuple());
    }

    return COR_I_SUCCESS;
}

int HashIndexSet::add_all_goals(StorageRelation *_reln, Context *_context,
						StorageRelation *owner_rel)
{
    // this is used in Ordered Search
    // tuples in (this) need to be converted to GoalNodes and
    // added to reln as well as context

    HashIndex *f_index = (HashIndex *) Indices[0];

    if (!f_index->is_argument_form()) {
	fprintf(exEnv.error_file, "ERROR: add_all_goals does not have proper index.\n");
	return COR_I_FAIL;
    }

    Tuple *tuple = NULL;
    GoalNode *g_node = NULL;
    HashNode *cur_pos = NULL;
    for (int i = 0; i < f_index->hash_size; i++) {
	for (cur_pos = f_index->hash_table[i]; cur_pos != NULL;
		cur_pos=cur_pos->next) {
	    tuple = cur_pos->tuple();
	    if (!(tuple->is_deleted())) {
	        g_node = new GoalNode(tuple->_args, tuple->bindenv, owner_rel);
		g_node->env_size = tuple->env_size;
		_reln->insert_tuple(g_node);
		_context->add_new_goal(g_node, NULL);
	    }
	}
    }

    for (cur_pos = f_index->var_chain; cur_pos != NULL; 
		cur_pos = cur_pos->next) {
	tuple = cur_pos->tuple();
	if (!(tuple->is_deleted())) {
	    g_node = new GoalNode(tuple->_args, tuple->bindenv, owner_rel);
	    g_node->env_size = tuple->env_size;
	    _reln->insert_tuple(g_node);
	    _context->add_new_goal(g_node, NULL);
	}
    }
    return COR_I_SUCCESS;
}

int HashIndexSet::move_all_entries ( StorageRelation *_reln)
{
    /* WARNING: This code assumes that the two relations have exactly the
	same indices in exactly the same order, and 
	both the relations are HashSimpleRelations*/

    HashSimpleRelation *otherrel = (HashSimpleRelation *) _reln;

    HashIndexSet *otherindset =  &(otherrel->index_set);
    
    if(num_in_array != otherindset->num_in_array) {
	fprintf(exEnv.error_file, "CORAL:: internal error in move_all_entries\n");
	return -1;
    }

    for (int i=0; i < num_in_array; i++) {
    	Indices[i]->move_all_entries(otherindset->Indices[i]);
    }
    return 1;
}

int HashIndex::is_subsumed(Tuple *newtuple)
{

    HashVal hash_val;
    HashNodePtr ppos;
    int ipos;	/* =-1 indicates looking at varchain, else indicates number of
			buckets left to look at after current one */

    StackMark  stackmark0;

    if (newtuple->args().count() == 0) {
	/*************************************
	  Normally this doesn't really mean that arity==0.
	  Rather, it means that the TupleIterator was
	  created with an ArgList parameter, which is a
	  convention meaning essentially (VAR1, VAR2, ..., VARn).
	  However, in pathological cases, it might really
	  be a nullary Relation. Do the same thing in either case.
	*************************************/
	hash_val = VarHashValue;
    } 
    else 
	hash_val = (HashVal) hash(&(newtuple->args()), newtuple->bindenv); 

    if (hash_val == UnmatchedHashValue) {
	return 0;
    }
    else if (hash_val != VarHashValue) {
	ipos = -2;	// Only one bucket to look at.
	ppos = hash_table[(int) hash_val % hash_size];
    } else {
	// the entire relation has to be searched
        ipos = non_empty_bucket_head;
	if (ipos >= 0) ppos = hash_table[ipos];
	else ppos = var_chain;
    }

    for (;ipos>=-2;) { // ipos == -1 indicates we are in varchain

	for(;ppos != NULL; ppos = ppos->next) {

	    if ((ppos->tuple())->is_deleted()) continue;
	        // the tuple has been logically deleted
	    if ((hash_val != VarHashValue) && 
		(hash_val != ppos->hash_val) && 
		(ppos->hash_val != VarHashValue)) continue;
	    
	        /** since many hash values could be mapped to the same
		    hash bucket, this preliminary check is more efficient. **/

		if (subsumes_tuple(ppos->tuple(), newtuple, exEnv.dbg_indexing))
		    return 1;

	}
        if (ipos == -1) return 0 ;

	if (ipos == -2) {
	  ipos = -1 ;
	  ppos = var_chain ;
	}
	else {
	  ipos = non_empty_buckets[ipos] ;
	  if (ipos >= 0)
	    ppos = hash_table[ipos];
	  else if (ipos==-1) ppos = var_chain;
	}
    }
    return 0;

}


