/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	compile.C

	Contains routines to 'compile' coral modules into an
	intermediate representation that can be executed when
	there is a query. It invokes code in compile_rule.C to
	compile each seminaive rule. The code to actually execute
	the query is in derived-rel.C, execute.C and apply.C.

 ***********************************************************************/

#include "externs.h"
#include "interp.h"
#include "pipelined.h"
#include "derived-rel.h"
#include "profile.h"
#include "apply.h"
#include "annotations.h"
#include "globals.h"
#include <stdlib.h>
#include <string.h>
#include "ordsearch.h"

extern void sprint_pred_name(char *buf, char *name, char *adorn);
extern void sprint_pred_name(char *buf, char *name, BitVector *adorn);
extern void sprint_magic_name(char *buf, char *name, BitVector *adorn);
extern void sprint_magic_name(char *buf, char *predname, char *adorn);
extern int print_new_line_after_rule;
extern int print_adorn_for_pred;

extern int getSuccessBacktrackPoint(SemiNaive *sn, Clause *clause, int);
extern int getfirstFailureBacktrackPoint(SemiNaive *sn, int rhs_index, int);
extern int getnextFailureBacktrackPoint(SemiNaive *sn, int rhs_index, int);
extern int has_grouping_args(ArgList &args);
// Defined in annotations.C
extern void add_make_index_annotation(PredAnnotations* &, Symbol *pred,
				      BitVector *bv,
				      ArgList *al1, ArgList *al2);


// Defined in compile_rule.C
extern RuleInfo *create_rule_info(Table *F, struct rule *);
extern void compile_nonrec_rule(SemiNaive &, Clause *, RuleInfo *r, int );

// Defined in execute.C
extern StorageRelation *do_query(QFModuleData *md, StorageRelation *magic,
		   int old_magic_relation, TupleIterator *iterator);

extern void AddProfileInfo(ProfileInfo *A, ProfileInfo *B) ;

// Defined in util.C
extern int is_magic(Name name);
extern int is_supp(Name name);

extern DatabaseStruct BuiltinDB ;

int get_offset(Table *F, Name name, int arity, int print_error)
{
 for (int i = 0; i< F->numRels; i++)
   if ((F->rels[i].relArity == arity) &&
       (F->rels[i].relName == name))
     break ;
 
 // The offset returned should be within the array bounds.
 if (i >= F->numRels) {
   if (print_error) {
     fprintf(exEnv.error_file,"Error: Couldn't find relation %s with arity %d\n",
	     name->string(), arity);
    }
   return(-1); // NOTE: divesh added this
  }
 ASSERT((i < F->numRels)) ;
 return(i) ;
}


// NOTE : Predicate field in Literal is often not set !! So use pred field.
int get_offset(Table *F, Literal *pred, int print_error)
{
	return get_offset(F, pred->pred, pred->arity(), print_error) ;
}

RuleEnv::RuleEnv(SemiNaive *parent) : bindenv(parent->rInfo->num_vars,4) 
{ 
 int max_literals = parent->rInfo->num_literals;

 // this is a hack to prevent memory overwrites for rules of the form p(X).
 // rInfo->num_literals == 0 for these rules, there are places in the code
 // where it is assumed that at least one rhs literal exists. This should actually
 // be fixed by either changing that code, or by introducing a true literal on the rhs.
 // PRAVEEN

 if (!max_literals) max_literals = 1;

 rmarks = new StackMark [max_literals];
 tuple_its = new TupleIterator * [max_literals];
 arglist_its = new ArgList * [max_literals];
 tuples = new Tuple * [max_literals];
 bindenvs = new BindEnv *[max_literals];
 parent_sn = parent;

 // allocate tuple iterators if possible
 if (parent->rule_id >= 0) {
   for (int i = 0; i < max_literals; i++)
     tuple_its[i] = new TupleIterator (NULL, parent->rInfo->arg_list[i], NULL, NULL, NULL, NULL);
  }

}

RuleEnv::~RuleEnv()
{
 delete rmarks;
 // delete each of the tuple iterators !
 delete tuple_its;
 delete arglist_its;
 delete tuples;
 delete bindenvs;
}

SccEnv::SccEnv(SccInfo *parent)
{
 int i;

 parent_scc = parent;
 numSNRules = parent->numSNRules;
 if (!parent->module->module_info.SaveModule) 
   rm_Arr = new RMarkArray(2*parent->numPreds) ;
  else
    rm_Arr = new RMarkArray(2*(parent->numPreds+parent->num_lowerPreds)) ;

 snrule_envs = new RuleEnv *[parent->numSNRules];
 for (i = 0; i < parent->numSNRules; i++)
   snrule_envs[i] = new RuleEnv(&(parent->sn_rules[i]));
}

SccEnv::~SccEnv()
{
 if (rm_Arr) delete rm_Arr;
 for (int i = 0; i < parent_scc->numSNRules; i++)
   delete snrule_envs[i];
 delete snrule_envs;
}

ModuleEnv::ModuleEnv(QFMaterializedModuleData *parent)
{
 parent_md = parent;
 numSccs = parent->numSccs;
 scc_envs = new SccEnv *[numSccs];

 for (int i=0; i< numSccs; i++)
   scc_envs[i] = new SccEnv(parent->sccArray[i]);
}

ModuleEnv::~ModuleEnv()
{
 for (int i=0; i< numSccs; i++)
   delete scc_envs[i];
 delete scc_envs;
}

void Table::AddValue(int i, Name relName, int arity)
{
 rels[i].AssignVal(relName, arity) ;
}

Table::Table(QFModuleData *md)
{
 Table *F = md->F;
 numRels = F->numRels ;
 rels = new RelationStruct[numRels];
 for (int i = 0; i< numRels; i++)
   rels[i].AssignVal(F->rels[i].relName, F->rels[i].relArity) ;
 next = NULL;
 context = NULL;
 moduledata = md;
 module_env = new ModuleEnv((QFMaterializedModuleData *)md);
}

Table::~Table()
{
 if (rels) delete rels;
 if (module_env) delete module_env;
}

void RelationStruct::AssignVal(Name name, int arity)
{
	relName = name ;
	relArity = arity ;
}

RuleInfoArray::RuleInfoArray()
{
	numRules = 0 ;
	rInfoPtrs = NULL;
}

void RuleInfoArray::AddRule(RuleInfo *rInfo)
{
	RuleInfo **temp = rInfoPtrs ;
	rInfoPtrs = new RuleInfo*[++numRules] ;

	for (int i=0; i< numRules-1; i++)
		rInfoPtrs[i] = temp[i] ;

	rInfoPtrs[i] = rInfo ;
	if (numRules > 1) delete [] temp ;

}

void RuleInfoArray::printon(FILE *outf)
{
	for (int i=0; i< numRules; i++)
	    fprintf(outf, "%s\n", rInfoPtrs[i]->pInfo.rule) ;

}

// This function finds the offset of an RMark in the RMarkArray for an scc
// It depends on the order in which the RMarks are actually allocated, which
// happens in SccInfo::Solve(). The function returns the offset of the first
// RMark for that predicate (i.e.) the old RMark. The new RMark is in the
// adjacent position.
static int Find_RMark_Offset(SccInfo *scc_info, Predicate *rpred)
{
    Predicate **pred_ptrs = scc_info->pred_ptrs;
    int total_numpreds = scc_info->numPreds + scc_info->num_lowerPreds;
    for (int i=0; i < total_numpreds; i++) {
	if (pred_ptrs[i] == rpred) return (2*i);
    }

    // Default --- this should never be reached !!
    fprintf(exEnv.error_file,
	    "CORAL:: Error - Find_RMark_Offset - can't find %s(arity %d)\n",
	    rpred->name->string(), rpred->arity);
    return(-1) ;
}

static int has_fail_in_body(struct rule *rule)
{ 
    int i;

    for(i=0; i < rule->num_preds; i++)
	if(rule->preds[i]->pred == FailSymbol)
	   return i;
    return -1;

}

int SccInfo::allocateRuleStructures(Predicate *ext_magicpred)
{

  // First, allocate enough memory to hold all seminaive rule structures
  numSNRules = 0 ;
  Predicate *pred; Clause *clause;
  int i, j;

  for (pred = scc->preds; pred != NULL; pred = pred->next)
    for (clause = pred->clauses; clause != *pred->clause_tail;
	 clause = clause->pnext) {
      if (!clause->seen) continue ;
      if (!clause->recursive) {
	if (! module->module_info.SaveModule) {
	  numSNRules++ ; continue ;
	}
	else {
	  // with SaveModules have to count the number of literals
	  // whose predicates are defined in lower SCCs.
	  // NOTE: something special needs to be done for non-magic modules
	  int thisclausesn = 0;
	  int ipred = 0;
	  for (; ipred < clause->num_preds; ipred++) {
	    // check to see if it is in a lower SCC
	    // versus being an external predicate
	    if ((clause->preds[ipred]->predicate->clauses) ||
		(clause->preds[ipred]->predicate->is_magic())) { 
	      numSNRules++; thisclausesn++;
	    }
	  }
	  if (thisclausesn == 0) { 
	    // all predicates in body are external to module
	    numSNRules++ ; 
	  }
	  continue;
	}
      }
      for (int ipred = 0; ipred <  clause->num_preds; ipred++) {

	/*
	 * Treat negated literals as base !!
	 */
	if (clause->preds[ipred]->negated) continue;

	/** The following is to special case the creation of the seminaive structures
	 * for the case of Ordered Search.  Three special things need to be done
	 * (1) `not p' is always treated as base, even if p is derived.
	 * (2) done_m_p is treated as derived even though there are no rules for it.
	 * (3) any literal following a done_m_p is treated as base, even if p
	 *     is derived. (this also covers case 1)
	 **/

	if (module->module_info.UseOrdSearch) {
	  if ((ipred > 0) && 
	      (clause->preds[ipred-1]->predicate->is_mp_done()))
	    continue;

	  /* Treat mp_done literals as derived */
	  if (clause->preds[ipred]->predicate->is_mp_done())
	    clause->preds[ipred]->predicate->scc = scc ;
	}
	if (clause->preds[ipred]->predicate->scc != scc) {
	  if (module->module_info.SaveModule) {
	    // check to see if it is in a lower SCC
	    // versus being an external predicate
	    if ((clause->preds[ipred]->predicate->clauses) ||
	        	(clause->preds[ipred]->predicate->is_magic())) 
	      numSNRules++;
	  }
	  continue;
	}
	numSNRules++ ;
      }
    }

  sn_rules = new SemiNaive[numSNRules]; 
  for (i = 0; i < numSNRules; i++)
    sn_rules[i].rule_id = i;

  sn_offset_arr = new int[numSNRules];

  // find the number of predicates in this Scc
  for (numPreds = 0, pred = scc->preds; pred != NULL; 
			++numPreds, pred = pred->next) ;

  // now fill in the information about lower Sccs, which is useful in
  // the case of SaveModules
  Predicate  *ext_magicquerypred = NULL;
  if (module->numSccs == 1) { // this is the first Scc
    if (module->module_info.SaveModule) {
      ext_magicquerypred = ext_magicpred;
    }
    if (ext_magicquerypred) num_lowerPreds = 1;
    else num_lowerPreds = 0;
  }
  else { // add the numPreds and num_lowerPreds of the previous scc
    ASSERT((module->numSccs > 1)) ;
    num_lowerPreds = module->sccArray[module->numSccs-2]->num_lowerPreds +
			module->sccArray[module->numSccs-2]->numPreds ;
  }

  // assign the predicate pointers and the offset array
  if (!module->module_info.SaveModule) {
    pred_ptrs = new Predicate *[numPreds] ;
    offset_arr = new int[numPreds];
  }
  else {
    pred_ptrs = new Predicate * [numPreds + num_lowerPreds];
    offset_arr = new int [numPreds + num_lowerPreds];
  }
  for (i = 0, pred = scc->preds; pred != NULL; i++,pred = pred->next){
    pred_ptrs[i] = pred ;
    // allow at least one iteration
  }
  if (module->module_info.SaveModule) {
    // get the predicate pointers from the lower scc
    if (module->numSccs > 1) {
      Predicate **lower_pred_ptrs = 
			module->sccArray[module->numSccs-2]->pred_ptrs;
      ASSERT((i == numPreds));
      for (j=0; j < num_lowerPreds; j++, i++) {
	pred_ptrs[i] = lower_pred_ptrs[j];
      }
    }
    else { // module->numSccs == 1
      ASSERT((num_lowerPreds <= 1));
      if (num_lowerPreds == 1) {
	pred_ptrs[numPreds] = ext_magicquerypred;
      }
    }
  }

  return numSNRules;
}

/*
 * Routine that creates the seminaive structures in each Scc structure 
 */
void SccInfo::createSNStructs(Predicate *ext_magicpred)
{
  // allocate sn_rules, pred_ptrs, offset_arr
  allocateRuleStructures(ext_magicpred);

  // Now fill in values in seminaive rule structures
  Predicate *pred; Clause *clause;

  int cur_snr = -1, i ,j ;

  /******************
   * There is a correspondence between the predicate positions in 
   * this->pred_ptrs, and the offsets in the frame given by 
   * this->offset_arr of those predicates in rm_Arr.
   ******************/
  
  for (i=0; i<numPreds; i++)
    offset_arr[i] = get_offset(module->F, pred_ptrs[i]) ;
  if (module->module_info.SaveModule) {
    for (i=numPreds; i < numPreds+num_lowerPreds; i++) 
      offset_arr[i] = get_offset(module->F, pred_ptrs[i]) ;
  }


  // Consider non-recursive clauses in this scc first.

  for (j=0, pred = scc->preds; pred != NULL; ++j, pred = pred->next) {
    for (clause = pred->clauses;
	 clause != *pred->clause_tail;
	 clause = clause->pnext) {
      if (!clause->seen) continue; 	
      if (clause->recursive) continue;

      // we know it is a non-recursive clause by now
      int thisclausesn = 0; 
      if (module->module_info.SaveModule) { 
	// have to generate SN rules corresponding to lower predicates
	int ipred, jpred;
	for (ipred=0; ipred < clause->num_preds; ipred++) {
	  // all body literals are in lower sccs or external predicates
	  ASSERT((clause->preds[ipred]->predicate->scc != scc));
	  if ((clause->preds[ipred]->predicate->clauses) ||
		(clause->preds[ipred]->predicate->is_magic())) { 
	    // lower scc or containing MagicQueryPredKind without any rules

	    thisclausesn++; // counting number of sn rules generated 
	    cur_snr++ ; 
	    ASSERT((cur_snr < numSNRules));
	    sn_rules[cur_snr].first = ipred;
	    sn_rules[cur_snr].recursive = 0 ;
	    sn_rules[cur_snr].scc = this;
	    sn_offset_arr[cur_snr] = j ;

	    sn_rules[cur_snr].Add_Rule_Info(
		       module->ruleArray->rInfoPtrs[clause->rule_number],
		        clause, module->module_info.NonGroundFacts);
	    for (jpred = 0; jpred < clause->num_preds; jpred++) {
	      Literal *bodypred = clause->preds[jpred];

	      if (ipred == jpred) {
	        // NOTE: check whether external predicates have 
		// such rmarks defined on it or not!
		// they may be needed 
		sn_rules[cur_snr].markOffsets[2*jpred] =
		  	Find_RMark_Offset(this, bodypred->predicate);
	      }
	      else 
		sn_rules[cur_snr].markOffsets[2*jpred] = -1;
	      
	      if (bodypred->predicate->clauses || 
			bodypred->predicate->is_magic()) {
		int tmp = Find_RMark_Offset(this, bodypred->predicate);
		if (ipred  > jpred)
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp;
		else
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
	      }
	      else 
		sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
	    }
	  }
	}
      }

      if (thisclausesn ==0) { 
	// Either no SaveModule or a rule with only external predicates
	// in a SaveModule
	// Same actions need to be performed
	cur_snr++ ;
	ASSERT((cur_snr < numSNRules));
	sn_offset_arr[cur_snr] = j ;
	sn_rules[cur_snr].scc = this;

	// the following function encapsulates the initialization of
	// a non-recursive seminaive rule. This is done so that 
	// imperative rules from the coral prompt can also be initialized
        // using the same code.
	compile_nonrec_rule(sn_rules[cur_snr], clause,
			    module->ruleArray->rInfoPtrs[clause->rule_number],
			    module->module_info.NonGroundFacts);

      }
    }
  }
  
  // Now handle recursive rules

  if (scc->recursive) { // If there are recursive rules:
    
    if (module->module_info.SaveModule) {
      // have to generate non-recursive SemiNaive rules from recursive rules
      for (j=0, pred = scc->preds; pred != NULL; ++j, pred = pred->next) {
	for (clause = pred->clauses;
	     clause != *pred->clause_tail;
	     clause = clause->pnext) {
	  if (!clause->seen) continue;
	  if (!clause->recursive) continue;

          int ipred, jpred, i;
	  for (ipred=0; ipred < clause->num_preds; ipred++) {
	    if (module->module_info.UseOrdSearch) {
	      if ((ipred > 0) &&
			(clause->preds[ipred-1]->predicate->is_mp_done()))
		continue;
	    }

	    // disregard literals defined in the same scc
	    if (clause->preds[ipred]->predicate->scc == scc) continue;
	    
	    if ((clause->preds[ipred]->predicate->clauses) ||
	    		(clause->preds[ipred]->predicate->is_magic())) {
	      // lower scc or containing MagicQueryPredKind without any rules

	      cur_snr++ ;
	      ASSERT((cur_snr < numSNRules));
	      i = has_fail_in_body(clause);
	      if (i >= 0) {
		sn_rules[cur_snr].first = i;
	      }
	      else {
		sn_rules[cur_snr].first = ipred;
	      }

	      sn_rules[cur_snr].recursive = 0 ;
	      sn_rules[cur_snr].scc = this;
	      sn_offset_arr[cur_snr] = j ;
	      sn_rules[cur_snr].Add_Rule_Info(
			 module->ruleArray->rInfoPtrs[clause->rule_number],
			 clause, module->module_info.NonGroundFacts);

	      for (jpred = 0; jpred < clause->num_preds; jpred++) {
		Literal *bodypred = clause->preds[jpred];

		if (ipred == jpred) {
		  sn_rules[cur_snr].markOffsets[2*jpred] =
			  Find_RMark_Offset(this, bodypred->predicate);
		}
		else
		  sn_rules[cur_snr].markOffsets[2*jpred] = -1;

		if (bodypred->predicate->scc == scc) {
		  // only until the global old mark; not the whole relation
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = 
		  		Find_RMark_Offset(this, bodypred->predicate);
		  // sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
		}
		else if (bodypred->predicate->clauses ||
			  bodypred->predicate->is_magic()) {

		  int tmp = Find_RMark_Offset(this, bodypred->predicate);
		  if (ipred  > jpred)
		    sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp;
		  else
		    sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
		}
		else { // external non querymagic predicate 
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
		}
	      } 
	    }
	  }
	}
      }
    }

    // now generate recursive SemiNaive rules from recursive rules
    for (j=0, pred = scc->preds; pred != NULL; j++, pred = pred->next)
      for (clause = pred->clauses;
	   clause != *pred->clause_tail;
	   clause = clause->pnext) {
	if (!clause->seen) continue;
	if (!clause->recursive) continue;

	int ipred, jpred, i;
	for (ipred = 0; ipred < clause->num_preds; ipred++) {

	  //
	  //  Treat negated literals as base !!
	  //
	  if (clause->preds[ipred]->negated) continue;

	  // with Ordered Search, have to special case the 
	  // negated literals as well as the grouped literal.
	  // the check below assumes that both follow a 
	  // done_m_p literal.

	  if (module->module_info.UseOrdSearch) {
	    if ((ipred > 0) && 
			(clause->preds[ipred-1]->predicate->is_mp_done()))
	      continue;
	  }

	  /* don't need to special case for mp_done since 
	     it is covered by the next if condition */

	  if (clause->preds[ipred]->predicate->scc != scc) continue;

	  cur_snr++ ; 
	  ASSERT((cur_snr < numSNRules));
	  i = has_fail_in_body(clause);
	  if (i >= 0) {
	    sn_rules[cur_snr].first = i;
	  }
	  else {
	    sn_rules[cur_snr].first = ipred;
	  }	    
	  sn_rules[cur_snr].recursive = 1 ;
	  sn_rules[cur_snr].scc = this;
	  sn_offset_arr[cur_snr] = j ;
	  sn_rules[cur_snr].Add_Rule_Info(
		    module->ruleArray->rInfoPtrs[clause->rule_number],
		    clause, module->module_info.NonGroundFacts);
	  
	  for (jpred = 0; jpred < clause->num_preds; jpred++) {
	    
	    Literal *bodypred = clause->preds[jpred];
	    
	    if (ipred == jpred)
	      sn_rules[cur_snr].markOffsets[2*jpred] = 
			Find_RMark_Offset(this, bodypred->predicate) ;
			// Find_RMark_Offset(scc, bodypred->predicate) ;
	    else 
	      sn_rules[cur_snr].markOffsets[2*jpred] = -1 ;
	    
	    if (bodypred->predicate->scc != scc)
	      sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
	    else {
	      int tmp = Find_RMark_Offset(this, bodypred->predicate) ;
			// Find_RMark_Offset(scc, bodypred->predicate) ;
	      if (ipred > jpred)
		sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp ;
	      else
		sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
	    }
	  }
	}
      }
  }
  
}


void SccInfo::display_annotation_info(Table *F) 
{
  FILE *file = exEnv.trace_file;
  PredAnnotations *cur;
  QFModuleData *md = F->moduledata;
  
  for(int i=0; i < numPreds; i++) {
    Name predname = F->rels[offset_arr[i]].relPtr->name;
    
    for(cur= md->agg_sel_annotations; cur; cur=cur->next()) {
      if (cur->pred==predname) {
	fprintf(file,"%s%s  ", "@",SymbolString(AggSelectionSymbol));
	cur->print(file);
	fprintf(file, ".\n");
      }
    }
    for(cur= md->prioritize_annotations; cur; cur=cur->next()) {
      if (cur->pred==predname) {
	fprintf(file,"%s%s  ", "@",SymbolString(PrioritizeSymbol));
	cur->print(file);
	fprintf(file, ".\n");
      }
    }
    fprintf(exEnv.trace_file, "%s: subsumption checking - %s \n\n",
	    F->rels[offset_arr[i]].relPtr->name->string(),
	    F->rels[offset_arr[i]].relPtr->check_subsum? "on" : "off");
    /* PRAVEEN:: Add the index information dumping routines here. */
  }
  fprintf(file,"---------------------------------------\n\n");
}



SccInfo::SccInfo(QFMaterializedModuleData *md, Scc *scc,
		 Predicate *ext_magicpred, int id)
{
  PredAnnotations *cur;
  int i;

  has_hidden_relations = 0;
  module = md ;
  this->scc = scc ;
  scc_id = id;
  this->createSNStructs(ext_magicpred);
  for (i=0; i < numPreds; i++) {
    for(cur = md->prioritize_annotations; cur; cur=cur->next()) {
      if (cur->pred==pred_ptrs[i]->name)
	has_hidden_relations = 1;
    }
  }
  rec_rule_start = 0;
}	

SccInfo::~SccInfo()
{
 if (numPreds)
  {
   delete pred_ptrs;
   delete offset_arr;
   delete pred_success_arr;
   delete sn_rules;
  }
}


QFMaterializedModuleData::QFMaterializedModuleData(Table *table,
						   RuleInfoArray *rArray,
						   ParserStruct& parserStruct)
{
  F = table ;
  ruleArray = rArray ;
  numSccs = 0 ;
  sccArray = NULL ;
  multiset_annotations = parserStruct.MultisetAnnotations;
  agg_sel_annotations  = parserStruct.agg_sel_annotations;
  prioritize_annotations  = parserStruct.prioritize_annotations;
  index_annotations    = parserStruct.make_index_annotations;
  diskrel_annotations    = parserStruct.diskrel_annotations;

  free_table_list = NULL;
  saved_state = 0;
  no_magic = 0;
  rel_options = 0;
  predicates = new ModulePred(NULL);
}

void QFMaterializedModuleData::AddScc(Scc *scc, Predicate *ext_magicpred)
{
	int i;

	SccInfo **temp = sccArray ;
	sccArray = new SccInfo*[++numSccs] ;
	for (i=0; i< numSccs-1; i++)
		sccArray[i] = temp[i] ;

	sccArray[i] = new SccInfo(this, scc, ext_magicpred, numSccs-1) ;
	if (numSccs > 1) delete [] temp ;

}

void add_pred_extern(Predicate *pred, enum PredKind kind,
			     ModulePred *predicates)
{
    Name name = pred->name;
    int arity = pred->arity;
    ModulePred *new_pred;

    for (new_pred = predicates->next; new_pred; new_pred = new_pred->next) {
      if (new_pred->pred->name->equals(name) &&
	  (new_pred->pred->arity == arity)) {
	    if (kind==LocalPredKind && new_pred->kind==ExternalPredKind)
		new_pred->kind = kind;
	    return;
       }
     }

    new_pred = new ModulePred(pred);
    new_pred->kind = kind;
    new_pred->next = predicates->next;
    predicates->next = new_pred;

}

Scc *SccList = NULL;
Scc **SccListTail = &SccList;

struct SccWork {
    int lowlink;
    int number;
    struct SccWork *next;
    Predicate *pred;
    int recursive;
};

struct SccWork *SccWorkList = NULL;
int SccCount = 0;
struct SccWork SccDone[1]; // dummy

static struct Scc *NewScc()
{
    struct Scc *scc = new Scc;
    scc->preds = NULL;
    scc->next = NULL;
    scc->recursive = 0;
    *SccListTail = scc;
    SccListTail = &scc->next;
    scc->emitted = -1;
    return scc;
}

void initialize_scc_structs()
{
    SccList = NULL ;
    SccListTail = &SccList;
    SccWorkList = NULL;
    SccCount = 0;
}

inline int min(int i, int j) { return i < j ? i : j; }

int find_predicates_invoked_from(Predicate *pred)
{
    // Based on code by Thomas Ball from an algorithm by Robert Tarjan

    struct SccWork *work = new SccWork();
    Clause *clause;
    int ipred;
    // number is a sequential node number
    // lowlink is lowest number of a node reachable from current node.
    work->lowlink = work->number = ++SccCount;

    work->next = SccWorkList;
    work->pred = pred;
    work->recursive = 0;
    pred->work = work;
    SccWorkList = work;
    // Iterate over all Clauses that define 'pred'.
    for (clause = pred->clauses; clause; clause = clause->pnext) {

	// This should take care of 'bogus' magic dependencies
	if (clause->head->predicate->is_magic() && clause->num_preds) {
	    Predicate *first_pred = clause->preds[0]->predicate;
	    if (first_pred->is_magic() && first_pred->mark == 0)
		continue ;
	}

        clause->seen = 1;
	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if (child->work == NULL) { /*  not visited yet */
		int child_lowlink = find_predicates_invoked_from(child);
		work->lowlink = min(work->lowlink, child_lowlink);
	    }
	    else if (child->work != SccDone) {
		if (child->work->number < work->number)
		    work->lowlink = min(work->lowlink, child->work->lowlink);
		child->work->recursive = 1;
	    }
	}
    }
    int lowlink = work->lowlink;
    if (work->lowlink == work->number) {
	struct Scc *scc = NewScc();
	Predicate **pred_tail = &scc->preds;
	while (SccWorkList && SccWorkList->number >= work->number) {
	    register SccWork *sccWork = SccWorkList;
	    Predicate *work_pred = sccWork->pred;
	    work_pred->next = scc->preds;
	    work_pred->scc = scc;
#ifdef DEBUG
	    if (work_pred->work != sccWork) abort();
#endif
	    work_pred->work = SccDone;
	    *pred_tail = work_pred;
	    pred_tail = &work_pred->next;
	    SccWorkList = sccWork->next;
	    scc->recursive |= work->recursive;
	//    delete sccWork;  PURIFY :: this frees memory used in later recursive calls !! 
	}
	*pred_tail = NULL;

    }
    return lowlink;
}

static void mark_predicates_invoked_from(Predicate *pred)
{
    Clause *clause;
    int ipred;

    pred->mark = 1 ;
    // Iterate over all Clauses that define 'pred'.
    for (clause = pred->clauses; clause; clause = clause->pnext) {

	if (clause->head->predicate->is_magic()) continue ;

	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if (child->mark == 0) { /*  not visited yet */
		mark_predicates_invoked_from(child);
	    }
	}
    }

}

static void clean_up_after_one_scc()
{
    // if there are no base predicates, need to clean up SccList
    struct Scc *base_scc = SccList;
    if (!(base_scc->preds)) {
	// there are no base predicates
	ASSERT((base_scc->next != NULL));
	ASSERT((base_scc->next->preds != NULL));
		// must at least have the query predicate

	SccList = base_scc->next;
	delete base_scc;
    }

    // NOTE: should the SccWorkList also be deleted?
}

int group_predicates_in_one_scc(Predicate *pred)
{
    // If there are base predicates, there have to be at least two
    // SCCs; the first one must be the base SCC

    struct Scc *base_scc;
    struct Scc *scc;
    if (!SccList) {
	base_scc = NewScc();
	scc = NewScc(); 	// this should link it to be the 2nd SCC
        scc->recursive = 1;		// NOTE: is this a right assumption?
    }
    else {
	base_scc = SccList;
	scc = SccList->next;
    }

    struct SccWork *work;
    if (!SccWorkList) {
	 work = new SccWork();
	 work->next = SccWorkList;
	 SccWorkList = work;
    }
    else work = SccWorkList;

    pred->work = work;  // visited

    Clause *clause;
    int ipred;
    
    pred->scc = scc;
    pred->next = scc->preds;
    scc->preds = pred;

    for (clause = pred->clauses; clause; clause = clause->pnext) {

	// This should take care of 'bogus' magic dependencies
	if (clause->head->predicate->is_magic() && clause->num_preds) {
	    Predicate *first_pred = clause->preds[0]->predicate;
	    if (first_pred->is_magic() && first_pred->mark == 0)
		continue ;
	}

	clause->seen = 1;

	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if ((child->work == NULL) && (child->clauses)) {
		// predicates not yet visited and those
		// which have rules defined in this module
		group_predicates_in_one_scc(child);
	    }
	    else if (child->work == NULL) {
		// predicates not yet visited
		// and which are external predicates
		child->work = work;
		if (child->is_mp_done()) { 
		    // NOTE: for Ordered Search
		    child->scc = scc;
		    child->next = scc->preds;
		    scc->preds = child;
		}
		else {
		    child->scc = base_scc;
		    child->next = base_scc->preds;
		    base_scc->preds = child;
		}
	    }
	}
    }

    return 1;
}


static void interpret_scc(QFMaterializedModuleData *md, Scc *scc,
		ExportInfo *export_info, Predicate *ext_magicpred)
{
    Predicate *pred; Clause *clause;
    int numRules = 0 ;

    // First emit code to solve Scc's needed by this Scc.
    // (This is, after all, bottom-up evaluation.)

    for (pred = scc->preds; pred != NULL; pred = pred->next) {
	for (clause = pred->clauses;
	     clause != *pred->clause_tail;
	     clause = clause->pnext) {
	    if (!clause->seen) continue;
	    int ipred;
	    numRules++ ;
	    for (ipred = 1; ipred <= clause->num_preds; ipred++) {
		Predicate *child = clause->preds[ipred-1]->predicate;
		if (child->scc == scc) continue;
		if (child->scc->emitted == export_info->sequence)
		    continue; // Already dumped.
		child->scc->emitted = export_info->sequence;
		interpret_scc(md, child->scc, export_info, ext_magicpred);
	    }
	}
    }

// Check that the scc has some rules to evaluate. Otherwise, it is useless
    if (numRules)  md->AddScc(scc, ext_magicpred) ;

}

Predicate *find_ext_magicquerypred(ParserStruct& parserStruct)
{
    for (Predicate *pred = parserStruct.AllPredicates; pred != NULL; 
				pred=pred->allprednext) {
	if ((!pred->clauses) && (pred->is_magic())) { // found it
	    return pred;
	}
    }
    return NULL;
}


int count_var_list(VarLink *rest)
{
    VarLink *cur;
    int max_var_num = -1;
    for( cur=rest;  cur; cur=cur->next) {
	if (cur->arg->var > max_var_num)
	    max_var_num = cur->arg->var;
    }
    return max_var_num+1;
}

int determine_subsum_status(QFModuleData *md,
			    Name name,
			    int default_val)
{
  PredAnnotations *cur;
  
  for(cur= md->multiset_annotations; cur; cur=cur->next()) {
    char buf[1000];
    sprint_pred_name(buf, SymbolString(cur->pred), cur->adorn);
    /*
     * Warning: multiset and checksubsumption annotations have
     * the name stored separate from the adornment 
     */
    if (name->equals(EnterSymbol(buf))) {
      default_val = cur->check_subsum;
    }
  } 
  /****  For pipelined modules, this used to be the code : Praveen
     for(cur= md->multiset_annotations; cur; cur=cur->next()) 
      {
       if (cur->pred==rule->head->pred)
	 F->rels[offset].relPtr->check_subsum = cur->check_subsum;
      }
  ****/

  return default_val;
}

static int diskrel_anno(QFModuleData *md, Name name, ArgList *& a_list)
{
  PredAnnotations *cur;

  // what about the schema arglist in this case ?? : PRAVEEN
  if (md->module_info.DiskRel) return 1;
  
  for(cur= md->diskrel_annotations; cur; cur=cur->next()) {
    char buf[1000];
    sprint_pred_name(buf, SymbolString(cur->pred), cur->adorn);
    /*
     * Warning: diskrel annotations have the name stored separate 
     * from the adornment 
     */
    if (EnterSymbol(buf) == name) {
      a_list = cur->arglist1;
      return 1;
     }
  }
  return 0;
}

void get_annos(QFModuleData *md, ModulePred *predicate)
{
 PredAnnotations *cur;
 struct AnnoStruct *new_anno;

 for(cur= md->index_annotations; cur; cur = cur->next())
   if (cur->pred->equals(predicate->pred->name)) { 
     VarLink *rest;
     rest = cur->arglist1->var_list(NULL);
     if (cur->arglist2 )
       rest = cur->arglist2->var_list(rest);
     
     int num_vars = count_var_list(rest);
     new_anno = new struct AnnoStruct(cur, num_vars, predicate->index_annos);
     predicate->index_annos = new_anno;
    }

 for(cur= md->agg_sel_annotations; cur; cur=cur->next())
   if (cur->pred->equals(predicate->pred->name)) {
     new_anno = new struct AnnoStruct(cur, 0, predicate->aggsel_annos);
     predicate->aggsel_annos = new_anno;
  }

 for(cur = md->prioritize_annotations; cur; cur=cur->next())
  if (cur->pred->equals(predicate->pred->name)) {
     new_anno = new struct AnnoStruct(cur, 0, predicate->prior_annos);
     predicate->prior_annos = new_anno;
 }
 
}

static void init_module_preds(QFModuleData *md)
{
 ModulePred *predicates = md->predicates->next;

 for (; predicates; predicates = predicates->next) { 
   Name name = predicates->pred->name;
   int arity = predicates->pred->arity;
   predicates->offset = get_offset(md->F, name, arity, 1);

   switch (predicates->kind) {
    case QueryPredKind:
       predicates->disk_rel = diskrel_anno(md, name, predicates->schema_def);
       predicates->check_subsum = determine_subsum_status(md, name, md->module_info.CheckSubsum);
       break;

    case LocalPredKind:
       predicates->disk_rel = diskrel_anno(md, name, predicates->schema_def);
       if (is_magic(name))
	 predicates->check_subsum = determine_subsum_status
	   (md,name, (md->module_info.CheckSubsum | md->module_info.MultiSet));
       else
	 predicates->check_subsum = determine_subsum_status
	   (md, name, md->module_info.CheckSubsum);
       break;
    case MagicQueryPredKind:
       predicates->disk_rel = diskrel_anno(md, name, predicates->schema_def);
       predicates->check_subsum = determine_subsum_status
	 (md, name, (md->module_info.CheckSubsum | md->module_info.MultiSet));
       break;
    case ExternalPredKind:
       break;
    }
   get_annos(md, predicates);
  }
}

/*------------------------------------------------------------------

  interpret_materialized_qf()

  This routine creates the structures within the QFMaterializedModuleData,
  and inserts it into the IterativeMethod table. It assumes that SCC
  analysis has already been performed w.r.t. the particular exported
  predicate.

 -------------------------------------------------------------------*/

static void interpret_materialized_qf(QFMaterializedModuleData *md,
				      ExportInfo *export_info,
				      ParserStruct& parserStruct)
{
    RuleInfoArray *rArray = md->ruleArray ;
    Name exported_name = export_info->pred_name;

    Scc *scc;
    ModulePred *predicates = md->predicates;
    // Make sure that all Relations used by the query are initialized
    // by init_rule_pointers. Use add_pred_extern to enter them into
    // the predicates collection to keep track of them.
    // First enter the the query and magic predicates:

    Predicate *query_predicate = export_info->predicate;
    add_pred_extern(query_predicate, QueryPredKind, predicates);

    int query_arity = query_predicate->arity;
    int magic_arity = export_info->adornment.count();
    char *magic_name = new char[exported_name->length()+4+query_arity];
    sprint_magic_name(magic_name, exported_name->string(), 
				&export_info->adornment);

    Predicate *magic_predicate =
	LookupPredicate(parserStruct, EnterSymbol(magic_name), magic_arity);
    if (magic_predicate) 
        add_pred_extern(magic_predicate, MagicQueryPredKind,
			 predicates);
    else {
	fprintf(exEnv.error_file,
		"Warning: No magic predicate for exported predicate %s\n",
		exported_name->string() );
	if (parserStruct.CurModule.SaveModule) {
	    md->no_magic = 1;
        }
    }

    // Then enter the remaining predicates.
    for (scc = SccList; scc != NULL; scc = scc->next) {
	Predicate *pred; Clause *clause; int ipred; Literal** literals;

	// Add predicates for non-recursive clauses in this scc.
	for (pred = scc->preds; pred != NULL; pred = pred->next) {
	    clause = pred->clauses;
	    // If pred->clauses == *pred->clause_tail, then
	    // the pred has no clauses, and hence is an ExternalPredKind.
	    if (clause != *pred->clause_tail) {
	        add_pred_extern(pred, LocalPredKind, predicates);
	    }
	    else if (pred->is_mp_done()) {
		// NOTE: this is for ordered search
		add_pred_extern(pred, LocalPredKind, predicates);
	    }
	    else add_pred_extern(pred, ExternalPredKind, predicates);

	    // This also determines if clause is recursive.
	    for (; clause != *pred->clause_tail; clause = clause->pnext) {
		if (!clause->seen) continue;
		clause->recursive = 0;
		for (literals = clause->preds, ipred = clause->num_preds;
		     --ipred>= 0; literals++) {
		    add_pred_extern(literals[0]->predicate,
				     ExternalPredKind,
				     predicates);

		    // with Ordered Search, have to special case the 
		    // negated literals
		    if (parserStruct.CurModule.UseOrdSearch) {
			if (literals[0]->negated) continue;
		    }
		    // don't need to special case for done_m_p since 
		    // it is covered by the next if condition

		    if (literals[0]->predicate->scc == scc)
			clause->recursive = 1;
		}
	    }
	}
    }

    md->export_pred = export_info->predicate;
    md->module_info = parserStruct.CurModule;
    if (md->module_info.UsePredicateSN) {
      if (md->module_info.UseOrdSearch) {
        fprintf(exEnv.error_file, 
                "CORAL error: Cannot use PSN with Ordered Search ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
      else if (md->module_info.SaveModule) {
        fprintf(exEnv.error_file,
                "CORAL error: Cannot use PSN with SaveModule ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
      else if (md->module_info.SupMagicIndexing) {
        fprintf(exEnv.error_file,
                "CORAL error: Cannot use PSN with SupMagicIndexing ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
    }
    else md->use_psn = md->module_info.UsePredicateSN;

    interpret_scc(md, export_info->predicate->scc, export_info,
	          find_ext_magicquerypred(parserStruct));

    // move some of the run time work from init_rule_pointers to
    // compile time here. Determine what flags need to be set, etc.
    init_module_preds(md);
    
    // Insert the module structure along with its associated exported
    // query form into the IterativeMethod table.
    if (exported_name) {
	DerivedMethod* query_method = new DerivedMethod(exported_name,
					export_info->adornment, do_query,
					(QFModuleData *)md); 
    }
}

static void create_module_table(ParserStruct& parserStruct,
                                Table* &module_table,
                                RuleInfoArray* &module_rules)
{
    // Create the Table structure (containing a pointer to each Relation used).
    int pred_count = 0 ;
    for (Predicate *pred = parserStruct.AllPredicates; pred != NULL;
					pred = pred->allprednext) 
	pred_count++ ;

    module_table = new Table(pred_count);
    module_rules = new RuleInfoArray;

    // This enters the name and arity of each relation in the Table into the
    // Table array 'module_table'. The relations are also initialized at the
    // same time.

    int i = 0 ;
    for (pred = parserStruct.AllPredicates; pred != NULL; 
				pred=pred->allprednext, i++){
	module_table->AddValue(i, pred->name, pred->arity) ;
	pred->frame_offset = i;
    }

// For each clause, create a rule-info structure that contains the 
// information necessary to evaluate it.

    int rule_number = 0;
/***
    FOR_ALL_RULES(clause) {
	clause->rule_number = rule_number++;
	module_rules->AddRule(create_rule_info(module_table, clause));
    } END_EACH_RULE
****/
    FOR_EACH_RULE(clause, parserStruct.rules) {
	clause->rule_number = rule_number++;
	module_rules->AddRule(create_rule_info(module_table, clause));
    } END_EACH_RULE
}

/*------------------------------------------------------------------
 interpret_module()

 Top-level routine to interpret a CORAL module. For each exported query form,
 a QFModuleData structure is created that encapsulates the information needed
 to evaluate that query. The exported query form and the associated 
 QFModuleData structure are stored in a table of IterativeMethods.

 -------------------------------------------------------------------*/
void interpret_module(ParserStruct &parserStruct)
{

    ExportInfo *export_info;
    Table *module_table;
    RuleInfoArray *module_rules;

    if (parserStruct.CurModule.ExportedPreds == NULL) {
	fprintf(exEnv.error_file, "CORAL: Error - No exported query form.\n");
	return;
    }

    // Create the Table structure (containing a pointer to each Relation used).
    create_module_table(parserStruct, module_table, module_rules);

    QFModuleData *module_data ;

    // Process each exported query form independently.
    for (export_info = parserStruct.CurModule.ExportedPreds;
	 export_info != NULL; export_info = export_info->next) {
	Name query_symbol;
	Name exported_name = export_info->pred_name;
	int query_arity;

	if (parserStruct.CurModule.UsePipelining) 
	  module_data = new 
	    QFPipelinedModuleData(module_table, module_rules, parserStruct);
	else
	  module_data = new 
	    QFMaterializedModuleData(module_table, module_rules, parserStruct);

	if (export_info->adorn_symbol == 0) {
	    query_arity = 0;
	    query_symbol = exported_name;
	}
	else 
	{
	    Name exported_adorn_name = export_info->adorn_symbol;
	    query_arity = (int) exported_adorn_name->length();
		
		if (parserStruct.CurModule.UsePipelining)
		{
			int len = (int) exported_name->length() + 1;
			char *query_name = new char[len+1];
			sprintf(query_name,"%s",exported_name->string());
			query_symbol = EnterSymbol(query_name);
		}
		else
		{
	    	int len = (int) exported_name->length() + 1 + query_arity;
	    	char *query_name = new char[len+1];
	   		sprint_pred_name(query_name, exported_name->string(), 
				exported_adorn_name->string());
	    	query_symbol = EnterSymbol(query_name);
		}

	}
	// Ensure that the exported query form is a valid one.
	export_info->predicate = 
	  LookupPredicate(parserStruct, query_symbol, query_arity);
	if (!export_info->predicate) {
	    fprintf(exEnv.error_file,
		"Error: Exported predicate %s not defined in module\n",
		query_symbol->string() );
	    continue;
	}

	if (parserStruct.CurModule.UsePipelining) {
	  interpretPipelined ((QFPipelinedModuleData *) module_data,
			      export_info, parserStruct);
	}

	else { // For MATERIALIZED modules
	  // Do Scc (Strongly Connected Components) analysis.
	  FOR_ALL_RULES(clause) {
	    clause->seen = 0;
	      
	    clause->recursive = 0;
	    for (int ipred = 0; ipred < clause->num_preds; ipred++) {
	      clause->preds[ipred]->predicate->work = NULL;
	      clause->preds[ipred]->predicate->mark = 0;
	    }
	  } END_EACH_RULE
	  initialize_scc_structs() ;

	  if (parserStruct.CurModule.SingleScc ||
	      parserStruct.CurModule.UseOrdSearch ||
	      parserStruct.CurModule.HasPrioritize) {
	    mark_predicates_invoked_from(export_info->predicate);
	    group_predicates_in_one_scc(export_info->predicate);
	    clean_up_after_one_scc();

	  }
	  else {
	    mark_predicates_invoked_from(export_info->predicate);
	    find_predicates_invoked_from(export_info->predicate);
	  }
	  
	  interpret_materialized_qf((QFMaterializedModuleData *)module_data,
				    export_info, parserStruct);
	}

    }

}

void compile_single_rule(ParserStruct &parserStruct, 
			 Table* &module_table,
			 RuleInfoArray* &module_rules,
			 SemiNaive &snrule)
{
    // Create the Table structure (containing a pointer to each Relation used).
    create_module_table(parserStruct, module_table, module_rules);

    // There is only one RuleInfo structure
    ASSERT((module_rules->numRules == 1));

    // There was only one rule in the parserStruct
    ASSERT((parserStruct.rule_list->link.next == NULL));

    Clause *clause = parserStruct.rule_list;

    compile_nonrec_rule(snrule, clause, module_rules->rInfoPtrs[0],
			exEnv.C_non_ground_facts_default);

}
