/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
CORAL Software :: U.W.Madison

arrays.C:

	This file contains definitions for an array abstract data type.  This
may be used as a template for creating new data types.  There are two parts to
this file.
 	The first part of the file defines the array data type, and its
methods.  Some of the methods are tagged as mandatory.  These have to be 
defined for all user defined abstract data types.  Others are specific to 
the array data type.
	The second part of the file defines predicates that are used to 
manipulate the abstract data type array.  Note that the procedures here may
perform _destructive_ update on the abstract data type.  In general this is
_very_ dangerous, and may destroy the logical semantics of programs.  The
destructive update features here are an efficiency hack, and should not 
be used unless the user understands the CORAL system enough to figure out
the operational implications of such updates.

***********************************************************************/

#include <stdio.h>
#include "arg.h"
#include "builtin-rel.h"
#include "unify.h"
#include "hash.h"
#include "externs.h"
#include "parser.h"
#include "interface.h"
#include "globals.h"

extern int C_linenum;
extern int scanner_at_eof;
extern char *strip_quotes(char *);

char *ArrayDestructorString = "destruct";
char *ArrayConstructorString = "array";
extern Name ArrayConstructSymbol;
char *LogicalArrayConstructorString = "logical_array";
char *LogicalArrayBindString = "logical_bind";
char *ArrayLookupString = "lookup";
char *ArrayBindString = "bind";

/*---------------------------------------------------------------- */

#define COR_ArrayKind 1
	// Make sure that the above is distinct from the other arg subkinds

class ArrayArg : public ConstArg {

  public:
  /*********** Optional methods *************/

  virtual Arg * lookup(int i) const = 0;
  virtual int bind(int i, Arg *arg) = 0;
  virtual ArrayArg* make_version() = 0;
  virtual int size() = 0;

  /************* Mandatory Methods *************/

  virtual arg_kind kindof() const { return COR_CONST_ARG;}
  virtual int subkind() { return (COR_ArrayKind);}
  virtual int equals(Arg *arg);
  virtual void print(BindEnv *context, FILE *file) const = 0;
  virtual void print(BindEnv *context, FILE *file, char *) const {
	print(context, file);
  }
  virtual void sprint(char *str, int *pos, BindEnv *context = NULL) const;
  virtual void dump(int arg_number, FILE *file);
  virtual int isConstant() { return 1;}
  virtual HashVal hash(BindEnv *) {return IntToHash((int)this);}
};

int ArrayArg::equals(Arg *arg) {
  if (arg->kindof() != COR_CONST_ARG || 
	((ConstArg*)arg)->subkind() != COR_ArrayKind)
    return 0;
  else return this == arg;
}

void ArrayArg::sprint(char *, int *, BindEnv *) const {
    fprintf(stderr, "Sorry:  Array::sprint() not implemented\n");
}

void ArrayArg::dump(int , FILE *) {
    fprintf(stderr, "Sorry:  Array::dump() not implemented\n");
}


/*---------------------------------*/

class FixedArrayArg : public ArrayArg {

  int _size;
  Arg **array;
  public:
  
  /*********** Optional methods *************/
  FixedArrayArg ( int size1);

  virtual inline Arg * lookup(int i) const { 
	if( i < 0 || i > _size)
	    return NULL;
	else return ((Arg**)array)[i];
  }
  virtual inline int bind(int i, Arg *arg) {
	if( i < 0 || i > _size)
	    return -1;
	else {
	    ((Arg**)array)[i] = arg;
	    return 1;
	}
      }

  virtual int size() { return _size; }

  virtual ArrayArg* make_version();
  /************* Mandatory Methods *************/

  virtual void print(BindEnv *context, FILE *file) const;
  virtual void print(BindEnv *context, FILE *file, char *) const {
    print(context, file) ;
  };
  // virtual void sprint(char *str, int *pos, BindEnv *context = NULL) const;
  // virtual void dump(int arg_number, FILE *file);
};

void FixedArrayArg::print(BindEnv *context, FILE *file) const {

    fprintf(file, " ARRAY(");
    for(int i=0; i < _size; i++) {
	array[i]->print(context, file);
	if (i < _size -1)
	    fprintf(file, ",\n\t");
    }
    fprintf(file, ")");
}

FixedArrayArg::FixedArrayArg(int size1) {
    _size = size1;
    array = new Arg*[_size];
    for(int i=0; i < _size; i++) 
      ((Arg**)array)[i] = NilSymbol;
}

ArrayArg* FixedArrayArg::make_version() {

    // WARNING:  be careful about using this function.  It is quite 
    //    inefficient since it copies out the whole array.
    //    Should probably use logical_array instead.

    ArrayArg *new_array = new FixedArrayArg(_size);
    for (int i=0; i < _size; i++) {
        new_array->bind(i,lookup(i));
    }
    return new_array;
}

/*---------------------------------*/

class LogicalArrayArg : public ArrayArg {

  int _size;
  VersionedBindEnv *array;
	// WARNING: Really ought to make a version of the VersionedBindEnv code
	// to deal with args rather than terms.  The current implementation
	// wastes some space.
  public:
  
  /*********** Optional methods *************/
  LogicalArrayArg ( int size1);
  LogicalArrayArg ( int size1, VersionedBindEnv *env);
  ArrayArg *make_version();
  virtual int size() {return _size;}

  virtual inline Arg * lookup(int i) const { 
	if( i < 0 || i >= _size)
	    return NULL;
	else return array->lookup(i).expr;
  }

  virtual inline int bind(int i, Arg *arg) {
	// Physical bind!
	if( i < 0 || i > _size)
	    return -1;
	else {
	    Term term(arg,NULL);
	    array->bind(i,term);
	    return 1;
	}
  }

  /************* Mandatory Methods *************/

  virtual void print(BindEnv *context, FILE *file) const;
  virtual void print(BindEnv *context, FILE *file, char *) const {
    print(context, file) ;
  };
  // virtual void sprint(char *str, int *pos, BindEnv *context = NULL) const;
  // virtual void dump(int arg_number, FILE *file);
};

LogicalArrayArg::LogicalArrayArg(int size1) {
    _size = size1;
    array = new VersionedBindEnv(_size);
    Term term0(NilSymbol,NULL);
    for(int i=0; i < _size; i++) array->forcebind(i,term0);
}

LogicalArrayArg::LogicalArrayArg ( int size1, VersionedBindEnv *env) {
    _size = size1;
    array = env;
}

void LogicalArrayArg::print(BindEnv *context, FILE *file) const {

    fprintf(file, " ARRAY(");
    for(int i=0; i < _size; i++) {
	lookup(i)->print(context, file);
	if (i < _size -1)
	    fprintf(file, ",\n\t");
    }
    fprintf(file, ")");
}

ArrayArg* LogicalArrayArg::make_version() {

    VersionedBindEnv *new_array = (VersionedBindEnv *)array->make_version();
    LogicalArrayArg *new_arg =  new LogicalArrayArg(_size, new_array);
    return new_arg;
}

/***************************************************************************/

/*** get_iterator_term is a utility routine to dereference an argument and 
	make sure that it is of a specified type / subtype
 ***/

#define GET_ITERATOR_TERM( term, iterator, argnum, kind, subkind, msg) \
	if( get_iterator_term( term, iterator, argnum, kind, subkind, msg)<0){\
	    iterator.set_no_match();					\
	    return NULL;						\
	} 

int get_iterator_term( Term& term, TupleIterator &iterator, int argnum, 
		arg_kind kind, int subkind, char *msg) {

    term.bindenv = iterator.bindenv;
    term.expr = iterator.arg_list[argnum];
    FULL_DEREFERENCE_TERM(term);
    
    if ( (term.expr->kindof() != kind) || 
    	 (kind == COR_CONST_ARG && 
		( subkind != ((ConstArg *)term.expr)->subkind()) )
	|| (kind == COR_NUM_CONST && 
		( subkind != (int) ((NumArg *)term.expr)->num_kindof()) ) 
       ) {
	fprintf(stderr, "Error:  %s: bad argument type to ", msg);
	term.printon(stderr);
	fprintf(stderr, "\n");
	return -1;
    }
    return 1;
}

/***************************************************************************/

/**  Predicate Definitions:
	The following procedures define predicates that are used to create,
lookup, update and destroy objects of the abstract data type array.
	The arguments to the predicate are in iterator.  iterator.arg_list
is an ArgList data structure; variables in the arguments must be "dereferenced"
through iterator.bindenv.  Dereferencing is required since the arguments
of iterator have variables that have bindings specified in iterator.bindenv.
There are two forms of dereferencing -- a shallow dereferencing and a 
deep dereferencing.  The shallow form is done by 
	FULL_DEREFERENCE_TERM  
while the deep dereferencing is done by the method
	simplify  
simplify is defined in the file arg.C.  The two differ in that the first
dereferences an argument till its outermost level is not a variable, or is a
free variable.  There may be bound variables inside the structure.  The second
dereferences _all_ variables in the argument, and returns a structure where
the variables have been dereferenced, and any unbound variables have been 
renamed.
**/

/***************************************************************************/

BindEnv *ArraySolver(BuiltinRelation& rel, TupleIterator& iterator);
BindEnv *ArrayDestructSolver(BuiltinRelation& rel, TupleIterator& iterator);
BindEnv *ArrayLookupSolver(BuiltinRelation& rel, TupleIterator& iterator);
BindEnv *ArrayBindSolver(BuiltinRelation& rel, TupleIterator& iterator);
BindEnv *LogicalArrayBindSolver(BuiltinRelation& rel, TupleIterator& iterator);

/*********   
        The following lines show a method for declaring new predicates 
	that does not work currently.  It will hopefully be implemented 
	in later versions of CORAL.  For now, look at file builtin-rel.C
	to see how these predicates are declared.
BuiltinRelation dummy1 (1, EnterSymbol(ArrayDestructorString),
                 ArrayDestructSolver);
BuiltinRelation dummy2 (1, EnterSymbol(ArrayConstructorString), ArraySolver);
BuiltinRelation dummy2a (1, EnterSymbol(LogicalArrayConstructorString),
		 LogicalArraySolver);
BuiltinRelation dummy3 (1, EnterSymbol(ArrayLookupString), ArrayLookupSolver);
BuiltinRelation dummy4 (1, EnterSymbol(ArrayBindString), ArrayBindSolver);
BuiltinRelation dummy4a (1, EnterSymbol(LogicalArrayBindString), 
		LogicalArrayBindSolver);
**********/

 
BindEnv * ArraySolver(BuiltinRelation& rel, TupleIterator& iterator)
{
  // array(array,size)
  // Construct a new array of the specified size;  Default size = 16 for now.
  // Plan to change to growing arrays at some point.  Bind the first argument
  // to the newly constructed array.
  // If first arg is bound to array, finds size.

  StackMark stackmark;

  int arity = iterator.arg_list.count();
  if (arity != 1 && arity != 2 ){
     fprintf(stderr,"CORAL :: Error -- bad number of arguments to %s :",
			ArrayConstructorString);
     iterator.arg_list.printon(stderr);
     fprintf(stderr, "\n");
     iterator.set_no_match();
     return NULL;
  }

  Term term1 (iterator.arg_list[0], iterator.bindenv);
  FULL_DEREFERENCE_TERM(term1);
  if (term1.expr->kindof() != COR_VARIABLE ) { // Being used to find the size.

        GET_ITERATOR_TERM(term1, iterator, 0, COR_CONST_ARG, COR_ArrayKind,
			 ArrayConstructorString);

  	Term term2 (iterator.arg_list[1], iterator.bindenv);
  	FULL_DEREFERENCE_TERM(term2);
	/***********
  	if (term2.expr->kindof() != COR_VARIABLE ) {
            fprintf(stderr,"CORAL :: Error -- bad argument to %s:",
			ArrayConstructorString);
            term22printon(stderr);
            fprintf(stderr,"\n");
            iterator.set_no_match();
            return NULL;
	}
	*************/
       Term term3( make_arg(((ArrayArg*)term1.expr)->size()), NULL);
       if (unify_args(term2, term3) == COR_U_SUCCEED) {
          iterator.reset_no_match();
          return iterator.bindenv ;
       }
       stackmark.pop_to();
       iterator.set_no_match();
       return NULL;
  }
  int size = 16;
  if (arity == 2) {
	Term term2;
	GET_ITERATOR_TERM(term2, iterator, 1, COR_NUM_CONST, COR_INTEGER,
			ArrayConstructorString);
	size = make_int(term2.expr);
  }
  
  Arg *newarray;

  if (! (rel.name->equals(ArrayConstructSymbol)) )
      newarray = new LogicalArrayArg(size);
  else newarray = new FixedArrayArg(size);
  unify_binding(term1.bindenv, ((VarArg *)term1.expr)->var, newarray);
  iterator.reset_no_match();
  return iterator.bindenv ;
}

BindEnv * ArrayDestructSolver(BuiltinRelation&, TupleIterator& iterator)
{ 
  // destruct(array):
  //   Destroy an array and deallocate space allocated for it.

  int arity = iterator.arg_list.count();
  if (arity != 1 ){
     fprintf(stderr,"CORAL :: Error -- bad number of arguments to %s:",
		ArrayDestructorString);
     iterator.arg_list.printon(stderr);
     fprintf(stderr, "\n");
     iterator.set_no_match();
     return NULL;
  }

  Term term0;
  GET_ITERATOR_TERM(term0, iterator, 0, COR_CONST_ARG, COR_ArrayKind, 
		ArrayDestructorString); 
  delete term0.expr;
  iterator.reset_no_match();
  return iterator.bindenv ;
}

BindEnv *ArrayLookupSolver(BuiltinRelation& , TupleIterator& iterator) 
{
  // lookup(array,i,val)
  //   Unify val with the i'th element of the array.  Array elements
  //   have a default value of [] (the nil list).
  //   Note unification will not affect the array elements since they
  //      are required to be ground.
  
  if (iterator.arg_list.count() != 3 ){
     fprintf(stderr,"CORAL :: Error -- bad number of arguments to %s:",
			ArrayLookupString);
     iterator.arg_list.printon(stderr);
     fprintf(stderr, "\n");
     iterator.set_no_match();
     return NULL;
  }

  Term term0;
  GET_ITERATOR_TERM(term0, iterator, 0, COR_CONST_ARG, COR_ArrayKind,
			 ArrayLookupString);

  Term term1;
  GET_ITERATOR_TERM(term1, iterator, 1, COR_NUM_CONST, COR_INTEGER,
			 ArrayLookupString);

  int i = make_int(term1.expr);
  Arg *arg = ((ArrayArg *)term0.expr)->lookup(i);
  if ( ! arg) {
      iterator.set_no_match();
      return NULL;
  }

  StackMark stackmark;
  Term term2(iterator.arg_list[2], iterator.bindenv);
  Term term3(arg, term0.bindenv);
  if (unify_args(term2, term3) == COR_U_SUCCEED) {
     iterator.reset_no_match();
     return iterator.bindenv ;
  }
  stackmark.pop_to();
  iterator.set_no_match();
  return NULL;
}

BindEnv *ArrayBindSolver(BuiltinRelation& , TupleIterator& iterator) 
{ // bind(array,i,val):
  //	bind array[i] to val.
  
  if (iterator.arg_list.count() != 3 ){
     fprintf(stderr,"CORAL :: Error -- bad number of arguments to %s:",
			ArrayBindString);
     iterator.arg_list.printon(stderr);
     fprintf(stderr, "\n");
     iterator.set_no_match();
     return NULL;
  }

  Term term0;
  GET_ITERATOR_TERM(term0, iterator, 0, COR_CONST_ARG, COR_ArrayKind,
			ArrayBindString);

  Term term1;
  GET_ITERATOR_TERM(term1, iterator, 1, COR_NUM_CONST, COR_INTEGER,
			 ArrayBindString);
  int i = make_int(term1.expr);
  
  // NOTE:: Should check that the arg. is a constant, and simplify it
  // before binding.  
  TermLink *renamed_vars = NULL;
  Arg *newval = iterator.arg_list[2]->simplify(iterator.bindenv, renamed_vars, 
	NULL, NULL);
  if (renamed_vars != NULL) {
	fprintf(stderr, "Error: terms containing variables cannot be stored in arrays! \n");
	fprintf(stderr, "\t - offending term = ");
	iterator.arg_list[2]->print(iterator.bindenv, stderr);
	iterator.set_no_match();
	return NULL;
  }

  ((ArrayArg *)term0.expr)->bind(i,newval);
  iterator.reset_no_match();
  return iterator.bindenv;
}

BindEnv *LogicalArrayBindSolver(BuiltinRelation& , TupleIterator& iterator)
{ // logical_bind(array,i,val,newarray):
  //    create a version of array with array[i] bound to val, and assign the
  //	result to newarray.
  // Note that newarray must be a variable.

  if (iterator.arg_list.count() != 4 ){
     fprintf(stderr,"CORAL :: Error -- bad number of arguments to %s:",
                        LogicalArrayBindString);
     iterator.arg_list.printon(stderr);
     fprintf(stderr, "\n");
     iterator.set_no_match();
     return NULL;
  }

  Term term0;
  GET_ITERATOR_TERM(term0, iterator, 0, COR_CONST_ARG, COR_ArrayKind,
                        ArrayBindString);

  Term term1;
  GET_ITERATOR_TERM(term1, iterator, 1, COR_NUM_CONST, COR_INTEGER,
                         ArrayBindString);
  int i = make_int(term1.expr);

  Term newarray_term (iterator.arg_list[3], iterator.bindenv);
  FULL_DEREFERENCE_TERM(newarray_term);
  if (newarray_term.expr->kindof() != COR_VARIABLE ) {
        fprintf(stderr,"CORAL :: Error -- bad argument to %s:",
			LogicalArrayBindString);
        newarray_term.printon(stderr);
        fprintf(stderr,"\n");
        iterator.set_no_match();
        return NULL;
  }

  // NOTE:: Should check that the arg. is a constant, and simplify it
  // before binding.  
  TermLink *renamed_vars = NULL;
  Arg *newval = iterator.arg_list[2]->simplify(iterator.bindenv, renamed_vars,
        NULL, NULL);
  if (renamed_vars != NULL) {
        fprintf(stderr, 
	   "Error: terms containing variables cannot be stored in arrays! \n");
        fprintf(stderr, "\t - offending term = ");
        iterator.arg_list[2]->print(iterator.bindenv, stderr);
        iterator.set_no_match();
        return NULL;
  }

  ArrayArg *new_array = ((ArrayArg *)term0.expr)->make_version();
	// The value returned by make_version is a LogicalArrayArg or
	// FixedArrayArg depending on the type  of term0.expr. 
  new_array->bind(i,newval);
  Term new_term(new_array,NULL);
  if (unify_args(newarray_term, new_term) == COR_U_SUCCEED) {
     // Should succeed, since newarray_term is supposed to be a variable.
     iterator.reset_no_match();
     return iterator.bindenv ;
  }
  iterator.set_no_match();
  return NULL;
}



