/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/*
 * scanner.C
 */


#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#ifndef UNIX_HP
#include <math.h>
#endif

#include "line_info.h"
#include "scanner.h"
#include "scan_input.h"
#include "mystrings.h"
#include "hashdefines.h"


#define NUM_OF_INCR 4
#define ARRAY_MAX 256

/* Internal functions: */

extern char *copy_string (char *);
static lexeme_type lookup_keyword (char *);
static token *scan_int (LINE *, int);
static token *scan_id_keyword (LINE *, int);
static token *scan_string (LINE *, int);
void fatal_error(char *);
int check_var_or_expr(char *);
int coral_var(char *);

extern mystring *coral_file_dump_fn;
extern FILE *error_file, *coral_file;
extern struct vars_list;
extern struct vars_list *tuple_var_list;
extern struct vars_list *relation_var_list;
extern struct vars_list *lookup_tuple_var(char *, struct vars_list *);
extern struct vars_list *lookup_relation_var(char *, struct vars_list *);
static type_name check_type(LINE *, char *&);
static type_name lookup_type(char *);
static char * scan_c_var(LINE *, type_name);
static char * get_var_name(LINE *);


/* SORTED list of the keywords recognized by scanner. */

typedef struct
{
  char *name;			/* Name of keyword */
  lexeme_type lx;		/* Lexeme for keyword */
} kw_record;


// NOTE : this array should be in sorted order !!
static kw_record keywords[] =
{
  {"Arg", KW_ARG},
  {"Relation", KW_RELATION},
  {"Tuple", KW_TUPLE},
  {EXPORT_DECL, KW_EXPORT},  // _coral_export
  {"as", KW_AS},
  {"char", KW_CHAR},
  {"class", KW_CLASS},
  {"do", KW_DO},
  {"double", KW_DOUBLE},
  {"each", KW_EACH},
  {"enddo", KW_ENDDO},
  {"float", KW_FLOAT},
  {"for", KW_FOR},
  {"in", KW_IN},
  {INIT_CORAL, KW_INIT_CORAL}, // init_coral
  {"int", KW_INT},
  {"long", KW_LONG},
  {MAKE_TUPLE, KW_MAKE_TUPLE}, // make_tuple
  {"short", KW_SHORT},
  {"string", KW_STRING},
  {"struct", KW_STRUCT},
  {"unsigned", KW_UNSIGNED},
};

/* SORTED list of the types of C++ variables that 
 * can occur in the declarative part of the C++ module
 */

typedef struct
{
  char *name;			/* Name of type*/
  type_name lx;		/* Lexeme for keyword */
} type_record;

static type_record typestrings[] = 
{
  {"char", TYPE_CHAR},
  {"double", TYPE_DOUBLE},
  /* {"file", TYPE_FILENAME}, */
  {"float", TYPE_FLOAT},
  {"int", TYPE_INT},
  {"long", TYPE_LONG},
  {"short", TYPE_SHORT},
  {"string", TYPE_STRING},
  /* {"unsigned", TYPE_UNSIGNED}, */
};


/* Return a token of type LX1 if the next character is D and LX2 if it
   is not. The next character is put back in the input stream if it is
   not D. */

#define RET1of2(LX1, D, LX2) \
{\
  int cc = lines[top]->get_char ();\
  if (cc == D)\
      return (new token (LX1));\
  lines[top]->unget_char (cc);\
  return (new token (LX2));\
 }

/*
 * get_var has to generate code which when executed would 
 * dump the code given in the input file with the runtime 
 * values replacing the C++ files. 
 * special care has to taken for special characters
 * such as \, ' etc.
 */

#define CUR_INFO info
#define CUR_TYPE CUR_INFO.type_array[CUR_INFO.no_of_params]
#define CUR_VARNAME CUR_INFO.var_names[CUR_INFO.no_of_params]

char * scan_input:: get_var(infoStruct& info)
{
  int line_number;
  int col_number;
  int max = ARRAY_MAX;
  char * buf = new char[max];
  char * name_buf = new char[max];
  int name_i = 0;
  int i = 1;
  int inside_string = 0;
  int c;

  
  if (buf == NULL)
    fatal_error ("new failed in get_var\n");

  buf[0] = ' ';
  buf[1] = '\0';
  
  if (name_buf == NULL)
    fatal_error ("new failed in get_var\n");
  name_buf[0] = '\0';
  
  c = lines[top]->get_char();
  while (1) {
    switch (c) {
      
    case '"' : 
      {
	inside_string = 1;
	buf[i++] = '\\';
	buf[i++] = c;
	while (inside_string) {
	  c = lines[top]->get_char();
	  if (c == '\n')  {
	    line_number = get_line_number();
	    col_number = get_line_position();
	    fprintf(error_file, "line %d col %d syntax error ",
		    line_number, col_number);
	    fprintf(error_file, "- new line char in string \n");
	    return NULL;
	  }
	  else 
	    if (c == EOF_CHAR) {
	      line_number = get_line_number();
	      col_number = get_line_position();
	      fprintf(error_file, "line %d , col %d unexpected eof reached \n",
		      line_number, col_number);
	      return NULL;
	    }
	    else 
	      if (c == '"') {
		if (buf[i - 1] == '\\') {
		  /* buf[i++] = c;*/
		}
		else {
		  /* end of string */
		  inside_string = 0;
		  /* buf[i++] = c; */
		}
		buf[i++] = '\\';
		buf[i++] = c;
	      }
	      else if (c == '\\') {
		buf[i++] = '\\';
		buf[i++] = c;
	      }
	      else {
		buf[i++] = c;
		
	      }
	  if ((i + NUM_OF_INCR) >= max)
	    {
	      char *new_buf = new (char [max * 2]);
	      
	      if (new_buf == NULL)
		fatal_error ("new failed in scan_id_keyword\n");
	      strncpy (new_buf, buf, i);
	      delete (buf);
	      buf = new_buf;
	      max *= 2;
	    }
	}
	break;
      }
    case '%' :
      {
	/* coral comment */
	while (c != '\n') 
	  c = lines[top]->get_char();
	buf[i++] = c;
	get_next_line();
	break;
      }
    case '/' :
      {
	/* could be a comment */
	c = lines[top]->get_char();
	if (c == '/') {
	  /* C++ style comment */
	  while (c != '\n') 
	    c = lines[top]->get_char();
	  buf[i++] = c;
	  get_next_line();
	}
	else 
	  if (c == '*') {
	    /* C style comment */
	    int loop = 1;
	    while (loop) {
	      c = lines[top]->get_char();
	      if ((c == '\n') || (c == EOLN_CHAR))
		get_next_line();
	      else 
		if (c == EOF_CHAR) {
		  line_number = get_line_number();
		  col_number = get_line_position();
		  fprintf(error_file, "Line %d Col %d unexpected eof reached \n",
			  line_number, col_number);
		  return NULL;
		}
		else 
		  if (c == '*') {
		    c = lines[top]->get_char();
		    if (c == '/') {
		      /* end of comment */
		      loop = 0;
		    }
		    else
		      lines[top]->unget_char(c);
		    
		    
		  }
	    }
	  }
	  else {
	    buf[i++] = c;
	  }
	break;
      }
    case '\'' : 
      {
	buf[i++] = c;
	c = lines[top]->get_char();
	if (c == '\n') 
	  get_next_line();
	else 
	  if (c == EOF_CHAR) {
	    line_number = get_line_number();
	    col_number = get_line_position();
	    fprintf(error_file, "Line %d Col %d - unexpected end of file reached \n",
		    line_number, col_number);
	    return NULL;
	  }
	  else 
	    if (c == '\\') {
	      buf[i++] = '\\';
	      buf[i++] = c;
	      c = lines[top]->get_char();
	    }
	/* else if (c == '"') {
	   // 	buf[i++] = '\\';
	   // 	// buf[i++] = c;
	   // 	}
	   */
	if ((c == '\\') || (c == '"')) {
	  buf[i++] = '\\';
	  buf[i++] = c;
	}
	else
	  buf[i++] = c;
	c = lines[top]->get_char();
	buf[i++] = c;
	if (c != '\'') {
	  line_number = get_line_number();
	  col_number = get_line_position();
	  fprintf(error_file, "Warning :: line %d col %d - single quote expected\n",
		  line_number, col_number);
	}
	break;
      }
    case '\\' :
      {
	/* buf[i++] = c; */
	c = lines[top]->get_char();
	if (c == ']') {
	  delete buf;
	  delete name_buf;
	  return "end_of_file";
	}
	if (c == '\n') {
	  get_next_line();
	}
	else 
	  if (c == EOF_CHAR) {
	    line_number = get_line_number();
	    col_number = get_line_position();
	    fprintf(error_file, "Error :: line %d col %d",
		    line_number, col_number);
	    fprintf(error_file, " -unexpected end of file reached \n");
	    return NULL;
	  }
	buf[i++] = '\\';
	buf[i++] = c;
	break;
      }
    case '(' : 
      {
	int cc = lines[top]->get_char();
	int line_number = get_line_number();
	int col_number = get_line_position();
	/* lines[top]->unget_char(cc); */
	buf[i] = '\0'; /* to get the length only */
	if (cc == '$')
	  {
	    char *s = NULL;
	    type_name t = check_type(lines[top], s);	
	    if (t != TYPE_NONE)  
	      {
		/* it is a valid type */
		cc = lines[top]->get_char();
		if (cc == ')') 
		  {
		    cc = lines[top]->get_char();
		    /* lines[top]->unget_char(cc); */
		    if (cc == '$')
		      {
			char tempbuf[ARRAY_MAX];
			char *str = scan_c_var(lines[top], t);
			//name_buf[name_i++] = ',';
			switch (t) 
			  {
			  case TYPE_CHAR :
			    buf[i++] = '$';
			    break;
			  case TYPE_DOUBLE :
			    buf[i++] = '$';
			    break;
			  case TYPE_FLOAT :
			    buf[i++] = '$';
			    break;
			  case TYPE_INT :
			    buf[i++] = '$';
			    break;
			  case TYPE_LONG :
			    buf[i++] = '$';
			    break;
			  case TYPE_SHORT :
			    buf[i++] = '$';
			    break;
			  case TYPE_STRING :
			    buf[i++] = '$';
			    break;
			  case TYPE_UNSIGNED :
			    buf[i++] = '$';
			    break;
			  case TYPE_FILENAME :
			    buf[i++] = '$';
			    break;

			    default :
			      fprintf(error_file, "type returned by check_type");
			      fprintf(error_file, "is not a valid type\n");
			    break;
			  }

			CUR_TYPE = t;
                        CUR_VARNAME = str;
			(CUR_INFO.no_of_params)++;
		      }
		    else {
		      lines[top]->unget_char(cc);
		      int buflen = strlen(buf);
		      int slen = strlen(s);
		      if ((buflen + slen + 4) >= max) 
			{
			  char *new_buf = new (char [max * 2]);
			  
			  if (new_buf == NULL)
			    fatal_error ("new failed in scan_id_keyword\n");
			  strncpy (new_buf, buf, buflen);
			  delete (buf);
			  buf = new_buf;
			  max *= 2;
			}
		      buf[i++] = '(';
		      buf[i++] = '$';
		      strncpy(buf+i, s, slen);
		      i += slen;
		      buf[i++] = ')';
		    }
		  }
		else {
		  lines[top]->unget_char(cc);
		  /* valid type but not ended with a )
		   * copy (, $ the string s 
		   */
		  fprintf(error_file, "Error :: line %d col %d ",
			  line_number, col_number);
		  fprintf(error_file, "- non coral var should be preceded by");
		  fprintf(error_file, "its type or this type is not a valid type\n");
		  
		  int buflen = strlen(buf);
		  int slen = strlen(s);
		  if ((buflen + slen + 2) >= max) 
		    {
		      char *new_buf = new (char [max * 2]);
		      
		      if (new_buf == NULL)
			fatal_error ("new failed in scan_id_keyword\n");
		      strncpy (new_buf, buf, buflen);
		      delete (buf);
		      buf = new_buf;
		      max *= 2;
		    }
		  buf[i++] = '(';
		  buf[i++] = '$';
		  strncpy(buf+i, s, slen);
		  i += slen;
		  /* buf[i++] = cc; */
		  
		}
	      }
	    else {
	      /* not a valid type
		 // copy (, $ the string s and then break
		 */
	      fprintf(error_file, "Error :: line %d col %d ",
		      line_number, col_number);
	      fprintf(error_file, "- non coral var should be preceded by");
	      fprintf(error_file, "its type or this type is not a valid type\n");
	      
	      int buflen = strlen(buf);
	      int slen = strlen(s);
	      if ((buflen + slen + 2) >= max) 
		{
		  char *new_buf = new (char [max * 2]);
		  
		  if (new_buf == NULL)
		    fatal_error ("new failed in scan_id_keyword\n");
		  strncpy (new_buf, buf, buflen);
		  delete (buf);
		  buf = new_buf;
		  max *= 2;
		}
	      buf[i++] = '(';
	      buf[i++] = '$';
	      strncpy(buf+i, s, slen);
	      i += slen;
	    }
	  }
	else {
	  lines[top]->unget_char(cc);
	  buf[i++] = c;
	}
	break;
      }
      
    case '\n' :
      {
	buf[i++] = ' ';
	buf[i] = '\0';
	name_buf[name_i] = '\0';
	coral_file_dump_fn->append("  fprintf(fd, \"%s\n\",");
	coral_file_dump_fn->append(buf);
	coral_file_dump_fn->append("); \n");
	return NULL;
      }
      
    case '.' :
      {
	buf[i++] = c;
	int cc = lines[top]->get_char();
	while ((cc == ' ') || (cc == '\t'))
	  cc = lines[top]->get_char();
	if ((cc != '\n') && (cc != EOF_CHAR)) {
	  lines[top]->unget_char(cc);
	  break;
	}
	buf[i++] = ' ';
       	buf[i] = '\0';
	name_buf[name_i] = '\0';
	coral_file_dump_fn->append("  fprintf(fd, \"%s\\n\", \"");
	coral_file_dump_fn->append(buf);
	coral_file_dump_fn->append("\"); \n");
	return NULL;
      }
      
    case '$' :
      {
	c = lines[top]->get_char();
	lines[top]->unget_char(c);
	if (('a' <= c && c <= 'z') || 
	    ('A' <= c && c <= 'Z') || 
	    (c == '_')) {
	  buf[i] = '\0';
	  line_number = get_line_number();
	  col_number = get_line_position();
	  fprintf(error_file, "Error :: line %d col %d ",
		  line_number, col_number);
	  fprintf(error_file, "- non coral var should be preceded by");
	  fprintf(error_file, "its type or this type is not a valid type\n");
	  char *var_name = get_var_name(lines[top]);
	  int buflen = strlen(buf);
	  int slen = strlen(var_name);
	  if ((buflen + slen) >= max) 
	    {
	      char *new_buf = new (char [max * 2]);
	      
	      if (new_buf == NULL)
		fatal_error ("new failed in scan_id_keyword\n");
	      strncpy (new_buf, buf, buflen);
	      delete (buf);
	      buf = new_buf;
	      max *= 2;
	    }
	  strncpy(buf+i, var_name, slen);
	  i += slen;
	  delete var_name;
	}
	else		
	  buf[i++] = '$';
	break;
      }
      
      default :
	{
	  if (('a' <= c && c <= 'z') || 
	      ('A' <= c && c <= 'Z') || 
	      (c == '_')) {
	    lines[top]->unget_char(c);
	    buf[i] = '\0';
	    char *var_name = get_var_name(lines[top]);
	    int buflen = strlen(buf);
	    int slen = strlen(var_name);
	    if ((buflen + slen) >= max) 
	      {
		char *new_buf = new (char [max * 2]);
		
		if (new_buf == NULL)
		  fatal_error ("new failed in scan_id_keyword\n");
		strncpy (new_buf, buf, buflen);
		delete (buf);
		buf = new_buf;
		max *= 2;
	      }
	    strncpy(buf+i, var_name, slen);
	    i += slen;
	    delete var_name;
	  }
	  else		
	    buf[i++] = c;
	  break;
	}
    }
    
    if ((i + NUM_OF_INCR) >= max)
      {
	char *new_buf = new (char [max * 2]);
	
	if (new_buf == NULL)
	  fatal_error ("new failed in scan_id_keyword\n");
	strncpy (new_buf, buf, i);
	delete (buf);
	buf = new_buf;
	max *= 2;
      }
    c = lines[top]->get_char();
  }
  
}

/* returns the next expression (argument) whose end is 
 * detected by the presence of a comma or closing parenthesis
 * It takes into account comments, nested brackets etc.
 */

token * 
 scan_input:: get_arg()
{
  token *t;
  int max = ARRAY_MAX;
  char *buf = new char[max];
  int i = 0;
  int num_of_paren = 0;	/* no of ( */
  int num_of_bracket = 0;	/* no of [ */
  int num_of_brases = 0;  	/* no of curly paren */
  int inside_string = 0;
  int c;
  
  if (buf == NULL)
    fatal_error ("new failed in get_arg\n");
  buf[0] = '\0';
  c = lines[top]->get_char();
  while (1) {
    switch (c) {
      
    case '\n':
      get_next_line();
      break;
    case ',' :
      if (i == 0)
	return (new token (LX_COMMA));
    case ')' :
      if (i == 0)
	return (new token (LX_RPAREN));
      if ((num_of_paren <= 0) && 
	  (num_of_bracket <= 0) && 
	  (num_of_brases <= 0))  {
	
	lines[top]->unget_char (c); /* put back the , */
	buf [i] = '\0';
	
	int j = check_var_or_expr(buf);
	if (j) {
	  /* it is a variable */
	  
	  /*
	    kw = lookup_keyword (buf);
	    if (kw == COR_RELATION_VAR)
	    t = new token(COR_RELATION_VAR, copy_string(buf));
	    else
	    if (kw == COR_TUPLE_VAR)
	    t = new token(COR_TUPLE_VAR, copy_string(buf));
	    else
	    if (kw != NOT_KEYWORD)
	    t = new token (kw);
	    else
	    */
	  
	  if (coral_var(buf))
	    t = new token(LX_COR_IDENTIFIER, copy_string(buf));
	  else
	    t = new token (LX_IDENTIFIER, copy_string (buf));
	  
	}
	else {
	  t = new token(LX_EXPR, copy_string(buf));
	}
	
	free(buf);
	return t;
      }
      
      buf[i++] = c;
      if (c == ')')
	--num_of_paren;
      break;
      
    case '(' :
      num_of_paren++;
      buf[i++] = c;
      break;
    case '{' :
      num_of_brases++;
      buf[i++] = c;
      break;
    case '[' :
      num_of_bracket++;
      buf[i++] = c;
      break;
    case '}' :
      num_of_brases--;
      buf[i++] = c;
      break;
    case ']' :
      num_of_bracket--;
      buf[i++] = c;
      break;
    case '"' : 
      inside_string = 1;
      buf[i++] = c;
      while (inside_string) {
	c = lines[top]->get_char();
	if (c == '\n') 
	  get_next_line();
	else 
	  if (c == EOF_CHAR) {
	    fprintf(error_file, "unexpected end of file reached \n");
	    return NULL;
	  }
	  else 
	    if (c == '"') {
	      if (buf[i - 1] == '\\')
		buf[i++] = c;
	      else {
		/* end of string */
		inside_string = 0;
		buf[i++] = c;
	      }
	    }
	    else {
	      buf[i++] = c;
	      
	    }
	if (i == max)
	  {
	    char *new_buf = new (char [max * 2]);
	    
	    if (new_buf == NULL)
	      fatal_error ("new failed in scan_id_keyword\n");
	    strncpy (new_buf, buf, max);
	    delete (buf);
	    buf = new_buf;
	    max *= 2;
	  }
      }
      break;
    case '/' :
      /* could be a comment */
      c = lines[top]->get_char();
      if (c == '/') {
	/* C++ style comment */
	while (c != '\n') 
	  c = lines[top]->get_char();
	get_next_line();
      }
      else 
	if (c == '*') {
	  /* C style comment */
	  int loop = 1;
	  while (loop) {
	    c = lines[top]->get_char();
	    if (c == '\n') 
	      get_next_line();
	    else 
	      if (c == EOF_CHAR) {
		fprintf(error_file, "unexpected end of file reached \n");
		return NULL;
	      }
	      else 
		if (c == '*') {
		  c = lines[top]->get_char();
		  if (c == '/') {
		    /* end of comment */
		    loop = 0;
		  }
		  else
		    lines[top]->unget_char(c);
		  
		  
		}
	  }
	}
	else {
	  buf[i++] = c;
	}
      break;
    case '\'' :
      fprintf(error_file, "warning : get_arg :: still have to take care of");
      fprintf(error_file, "array out of bound for quotes\n");
      buf[i++] = c;
      c = lines[top]->get_char();
      if (c == '\n') 
	get_next_line();
      else 
	if (c == EOF_CHAR) {
	  fprintf(error_file, "unexpected end of file reached \n");
	  return NULL;
	}
	else 
	  if (c == '\\') {
	    buf[i++] = c;
	    c = lines[top]->get_char();
	  }
      buf[i++] = c;
      c = lines[top]->get_char();
      buf[i++] = c;
      if (c != '\'')
	fprintf(error_file, "warning : something wrong in single quotes \n");
      break;
    case '\\' :
      fprintf(error_file, "warning : get_arg :: still have to take care of");
      fprintf(error_file, "array out of bound for backslash\n");
      buf[i++] = c;
      c = lines[top]->get_char();
      if (c == '\n') 
	get_next_line();
      else 
	if (c == EOF_CHAR) {
	  fprintf(error_file, "unexpected end of file reached \n");
	  return NULL;
	}
      buf[i++] = c;
      break;
      
      default :
	buf[i++] = c;
      /* c = lines[top]->get_char(); */
      break;
      
    }
    if (i == max)
      {
	char *new_buf = new (char [max * 2]);
	
	if (new_buf == NULL)
	  fatal_error ("new failed in scan_id_keyword\n");
	strncpy (new_buf, buf, max);
	delete (buf);
	buf = new_buf;
	max *= 2;
      }
    c = lines[top]->get_char();
  }
  
}


token *
 scan_input::get_token()
{
  
  if (eof_reached) 
    return (new token (LX_EOF));
  
  if (eoln_reached) 
    return (new token (LX_EOLN));
  
  if (top < 0) {
    fprintf(error_file, "get_token error - no input \n");
    fprintf(error_file, "call get_line (??) first \n");
    return (new token (LX_NOLINE));
  }
  
  if (lines[top] == NULL) {
    fprintf(error_file, "get_token error - no top input line %d \n", top);
    fprintf(error_file, "call get_line (??) first \n");
    return (new token (LX_NOLINE));
  }
  
  if (next_token) {
    token * current_token = next_token;
    next_token = NULL;
    return current_token; 
  }
  
  int c = lines[top]->get_char();
  while (1)
    {
      switch (c)
	{
	case EOF_CHAR:
	  eof_reached = TRUE;
	  return (new token (LX_EOF));
	  
	case '"':		// String
	  return (scan_string (lines[top], c));
	  
	case '(':		/* ( */
	  return (new token (LX_LPAREN));
	  
	case ')':		/* ) */
	  return (new token (LX_RPAREN));
	  
	case '[':		/* [ */
	  return (new token (LX_LBRACKET));
	  
	case ']':		/* ] */
	  return (new token (LX_RBRACKET));
	  
	case '.':		// . 
	  return (new token (LX_DOT));
	  
	case ';':		/* ; */
	  return (new token (LX_SEMICOLON));
	  
	case ':':		/* ; */
	  return (new token (LX_COLON));
	  
	case ',':		/* , */
	  return (new token (LX_COMMA));
	  
	case '/':		/* div or comment */
	  {
	    int cc = lines[top]->get_char ();
	    int ccc;
	    
	    if (cc == '/')
	      {
		while (cc != '\n' && cc != EOF_CHAR) cc = lines[top]->get_char ();
		c = cc;
		break;		// Get next token 
		}
	    
	    if (cc == '*')
	      {
		cc = 'a';
		ccc = 'b';
		while ((cc != EOF_CHAR) && (ccc != EOF_CHAR) 
		       && (cc != '*') && (ccc != '/')) { 
		  ccc = 'b';
		  while (cc != '*' && cc != EOF_CHAR) {
		    if (cc == EOLN_CHAR) {
		      /* PRAVEEN : This was causing some wierd problems
			 with comments, so i'm getting rid of this. Now
			 some of the original comments do not make it
			 to the output, but thats okay !

			dump();     
		      */
			get_next_line(1);
		      }
		    cc = lines[top]->get_char ();
		  }
		  c = cc;
		  if (cc == '*') {
		    ccc = lines[top]->get_char ();
		    c = ccc;
		  }
		}
		if ((cc != EOF_CHAR) && (ccc != EOF_CHAR)) { 
		  c = lines[top]->get_char();
		}
		break;		// Get next token 
		}
	    
	    lines[top]->unget_char (cc);
	    return (new token (LX_SLASH));
	  }
	  
	case '\\':
	  {
	    int cc = lines[top]->get_char ();
	    if (cc == '[')
	      return (new token (CORAL_BEGIN));
	    if (cc == ']')
	      return (new token (CORAL_END));
	    
	    lines[top]->unget_char (cc);
	    return (new token (LX_BACKSLASH));
	  }
	  
	case '+':		// + 
	  return (new token (LX_PLUS));
	  
	case '-':		/* - or -- or negative integer*/
	  {
	    int cc = lines[top]->get_char ();
	    if (cc == '>') {
	      return (new token(LX_ARROW));
	    }
	    
	    
	    if (cc == '-')
	      {
		while (cc != '\n' && cc != EOF_CHAR) cc = lines[top]->get_char ();
		c = cc;
		break;		// Get next token 
		}
	    lines[top]->unget_char (cc);
	    if (!('0' <= cc && cc <= '9'))
	      return (new token (LX_MINUS));
	    // else fall-through 
	    }
	  
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  return (scan_int (lines[top], c));
	  
	case '*':		/* * */
	  return (new token (LX_STAR));
	  
	case '=':		/* = */
	  return (new token (LX_EQ));
	  
	case '#':		/* # */
	  return (new token (LX_HASH));
	  
	case '<':		/* < */
	  return (new token (LX_LEFTARROW));
	  
	case '>':		/* > */
	  return (new token (LX_RIGHTARROW));
	  
	case '{':		/* { */
	  return (new token (LX_LEFTBRACE));
	  
	case '}':		/* } */
	  return (new token (LX_RIGHTBRACE));
	  
	  
	case '!':               /* ! */
	  return (new token (LX_EXCLAMATION));

	case '&':               /* & */
	  return (new token (LX_AMPERSAND));
	  
	case ' ':		/* White space */
	case '\t':
	case '\f':
	  c = lines[top]->get_char ();
	  break;
	case '\n':
	  eoln_reached = TRUE;
	  return (new token (LX_EOLN));
	  
	case '$':		/* $ */
	  {
	    int cc = lines[top]->get_char ();
	    lines[top]->unget_char (cc);
	    if (('a' <= cc && cc <= 'z')
		|| ('A' <= cc && cc <= 'Z')
		|| cc == '_')
	      return (scan_id_keyword (lines[top], c));
	    
	    return (new token (LX_DOLLAR));
	  }
	  
	  
	default:		/* Identifier, keywords, garbage */
	  if (('a' <= c && c <= 'z')
	      || ('A' <= c && c <= 'Z')
	      || c == '_')
	    return (scan_id_keyword (lines[top], c));
	  //fprintf(error_file, "Unknown character %c ignored", c);
	  c = lines[top]->get_char ();
	}
    }
}


token *
 scan_input::get_next_token()
{
  
  if (eof_reached) 
    return (new token (LX_EOF));
  if (eoln_reached) 
    get_next_line();
  token * current_token = get_token();
  if (current_token->token_type() == LX_EOLN)
    return get_next_token();
  return current_token;
}

int
 scan_input::unget_token(token *t)
{
  if (next_token) {
    fprintf(error_file, "cannot put back more than 1 token \n");
    return 0;
  }
  next_token = t;
  return 1;
}



/* Complete scanning a string whose opening " delimiter has been seen.
   Return the appropriate token. There is no limit on string size beyond
   the normal memory limits. */

static token *
  scan_string (LINE *line_info, int c)
{
  token *t;
  int max = ARRAY_MAX;
  char *buf = new char[max];
  int i = 0;
  
  if (buf == NULL) {
    fprintf(error_file, "new failed in scan_string\n");
    exit(0);
  }
  buf[0] = '\0';
  c = line_info->get_char ();
  while (c != '"' && c != '\n' && c != EOF_CHAR)
    {
      buf[i++] = c;
      if (i == max)
	{
	  char *new_buf = new (char [max * 2]);
	  
	  if (new_buf == NULL)
	    fatal_error ("new failed in scan_string\n");
	  strncpy (new_buf, buf, max);
	  delete (buf);
	  buf = new_buf;
	  max *= 2;
	}
      c = line_info->get_char ();
    }
  if (c != '"')
    fprintf(error_file, "Unterminated string");
  buf [i] = '\0';
  t = new token (LX_STRING, copy_string (buf));
  delete (buf);
  return (t);
}


/* Complete scanning an integer whose leading digit has been seen.
   Return the appropriate token. Does NOT handle overflow at all. */

static token *
  scan_int (LINE *line_info, int c)
{
  token *t;
  int max = ARRAY_MAX;
  char *buf = new (char [max]);
  int i = 0;
  
  if (buf == NULL)
    fatal_error ("new failed in scan_int\n");
  buf[0] = '\0';
  do
    {
      buf[i++] = c;
      if (i == max)
	{
	  char *new_buf = new (char [max * 2]);
	  
	  if (new_buf == NULL)
	    fatal_error ("new failed in scan_int\n");
	  strncpy (new_buf, buf, max);
	  delete (buf);
	  buf = new_buf;
	  max *= 2;
	}
      c = line_info->get_char ();
    } while ('0' <= c && c <= '9');
  line_info->unget_char (c);
  buf [i] = '\0';
  t = new token (LX_INTEGER, atoi (buf));
  free (buf);
  return (t);
}


/* Complete scanning an identifier or keyword whose leading character
   has been seen.  Return the appropriate token. There are no limits on
   identifier length, beyond the normal memory limits. */

static token *
  scan_id_keyword (LINE *line_info, int c)
{
  token *t;
  int max = ARRAY_MAX;
  char *buf = new (char [max]);
  int i = 0;
  lexeme_type kw;
  
  if (buf == NULL)
    fatal_error ("new failed in scan_id_keyword\n");
  buf[0] = '\0';
  do
    {
      buf[i++] = c;
      if (i == max)
	{
	  char *new_buf = new (char [max * 2]);
	  
	  if (new_buf == NULL)
	    fatal_error ("new failed in scan_id_keyword\n");
	  strncpy (new_buf, buf, max);
	  delete (buf);
	  buf = new_buf;
	  max *= 2;
	}
      c = line_info->get_char ();
    } while (('a' <= c && c <= 'z')
	     || ('A' <= c && c <= 'Z')
	     || ('0' <= c && c <= '9')
	     || c == '_');
  line_info->unget_char (c);
  buf [i] = '\0';
  kw = lookup_keyword (buf);
  if (kw == COR_RELATION_VAR)
    t = new token(COR_RELATION_VAR, copy_string(buf));
  else
    if (kw == COR_TUPLE_VAR)
      t = new token(COR_TUPLE_VAR, copy_string(buf));
    else
      if (kw != NOT_KEYWORD)
	t = new token (kw);
      else
	t = new token (LX_IDENTIFIER, copy_string (buf));
  free (buf);
  return (t);
}

static char *
  get_var_name(LINE *line_info)
{
  int max = ARRAY_MAX;
  char *buf = new (char [max]);
  int i = 0;
  int c;
  
  if (buf == NULL)
    fatal_error ("new failed in scan_id_keyword\n");
  buf[0] = '\0';
  c = line_info->get_char ();
  do
    {
      buf[i++] = c;
      if (i == max)
	{
	  char *new_buf = new (char [max * 2]);
	  
	  if (new_buf == NULL)
	    fatal_error ("new failed in scan_id_keyword\n");
	  strncpy (new_buf, buf, max);
	  delete (buf);
	  buf = new_buf;
	  max *= 2;
	}
      c = line_info->get_char ();
    } while (('a' <= c && c <= 'z')
	     || ('A' <= c && c <= 'Z')
	     || ('0' <= c && c <= '9')
	     || c == '_');
  line_info->unget_char (c);
  buf [i] = '\0';
  return (buf);
}



static char *
  scan_c_var(LINE *line_info, type_name lx)
{
  int max = ARRAY_MAX;
  char *buf = new (char [max]);
  int i = 0;
  int c;
  
  if (buf == NULL)
    fatal_error ("new failed in scan_id_keyword\n");
  buf[0] = '\0';
  c = line_info->get_char ();
  if (lx == TYPE_FILENAME) {
    do
      {
	buf[i++] = c;
	if (i == max)
	  {
	    char *new_buf = new (char [max * 2]);
	    
	    if (new_buf == NULL)
	      fatal_error ("new failed in scan_id_keyword\n");
	    strncpy (new_buf, buf, max);
	    delete (buf);
	    buf = new_buf;
	    max *= 2;
	  }
	c = line_info->get_char ();
      } while (('a' <= c && c <= 'z')
	       || ('A' <= c && c <= 'Z')
	       || ('0' <= c && c <= '9')
	       || c == '_'
	       || c == '-'
	       || c == '.');
  }
  else {
    do
      {
	buf[i++] = c;
	if (i == max)
	  {
	    char *new_buf = new (char [max * 2]);
	    
	    if (new_buf == NULL)
	      fatal_error ("new failed in scan_id_keyword\n");
	    strncpy (new_buf, buf, max);
	    delete (buf);
	    buf = new_buf;
	    max *= 2;
	  }
	c = line_info->get_char ();
      } while (('a' <= c && c <= 'z')
	       || ('A' <= c && c <= 'Z')
	       || ('0' <= c && c <= '9')
	       || c == '_');
  }
  line_info->unget_char (c);
  buf [i] = '\0';
  return (buf);
}

static type_name 
  check_type(LINE *line_info, char* &s)
{
  int max = ARRAY_MAX;
  char *buf = new char[max];
  int i = 0;
  type_name lx;
  int c = line_info->get_char ();
  if (buf == NULL)
    fatal_error ("new failed in scan_id_keyword\n");
  buf[0] = '\0';
  do
    {
      buf[i++] = c;
      if (i == max)
	{
	  char *new_buf = new (char [max * 2]);
	  
	  if (new_buf == NULL)
	    fatal_error ("new failed in scan_id_keyword\n");
	  strncpy (new_buf, buf, max);
	  delete (buf);
	  buf = new_buf;
	  max *= 2;
	}
      c = line_info->get_char ();
    } while (('a' <= c && c <= 'z')
	     || ('A' <= c && c <= 'Z')
	     || ('0' <= c && c <= '9')
	     || c == '_');
  line_info->unget_char (c);
  buf [i] = '\0';
  lx = lookup_type(buf);
  s = buf;
  return lx;
}

/* Lookup a string in the sorted table of keywords.  If the string is
   the name of a keyword, return its lexeme_type.  If not, return
   NOT_KEYWORD. */

static lexeme_type
  lookup_keyword (char *id)
{
  int low = 0;
  int hi = sizeof (keywords) / sizeof (kw_record) - 1;
  
  while (low <= hi)
    {
      int mid = (low + hi) / 2;
      int cmp = strcmp (id, keywords[mid].name);
      
      if (cmp < 0)
	hi = mid - 1;
      else if (cmp == 0)
	return (keywords[mid].lx);
      else
	low = mid + 1;
    }
  /*
    // check if it is a Relation var
    
    // if (lookup_relation_var(id, relation_var_list))
    // 	return COR_RELATION_VAR;
    
    // check if it is a Tuple var
    // if (lookup_tuple_var(id, tuple_var_list))
    // 	return COR_TUPLE_VAR;
    */
  return (NOT_KEYWORD);
}


static type_name lookup_type(char *id)
{
  int low = 0;
  int hi = sizeof (typestrings) / sizeof (type_record) - 1;
  
  while (low <= hi)
    {
      int mid = (low + hi) / 2;
      int cmp = strcmp (id, typestrings[mid].name);
      
      if (cmp < 0)
	hi = mid - 1;
      else if (cmp == 0) {
	return (typestrings[mid].lx);
      }
      else
	low = mid + 1;
    }
  
  return (TYPE_NONE);
}

void fatal_error(char *s) {
  
  fprintf(error_file, "%s \n", s);
  exit(1);
}


int check_var_or_expr(char *buf) 
{
  /* return 1 for variable
     return 0 for expr 
     */
  
  int i = 0;
  int c; 
  
  if (buf == NULL)
    return 0;
  
  c = buf[0];
  if (('a' <= c && c <= 'z')
      || ('A' <= c && c <= 'Z')
      || (c == '_') 
      || (c == '$')) {
    while ((c = buf[++i]) != '\0') {
      if   (('a' <= c && c <= 'z')
	    || ('A' <= c && c <= 'Z')
	    || ('0' <= c && c <= '9')
	    || c == '_') {
      }
      else
	return 0;
    }
    return 1;
  }
  
  else 
    return 0;
  
}



int coral_var(char *buf) {
  
  /* return 1 for coral variable
     else return 0 
     */
  
  int i = 0;
  int c; 
  
  if (buf == NULL)
    return 0;
  
  c = buf[0];
  if (c == '$') {
    while ((c = buf[++i]) != '\0') {
      if   (('a' <= c && c <= 'z')
	    || ('A' <= c && c <= 'Z')
	    || ('0' <= c && c <= '9')
	    || c == '_') {
      }
      else
	return 0;
    }
    return 1;
  }
  
  else 
    return 0;
  
}

