/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************

	CORAL Software :: U.W.Madison

	term.h : Header file

	Declarations for manipulating terms, tuples and variable bindings.

	Contains the following class/type  declarations	 :
	  Term
	  Tuple
	  BindEnv
	  Trail
	  StackMark
	  TupleLink

 ***********************************************************************/

#ifndef CORAL_TERM_H
#define CORAL_TERM_H

#include "arg.h"

extern int C_NewGArraySize, C_DeleteGArraySize;
#define ceil_div(size, incr) ((size+incr-1)/incr)

// A bindenv is a set of substitutions, mapping variables to Terms.
// Variables in a Clause are numbered sequentially,
// with each variable's number stored in the 'var' field of a VarArg.
// A binding can be implemented as an array, mapping
// 'var' numbers to Terms.

typedef struct BaVarPair TrailElement ;
class Term;
class BindEnv;
class ArrayBindEnv;
class VersionedBindEnv;
class GArrayBindEnv;
class Trail;
class StackMark;
extern class Relation;
extern class BitVector;
extern class ModuleInfo;
extern class ExecutionEnv ;
extern Term GlobalNewt;

extern Term NullTerm ;
extern Trail trail ;
/* points to the current moduleInfo defaults */
extern ModuleInfo *CurrModuleInfo;
extern ExecutionEnv exEnv ;


#define NOBINDENVHACK

#define USE_TRIES 

//
// A Term is implemented using a variant of structure sharing (though this 
// may change):
// The 'expr' field contains the static structure of the term,
// while 'bindenv' maps (some of) the variables in 'expr' to terms.
// Variables in these `terms' are not interpreted in BindEnv, unlike what is
// done in Boyer and Moore structure sharing.
//

class Term {
  public:
                          /* arg  is NULL if unbound ??? */
    BindEnv *bindenv;     /* context for variables in u.arg */
    Arg *expr;

    inline Term() { expr = NULL; bindenv = NULL; } 
    inline Term (Arg *expr1, BindEnv *bindenv1 = NULL) {
	expr = expr1;
	bindenv = bindenv1; }

    inline void printon(FILE*file) const
	{ if (expr) expr->print(bindenv, file); else fputc('_', file); }

    // Assuming 'expr' is a VarArg*, get its bindings.
    inline Term dereference_var() const ; 
    inline Variable varnum() 
	{ return (((VarArg*)expr)->var) ;}
    Term dereference();
    inline int hash() const { return expr->hash(bindenv); }
    inline arg_kind kindof() { return expr->kindof(); }
    inline int isConstant()
	{ return (expr->isConstant());}
};

/***************  MACROS *******************************************/


/*
 * The macro ARITH_DEREFERENCE_TERM overwrites its argument.  
 *	Use it carefully.
 *  It takes as argument a term, and if it is an arithmetic expression
 *  it is simplified.
 */

#define ARITH_DEREFERENCE_TERM(term) {                                 \
	   TermLink *renamed_vars = NULL;                                  \
	   Term newt;      						\
	   newt.expr = (term).expr->simplify((term).bindenv, renamed_vars, \
			(term).bindenv, NULL);                          \
	   newt.bindenv = NULL;			    			\
	   if (newt.expr != NULL) 					\
		(term) = newt;						\
	}

/*
 * The macro FULL_DEREFERENCE_TERM overwrites its argument.  
 *	Use it carefully.
 *  It takes as argument a term, and dereferences the term fully, till the 
 *  term is either a free variable or a non-variable structure or a constant.
 *  Note that this merely chases down chains of variable bindings. It does NOT
 * go into structures.
 */
/*******
#define FULL_DEREFERENCE_TERM(term) { 					\
     if ((term).expr) {                                                 \
	Term newt;      						\
	if (CurrModuleInfo) {                                           \
	  if (CurrModuleInfo->NonGroundFacts)                           \
	    while ((term).bindenv                                       \
	       && (term).expr->kindof() == COR_VARIABLE) {	        \
	      newt = (term).bindenv->lookup((((VarArg*)((term).expr))->var));\
	      if (newt.expr) 					        \
	        (term) = newt;						\
	      else break;						\
	      }                                                         \
	}	 							\
	else if (exEnv.C_non_ground_facts_default) {                    \
	     while ((term).bindenv                                       \
	       && (term).expr->kindof() == COR_VARIABLE) {	        \
	      newt = (term).bindenv->lookup((((VarArg*)((term).expr))->var));\
	      if (newt.expr) 					        \
	        (term) = newt;						\
	      else break;						\
	      }                                                         \
	}	 							\
	if ((term).expr->kindof() == COR_NUM_CONST &&                   \
	    ((NumArg *) (term).expr)->num_kindof() == COR_ARITH_EXPR)   \
        ARITH_DEREFERENCE_TERM(term)                                    \
     }                                                                  \
     }
******/

#define FULL_DEREFERENCE_TERM(term) { 					\
     if ((term).expr) {                                                 \
        arg_kind temp_kind = (term).expr->kindof();                     \
	while ((term).bindenv                                           \
	       && (temp_kind == COR_VARIABLE)) {	                \
	   GlobalNewt =                                                 \
	     (term).bindenv->lookup((((VarArg*)((term).expr))->var));   \
	   if (GlobalNewt.expr) 				        \
	     (term) = GlobalNewt;			     		\
	   else break;			       				\
	   temp_kind = (term).expr->kindof();                           \
	}								\
	if (temp_kind == COR_NUM_CONST &&                               \
	    ((NumArg *) (term).expr)->num_kindof() == COR_ARITH_EXPR)   \
        ARITH_DEREFERENCE_TERM(term)                                    \
     }                                                                  \
   } 

/*
 * The macro DEEP_DEREFERENCE_TERM overwrites its argument.  
 *	Use it carefully.
 *  It takes as argument a term, and deeply dereferences the term, by replacing
 *  all bound variables by the appropriate constant.
 */

#define DEEP_DEREFERENCE_TERM(term) { 					\
     FULL_DEREFERENCE_TERM(term);                                        \
     if (!term.isConstant() && isGroundHashValue(term.hash())) {          \
        Arg *new_arg;                                                   \
	CopyArg(&new_arg, term.expr, term.bindenv);	                \
	term.expr = new_arg;                                            \
	term.bindenv = NULL;                                            \
      }                                                                 \
   } 


/*
 *  ISFREE_TERM - returns 1 if argument  is a free variable, 0 otherwise 
 */
#define ISFREE_TERM(term)  ((term)->expr->kindof()==COR_VARIABLE &&	\
	((term)->bindenv)->lookup(((VarArg*)((term)->expr))->var).expr == NULL)

/***************  END MACROS *******************************************/

/***  WARNING:
	IF      a bindenv is allocated within a function and deleted on 
			function exit, 
	THEN    1) you MUST create a StackMark first.
		2) before returning from the function you MUST pop back to 
			the stackmark.  Otherwise the trail may contain 
			references to the defunct bindenv, and cause 
			a crash later.
****/

class BindEnv{
	friend class ArrayBindEnv;
	friend class VersionedBindEnv;
	friend class GArrayBindEnv;

    	int _limit;
	int _is_versioned;
    public:

	// The following can be restored as inline functions 
	// if we have no subclass of BindEnv other than ArrayBindEnv
	// 
#ifdef NOBINDENVHACK
	virtual inline Term lookup(Variable var) const = 0;
	virtual inline int bind(Variable var, Term binding)=0;
#else
	inline Term lookup(Variable var) const;
	inline int bind(Variable var, Term binding);
#endif
	virtual BindEnv *make_version()=0;
	virtual BindEnv *make_version_and_swap()=0;
	virtual BindEnv *copy_shell()=0;
	virtual void restore(BindEnv *env)=0;
	virtual void resize(int newsize)=0;
	virtual int all_free()=0;
	inline int max_var_number() { return _limit;}
	inline int is_versioned() { return _is_versioned;}
};

class ArrayBindEnv: public BindEnv{
	friend class BindEnv;
	friend class VersionedBindEnv;
	Term   *bindarray;
	ArrayBindEnv();
    public:
	ArrayBindEnv(int size, int incr=-1);
#ifdef NOBINDENVHACK
	virtual inline Term lookup(Variable var) const
		{return (bindarray[var]);}
	virtual inline int  bind(Variable var, Term binding)
		{bindarray[var]=binding; return 1;}
#endif
	virtual void resize(int newsize);  /* Aborts */
	virtual int all_free();
	virtual BindEnv *make_version();  // returns a new 
		// copy of the bindenv
	virtual BindEnv *make_version_and_swap();
	virtual BindEnv *copy_shell();
	virtual void restore(BindEnv *env);
	virtual ~ArrayBindEnv();	

};

class TrieNode {
    public:
    int height;
    /* NOTE: The following two should be a union, but C++ doesn't like it */
    TrieNode *pointers[2];
    Term      binding;
    TrieNode *copy(); 
};

class VersionedBindEnv: public BindEnv{
	friend class BindEnv;
	friend class ArrayBindEnv;
#ifdef USE_TRIES
	TrieNode   *trie;
#else
	Term   *bindarray;
#endif
	VersionedBindEnv();
    public:
	VersionedBindEnv(int size, int incr=-1);
#ifdef NOBINDENVHACK
	virtual Term lookup(Variable var) const;
	virtual int bind(Variable var, Term binding);
#else
	virtual Term lookupV(Variable var) const;
	virtual int bindV(Variable var, Term binding);
#endif
	virtual int forcebind(Variable var, Term binding);
	virtual void resize(int newsize);  /* Aborts */
	virtual int all_free();
	virtual BindEnv *make_version();  // returns a new version
			// of the bindenv
	virtual BindEnv *make_version_and_swap();
	virtual BindEnv *copy_shell();
	virtual void restore(BindEnv *env);
	virtual ~VersionedBindEnv();	
};

class Term_garray {
  Term **tarray;
  int size;		
  int _limit;		
  int incr; 		
public:			
  Term_garray(int size =128*128, int incr = 128, int tlimit=-1);

  inline Term& operator[](int i)
  {						
    ASSERT(tarray!=NULL && i<_limit && i >= 0);
    if (i >= size) resize(2*size);			
    if (! tarray[i/incr]) { 				
      tarray[i/incr] = new Term[incr];	
      C_NewGArraySize += incr*sizeof(Term);
    }				
    return ( tarray[i/incr][i%incr] );
  }						
  int limit() {return _limit;}		
  int resize(int newsize); 			
  Term_garray *make_copy();		
  int get_next(int i);			
  ~Term_garray();			
};	


class GArrayBindEnv: public BindEnv{
    friend class BindEnv;
    Term_garray  *bindarray;
    GArrayBindEnv();
    public:

    GArrayBindEnv(int size, int incr = 128);

#ifdef NOBINDENVHACK
	virtual inline Term lookup(Variable var) const /* HACK */
		{return ((*bindarray)[var]);}
	virtual inline int  bind(Variable var, Term binding) /* HACK */
		{(*bindarray)[var]=binding; return 1;}
#endif
	virtual void resize(int newsize) {
		bindarray->resize(newsize);
		_limit = newsize;
	}

	virtual int all_free();
	virtual BindEnv *make_version();  // returns a copy of the bindenv
	virtual BindEnv *make_version_and_swap();
	virtual BindEnv *copy_shell();
	virtual void restore(BindEnv *env);
	virtual ~GArrayBindEnv();	

};


struct BaVarPair{
	BindEnv *bindenv;
	Variable var;
	};

// The Trail is a means of allowing environments to be created (during
// rule evaluation) and then discarded when they are not needed any more.
// It is a stack of environments (BindEnv/ Variable) pairs which can
// be pushed to grow it, and popped back to a fixed StackMark, to bring
// the environment back to its old state.
class Trail{
    TrailElement *Stack;
    int Limit;
  public:
    // Size of current Trail stack.
    int TrailSize;

    // Creates a trail with the specified initial sizes.
    Trail(int trailsize);

    // Return specified element of trail
    inline TrailElement& operator [](int i){
      return (Stack[i]);
    }

    // Pop trail to specified size. Clear (in BindArray) variables that are 
    // found on the popped portion of the trail.
    inline void PopTrail(int new_size)
      {
	while (TrailSize > new_size) {
	  TrailSize--;
	  if (Stack[TrailSize].var >= 0) {
	    ((Stack[TrailSize]).bindenv)->bind(Stack[TrailSize].var,
					       NullTerm);
	    /* if ( ! Stack[TrailSize].bindenv->is_versioned)
	     *(*(Stack[TrailSize].bindenv))[Stack[TrailSize].var] = 
	     *                                               NullTerm;
	     * Don't undo bindings for persistent versioned bindenvs! 
	     */
	  }
	  else {
	    delete Stack[TrailSize].bindenv ;
	  }
	  
	}
      }
				
    inline void PushTrail(BindEnv *env, Variable var ) {
      if (TrailSize >= Limit)
	if (Limit == 0) {
	  Limit = 255;
	  Stack = (TrailElement*)malloc(Limit * sizeof(TrailElement));
	}
	else {
	  Limit *= 2;
	  Stack = (TrailElement*)realloc((char *)Stack,
					 Limit * sizeof(TrailElement));
	}
      Stack[TrailSize].bindenv = env;
      Stack[TrailSize].var = var;
      TrailSize++;
    }

    // Push a bindenv on trail; 
    // The argument -1 indicates the whole bindenv is to be 
    // deleted on backtracking.
    inline void PushTrail(BindEnv *env)
      { PushTrail(env, -1);}
  };

/*
 * Currently trail is a global variable.  It should probably be made a 
 * parameter to StackMark, but currently isn't since it is less efficient. 
 */	
class StackMark {
    int trail_size;		// Size of trail when mark is made.
    public:
    // The following form should be used eventually.
    // StackMark(Trail &trail=NULL) { trail_size = trail.TrailSize; }
    
    inline trail_index() { return trail_size; }
    inline StackMark()
	{trail_size = trail.TrailSize;}
    inline void get_mark() 
	{trail_size = trail.TrailSize;}
    inline void pop_to() { 		// Pop the Trail upto stackmark; 
	trail.PopTrail(trail_size) ;
    }
};


extern int C_NewTupleCount, C_NewTupleSize;
extern int C_DeleteTupleCount, C_DeleteTupleSize;

//
// A Tuple has an ArgList and a BindEnv to interpret the ArgList in.
// If the BindEnv is NULL, then we assume that all variables in the tuple are
// free.  
//
class Tuple {
    public:
    unsigned int deleted : 1;
    BindEnv *bindenv;  
    ArgList* _args;             // array of terms. Length==relation's arity
    int env_size;		// Gives number of variables in the tuple---
	                        // WARNING:  set this to -1 if it's not known

    int contid;			//  This field and the next are related to
    int parid;			//  return unification optimizations.

    int ref_count;              // tuple reference count

	
	// Used to be copy_env
    BindEnv *make_bindenv_version();   	
				// Returns a copy of the bindenv, but default
				// bindenvs result in full copies, not default
				// copies.
		
    Tuple()  {_args = NULL; bindenv = NULL; env_size = 0; deleted = 0;
			contid = parid = ref_count = 0;
#ifdef DEBUG
			C_NewTupleCount ++;
			C_NewTupleSize += sizeof(Tuple);
#endif
			}
    Tuple(int arity)  {_args = ArgList::New(arity); bindenv = NULL;
			env_size = 0; deleted = 0;
			contid = parid = ref_count = 0;
#ifdef DEBUG
			C_NewTupleCount ++;
			C_NewTupleSize += sizeof(Tuple); 
#endif
			}

    Tuple(ArgList *arglist, BindEnv *env) { _args = arglist; bindenv = env;
			env_size = 0; deleted = 0;
			contid = parid = ref_count = 0;
#ifdef DEBUG
			C_NewTupleCount ++;
			C_NewTupleSize += sizeof(Tuple);
#endif
			}

    Tuple(ArgList&a) : _args(&a) { bindenv = NULL; env_size = 0; deleted = 0;
			contid = parid = ref_count = 0;
#ifdef DEBUG
			C_NewTupleCount ++;
			C_NewTupleSize += sizeof(Tuple);
#endif
   		 	}

    Tuple(ArgList *arglist) { _args = arglist; bindenv = NULL; 
			env_size = 0; deleted = 0;
			contid = parid = ref_count = 0;
#ifdef DEBUG
			C_NewTupleCount ++;
			C_NewTupleSize += sizeof(Tuple);
#endif
    			}
    ~Tuple()  { 
#ifdef DEBUG
		C_DeleteTupleCount++; C_DeleteTupleSize += sizeof(Tuple);
#endif
		}

    // true if all vars are free, so no bindenv is allocated.
    inline int default_bindenv() const {return (bindenv == NULL);}

    // True if all arguments are ground.
    int isConstant();
    inline void do_delete()  {deleted = 1;}
    inline int is_deleted()  {return deleted;}
    inline ArgList& args() { return *_args;}
    inline int arity() const { return _args->count(); }

    // Returns a term that is the i'th argument of
    // the tuple.
    inline Term *term(int i) const
        { Term *term = new Term ((*_args)[i], bindenv); return term; }
    inline ArgPtr arg(int i) const {return (*_args)[i];}
    ArgPtr& operator [] (int i) const;  // Cannot make this inline !! - Praveen
    Tuple * rename(BindEnv *env);  // Renames variables in tuple to start from
		// after the highest var in env;  also adds new variables to
		// env

    inline int hash( BitVector& bv ) const 
	{ return _args->hash( bv, bindenv ); }
    inline int hash() const
	{ return _args->hash(bindenv); }
    void printon(FILE *file) const;
    void update1(int argnum, ArgList *arg);
#ifdef DO_GC
    GCCLASS(Tuple);
#endif
};

class TupleLink {
  public:
    TupleLink *next;
    Tuple *tuple;

    TupleLink(Tuple *v, TupleLink *n = NULL) 
	     { tuple = v; next = n; }
 };



/*
 * Inline functions defined here.  They cannot be defined earlier since
 * they need to know the size of classes defined later.
 */


inline Term Term::dereference_var() const { 
   ASSERT(expr->kindof()==COR_VARIABLE);
   return bindenv->lookup(((VarArg*)expr)->var); 
 }


/*
 *  The following are from arg.h, but they were moved here since they use 
 *  bindenvs and terms.  Anyone who uses arg.h but not term.h could
 *  get into trouble.
 */

inline Term VarArg::dereference_var(BindEnv* env) { 
				// Does a one level dereference
	 	ASSERT(env!=NULL);
		return env->lookup(var);
   }

inline Term  VarArg::dereference(BindEnv* env) { 	  
				// Does a full dereference
		ASSERT(env!=NULL);
		return env->lookup(var).dereference();
    }

#endif /* CORAL_TERM_H */
