

module nqueens.
export queens(bf), queensa(bf).
@pipelining.

/*  The following nqueens program is due to Thom Fruewirth:

This introduces an n-queens program that is simpler, smaller and an order
of magnitude faster (for n>=8) than optimized standard version as described 
in the book Art of Prolog.

The idea for the algorithm and so its complexity is the same.
However instead of encoding the problem of finding attacking queens
with arithmetic operations, the power of unification and the logical
variable is exploited. This shows once more that "elegance is not optional" 
(as Richard O'Keefe likes to put it).

Oberving that no two queens can be positioned on the same row, column
or diagonals, we place only one queen on each row. Hence we can identify
the queen by its row-number. Now imagine that the chess-board is divided
into three layers, one that deals with attacks on columns and two for the
diagonals going up and down respectively. We indicate that a field is
attacked by a queen by putting the number of the queen there.
Now we solve the problem by looking at one row at a time, placing one queen
on the column and the two diagonal-layers. For the next row/queen we use
the same column layer, to get the new up-diagonals we have to move the 
layer one field up, for the down-diagonals we move the layer one field down.

*/

/* n-queens problem		Thom Fruehwirth 1988 modified 910308 */

% -------- Meaning of Variables ------
% N, M  ... Size of the board
% I, J  ... Number of the row current queen is on
% Qs, L ... List of length N used to represent the solution
% Cs ... Column as a list of fields of length N
% Us ... Up-Diagonal as an open list of fields
% Ds ... Down-Diagonal as an open list of fields

#define is =

% Version with arithmetic increment

queens(N,Qs):- gen_list(N,Qs), place_queens(N,Qs,_,_).

gen_list(0,[]).
gen_list(N,[_|L]):-
	N>0, M is N-1,
	gen_list(M,L).

place_queens(0,_,_,_).
place_queens(I,Cs,Us,[_|Ds]):-
	I>0, J is I-1,
	place_queens(J,Cs,[_|Us],Ds),
	place_queen(I,Cs,Us,Ds).

% place_queen(Queen,Column,Updiagonal,Downdiagonal) places a single queen

	place_queen(I,[I|T1],[I|T2],[I|T3]).
	place_queen(I,[T1|Cs],[T2|Us],[T3|Ds]):-
		place_queen(I,Cs,Us,Ds).



% Art-of-Prolog version	(from Sterling and Shapiro's book)

queensa(N,Qs) :- range(1,N,Ns), queens(Ns,[],Qs).

queens(UnplacedQs,SafeQs,Qs) :-
        select(Q,UnplacedQs,UnplacedQs1),
	notattack2(Q,SafeQs),
        queens(UnplacedQs1,[Q|SafeQs],Qs).
queens(U,Qs,Qs) :- U = [].

select(X,U,L) :- U = [X|L] .
select(Y,U,L3) :- U = [X|L1], select(Y,L1,L2), L3 = [X|L2].

range(M,N,[M|Ns]) :- M<N, M1 is M+1, range(M1,N,Ns).
range(N,N,[N]).

% end-of-nqueens

notattack2(Q,Qs):- notattack3(Q,1,Qs).

notattack3(X,N,U) :- U = [].
notattack3(X,N,U):- U = [Y|Ys], not X=Y, not X = Y+N, not X=Y-N,
			N1 is N+1, notattack3(X,N1,Ys).

end_module.

