/*
 * @(#) plread.c [2.3b] 02-17-91 VS
 *
 * A general Prolog (or similar) expression parser.
 * See the file "plread.doc" for how to use this file with your
 * application.
 *
 * Copyright (C) 1990,1991 V.Siebert
 *
 * History:
 *   90/10/10 - First implementation, deterministic operator stack
 *	 	reduction, cannot handle cases like "+ + +" yet.
 *   90/10/14 - Added backtracking parser. This parser can handle
 *		handle any case of operator order correctly, but
 *		it's terrible slow...
 *   90/10/15 - Enhanced the deterministic parser. This parser can now
 *		e.g. handle any aggregation of "+ + + + + + ...".
 *		This parser can also find the unique correct order
 *		in one pass. So stack reduction is now done in
 *		O(ts->ts_used), where ts is a 'struct TokenStack'...
 *		(removed the backtracking parser again)
 *	      - added define TRACK_VARIABLES, if plread itself should keep
 *		track of already used variables and reuse TERMs returned
 *		for them.
 *	      - increased the speed of the stack reducing loop.
 *		Now it not looks up the `absolute' but the `first local'
 *		lowest precedence operator.
 *   90/11/10 - removed the recursive call in the three parser functions
 *		PARSE_FUNCTION, PARSE_LIST and PARSE_SUBTERM. Now there
 *		is only one function left that is called PARSE_TERM.
 *		On stack-parameter-passing machines, this does not take
 *		much affect, but on systems like the Sparc it acts
 *		much faster because we stay inside one function.
 *		On my machine (Intel 386SX, 16MHz) a test file with
 *		32 lines of s^64(0) terms is parsed 20% faster now.
 *	      - Changed the #define'd constants to enum's for debugging
 *		reasons.
 *   90/11/11 - Made the stack reduction (operator resolution) to run
 *		completely in linear time depending on the number of
 *		tokens on the stack.
 *	      - Changed the stack reduction to avoid unnecessary bcopy's
 *		for operators at the bottom or at the top of the
 *		stack by also changing the "low water mark" of the stack.
 *   90/11/28 - Made the scanner a little bit more intelligent. Now it
 *		depends on the last token emitted, if the current token
 *		can even be a functor. So now 5+(2*3) is parsed correctly,
 *		even without a space between the + and the (. (This is
 *		very important, if `.' is defined as operator and you
 *		have to brace some terms because of operator precedence,
 *		here you cannot add a space between the '.' and the '('
 *		'cause it would mean the end-of-input)
 *	      - Now the scanner produces context error messages like
 *		the parser (just for Michael)
 *   91/02/17 - Introduced two new #define's named C_STYLE_NUMBERS and
 *		C_STYLE_ESCAPES.  With C_STYLE_NUMBERS, octal and hexadecimal
 *		integer constants like in 033 or 0x1b are allowed.
 *		With C_STYLE_ESCAPES, C like escape sequences in strings
 *		(both '' and "") are allowed.
 *	      - Fixed a bug noticed by Andreas Schwab: the parser was
 *		not able to parse [] as the empty list (though I don't
 *		know how nobody recognized this during 2 months ???).
 *   91/02/19 - Rewrote the stack resolving mechanism, was completely
 *		weird.  Now we can parse e.g.:
 *			X :- + 2 == 3	to	(X :- ((+ 2) == 3))
 *		and	X :- + 2 * 3	to	(X :- (+ (2 * 3)))
 *	      - The detection of impossible functors (even if an opening
 *		brace follows), works much better now.  This was essentially
 *		for expressions using the '.' as a binary functor, where
 *		we cannot insert a whitespace between the '.' and the '('.
 *		So we can parse e.g.:
 *			(a).(b).(c).[]	to	[a,b,c]
 *	      - Removed a (hard) bug in `reduce_stack_to_term' : called
 *		MALLOC for N bytes to store N integers;  nobody recognized
 *		this yet since in 99% of all expression you only need
 *		1/4 of the stack.
 */

#include <ctype.h>
#include "prologdef.h"		/* System dependent information */
#include "plstd.h"
#include "plrdwr.h"
#include "plprintf.h"		/* Includes also stdarg.h or varargs.h */

enum TokenType {
    TOK_END,		/* end-of-stack marker */
    TOK_STATE,		/* a pushed parser state */
    TOK_ATOM,		/* an atom or an operator, contents is TO_NAME */
    TOK_QATOM,		/* same as above, but quoted */
    TOK_FUNCTOR,	/* surely a functor, contents in TO_NAME */
    TOK_QFUNCTOR,	/* same as above, but quoted */
    TOK_VAR,		/* a variable, name in TO_NAME */
    TOK_STRING,		/* a string, contents are in TO_NAME */
    TOK_LITERAL,	/* a literal character like ')', ';' ... */
    TOK_INTEGER,	/* a constant integer number */
    TOK_FLOAT,		/* a constant floating point number */
    TOK_TERM		/* an already parsed (sub-)term */
};

enum ParserState {
    PS_SUBTERM,		/* parse a subterm finished by a distinct end-marker */
    PS_LIST_HEAD,	/* parse a list before the '|' */
    PS_LIST_TAIL,	/* parse a list after the '|' */
    PS_FUNCTION		/* parse a function */
};

/*
 * A struct Token is used to represent one entry on the parser stack.
 * This can either be an unsafe (OPERATOR/ATOM ??) token with it's name
 * stored in `to_name' or an already converted TERM with it's contents
 * in `to_term'. `_to_saved_state' is used to protect different levels
 * of terms (i.e. subterms, lists, or arguments) from the things that
 * parsed at the moment...
 */
struct Token {
    enum TokenType to_kind;	/* the kind of token */
#if CONTEXT_ERRORS
    char *to_context;		/* the token's ascii context */
    int to_ctxcount;		/* the # of ascii contexts */
#endif
    union {
	struct {
	    union {
		TERM __to_term;		/* the term representation */
		char *__to_name;	/* the atom/functor name */
	    } _to_contents;
	    int _to_opdef;		/* the possible operator types */
	    int _to_precedence;		/* the operator precedence */
	} _to_token;
	struct {
	    enum ParserState _to_state;	/* the state before last push */
	    int _to_pred;		/* the previous state pointer */
	    int _to_endch;		/* the termination character for subterms */
	} _to_saved_state;
    } to_contents;
};

/*
 * Some shortcuts
 */
#define to_name		to_contents._to_token._to_contents.__to_name
#define to_term		to_contents._to_token._to_contents.__to_term
#define to_opdef	to_contents._to_token._to_opdef
#define to_precedence	to_contents._to_token._to_precedence
#define to_state	to_contents._to_saved_state._to_state
#define to_pred		to_contents._to_saved_state._to_pred
#define to_endch        to_contents._to_saved_state._to_endch

struct TokenStack {
    struct Token *ts_stack;	/* the token-stack */
    int ts_size;		/* the allocated size */
    int ts_used;		/* the used size */
};

#if TRACK_VARIABLES
struct VarEntry {
    char	 *ve_name;		/* the variables name */
    TERM	  ve_term;		/* the variables term repr. */
};

struct VarStack {
    int		  vs_size;		/* the allocated stack size */
    int		  vs_used;		/* the used stack size */
    struct VarEntry *vs_stack;		/* the variable stack */
};

struct VarStack varstack	= { 0, 0, (struct VarEntry *) 0 };

#define allocate_vs() \
    if(varstack.vs_size == 0) \
	varstack.vs_stack = TMALLOC(varstack.vs_size = 32,struct VarEntry)
#define grow_vs() \
    if(varstack.vs_used >= varstack.vs_size) \
	varstack.vs_stack = TREALLOC(varstack.vs_stack,varstack.vs_size += 32,struct VarEntry)
#define reset_vs()	allocate_vs(); varstack.vs_used = 0
#endif

#define PL_SPEC_ATOMS	";!"
#define ATOM1CHAR(c)	((c) >= 'a' && (c) <= 'z')
#define ATOM2CHAR(c)	(index("#$&*+-./:<=>?@\\^`~",c) != NULL)
#define VARCHAR(c)	(((c) >= 'A' && (c) <= 'Z') || (c) == '_')
#define AVCHAR(c)	(ATOM1CHAR(c) || VARCHAR(c) || ((c) >= '0' && (c) <= '9'))
#define HEXCHAR(c)	(((c) >= '0' && (c) <= '9') \
			 || ((c) >= 'A' && (c) <= 'F') \
			 || ((c) >= 'a' && (c) <= 'f'))
#define HEXVAL(c)	(((c) >= '0' && (c) <= '9') \
			 ? ((c) - '0') \
			 : (c)+10 - (((c) >= 'A' && (c) <= 'F') ? 'A' : 'a'))
#define OCTCHAR(c)	((c) >= '0' && (c) <= '7')

#define STORECH(c)	(pread_bufused >= pread_bufsize \
			 ? expand_buffer(c) \
			 : (pread_buffer[pread_bufused++] = (c)))
#define GETRESET()	(savedchcount = 0)
#define GETCH()		(savedchcount \
			 ? savedchars[--savedchcount] \
			 : prolog_getchar())
#define UNGETCH(ch)	(savedchars[savedchcount++] = (ch))

static char	 E_prefix[]	= "prefix";
static char	 E_infix[]	= "infix";
static char	 E_postfix[]	= "postfix";
static char	 E_noneofall[]	= "<???>";

#define OPDEF1(od)	(((od) & OP_PREFIX) \
			 ? E_prefix \
			 : ((od) & OP_INFIX) ? E_infix : E_noneofall)
#define OPDEF2(od)	(((od) & OP_POSTFIX) \
			 ? E_postfix \
			 : ((od) & OP_INFIX) ? E_infix : E_noneofall)

#define grow_ts(ts)	(((ts)->ts_stack = TREALLOC((ts)->ts_stack,(ts)->ts_size += 32,struct Token)),ts)
#define free_ts(ts)	FREE((ts)->ts_stack);FREE(ts)
#define swap_ts(ts)	(ts_next(ts) = ts_top(ts),ts_top(ts) = ts_pickn(ts,2),ts_pickn(ts,2) = ts_pickn(ts,0))
#define swappop_ts(ts)	(ts_pickn(ts,2) = ts_top(ts), (ts)->ts_used--)
#define ts_pick(ts,n)	((ts)->ts_stack[n])
#define ts_pickn(ts,n)	((ts)->ts_stack[(ts)->ts_used-(n)])
#define ts_top(ts)	ts_pickn(ts,1)
#define ts_next(ts)	ts_pick((((ts)->ts_used >= (ts)->ts_size) ? grow_ts(ts) : (ts)),(ts)->ts_used)

#define onebit(b)	((b) != 0 && ((b)|((b)-1)) == (b)+(b)-1)
#define cut_off_opdef(tp,cutmask) \
{ \
  (tp)->to_opdef &= cutmask; \
  if ((tp)->to_opdef == 0) (tp)->to_opdef = OP_ATOM; \
  if ((tp)->to_opdef == OP_ATOM) \
    { \
      create_atom(&(tp)->to_term,(tp)->to_name); \
      (tp)->to_kind = TOK_TERM; \
    } \
}

#if CONTEXT_ERRORS
#  define set_context(d,s)	((d)=(s))
#  define set_ctxcount(d,s)	((d)=(s))
#  define add_ctxcount(d,s)	((d)+=(s))
#else
#  define set_context(d,s)
#  define set_ctxcount(d,s)
#  define add_ctxcount(d,s)
#endif

/*
 * Prototypes of local functions
 */
#if TRACK_VARIABLES
STATIC void lookup_variable PROTO((TERM *,char *));
#endif
#if CONTEXT_ERRORS
STATIC char *print_context PROTO((char *,int));
#endif
#ifndef __STDC__
STATIC int print_scanner_error ();
STATIC int print_error ();
#else
STATIC int print_scanner_error PROTO((char *,char *,...));
STATIC int print_error PROTO((struct TokenStack *,int,int,int,char *,...));
#endif
STATIC int expand_buffer PROTO((int));
STATIC struct TokenStack *allocate_ts PROTO((void));
STATIC int prolog_lex PROTO((void));
STATIC int get_token_to_stack PROTO((struct TokenStack *,int));
STATIC int reduce_stack_to_term PROTO((struct TokenStack *));
STATIC int parse_term PROTO((TERM *));

/*
 * Error messages
 */
#ifdef DEUTSCH
STATIC char	E_read_error[]	 = "Parserfehler: ";
STATIC char	E_in_context[]	 = "im Kontext: ";
STATIC char	E_here[]	 = " <<HIER>>";
STATIC char	E_unexpeof[]	 = "Unerwartetes Dateiende"
STATIC char	E_illchar[]	 = "Unerlaubtes Zeichen (hex 0x%x)";
STATIC char	E_ambigious[]	 = "Der Operator '%s' ist mehrdeutig (%s/%s)";
STATIC char	E_assoc[]	 = "Praezedenzkonflikt der Operatoren '%s' und '%s'";
STATIC char	E_nooperator[]	 = "Unerlaubter oder fehlender Operator";
STATIC char	E_opusage[]	 = "Unerlaubter Gebrauch des %s-Operators '%s'";
STATIC char	E_unclosedexpr[] = "Nicht abgeschlossener Ausdruck";
STATIC char	E_emptyexpr[]	 = "Leerer Ausdruck";
STATIC char	E_syntax[]	 = "Syntaxfehler (Zeichen '%c')";
STATIC char	E_illbrace[]	 = "Die Klammer '%c' wurde nicht geoeffnet";
STATIC char	E_internal[]	 = "Interner Fehler";
#else
STATIC char	E_read_error[]	 = "read error: ";
STATIC char	E_in_context[]	 = "in context: ";
STATIC char	E_here[]	 = " <<HERE>>";
STATIC char	E_unexpeof[]	 = "unexpected end-of-file";
STATIC char	E_illchar[]	 = "illegal character (hex 0x%x)";
STATIC char	E_ambigious[]	 = "operator '%s' is ambigious (%s/%s)";
STATIC char	E_assoc[]	 = "operators '%s' and '%s' have conflicting precedence";
STATIC char	E_nooperator[]	 = "bad or missing operator";
STATIC char	E_opusage[]	 = "invalid use of %s operator '%s'";
STATIC char	E_unclosedexpr[] = "unclosed expression";
STATIC char	E_emptyexpr[]	 = "empty expression";
STATIC char	E_syntax[]	 = "syntax error (char '%c')";
STATIC char	E_illbrace[]	 = "unmatched brace '%c'";
STATIC char	E_internal[]	 = "internal error";
#endif

/*
 * local (but file global) variables
 */
STATIC char		 *pread_buffer	= NULL;	/* the input buffer */
STATIC char		 *pread_bufscan;	/* to scan the buffer */
STATIC int		  pread_bufsize	= 0;	/* the allocated size of 'pread_buffer' */
STATIC int		  pread_bufused	= 0;	/* the used size of 'pread_buffer' */
STATIC char		  savedchars[32];	/* UNGETCH() characters */
STATIC int		  savedchcount	= 0;
#if CONTEXT_ERRORS
STATIC char		 *pread_context;	/* context of last token */
STATIC struct TokenStack *frame_tsp;		/* frame for empty expression context */
#endif
#ifndef OLD_FUNCTORS
STATIC int		  next_is_never_functor;
#endif

/*
 * Utility functions:
 *     read_escaped_char - reads a character that was preceeded
 *			 by a backslash (if C_STYLE_ESCAPES)
 *     lookup_variable - looks up a list for previously used variables
 *     print_context   - print out an errors context
 *     print_error     - print a formatted error message and the
 *			 errors context (if CONTEXT_ERRORS)
 *     expand_buffer   - expands the allocated size for 'pread_buffer'
 *     allocate_ts     - allocates a small initial 'struct TokenStack'
 */
#if C_STYLE_ESCAPES
STATIC int
read_escaped_char ()
{
  int ch;

 again:
  switch (ch = GETCH())
    {
    case 'a':
      ch = '\a';
      break;
    case 'b':
      ch = '\b';
      break;
    case 'E':
      ch = '\033';
      break;
    case 'f':
      ch = '\f';
      break;
    case 'M':
      ch = GETCH ();
      if (ch == '\\')
	ch = read_escaped_char ();
      ch |= 0x180;
      break;
    case 'n':
      ch = '\n';
      break;
    case 'r':
      ch = '\r';
      break;
    case 't':
      ch = '\t';
      break;
    case '\n':
      ch = GETCH ();
      if (ch == '\\')
	goto again;
      break;
    case 'x':
    case 'X':
      {
	int ch2 = GETCH();

	if (HEXCHAR (ch))
	  {
	    ch = HEXVAL (ch);
	    ch2 = GETCH();
	    if (HEXCHAR (ch2))
	      ch = (ch << 16) + HEXVAL (ch2);
	    else
	      UNGETCH (ch2);
	  }
	else
	  UNGETCH (ch2);
	ch |= 0x100;
      }
      break;
    case '0':
    case '1':
    case '2':
    case '3':
      {
	int value = ch - '0';
	    	
	ch = GETCH ();
	if (OCTCHAR (ch))
	  {
	    value = (value << 3) + (ch - '0');
	    ch = GETCH ();
	    if (OCTCHAR (ch))
	      value = (value << 3) + (ch - '0');
	    else
	      UNGETCH (ch);
	  }
	else
	  UNGETCH (ch);
	ch = value | 0x100;
      }
    case '^':
      ch = GETCH ();
      if (ch >= 'A' && ch <= '_')
	ch = (ch - 'A') | 0x100;
      else if (ch >= 'a' && ch <= '~')
	ch = (ch - 'a') | 0x100;
      else if (ch == '?')
	ch = 127;
      else
	{
	  UNGETCH (ch);
	  ch = '^' | 0x100;
	}
      break;
    default:
      ch |= 0x100;
      break;
    }
  if ((ch & 0xff) == 0)
    goto again;
  return ch;
}
#endif /* C_STYLE_ESCAPES */

#if TRACK_VARIABLES
STATIC void
lookup_variable (termp, name)
    TERM *termp;
    char *name;
{
  int i;

  if (name[0] == '_' && name[1] == 0)
    {
#if ANONYMOUS_SPECIAL
      create_anonymous (termp);
#else
      create_variable (termp, name);
#endif
      return;
    }

  for (i = 0; i < varstack.vs_used; i++)
    if (strcmp (name, varstack.vs_stack[i].ve_name) == 0)
      {
	TERMASSIGN (*termp, varstack.vs_stack[i].ve_term);
	return;
      }

  grow_vs ();
  varstack.vs_stack[varstack.vs_used].ve_name = name;
  create_variable (termp, name);
  TERMASSIGN (varstack.vs_stack[varstack.vs_used].ve_term, *termp);
  varstack.vs_used++;
}

#undef create_variable
#define create_variable(tp,n)	lookup_variable(tp,n)
#endif

#if CONTEXT_ERRORS
STATIC char *
print_context (start, count)
    char *start;
    int count;
{
  int i;

  for (i = 0; i < count; i++)
    {
      prolog_putchar (' ');
      switch (*start++)
	{
	case TOK_ATOM:
	case TOK_INTEGER:
	case TOK_VAR:
	case TOK_FUNCTOR:
	case TOK_FLOAT:
	  prolog_printf ("%s", start);
	  break;
	case TOK_QATOM:
	case TOK_QFUNCTOR:
	  prolog_printf ("'%'s'", start);
	  break;
	case TOK_STRING:
	  prolog_printf ("\"%\"s\"", start);
	  break;
	case TOK_LITERAL:
	  prolog_printf ("%c", *start);
	  break;
	default:
	  /*
	   * Just return here for TOK_END or TOK_STATE
	   */
	  return NULL;
	}
      start += strlen (start) + 1;
    }
  return start;
}
#endif

#ifndef __STDC__
STATIC int
print_scanner_error (finaltoken, err, va_alist)
     char *finaltoken;
     char *err;
     va_dcl
#else
STATIC int
print_scanner_error (char *finaltoken, char *err, ...)
#endif
{
  va_list argv;
  int ctxcount = 0;
  char *ctxscan = pread_buffer;

  prolog_printf (E_read_error);
#ifndef __STDC__
  va_start (argv);
#else
  va_start (argv, err);
#endif
  prolog_vprintf (err, argv);
  va_end (argv);
  prolog_putchar ('\n');
#if CONTEXT_ERRORS
  while (ctxscan < finaltoken && *ctxscan != TOK_END)
    {
      ctxscan += strlen (ctxscan) + 1;
      ctxcount++;
    }
  ctxscan = print_context (pread_buffer, ctxcount - 1);
  prolog_printf (E_here);
  if (ctxscan < finaltoken)
    print_context (ctxscan, 1);
  prolog_putchar ('\n');
#endif
  return 0;
}

#ifndef __STDC__
STATIC int
print_error (tsp, from, upto, mark, err, va_alist)
     struct TokenStack *tsp;
     int from, upto, mark;
     char *err;
     va_dcl
#else
STATIC int
print_error (struct TokenStack *tsp, int from, int upto,
	     int mark, char *err,...)
#endif
{
  va_list argv;

  prolog_printf (E_read_error);
#ifndef __STDC__
  va_start (argv);
#else
  va_start (argv, err);
#endif
  prolog_vprintf (err, argv);
  va_end (argv);
  prolog_putchar ('\n');
#if CONTEXT_ERRORS
  if (tsp != NULL)
    {
      char *lastctx;
      int i;

      prolog_printf (E_in_context);
      if (upto > tsp->ts_used)
	upto = tsp->ts_used;
      if (from < 0)
	{
	  for (from = upto - 1; from >= 0; from--)
	    if (ts_pick (tsp, from).to_kind == TOK_END)
	      break;
	  from++;
	}
      if (from > 0 || ts_pick (tsp, 0).to_context != pread_buffer)
	prolog_printf (" ...");
      lastctx = tsp->ts_stack[from].to_context;
      for (i = from; i < upto; i++)
	if (ts_pick(tsp,i).to_kind != TOK_END &&
	    ts_pick(tsp,i).to_kind != TOK_STATE)
	  {
	    lastctx = print_context (ts_pick(tsp,i).to_context,
				     ts_pick(tsp,i).to_ctxcount);
	    if (i == mark)
	      prolog_printf (E_here);
	  }
      if (*lastctx != TOK_END)
	prolog_printf (" ...");
      prolog_putchar ('\n');
    }
#endif
  return 0;
}

/*
 * This could also be a macro, but why???
 */
STATIC int
expand_buffer (ch)
    int ch;
{
  if (pread_bufsize)
    pread_buffer = (char *) REALLOC (pread_buffer, pread_bufsize += 1024);
  else
    pread_buffer = (char *) MALLOC (pread_bufsize = 1024);
  return pread_buffer[pread_bufused++] = ch;
}

STATIC struct TokenStack *
allocate_ts ()
{
  struct TokenStack *tmp;

  tmp = TMALLOC (1, struct TokenStack);
  tmp->ts_stack = TMALLOC (tmp->ts_size = 32, struct Token);
  tmp->ts_used = 0;
  return tmp;
}

/*
 * prolog_lex -
 *	read a complete expression from the input via the user's
 *	'prolog_getchar' function. The input is broken into tagged
 *	'\0'-terminated strings.
 */
STATIC int
prolog_lex ()
{
  int laststart,lastch,ch;
  int bracecount = 0;

#ifndef OLD_FUNCTORS
  /*
   * allow functors at the start of an expression...
   */
  next_is_never_functor = 0;
#endif
  pread_bufused = 0;
  GETRESET ();
  for (;;)
    {
      ch = GETCH ();
      switch (ch)
	{
	case EOF:
	handle_eof:
	  if (pread_bufused)
	    {
	      /*
	       * If we already read something, this is an error!
	       */
	      STORECH (TOK_END);
	      STORECH (0);
	      print_scanner_error (&pread_buffer[pread_bufused], 
				   E_unexpeof);
	      /* print_error(NULL,0,0,-1,E_unexpeof); */
	      return 0;
	    }
	  STORECH (TOK_END);
	  STORECH (0);
	  return 1;
	case '(':
	case '[':
	case '{':
	  /*
	   * an opening brace --> update counter...
	   */
	  bracecount++;
	  goto literal_char;
	case ')':
	case ']':
	case '}':
	  /*
	   * a closing brace --> update counter...
	   */
	  if (bracecount-- == 0)
	    {
	      STORECH (TOK_LITERAL);
	      STORECH (ch);
	      STORECH (0);
	      print_scanner_error (&pread_buffer[pread_bufused],
				   E_illbrace, ch);
	      return 0;
	    }
	  /* fall through */
	case '!':
	case ';':
	case ',':
	case '|':
	literal_char:
	  /*
	   * special Prolog characters, which can be used
	   * without quotes...
	   */
	  STORECH (TOK_LITERAL);
	  STORECH (ch);
	  STORECH (0);
	  break;
	case '%':
	  /*
	   * comment --> skip input up to the next end-of-line ...
	   */
	  while ((ch = GETCH ()) != EOF && ch != '\n');
	  UNGETCH (ch);
	  break;
	case '\'':
	  /*
	   * a Prolog atom enclosed in single quotes...
	   */
	  laststart = pread_bufused;
	  STORECH (TOK_QATOM);
	  for (;;)
	    {
	      ch = GETCH ();
#if C_STYLE_ESCAPES
	      if (ch == '\\')
		ch = read_escaped_char ();
#endif
	      if (ch == '\'')
		{
		  ch = GETCH ();
		  if (ch != '\'')
		    {
		      UNGETCH (ch);
		      STORECH (0);
		      break;
		    }
		}
	      else if (ch == EOF)
		{
		  STORECH (0);
		  goto handle_eof;
		}
	      STORECH (ch);
	    }
	  goto handle_functor;
	case '\"':
	  /*
	   * a Prolog integer-list given as a string enclosed
	   * in double quotes...
	   */
	  STORECH (TOK_STRING);
	  for (;;)
	    {
	      ch = GETCH ();
#if C_STYLE_ESCAPES
	      if (ch == '\\')
		ch = read_escaped_char ();
#endif
	      if (ch == '\"')
		{
		  ch = GETCH ();
		  if (ch != '\"')
		    {
		      UNGETCH (ch);
		      STORECH (0);
		      break;
		    }
		}
	      else if (ch == EOF)
		{
		  STORECH (0);
		  goto handle_eof;
		}
	      STORECH (ch);
	    }
	  break;
#if C_STYLE_COMMENTS
	case '/':
	  ch = GETCH ();
	  if (ch != '*')
	    {
	      UNGETCH (ch);
	      ch = '/';
	      goto handle_atom2;
	    }
	  lastch = 0;
	  for (ch = GETCH ();
	       ch != EOF && (ch != '/' || lastch != '*');
	       ch = GETCH ())
	    lastch = ch;
	  if (ch == EOF)
	    {
	      STORECH (0);
	      goto handle_eof;
	    }
	  break;
#endif /* C_STYLE_COMMENTS */
	case '.':
	  /*
	   * This could be the end of the term.
	   */
	  if (bracecount)
	    /*
	     * not if there are opened braces ...
	     */
	    goto handle_atom2;
	  ch = GETCH ();
	  if (ch != EOF && !isspace (ch))
	    {
	      /*
	       * a non-blank character follows --> treat as atom.
	       */
	      UNGETCH (ch);
	      ch = '.';
	      goto handle_atom2;
	    }
	  if (pread_bufused == 0)
	    {
	      STORECH (TOK_END);
	      STORECH (0);
	      print_scanner_error (&pread_buffer[pread_bufused],
				   E_emptyexpr);
	      /* print_error(NULL,0,0,-1,E_emptyexpr); */
	      return 0;
	    }
	  STORECH (TOK_END);
	  STORECH (0);
	  return 1;
#if SIGNED_NUMBERS
	case '-':
	  /*
	   * Check if it is followed by a digit
	   */
	  lastch = GETCH ();
	  UNGETCH (lastch);
	  if (lastch >= '0' && lastch <= '9')
	    goto handle_number;
	  goto handle_atom2;
#endif /* SIGNED_NUMBERS */
#if C_STYLE_NUMBERS
	case '0':
	  ch = GETCH ();
	  if (ch == 'x' || ch == 'X')
	    {
	      int ch2 = GETCH ();

	      if (HEXCHAR (ch2))
		{
		  STORECH (TOK_INTEGER);
		  STORECH ('0');
		  STORECH (ch);
		  STORECH (ch2);
		  for (ch = GETCH(); HEXCHAR (ch); ch = GETCH())
		    STORECH (ch);
		  UNGETCH (ch);
		  STORECH (0);
		}
	      else
		{
		  UNGETCH (ch2);
		  UNGETCH (ch);
		  ch = '0';
		  goto handle_number;
		}
	    }
	  else
	    {
	      STORECH (TOK_INTEGER);
	      STORECH ('0');
	      for (; OCTCHAR (ch); ch = GETCH())
		STORECH (ch);
	      UNGETCH (ch);
	      STORECH (0);
	    }
	  break;
#endif /* C_STYLE_NUMBERS */
	default:
	  if (isspace (ch))
	    /*
	     * ignore whitespaces...
	     */
	    break;
	  if (ch >= '0' && ch <= '9')
	    {
	      /*
	       * integer number
	       */
	    handle_number:
#if BUILTIN_FLOATS
	      laststart = pread_bufused;
#endif
	      STORECH (TOK_INTEGER);
	      STORECH (ch);
	      for (ch = GETCH (); ch >= '0' && ch <= '9'; ch = GETCH ())
		STORECH (ch);
#if BUILTIN_FLOATS
	      /*
	       * Syntax of floating point numbers is:
	       *	NUM	= [0-9]+
	       *	FRAC	= \. NUM
	       *	EXPO	= ([eE] [+-]? NUM)
	       *	float	= -? NUM (FRAC | EXPO | FRAC EXPO)
	       * you see, either the fractal or the exponential part
	       * must be given to identify a floating point number.
	       */
	      if (ch == '.')
		{
		  /*
		   * A dot, this may be a floating point number...
		   */
		  pread_buffer[laststart] = TOK_FLOAT;
		  STORECH (ch);
		  for (ch = GETCH ();
		       ch >= '0' && ch <= '9';
		       ch = GETCH ())
		    STORECH (ch);
		  if (pread_buffer[pread_bufused-1] == '.')
		    {
		      /*
		       * No digits after the dot, no float
		       */
		      UNGETCH (ch);
		      ch = '.';
		      pread_bufused--;
		      pread_buffer[laststart] = TOK_INTEGER;
		    }
		  else if (ch == 'e' || ch == 'E')
		    {
		      STORECH (ch);
		      ch = GETCH ();
		      if (ch == '-' || ch == '+')
			{
			  STORECH (ch);
			  ch = GETCH ();
			}
		      for (; ch >= '0' && ch <= '9'; ch = GETCH())
			STORECH (ch);
		      if (pread_buffer[pread_bufused-1] == '-'
			  || pread_buffer[pread_bufused-1] == '+')
			{
			  UNGETCH (ch);
			  ch = pread_buffer[--pread_bufused];
			}
		      if (pread_buffer[pread_bufused-1] == 'e'
			  || pread_buffer[pread_bufused-1] == 'E')
			{
			  UNGETCH (ch);
			  ch = pread_buffer[--pread_bufused];
			}
		    }
		}
	      else if (ch == 'e' || ch == 'E')
		{
		  pread_buffer[laststart] = TOK_FLOAT;
		  STORECH (ch);
		  ch = GETCH ();
		  if (ch == '-' || ch == '+')
		    {
		      STORECH (ch);
		      ch = GETCH ();
		    }
		  for (; ch >= '0' && ch <= '9'; ch = GETCH ())
		    STORECH (ch);
		  if (pread_buffer[pread_bufused-1] == '-'
		      || pread_buffer[pread_bufused-1] == '+')
		    {
		      UNGETCH (ch);
		      ch = pread_buffer[--pread_bufused];
		    }
		  if (pread_buffer[pread_bufused-1] == 'e'
		      || pread_buffer[pread_bufused-1] == 'E')
		    {
		      UNGETCH (ch);
		      ch = pread_buffer[--pread_bufused];
		      pread_buffer[laststart] = TOK_INTEGER;
		    }
		}
#endif				/* BUILTIN_FLOATS */
	      UNGETCH (ch);
	      STORECH (0);
	    }
	  else if (VARCHAR (ch))
	    {
	      /*
	       * upper-case letter or underscore --> variable
	       */
	      STORECH (TOK_VAR);
	      STORECH (ch);
	      for (ch = GETCH (); AVCHAR (ch); ch = GETCH ())
		STORECH (ch);
	      UNGETCH (ch);
	      STORECH (0);
	    }
	  else if (ATOM1CHAR (ch))
	    {
	      /*
	       * an atom starting with a lower-case letter...
	       */
	      laststart = pread_bufused;
	      STORECH (TOK_ATOM);
	      STORECH (ch);
	      for (ch = GETCH (); AVCHAR (ch); ch = GETCH ())
		STORECH (ch);
	      UNGETCH (ch);
	      STORECH (0);
	      goto handle_functor;
	    }
	  else if (ATOM2CHAR (ch))
	    {
	    handle_atom2:
	      /*
	       * an atom made up of special characters ...
	       */
	      laststart = pread_bufused;
	      STORECH (TOK_ATOM);
	      STORECH (ch);
	      for (ch = GETCH (); ATOM2CHAR (ch); ch = GETCH ())
		STORECH (ch);
	      UNGETCH (ch);
	      STORECH (0);
	    handle_functor:
	      if (ch == '(')
		pread_buffer[laststart] =
		  ((pread_buffer[laststart] == TOK_QATOM)
		   ? TOK_QFUNCTOR
		   : TOK_FUNCTOR);
	    }
	  else
	    {
	      /*
	       * anything else is an illegal character...
	       */
	      STORECH (TOK_END);
	      STORECH (0);
	      print_scanner_error (&pread_buffer[pread_bufused],
				   E_illchar, ch & 0xff);
	      /* print_error(NULL,0,0,-1,E_illchar, ch & 0xff); */
	      return 0;
	    }
	}
    }
}

/*----------------------------------------------------------------------*
 * get_token_to_stack	- gets the next token from 'pread_buffer' and	*
 *		  	  transforms it to a 'struct Token' in the next *
 *			  free token in the given stack.
 *----------------------------------------------------------------------*
 * Return values:
 *	0 - end of term
 *	1 - return value is the term-token in 'tokp'
 *   else - return value is an immediate character
 * Term kind tags are reduced to:
 *	TOK_TERM	- if tokp->to_term holds an
 *			  already converted term.
 *	TOK_ATOM	- only atoms which are defined operators
 *			  are kept as tokens; tokp->to_opdef and
 *			  tokp->to_precedence hold the operator
 *			  declaration for them.
 *	TOK_FUNCTOR	- tokp->to_name points to the
 *			  functors name and the next token to be read
 *			  will ALWAYS be an opening brace '('.
 * All other term kind tags (TOK_INTEGER, TOK_STRING and TOK_ATOM for
 * non-operators) are converted to terms before returning them.
 *----------------------------------------------------------------------*/
STATIC int
get_token_to_stack (tsp, comma_is_op)
    struct TokenStack *tsp;
    int comma_is_op;
{
  register struct Token *tokp = &ts_next (tsp);
  TERM tl1,tl2,tlr;
  char *sl1;

  set_context (pread_context, pread_bufscan);
  set_context (tokp->to_context, pread_bufscan);
  set_ctxcount (tokp->to_ctxcount, 1);
  switch (*pread_bufscan++)
    {
    case TOK_END:
      pread_bufscan--;
      return 0;
    case TOK_LITERAL:
      if ((*pread_bufscan != ',' || !comma_is_op)
	  && index (PL_SPEC_ATOMS, *pread_bufscan) == NULL)
	{
#ifndef OLD_FUNCTORS
	  /*
	   * A functor can never occur after a closing brace...
	   */
	  next_is_never_functor = index (")]}", *pread_bufscan) != NULL;
#endif
	  pread_bufscan += 2;
	  return pread_bufscan[-2];
	}
      /* else fall through and treat it as an atom */
    case TOK_ATOM:
    case TOK_QATOM:
    handle_operator:
      tokp->to_kind = TOK_ATOM;
      tokp->to_name = pread_bufscan;
#if QUOTED_CAN_BE_OP
      get_operator_type (pread_bufscan, &tokp->to_precedence,
			 &tokp->to_opdef);
#else
      if (pread_bufscan[-1] == TOK_QATOM)
	tokp->to_precedence = 0;
      else
	get_operator_type (pread_bufscan, &tokp->to_precedence,
			   &tokp->to_opdef);
#endif
      if (tokp->to_precedence == 0 || tokp->to_opdef == 0)
	{
	  sl1 = pread_bufscan + strlen (pread_bufscan) + 1;
	  if (sl1[0] == TOK_LITERAL && sl1[1] == '(')
	    {
	      tokp->to_kind = TOK_FUNCTOR;
	      tokp->to_precedence =
		tokp->to_opdef = 0;
	      add_ctxcount (tokp->to_ctxcount, 1);
	    }
	  else
	    {
	      create_atom (&tokp->to_term, pread_bufscan);
	      tokp->to_kind = TOK_TERM;
	    }
	}
      else
	tokp->to_opdef |= OP_ATOM;
#ifndef OLD_FUNCTORS
      /*
       * A functor can never occur after atoms that cannot be
       * operators...
       */
      next_is_never_functor = (tokp->to_opdef & ~OP_ATOM) == 0;
#endif
      break;
    case TOK_FUNCTOR:
    case TOK_QFUNCTOR:
#ifndef OLD_FUNCTORS
      /*
       * If this one cannot be a functor, try to make an atom of it...
       */
      if (next_is_never_functor)
	goto handle_operator;
#endif
      tokp->to_kind = TOK_FUNCTOR;
      tokp->to_name = pread_bufscan;
      tokp->to_precedence =
	tokp->to_opdef = 0;
      add_ctxcount (tokp->to_ctxcount, 1);
      break;
    case TOK_VAR:
      tokp->to_kind = TOK_TERM;
#if ANONYMOUS_SPECIAL
      if (pread_bufscan[0] == '_' && pread_bufscan[1] == 0)
	create_anonymous (&tokp->to_term);
      else
	create_variable (&tokp->to_term, pread_bufscan);
#else
      create_variable (&tokp->to_term, pread_bufscan);
#endif
#ifndef OLD_FUNCTORS
      /*
       * A functor can never occur after a variable...
       */
      next_is_never_functor = 1;
#endif
      break;
    case TOK_STRING:
      tokp->to_kind = TOK_TERM;
#if STRING_SPECIAL
      create_string (&tokp->to_term, pread_bufscan);
#else
      sl1 = pread_bufscan + strlen (pread_bufscan);
      create_atom (&tl1, "[]");
      while (--sl1 >= pread_bufscan)
	{
	  create_intfval (&tl2, *sl1 & 0xff);
	  create_list (&tlr, &tl2, &tl1);
	  TERMASSIGN (tl1, tlr);
	}
      TERMASSIGN (tokp->to_term, tlr);
#endif
#ifndef OLD_FUNCTORS
      /*
       * a functor can never occur after a constant string...
       */
      next_is_never_functor = 1;
#endif
      break;
    case TOK_INTEGER:
      tokp->to_kind = TOK_TERM;
      create_integer (&tokp->to_term, pread_bufscan);
#ifndef OLD_FUNCTORS
      /*
       * a functor can never occur after a constant integer...
       */
      next_is_never_functor = 1;
#endif
      break;
#if BUILTIN_FLOATS
    case TOK_FLOAT:
      tokp->to_kind = TOK_TERM;
      create_float (&tokp->to_term, pread_bufscan);
#ifndef OLD_FUNCTORS
      /*
       * a functor can never occur after a constant float ...
       */
      next_is_never_functor = 1;
#endif
      break;
#endif
    }
  pread_bufscan += strlen (pread_bufscan) + 1;
  tsp->ts_used++;
  return 1;
}

/*----------------------------------------------------------------------*
 * reduce_stack_to_term	- takes a token-stack and reduces it until	*
 * 			  there is only one term (token) left on it.	*
 *----------------------------------------------------------------------*/

/*
 * The following macros test, if ME allows another operator named HE
 * on his right/left side respecting precedences and ME's right/left
 * associativity...
 *
 *	 [1]               [2]
 *      INFIX	   -> ( PREFIX ... )	ALLOWS_R ([2], [1], OP_INFIX)
 *      PREFIX	   -> ( PREFIX ... )	ALLOWS_R ([2], [1], OP_PREFIX)
 * ( ... POSTFIX ) <-     INFIX		ALLOWS_L ([1], [2], OP_INFIX)
 * ( ... POSTFIX ) <-    POSTFIX	ALLOWS_L ([1], [2], OP_POSTFIX)
 */
#define ALLOWS_R(Me,He,Optype) \
    ((Me).to_precedence > (He).to_precedence || \
     ((Me).to_precedence == (He).to_precedence && \
      (Me).to_opdef & (OP_RIGHTASSOC & (Optype))))

#define ALLOWS_L(Me,He,Optype) \
    ((Me).to_precedence > (He).to_precedence || \
     ((Me).to_precedence == (He).to_precedence && \
      (Me).to_opdef & (OP_LEFTASSOC & (Optype))))

STATIC int
reduce_stack_to_term (ts)
    struct TokenStack *ts;
{
  int i,bottom;
  struct Token *tokp;

  for (bottom = ts->ts_used - 1; bottom >= 0; bottom--)
    if ((ts_pick (ts, bottom).to_kind == TOK_END
	 || ts_pick (ts, bottom).to_kind == TOK_STATE))
      break;
  bottom++;

  if (bottom == ts->ts_used)
    {
      print_error (frame_tsp, 0, frame_tsp->ts_used, frame_tsp->ts_used-1,
		   E_emptyexpr);
      return 0;
    }

  /*
   * deterministic elemination of impossible operator types.
   */
  if (ts_pick (ts, bottom).to_kind == TOK_ATOM)
    /*
     * The first one can only be a real atom or prefix
     */
    cut_off_opdef (&ts_pick (ts, bottom), OP_ATOM | OP_PREFIX);

  if (ts_top (ts).to_kind == TOK_ATOM)
    /*
     * The last one can only be a real atom or postfix
     */
    cut_off_opdef (&ts_top (ts), OP_ATOM | OP_POSTFIX);

  if (bottom == ts->ts_used - 1)
    /*
     * If there is only one token on the stack (mostly it is so)
     * we can immediately return
     */
    return 1;

  /* -------------------------- *
   * reduce_stack_once -- start *
   * -------------------------- */

  tokp = &ts_pick (ts, bottom);
  for (i = bottom; i < ts->ts_used; i++, tokp++)
    {
      int mask = 0;

      if (i > bottom && tokp[-1].to_kind == TOK_ATOM)
	{
	  if (tokp->to_kind == TOK_ATOM)
	    {
	      if (tokp->to_opdef & OP_PREFIX)
		{
		  /*
		   * If this token is PREFIX, the previous one may also be
		   * PREFIX or INFIX, if HIS precedence and associtivity
		   * allows ME...
		   */
		  if (ALLOWS_R (tokp[-1], tokp[0], OP_PREFIX))
		    mask |= OP_PREFIX;
		  if (ALLOWS_R (tokp[-1], tokp[0], OP_INFIX))
		    mask |= OP_INFIX;
		}
	      if (tokp->to_opdef & OP_POSTFIX)
		{
		  /*
		   * If this token is POSTFIX, the previous one may also be
		   * POSTFIX, if MY precedence and associativity allow
		   * HIM ...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_L (tokp[0], tokp[-1], OP_POSTFIX))
		    mask |= OP_POSTFIX;
		}
	      if (tokp->to_opdef & OP_INFIX)
		{
		  /*
		   * If this token is INFIX, the previous one may be
		   * POSTFIX, if MY precedence and associativity
		   * allows HIM ...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_L (tokp[0], tokp[-1], OP_INFIX))
		    mask |= OP_POSTFIX;
		}
	      if (tokp->to_opdef & OP_ATOM)
		mask |= OP_PREFIX | OP_INFIX;
	    }
	  else
	    mask |= OP_PREFIX | OP_INFIX;
	  cut_off_opdef (&tokp[-1], mask);
	}

      mask = 0;
      if (i < ts->ts_used - 1 && tokp[1].to_kind == TOK_ATOM)
	{
	  if (tokp->to_kind == TOK_ATOM)
	    {
	      if (tokp->to_opdef & OP_POSTFIX)
		{
		  /*
		   * If this token is POSTFIX, the following one may also
		   * be POSTFIX or INFIX, if HIS precedence allows ME ...
		   */
		  if (ALLOWS_L (tokp[1], tokp[0], OP_POSTFIX))
		    mask |= OP_POSTFIX;
		  if (ALLOWS_L (tokp[1], tokp[0], OP_INFIX))
		    mask |= OP_INFIX;
		}
	      if (tokp->to_opdef & OP_PREFIX)
		{
		  /*
		   * If this token is PREFIX, the following one may also be
		   * PREFIX, if MY precedence allows HIM...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_R (tokp[0], tokp[1], OP_PREFIX))
		    mask |= OP_PREFIX;
		}
	      if (tokp->to_opdef & OP_INFIX)
		{
		  /*
		   * If this token is INFIX, the following one may also be
		   * PREFIX, if MY precedence allows HIM...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_R (tokp[0], tokp[1], OP_INFIX))
		    mask |= OP_PREFIX;
		}
	      if (tokp->to_opdef & OP_ATOM)
		mask |= OP_INFIX | OP_POSTFIX;
	    }
	  else
	    mask |= OP_POSTFIX | OP_INFIX;
	  cut_off_opdef (&tokp[1], mask);
	}
    }

  tokp = &ts_top (ts);
  for (i = ts->ts_used - 1; i >= bottom; i--, tokp--)
    {
      int mask = 0;

      if (i > bottom && tokp[-1].to_kind == TOK_ATOM)
	{
	  if (tokp->to_kind == TOK_ATOM)
	    {
	      if (tokp->to_opdef & OP_PREFIX)
		{
		  /*
		   * If this token is PREFIX, the previous one may also be
		   * PREFIX or INFIX, if HIS precedence and associtivity
		   * allows ME...
		   */
		  if (ALLOWS_R (tokp[-1], tokp[0], OP_PREFIX))
		    mask |= OP_PREFIX;
		  if (ALLOWS_R (tokp[-1], tokp[0], OP_INFIX))
		    mask |= OP_INFIX;
		}
	      if (tokp->to_opdef & OP_POSTFIX)
		{
		  /*
		   * If this token is POSTFIX, the previous one may also be
		   * POSTFIX, if MY precedence and associativity allow
		   * HIM ...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_L (tokp[0], tokp[-1], OP_POSTFIX))
		    mask |= OP_POSTFIX;
		}
	      if (tokp->to_opdef & OP_INFIX)
		{
		  /*
		   * If this token is INFIX, the previous one may be
		   * POSTFIX, if MY precedence and associativity allows
		   * HIM ...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_L (tokp[0], tokp[-1], OP_INFIX))
		    mask |= OP_POSTFIX;
		}
	      if (tokp->to_opdef & OP_ATOM)
		mask |= OP_PREFIX | OP_INFIX;
	    }
	  else
	    mask |= OP_PREFIX | OP_INFIX;
	  cut_off_opdef (&tokp[-1], mask);
	}

      mask = 0;
      if (i < ts->ts_used - 1 && tokp[1].to_kind == TOK_ATOM)
	{
	  if (tokp->to_kind == TOK_ATOM)
	    {
	      if (tokp->to_opdef & OP_POSTFIX)
		{
		  /*
		   * If this token is POSTFIX, the following one may also
		   * be POSTFIX or INFIX, if HIS precedence allows ME ...
		   */
		  if (ALLOWS_L (tokp[1], tokp[0], OP_POSTFIX))
		    mask |= OP_POSTFIX;
		  if (ALLOWS_L (tokp[1], tokp[0], OP_INFIX))
		    mask |= OP_INFIX;
		}
	      if (tokp->to_opdef & OP_PREFIX)
		{
		  /*
		   * If this token is PREFIX, the following one may also be
		   * PREFIX, if MY precedence allows HIM...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_R (tokp[0], tokp[1], OP_PREFIX))
		    mask |= OP_PREFIX;
		}
	      if (tokp->to_opdef & OP_INFIX)
		{
		  /*
		   * If this token is INFIX, the following one may also be
		   * PREFIX, if MY precedence allows HIM...
		   */
		  /* mask |= OP_ATOM; */
		  if (ALLOWS_R (tokp[0], tokp[1], OP_INFIX))
		    mask |= OP_PREFIX;
		}
	      if (tokp->to_opdef & OP_ATOM)
		mask |= OP_INFIX | OP_POSTFIX;
	    }
	  else
	    mask |= OP_POSTFIX | OP_INFIX;
	  cut_off_opdef (&tokp[1], mask);
	}
    }

  /* -------------------------- *
   *  reduce_stack_once -- end  *
   * -------------------------- */

  /*
   * check the stack for correct usage of operators.
   */
  tokp = &ts_pick (ts, bottom);
  for (i = bottom; i < ts->ts_used; i++, tokp++)
    if (tokp->to_kind == TOK_ATOM && !onebit(tokp->to_opdef))
      return print_error (ts, bottom, ts->ts_used, i, E_ambigious,
			  tokp->to_name,
			  OPDEF1 (tokp->to_opdef),
			  OPDEF2 (tokp->to_opdef));

  /*
   * Now reduce it until we get an error or have only one
   * element left on the stack.
   */
  {
    static int *opstack = NULL;
    static int opsize = 0;
    int lopos, opused = 0;
    int checkpos = bottom;
    int startbottom = bottom;

    if (opsize < (ts->ts_used - bottom))
      {
	if (opsize == 0)
	  opstack = TMALLOC (ts->ts_used - bottom, int);
	else
	  opstack = TREALLOC (opstack, ts->ts_used - bottom, int);
	opsize = ts->ts_used - bottom;
      }

    while (ts->ts_used > bottom + 1)
      {
	tokp = &ts_pick (ts, 0);
	for (; checkpos < ts->ts_used; checkpos++)
	  {
	    if (tokp[checkpos].to_kind != TOK_ATOM)
	      continue;
	    if (opused <= 0 || ALLOWS_R (tokp[opstack[opused-1]],
					 tokp[checkpos],
					 OP_PREFIX | OP_INFIX))
	      {
		/*
		 * If the last token allows this one on his
		 * right-hand-side, this one is smaller...
		 */
		opstack[opused++] = checkpos;
	      }
	    else if (!ALLOWS_L (tokp[checkpos],
				tokp[opstack[opused-1]],
				OP_POSTFIX | OP_INFIX))
	      {
		/*
		 * If this token does not allow the last token on
		 * his left-hand-side, there is an operator-clash.
		 * [change 90/10/16 - to know the conflicting operator,
		 *  we print the context with the <<HERE>> after the
		 *  left and put the end after the right operator]
		 */
		/* FREE (opstack); */
		return print_error (ts, bottom, checkpos,
				    opstack[opused-1], E_assoc,
				    tokp[opstack[opused-1]].to_name,
				    tokp[checkpos].to_name);
	      }
	    else
	      /*
	       * Otherwise the opstack[opused-1] can be reduced...
	       * Note: the break also causes CHECKPOS to stay here,
	       * so we can check it with the next operator on the
	       * stack when we come here again.
	       */
	      break;
	  }
	if (opused < 1)
	  {
	    /*
	     * No operator left but still more than 1 element.
	     */
	    /* FREE (opstack); */
	    return print_error (ts, bottom, ts->ts_used, bottom,
				E_nooperator);
	  }

	/*
	 * We found an independent low operator
	 * --> reduce to a structure term.
	 */
	lopos = opstack[--opused];
	tokp = &ts_pick (ts, lopos);
	if (tokp->to_opdef & OP_INFIX)
	  {
	    TERM arguments[2];

	    if ((tokp[-1].to_kind == TOK_ATOM
		 || tokp[1].to_kind == TOK_ATOM))
	      {
		/* FREE (opstack); */
		return print_error (ts, bottom, ts->ts_used, lopos,
				    E_opusage, OPDEF1 (tokp->to_opdef),
				    tokp->to_name);
	      }
	    TERMASSIGN (arguments[0], tokp[-1].to_term);
	    TERMASSIGN (arguments[1], tokp[1].to_term);
#if CONVERT_DOT_TO_LIST
	    if (tokp->to_name[0] == '.' && tokp->to_name[1] == 0)
	      {
		if (lopos == bottom + 1)
		  {
		    create_list (&tokp[1].to_term,
				 &arguments[0], &arguments[1]);
		    set_context (tokp[1].to_context, tokp[-1].to_context);
		    add_ctxcount (tokp[1].to_ctxcount,
				  tokp->to_ctxcount + tokp[-1].to_ctxcount);
		    bottom += 2;
		  }
		else
		  {
		    create_list (&tokp[-1].to_term,
				 &arguments[0], &arguments[1]);
		    add_ctxcount (tokp[-1].to_ctxcount,
				  tokp->to_ctxcount + tokp[1].to_ctxcount);
		    if (lopos < ts->ts_used - 2)
		      bcopy (&tokp[2], &tokp[0],
			     ((ts->ts_used - lopos - 2)
			      * sizeof(struct Token)));
		    ts->ts_used -= 2;
		    checkpos -= 2;
		  }
	      }
	    else
#endif
	      if (lopos == bottom + 1)
		{
		  create_structure (&tokp[1].to_term, tokp->to_name,
				    2, arguments);
		  set_context (tokp[1].to_context, tokp[-1].to_context);
		  add_ctxcount (tokp[1].to_ctxcount,
				tokp->to_ctxcount + tokp[-1].to_ctxcount);
		  bottom += 2;
		}
	      else
		{
		  create_structure (&tokp[-1].to_term, tokp->to_name,
				    2, arguments);
		  add_ctxcount (tokp[-1].to_ctxcount,
				tokp->to_ctxcount + tokp[1].to_ctxcount);
		  if (lopos < ts->ts_used - 2)
		    bcopy (&tokp[2], &tokp[0],
			   ((ts->ts_used - lopos - 2)
			    * sizeof (struct Token)));
		  ts->ts_used -= 2;
		  checkpos -= 2;
		}
	  }
	else if (tokp->to_opdef & OP_PREFIX)
	  {
	    if (tokp[1].to_kind != TOK_TERM)
	      {
		/* FREE (opstack); */
		return print_error (ts,bottom, ts->ts_used, lopos,
				    E_opusage, OPDEF1 (tokp->to_opdef),
				    tokp->to_name);
	      }
	    if (lopos == bottom)
	      {
		TERM argument;

		TERMASSIGN (argument, tokp[1].to_term);
		create_structure (&tokp[1].to_term, tokp->to_name,
				  1, &argument);
		set_context (tokp[1].to_context, tokp->to_context);
		add_ctxcount (tokp[1].to_ctxcount, tokp->to_ctxcount);
		bottom++;
	      }
	    else
	      {
		create_structure (&tokp->to_term, tokp->to_name,
				  1, &tokp[1].to_term);
		tokp->to_kind = TOK_TERM;
		add_ctxcount (tokp->to_ctxcount, tokp[1].to_ctxcount);
		if (lopos < ts->ts_used - 2)
		  bcopy (&tokp[2], &tokp[1],
			 ((ts->ts_used - lopos - 2)
			  * sizeof (struct Token)));
		ts->ts_used--;
		checkpos--;
	      }
	  }
	else
	  {
	    /* OP_POSTFIX */
	    if (tokp[-1].to_kind == TOK_ATOM)
	      {
		/* FREE (opstack); */
		return print_error (ts, bottom, ts->ts_used, lopos,
				    E_opusage, OPDEF1(tokp->to_opdef),
				    tokp->to_name);
	      }
	    if (lopos == bottom + 1)
	      {
		create_structure (&tokp->to_term, tokp->to_name,
				  1, &tokp[-1].to_term);
		tokp->to_kind = TOK_TERM;
		set_context (tokp->to_context, tokp[-1].to_context);
		add_ctxcount (tokp->to_ctxcount, tokp[-1].to_ctxcount);
		bottom++;
	      }
	    else
	      {
		TERM argument;

		TERMASSIGN (argument, tokp[-1].to_term);
		create_structure (&tokp[-1].to_term, tokp->to_name,
				  1, &argument);
		add_ctxcount (tokp[-1].to_ctxcount, tokp->to_ctxcount);
		if (lopos < ts->ts_used - 1)
		  bcopy (&tokp[1], &tokp[0],
			 ((ts->ts_used - lopos - 1)
			  * sizeof (struct Token)));
		ts->ts_used--;
		checkpos--;
	      }
	  }
      }
    /* FREE(opstack); */
    if (bottom != startbottom)
      ts_pick (ts, startbottom) = ts_pick (ts, bottom);
    ts->ts_used = 1 + startbottom; /* ??? */
  }
  return 1;
}

/*----------------------------------------------------------------------*
 * parse_term - parse a term from the input and store the result in	*
 *		RESULT							*
 *----------------------------------------------------------------------*/
#if CONTEXT_ERRORS

#define push_state(tsp,newstate) \
  ( ts_next(tsp).to_kind = TOK_STATE,		\
    (tsp)->ts_used++,				\
    ts_top(tsp).to_state = pstate,		\
    pstate = newstate,				\
    ts_top(tsp).to_pred = __laststate,		\
    __laststate = (tsp)->ts_used-1,		\
    ts_top(tsp).to_endch = endch,		\
    ts_top(tsp).to_context = context,		\
    context = pread_context,			\
    ts_top(tsp).to_ctxcount = ctxcount-1,	\
    ctxcount = 1 )

#define restore_state() \
  ( assign_token = laststate,		\
    pstate = laststate->to_state,	\
    endch = laststate->to_endch,	\
    context = laststate->to_context,	\
    ctxcount += laststate->to_ctxcount,	\
    __laststate = laststate->to_pred )

#else	/* not CONTEXT_ERRORS */

#define push_state(tsp,newstate) \
  ( ts_next(tsp).to_kind = TOK_STATE,	\
    (tsp)->ts_used++,			\
    ts_top(tsp).to_state = pstate,	\
    pstate = newstate,			\
    ts_top(tsp).to_pred = __laststate,	\
    __laststate = (tsp)->ts_used-1,	\
    ts_top(tsp).to_endch = endch )

#define restore_state() \
  ( assign_token = laststate,		\
    pstate = laststate->to_state,	\
    endch = laststate->to_endch,	\
    __laststate = laststate->to_pred )

#endif	/* not CONTEXT_ERRORS */

#define laststate		(&ts_pick(tsp,__laststate))
#define pushed_tokens(tsp)	((tsp)->ts_used - __laststate - 1)
#define abort_with_value(n)	{ free_ts(tsp); return (n); }
#define REDUCE_STACK(ts)	reduce_stack_to_term(ts)

STATIC int
parse_term (result)
    TERM *result;
{
  static TERM *arguments = NULL;
  static int argssize = 0;

  int i,back;
  struct TokenStack *tsp = allocate_ts ();
  struct Token *assign_token;
  enum ParserState pstate = PS_SUBTERM;
  int __laststate = 0;
  int endch = 0;
#if CONTEXT_ERRORS
  char *context = pread_context;
  int ctxcount = 0;
#endif

  set_context (frame_tsp, tsp);
  for (;;)
    {
      back = get_token_to_stack (tsp, (pstate == PS_SUBTERM
				       || pstate == PS_LIST_TAIL));
      add_ctxcount (ctxcount, 1);
      switch (back)
	{
	case 0:			/* end of input */
	  if (pstate == PS_SUBTERM || endch == 0)
	    {
	      if (!REDUCE_STACK (tsp))
		abort_with_value (0);
	      TERMASSIGN (*result, ts_top (tsp).to_term);
	      abort_with_value (1);
	    }
	  print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
		       E_unclosedexpr);
	  abort_with_value (0);
	case 1:			/* returned a token */
	  if (ts_top (tsp).to_kind == TOK_FUNCTOR)
	    {
	      push_state (tsp, PS_FUNCTION);
	      ts_next (tsp).to_kind = TOK_END;
	      tsp->ts_used++;
	      if ((get_token_to_stack (tsp, (pstate == PS_SUBTERM
					     || pstate == PS_LIST_TAIL))
		   != '('))
		{
		  print_error (tsp, 0, 0, 0, E_internal);
		  abort_with_value (0);
		}
	      add_ctxcount (ctxcount, 1);
	    }
	  break;
	case '(':
	  push_state (tsp, PS_SUBTERM);
	  endch = ')';
	  break;
	case '{':
	  push_state (tsp, PS_SUBTERM);
	  endch = '}';
	  break;
	case '[':
	  push_state (tsp, PS_LIST_HEAD);
	  ts_next(tsp).to_kind = TOK_END;
	  tsp->ts_used++;
	  break;
	case '|':
	  if (pstate != PS_LIST_HEAD)
	    {
	      print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
			   E_syntax, back);
	      abort_with_value (0);
	    }
	  if (!REDUCE_STACK (tsp))
	    abort_with_value (0);
	  add_ctxcount(ts_top(tsp).to_ctxcount, 1);
	  swap_ts (tsp);
	  pstate = PS_LIST_TAIL;
	  break;
	case ',':
	  /*
	   * This can only happen when we are in PS_LIST_HEAD or
	   * PS_FUNCTION. So we do not need to check for errors
	   */
	  if (!REDUCE_STACK (tsp))
	    abort_with_value (0);
	  add_ctxcount(ts_top(tsp).to_ctxcount, 1);
	  swap_ts (tsp);
	  break;
	case ')':		/* end of subterm/structure */
	  if (pstate == PS_FUNCTION)
	    {
	      if (!REDUCE_STACK (tsp))
		abort_with_value (0);
	      swappop_ts (tsp);

	      if (argssize < pushed_tokens (tsp))
		{
		  argssize = pushed_tokens (tsp);
		  if (arguments == NULL)
		    arguments = TMALLOC (argssize, TERM);
		  else
		    arguments = TREALLOC (arguments, argssize, TERM);
		}

	      for (i = 0; i < pushed_tokens (tsp); i++)
		{
		  TERMASSIGN (arguments[i],
			      ts_pickn (tsp, pushed_tokens (tsp)-i).to_term);
		}
#if CONVERT_DOT_TO_LIST
	      if ((pushed_tokens (tsp) == 2
		   && laststate[-1].to_name[0] == '.'
		   && laststate[-1].to_name[1] == 0))
		create_list (&laststate[-1].to_term, &arguments[0],
			     &arguments[1]);
	      else
		create_structure (&laststate[-1].to_term,
				  laststate[-1].to_name,
				  pushed_tokens(tsp),
				  arguments);
#else
	      create_structure (&laststate[-1].to_term,
				laststate[-1].to_name,
				pushed_tokens (tsp), arguments);
#endif
	      tsp->ts_used -= pushed_tokens (tsp);
	      set_ctxcount (laststate[-1].to_ctxcount, ctxcount);
	      set_context (laststate[-1].to_context, context);
	      restore_state ();
	      tsp->ts_used--;
	      ts_top(tsp).to_kind = TOK_TERM;
	    }
	  else if (pstate == PS_SUBTERM && endch == ')')
	    {
	      if (!REDUCE_STACK (tsp))
		abort_with_value (0);
	      set_ctxcount (ts_top(tsp).to_ctxcount, ctxcount);
	      set_context (ts_top(tsp).to_context, context);
	      restore_state ();
	      *assign_token = ts_top (tsp);
	      tsp->ts_used--;
	    }
	  else
	    {
	      print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
			   E_illbrace, back);
	      abort_with_value (0);
	    }
	  break;
	case '}':		/* end of conditional */
	  if (pstate != PS_SUBTERM || endch != '}')
	    {
	      print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
			   E_illbrace, back);
	      abort_with_value (0);
	    }
	  if (!REDUCE_STACK (tsp))
	    abort_with_value (0);
	  set_ctxcount (ts_top(tsp).to_ctxcount, ctxcount);
	  set_context (ts_top(tsp).to_context, context);
	  restore_state ();
#if CONDITIONAL_SPECIAL
	  create_conditional (&assign_token->to_term,
			      &ts_top(tsp).to_term);
#else
	  create_structure (&assign_token->to_term, "{}",
			    1, &ts_top(tsp).to_term);
#endif
	  assign_token->to_kind = TOK_TERM;
	  set_context (assign_token->to_context, ts_top(tsp).to_context);
	  set_ctxcount (assign_token->to_ctxcount,
			ts_top(tsp).to_ctxcount);
	  tsp->ts_used--;
	  break;
	case ']':
	  if (pstate != PS_LIST_HEAD && pstate != PS_LIST_TAIL)
	    {
	      print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
			   E_illbrace, back);
	      abort_with_value (0);
	    }
	  if (pushed_tokens (tsp) == 1)
	    {
	      create_atom (&ts_top(tsp).to_term, "[]");
	      ts_top(tsp).to_kind = TOK_TERM;
	    }
	  else if (pstate == PS_LIST_HEAD)
	    {
	      /*
	       * Create the NIL atom []. For PS_LIST_HEAD this
	       * is needed as an additional term to close the
	       * list (this atom overwrites the TOK_END token).
	       */
	      if (!REDUCE_STACK (tsp))
		abort_with_value (0);
	      swap_ts (tsp);
	      create_atom (&ts_top(tsp).to_term, "[]");
	      ts_top(tsp).to_kind = TOK_TERM;
	    }
	  else
	    {
	      /*
	       * Reduce the list tail. After that we can drop
	       * the TOK_END marker that's still on the stack.
	       */
	      if (!REDUCE_STACK (tsp))
		abort_with_value (0);
	      swappop_ts (tsp);
	    }
	  /*
	   * Now we loop until we have less that two tokens on the
	   * stack and reduce:	...,L1|L2] to ...|[L1|[L2]]]
	   */
	  while (pushed_tokens (tsp) > 1)
	    {
	      TERM lres;

	      create_list (&lres, &ts_pickn(tsp,2).to_term,
			   &ts_top(tsp).to_term);
	      tsp->ts_used--;
	      TERMASSIGN (ts_top(tsp).to_term, lres);
	    }
	  set_ctxcount (ts_top(tsp).to_ctxcount, ctxcount);
	  set_context (ts_top(tsp).to_context, context);
	  restore_state ();
	  *assign_token = ts_top(tsp);
	  tsp->ts_used--;
	  break;
	default:
	  print_error (tsp, 0, tsp->ts_used, tsp->ts_used-1,
		       E_illchar, back);
	  abort_with_value (0);
	  break;
	}
    }
}

/*----------------------------------------------------------------------*
 * prolog_read	- the main interface function. Reads a term and		*
 *		  returns a TERM value.					*
 * Return values:							*
 *	1 - term was correct						*
 *	0 - an error occured, message was written using 'prolog_putchar'*
 *----------------------------------------------------------------------*/

int
prolog_read (result)
    TERM *result;
{
#if TRACK_VARIABLES
  reset_vs ();
#endif
  if (!prolog_lex ())
    return 0;
  pread_bufscan = pread_buffer;
  if (*pread_bufscan == TOK_END)
    {
      create_eof (result);
      return 1;
    }
  return parse_term (result);
}
